<?php
/**
 * View for display photo page
 *
 * This file is part of Zoph.
 *
 * Zoph is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Zoph is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with Zoph; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @package Zoph
 * @author Jeroen Roos
 */

namespace person\view;

use conf\conf;
use template\block;
use template\template;
use web\request;

use person;
use selection;
use user;

use photoNoSelectionException;
use pageException;

/**
 * This view displays the "person" page
 */
class display extends common {

    /**
     * Get action links
     * @return array action links
     */
    protected function getActionlinks() {
        $actionlinks=parent::getActionlinks();
        unset($actionlinks["cancel"]);
        unset($actionlinks["return"]);
        return $actionlinks;
    }


    /**
     * Output view
     */
    public function view() {
        $user=user::getCurrent();
        $photosOf = $this->person->getPhotoCount();
        $photosBy = $this->person->getPhotographer()->getPhotoCount();


        $selection=null;
        $actionlinks=null;

        if ($user->canEditOrganizers()) {
            $actionlinks=array(
                translate("edit")   => "person.php?_action=edit&amp;person_id=" . $this->person->getId(),
                translate("delete") => "person.php?_action=delete&amp;person_id=" . $this->person->getId(),
                translate("new")    => "person.php?_action=new"
            );
            if ($this->person->get("coverphoto")) {
                $actionlinks[translate("unset coverphoto")]="person.php?_action=update&amp;person_id=" .
                    $this->person->getId() . "&amp;coverphoto=NULL";
            }

            try {
                $selection=new selection($_SESSION, array(
                    "coverphoto"    => "person.php?_action=update&amp;person_id=" . $this->person->getId() . "&amp;coverphoto=",
                    "return"        => "_return=person.php&amp;_qs=person_id=" . $this->person->getId()
                ));
            } catch (photoNoSelectionException $e) {
                $selection=null;
            }
        }

        try {
            $pageset=$this->person->getPageset();
            $page=$this->person->getPage($this->request, $this->request["pagenum"]);
            $showOrig=$this->person->showOrig($this->request["pagenum"]);
        } catch (pageException $e) {
            $showOrig=true;
            $page=null;
        }

        $mainActionlinks=array();

        if ($photosOf > 0) {
            $mainActionlinks[$photosOf . " " . translate("photos of")]  = "photos.php?person_id=" . $this->person->getId();
        }

        if ($photosBy > 0) {
            $mainActionlinks[$photosBy . " " . translate("photos by")]  = "photos.php?photographer_id=" . $this->person->getId();
        }

        $tpl=new template("display", array(
            "title"             => $this->getTitle(),
            "obj"               => $this->person,
            "actionlinks"       => $actionlinks,
            "mainActionlinks"   => $mainActionlinks,
            "selection"         => $selection,
            "page"              => $page,
            "pageTop"           => $this->person->showPageOnTop(),
            "pageBottom"        => $this->person->showPageOnBottom(),
            "showMain"          => $showOrig
        ));

        /**
         * @todo All the link blocks here could be generated by the objects themselves
         *       saving a huge amount of more-or-less duplicate code
         */
        if ($user->canSeePeopleDetails()) {
            $dl=$this->person->getDisplayArray();
            if ($this->person->getEmail()) {
                $mail=new block("link", array(
                    "href"      => "mailto:" . e($this->person->getEmail()),
                    "target"    =>  "",
                    "link"      => e($this->person->getEmail())
                ));
                $dl[translate("email")]=$mail;
            }

            if ($this->person->home) {
                $home=new block("link", array(
                    "href"      => "place.php?place_id=" . $this->person->get("home_id"),
                    "target"    => "",
                    "link"      => $this->person->home->get("title")
                ));
                $dl[translate("home location")]=$home;
            }

            if ($this->person->work) {
                $home=new block("link", array(
                    "href"      => "place.php?place_id=" . $this->person->get("work_id"),
                    "target"    => "",
                    "link"      => $this->person->work->get("title")
                ));
                $dl[translate("work location")]=$home;
            }
        }

        if ($this->person->get("notes")) {
            $dl[translate("notes")]=$this->person->get("notes");
        }

        $circles=$this->person->getCircles();
        if ($circles) {
            $circleLinks=array();
            foreach ($circles as $circle) {
                $circle->lookup();
                $circleLinks[]= new block("link", array(
                                    "href"      => $circle->getURL(),
                                    "target"    => "",
                                    "link"      => $circle->getName()
                                ));
            }
            $dl[translate("circles")]=implode($circleLinks, ", ");
        }
        $tpl->addBlock(new block("definitionlist", array(
            "dl"    => $dl,
            "class" => ""
        )));
        return $tpl;
    }

    /**
     * Get the title for this view
     */
    public function getTitle() {
        return $this->person->getName();
    }

}
