use lang;
use lang:bs;
use lang:bs:macro;
use core:io;
use core:asm;
use core:lang;
use core:debug;

class BsRepl extends Repl {

	// Shall we terminate?
	package Bool done;

	// Scope.
	package Scope scope;

	// Pre-processing.
	init() {
		init() {
			done = false;
			scope(named{ReplSandbox}, BSLookup());
		}

		scope.addInclude(named{core});
		scope.addInclude(named{lang:bs});
		scope.addInclude(named{lang:bs:macro});
	}

	void greet() {
		print("Type 'licenses' to show licenses for all currently used modules.");
	}

	Bool eval(Str line) {
		Parser<SSkipBlock> skipBlock;
		skipBlock.parse(line, Url());
		if (skipBlock.hasError) {
			// Mismatched something. Let the user complete their input.
			return false;
		}

		Parser<SReplUse> useStmt;
		useStmt.addSyntax(named{lang:bs});
		useStmt.parse(line, Url());
		if (!useStmt.hasError) {
			addUse(useStmt.tree.transform);
			return true;
		}


		Parser<SReplExpr> root;
		scope.addSyntax(root);
		root.parse(line, Url());
		if (root.hasError) {
			var error = root.error;
			print("At character " + error.pos.start.toS + ", syntax error: " + error.text);
			return true;
		}

		evalExpr(root.tree);
		true;
	}

	Bool exit() {
		done;
	}

private:
	void addUse(SrcName name) {
		Named? pkg = rootScope.find(name);
		if (pkg as Package) {
			if (scope.addInclude(pkg)) {
				print("Using " # name # ".");
			} else {
				print(name # " already in use.");
			}
		} else {
			print("Package " # name # " not found.");
		}
	}

	void evalExpr(SReplExpr tree) {
		Type sandboxType = named{ReplSandbox};
		ReplBlock block(scope);
		ThisVar thisPtr(thisPtr(sandboxType), SrcPos(), true);
		block.add(thisPtr);

		Expr expr = tree.transform(block);
		block.add(expr);

		if (var = expr as Var) {
			// We are declaring a variable.
			Str name = var.var.name;
			Value result = block.result.type.asRef(false);

			if (result.type) {
				ReplCode code(block, thisPtr, name);
				Function fn(result, "init", [thisPtr.result]);
				fn.parentLookup = sandboxType;
				fn.setCode(LazyCode(&code.loadGenerator));

				ReplSandbox sandbox(this);
				GlobalVar var(name, result, named{Compiler}, fn.pointer(sandbox));
				if (old = sandboxType.find(name, scope) as GlobalVar)
					sandboxType.remove(old);
				if (into = thisPtr.result.type)
					into.add(var);
				var.create();
				return;
			}

			// Otherwise, just try to evaluate the expression normally. Doing so will most likely
			// result in an error anyway, and that error is more relevant than anything we can say
			// at this point.
		}

		// Create a new function that we execute, so that we can run multiple REPL:s at the same time.
		Function evalFn(Value(named{Str}), "<interactive>", [named{ReplSandbox}]);
		evalFn.parentLookup = named{ReplSandbox};
		ReplCode code(block, thisPtr);
		evalFn.setCode(LazyCode(&code.load()));

		if (ptr = evalFn.pointer as Fn<Str, ReplSandbox>) {
			ReplSandbox sandbox(this);
			printPrompt("=> ", ptr.call(sandbox));
		}

		// Function replaceFn = named{ReplSandbox:evalFn<ReplSandbox>};
		// ReplCode code(block, thisPtr);
		// replaceFn.setCode(LazyCode(&code.load()));
		//
		// // Now, run the replaced function!
		// ReplSandbox sandbox(this);
		// print("=> " # sandbox.evalFn());
	}
}

private void outputLicense(core:info:License l, Bool full) on Compiler {
	print(l.identifier # ": " # l.title # " (" # l.author # ")");
	if (full) {
		Str line = "-" * 80;
		print(line);
		print(l.body);
		print(line);
		print("");
	}
}

/**
 * Sandbox class for the Repl to execute in.
 */
class ReplSandbox on Compiler {
	package BsRepl owner;

	init(BsRepl owner) {
		init() {
			owner = owner;
		}
	}

	// Show about the repl.
	void help() {
		print("Special commands in the REPL:");
		print("licenses       Print license information");
		print("version        Print version information");
		print("variables      Print all variables in this top loop");
		print("exit           Terminate this Storm instance");
		print("<var> = <expr> Create a variable in the top loop");
		print("               Regular variable definitions are also accepted");
		print("del <var>      Delete variable <var> from the top loop");
		print("use <package>  Import <package> into the current namespace");
		print("help <name>    Print documentation about <name>");
	}

	// Show all licenses in the system.
	void licenses() {
		print("For full license information, type 'fullLicenses'.");
		print("Licenses of modules currently in use by the system:");
		print("");
		for (l in core:info:licenses) {
			outputLicense(l, false);
		}
	}

	// Show the long version of the licenses.
	void fullLicenses() {
		print("Licenses of modules currently in use by the system:");
		print("");
		for (l in core:info:licenses) {
			outputLicense(l, true);
		}
	}

	// Show version information.
	void version() {
		print("Versions for modules currently in use by the system:");
		print("");
		for (v in core:info:versions) {
			print(v.toS);
		}
	}

	// Show all variables.
	void variables() {
		Type here = named{ReplSandbox};
		Array<GlobalVar> all;
		for (v in here)
			if (v as GlobalVar)
				all << v;

		if (all.empty) {
			print("No variables declared.");
		} else {
			all.sort(&compareMembers);

			print("Name       Value");
			print("-------------------");
			for (v in all) {
				StrBuf b;
				b << left << width(10) << v.name << " " << v.strValue;
				print(b.toS);
			}
		}
	}

	// This is the exit command.
	void exit() {
		owner.done = true;
	}

	// The REPL machinery will inject a function here, so that everything typed by the user will be
	// executed in this context.
}

class ReplBlock extends ExprBlock {
	init(Scope scope) {
		init(SrcPos(), scope) {}
	}
}

class ReplCode on Compiler {
	private ReplBlock root;
	private LocalVar thisPtr;
	private Value strType;
	private Str varName;

	init(ReplBlock root, LocalVar thisPtr) {
		init() {
			root = root;
			thisPtr = thisPtr;
			strType = named{Str};
		}
	}

	init(ReplBlock root, LocalVar thisPtr, Str varName) {
		init() {
			root = root;
			thisPtr = thisPtr;
			strType = named{Str};
			varName = varName;
		}
	}

	CodeGen load() {
		CodeGen g(RunOn(named{Compiler}), true, strType);
		VarInfo resultVar = g.createVar(strType);

		g.l << prolog();

		thisPtr.createParam(g);

		Value result = root.result.type;
		createStr(g, result, root, resultVar);

		g.l << fnRet(resultVar.v);

		g;
	}

	// Generate code that makes sure to return the value created by 'root' after printing it.
	CodeGen loadGenerator() {
		Value result = root.result.type.asRef(false);

		CodeGen g(RunOn(named{Compiler}), true, result);
		VarInfo resultVar = g.createVar(result);
		VarInfo resultStr = g.createVar(strType);

		g.l << prolog();

		thisPtr.createParam(g);

		// Compute the result.
		root.code(g, CodeResult(result, resultVar));

		// Generate a string representation.
		createStr(g, result, BareVarAccess(SrcPos(), result, resultVar.v), resultStr);

		// Print the result.
		Actuals printParams;
		printParams.add(strConstant(SrcPos(), varName + " = "));
		printParams.add(BareVarAccess(SrcPos(), strType, resultStr.v));
		FnCall printCall(SrcPos(), root.scope, named{printPrompt<Str, Str>}, printParams);
		printCall.code(g, CodeResult());

		g.l << fnRet(resultVar.v);

		g;
	}

private:
	// Compute a string representation of an object.
	void createStr(CodeGen g, Value resultType, Expr expr, VarInfo resultVar) {
		Name name("toS", [resultType]);
		Named? toSFn = root.scope.find(name);
		if (toSFn as Function) {
			// It exists, use the result.
			FnCall toSCall(SrcPos(), root.scope, toSFn, Actuals(expr));
			toSCall.code(g, CodeResult(strType, resultVar));
		} else {
			// Just evaluate and return a dummy output.
			expr.code(g, CodeResult());
			createDefaultResult(g, resultType, resultVar);
		}
	}

	// Create an empty string for output.
	void createDefaultResult(CodeGen g, Value resultType, VarInfo result) {
		StrBuf s;
		s.add("<").add(resultType.toS).add(">");
		Expr constant = strConstant(SrcPos(), s.toS);
		constant.code(g, CodeResult(strType, result));
	}
}

// Helper to print results nicely.
private void printPrompt(Str prompt, Str contents) on Compiler {
	StrBuf out;
	out << prompt;
	out.indentBy(" " * prompt.count);
	out.indent();
	out << contents;
	print(out.toS);
}

// Implementation of the 'del' statement in the Repl.
Expr delStmt(Block block, SStr name) on Compiler {
	SrcName fnName(name.pos);
	fnName.add("lang");
	fnName.add("bs");
	fnName.add("replimpl");
	fnName.add("replDel");
	namedExpr(block, fnName, Actuals(strConstant(name)));
}

// Remove variables from the repl class.
void replDel(Str name) on Compiler {
	Type inside = named{ReplSandbox};
	if (var = inside.find(name, rootScope) as GlobalVar) {
		inside.remove(var);

		print("Deleted " + name + " (was " + var.strValue + ")");
	} else {
		print("No variable " + name + ".");
	}
}

// Implementation of the 'help' statement in the Repl.
Expr helpStmt(Block block, SrcName name) on Compiler {
	Expr thisVar = namedExpr(block, name.pos, "this", Actuals());
    Expr type = macro:createType(name.pos, block, name);
	SrcName fnName(name.pos);
	fnName.add("lang");
	fnName.add("bs");
	fnName.add("replimpl");
	fnName.add("replHelp");

	Actuals params;
	params.add(thisVar);
	params.add(type);
	namedExpr(block, fnName, params);
}

// Execute the help statement in the repl.
void replHelp(ReplSandbox me, SrcName name) on Compiler {
	Scope root = me.owner.scope;

	if (entity = root.find(name)) {
		printHelp(entity);
	} else if (entity = root.find(name.parent) as NameSet) {
		printHelp(entity, name.last.name);
	} else {
		print("I do not know anything called " + name.toS);
	}
}


void printHelp(Named entity) on Compiler {
	print(entity.findDoc.toS);
	print("\nAt: " # entity.pos);

	if (entity as NameSet) {
		entity.forceLoad();
		print("\nMembers:");

		Array<Named> members;
		for (i in entity)
			members << i;

		members.sort(&compareMembers);
		for (m in members)
			print(" " + doc(m).summary());
	}
}

void printHelp(NameSet in, Str last) on Compiler {
	Array<Named> found = in.findName(last);

	if (found.empty) {
		print("No members named '" # last # "' in " # in);
		return;
	}

	print("Found " # found.count # " overloads for " # last # ":");

	found.sort(&compareMembers);
	for (i in found) {
		print("");
		printHelp(i);
	}
}

Bool compareMembers(Named a, Named b) on Compiler {
	if (a.name == b.name)
		return a.params.count < b.params.count;

	return a.name < b.name;
}
