/*
The stellarium telescope library helps building
telescope server programs, that can communicate with stellarium
by means of the stellarium TCP telescope protocol.
It also contains sample server classes (dummy, Meade LX200).

Author and Copyright of this file and of the stellarium telescope library:
Johannes Gajdosik, 2006

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335, USA.
*/

#ifndef SERIALPORT_HPP
#define SERIALPORT_HPP

#include "Connection.hpp"

#ifdef Q_OS_WIN
  #include <Windows.h>
#else
  #include <termios.h>
#endif

//! Serial interface connection.
class SerialPort : public Connection
{
public:
	//! Class constructor.
	//! @param serial_device A string containing the name of a serial port.
	//! On UNIX systems, this should be something like "/dev/ttyS0".
	//! On Microsoft Windows systems, this should be something like "COM1:".
	SerialPort(Server &server, const char *serial_device);
	~SerialPort(void) override;
	//! Returns true if the connection is closed.
	//! This method has different platform-dependent implementations.
	virtual bool isClosed(void) const override
	{
	#ifdef Q_OS_WIN
		return (handle == INVALID_HANDLE_VALUE);
	#else
		return IS_INVALID_SOCKET(fd);
	#endif
	}
	
protected:
	void prepareSelectFds(fd_set&, fd_set&, int&) override;
	
private:
	//! Returns false, as SerialPort implements a serial port connection.
	virtual bool isTcpConnection(void) const override {return false;}
	//! Returns true, as SerialPort implements a serial port connection.
	virtual bool isAsciiConnection(void) const override {return true;}
	
private:
#ifdef Q_OS_WIN
	virtual int readNonblocking(char *buf, int count) override;
	virtual int writeNonblocking(const char *buf, int count) override;
	virtual void handleSelectFds(const fd_set&, const fd_set&) override {}
	HANDLE handle;
	DCB dcb_original;
#else
	struct termios termios_original;
#endif
};

#endif
