/*
 *  Copyright (c) 2008,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstring>

// GTLCore Headers
#include <GTLCore/AbstractImage.h>
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Image.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Region.h>
#include <GTLCore/StdTypes.h>
#include <GTLCore/Type.h>

// GTLFragment Headers
#include <GTLFragment/LibrariesManager.h>

// OpenRijn Headers
#include <OpenRijn/Sketch.h>
#include <OpenRijn/Version.h>
#include <OpenRijn/ImageEngine/Canvas.h>
#include <OpenRijn/ImageEngine/Palette.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>

void printVersion()
{
  std::cout << OpenRijn::LibraryShortName() << " - " << OpenRijn::LibraryName() << " - " << OpenRijn::LibraryVersionString() << std::endl;
  std::cout << OpenRijn::LibraryCopyright() << std::endl;
  std::cout << "Rijn Version : " << OpenRijn::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : rijntester [option] fileName.rijn" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -r --run-test           run the function 'run-test'" << std::endl;
  std::cout << "  -c --compare            compare the result of the program with a file stored on disk" << std::endl;
//   std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

enum RijnTesterMode {
  RTM_DEFAULT,
  RTM_RUN_TEST,
  RTM_COMPARE_RESULT
};

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  std::list< GTLCore::String > fileNames;
  RijnTesterMode mode = RTM_DEFAULT;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-c","--compare") ) {
      mode = RTM_COMPARE_RESULT;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLFragment::LibrariesManager::instance()->addDirectory(argv[ai]);
      }
    } else {
      fileNames.push_back( argv[ai] );
    }
  }
  if( fileNames.size() > 0 )
  {
    fileName = *( fileNames.begin() );
    fileNames.pop_front();
  }
  if( fileName == "" or (fileNames.size() < 1 and mode == RTM_COMPARE_RESULT) )
  {
    std::cerr << "Invalid command line parameters." << std::endl;
    printHelp();
  } else {
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    std::string str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    
    GTLCore::String errMsg;
    GTLCore::RegionI region;
    GTLCore::AbstractImage* resultImage = 0;
    if(mode == RTM_COMPARE_RESULT)
    {
      GTLCore::String resultFileName = *( --fileNames.end() );
      fileNames.pop_back();
      const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( resultFileName );
      if( not decoder )
      {
        std::cerr << "Can't find decoder for " << resultFileName << std::endl;
        return EXIT_FAILURE;
      }
      resultImage = decoder->decode( resultFileName, &region, &errMsg );
      if(not resultImage)
      {
        std::cerr << "Can't open result file: " << resultFileName << std::endl;
        return EXIT_FAILURE;
      }
    }
    OpenRijn::Sketch sketch;
    sketch.setSource( source );
    sketch.compile();
    if(not sketch.isCompiled())
    {
      std::cout << "Error: " << std::endl << sketch.compilationMessages().toString() << std::endl;
      return EXIT_FAILURE;
    }
    switch( mode )
    {
      case RTM_COMPARE_RESULT:
      {
        GTL_ASSERT(resultImage);
        int width = region.columns();
        int height = region.rows();
        std::cout << "Start evaluation..." << std::endl;
        GTLCore::Image image(width, height, GTLCore::PixelDescription(GTLCore::Type::Float32, 4, 3));
        std::memset(image.rawData(0,0),0, width * height * sizeof(float));
        OpenRijn::ImageEngine::Canvas c(&image);
        OpenRijn::ImageEngine::Palette p;
        sketch.draw(&c, &p);
        std::cout << "Start comparison..." << std::endl;
        
        GTLCore::Image image2(width, height, GTLCore::PixelDescription(GTLCore::Type::UnsignedInteger16, 4, 3));
        for(int x = 0; x < width; ++x)
        {
          for(int y = 0; y < height; ++y)
          {
            for(int k = 0; k < 4; ++k)
            {
              image2.data<gtl_uint16>(x, y)[k] = 65535 * image.data<float>(x, y)[k];
            }
          }
        }
        int errorCount = image2.compare( resultImage, region );
        if( errorCount != 0 )
        {
          std::cout << errorCount << " different pixels" << std::endl;
          return EXIT_FAILURE;
        }
        break;
      }
      case RTM_RUN_TEST:
      case RTM_DEFAULT:
        if( sketch.hasTestFunction() )
        {
          return sketch.runTest();
        } else {
          std::cout << "No test function" << std::endl;
          return EXIT_FAILURE;
        }
    }
  }
  return EXIT_SUCCESS;
}
