/* NVTV main -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: nvtv.c,v 1.7 2004/03/01 21:08:10 dthierbach Exp $
 *
 * Contents:
 *
 * Main routine. Parse options, execute actions or call gui.
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <getopt.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_X
#include <X11/Xlib.h>
#endif

#ifdef HAVE_GTK
#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>

#include "gui.h"
#endif

#include "backend.h"
#include "back_null.h"
#ifdef USE_UNIX_BACKEND
#include "back_unix.h"
#endif
#ifdef USE_MSWIN_BACKEND
#include "back_mswin.h"
#endif
#ifdef USE_CLIENT_BACKEND
#include "back_client.h"
#endif

#include "nvtv.h"
#include "print.h"
#include "data_vesa.h"

#ifdef HAVE_X
#include "actions.h"
#endif

#ifdef DEBUG_PROBE
#include "back_direct.h"
#endif

/* Debug: */

#include "tv_nv.h"

/* -------- Global options -------- */

Bool opt_debug = FALSE;

Bool opt_no_root = FALSE;

Bool opt_nvdev = FALSE;

int opt_res_x = -1, opt_res_y = -1;
float opt_hoc = -1.0, opt_voc = -1.0;

int opt_head = -1;

char *opt_size = NULL;
char *opt_dpy = NULL;
char *opt_window_name = NULL;

BackAccessPtr back_access = NULL;  /* global backend */
BackCardPtr back_card = NULL;

int opt_tv_bus = -1, opt_tv_addr = -1;

TVChip opt_tv_chip = TV_NO_CHIP;
TVSystem opt_system = TV_SYSTEM_NONE;
TVConnect opt_connect = CONNECT_AUTO;

#if HAVE_X
Window opt_window = None;
#endif

int opt_service_flags = 0;
int opt_service_mask = 0;

int opt_mode_flags = 0;
int opt_mode_mask = 0;

int opt_pci_bus  = -1;
int opt_pci_slot = -1;
int opt_pci_func = -1;

TVSettings opt_set;

static const char *short_options = "?12bcdf:ghlmno:pqr:s:tw:A:C:FNPS:T:W:X";

static struct option long_options[] =
  {{"tv-bars",        no_argument,       NULL, 'b'},
   {"center",         no_argument,       NULL, 'c'},
   {"debug",          no_argument,       NULL, 'd'},
   {"set-flicker",    required_argument, NULL, 'f'},
   {"gui",            no_argument,       NULL, 'g'},
   {"help",           no_argument,       NULL, 'h'},
   {"list",           no_argument,       NULL, 'l'},
   {"tv-off",         no_argument,       NULL, 'm'},
   {"no-root",        no_argument,       NULL, 'n'},
   {"overscan",       required_argument, NULL, 'o'},
   {"print",          no_argument,       NULL, 'p'},
   {"query",          no_argument,       NULL, 'q'},
   {"resolution",     required_argument, NULL, 'r'},
   {"size",           required_argument, NULL, 's'},
   {"tv-on",          no_argument,       NULL, 't'},
   {"win-name",       required_argument, NULL, 'w'},
   {"card-addr",      required_argument, NULL, 'A'},
   {"connector",      required_argument, NULL, 'C'},
   {"display",        required_argument, NULL, 'D'},
   {"fetch",          no_argument,       NULL, 'F'},
   {"nvdev",          no_argument,       NULL, 'N'},
   {"probe",          no_argument,       NULL, 'P'},
   {"system",         required_argument, NULL, 'S'},
   {"chip",           required_argument, NULL, 'T'},
   {"win-id",         required_argument, NULL, 'W'},
   {"switch-mode",    no_argument,       NULL, 'X'},
   {"set",            required_argument, NULL, 300},
   {"sync",           no_argument,       NULL, '.'},
   {"no-xshm",        no_argument,       NULL, '.'},
   {"name",           required_argument, NULL, '.'},
   {"class",          required_argument, NULL, '.'},
   {NULL,             0,                 NULL, 0}
};

OptIntVal *opt_set_act = NULL;

#define SET_FIELD(f) addr:&opt_set.f

static OptIntDecl opt_set_list [] = {
  {"Contrast",        min:-100, max:100, SET_FIELD(contrast)},
  {"Saturation",      min:-100, max:100, SET_FIELD(saturation)},
  {"Brightness#",     min: -50, max: 50, SET_FIELD(brightness_sig)}, 
  {"Contrast#",       min: -50, max: 50, SET_FIELD(contrast_sig)},
  {"Saturation#",     min: -50, max: 50, SET_FIELD(saturation_sig)},
  {"Phase",           min: -60, max: 60, SET_FIELD(phase)},
  {"Hue",             min: -60, max: 60, SET_FIELD(hue)},
  {"Flicker",         min:   0, max:100, SET_FIELD(flicker)},
  {"AdaptFlicker",    min:   0, max:100, SET_FIELD(flicker_adapt)},
  {"LumaBandwidth",   min:   0, max:100, SET_FIELD(luma_bandwidth)},
  {"ChromaBandwidth", min:   0, max:100, SET_FIELD(chroma_bandwidth)},
  {"Sharpness",       min:   0, max:100, SET_FIELD(sharpness)},
  {"CrossColor",      min:   0, max:100, SET_FIELD(cross_color)},
  {NULL}
};

#define SET_FLAG(x) addr_val:&opt_##x##_flags, addr_mask:&opt_##x##_mask

static OptFlagDecl opt_flag_list [] = {
  {"Dualview",        mask: TV_DESC_DUALVIEW,      SET_FLAG(mode)},
  {"Monochrome",      mask: TV_DESC_MONOCHROME,    SET_FLAG(mode)},
  {"Non-Interlaced",  mask: TV_DESC_NONINTERLACED, SET_FLAG(mode)},
  {"Macrovision",     mask: TV_DESC_MACROVISION,   SET_FLAG(mode)},
  {"FreeCarrier",     mask: TV_DESC_CARRIER_LOCK,  SET_FLAG(mode)},
  {"Colorfix",        mask: TV_DESC_COLORFIX,      SET_FLAG(mode)},
  {"AdjustCursor",    mask: BACK_SERVICE_CURSOR,      SET_FLAG(service)},
  {"ViewportCursor",  mask: BACK_SERVICE_VIEW_CURSOR, SET_FLAG(service)},
  {"ViewportMonitor", mask: BACK_SERVICE_VIEW_MAIN,   SET_FLAG(service)},
  {NULL}
};

/* -------- Actions -------- */

typedef enum {
  ACTION_NONE    = 0,
  ACTION_PRINT   = 1,
  ACTION_GUI     = 2,
  ACTION_TVON    = 3,
  ACTION_TVOFF   = 4,
  ACTION_TVBARS  = 5,
  ACTION_DEBUG   = 6,
  ACTION_PROBE   = 7,
  ACTION_QUERY   = 8,
  ACTION_LIST    = 9,
} ActionMain;

typedef enum {
  ACTION_CENTER  = (1 << 0),
  ACTION_XMODE   = (1 << 1),
  ACTION_FETCH   = (1 << 2),
} ActionAux;

/* -------- Error handling -------- */

void raise_msg (int class, char *format, ...)
{
  va_list args;

  if (class >= MSG_DEBUG && !opt_debug) return;
  switch (class) {
    case MSG_ABORT:
      fprintf (stderr, "Fatal: ");
      break;
    case MSG_WARNING:
      fprintf (stderr, "Warning: ");
      break;
  }
  va_start(args, format);
  vfprintf (stderr, format, args);
  va_end(args);
  if (class != MSG_DEBUG_NL) fprintf (stderr, "\n");
  if (class == MSG_ABORT) exit (1);
}

/* -------- Usage -------- */

void usage (void)
{
  OptIntDecl *p;
  OptFlagDecl *r;
  int i;

  fprintf (stderr,
      "NVTV version " PACKAGE_VERSION "\n\n");
  fprintf (stderr,
      "Usage: nvtv [-options ...]\n\n");
  fprintf (stderr,
      "where options include:\n");
  fprintf (stderr,
      "  -h --help             print this message\n");
  fprintf (stderr,	       
      "     --display hst:scrn X server to contact\n");
  fprintf (stderr,	       
      "  -m --tv-off           TV off, switch to monitor\n");
  fprintf (stderr,	       
      "  -t --tv-on            TV on\n");
  fprintf (stderr,	       
      "  -b --tv-bars          Color bars\n");
  fprintf (stderr,	       
      "  -p --print            calculate and print register values\n");
  fprintf (stderr,	       
      "  -F --fetch            fetch current mode (use only for printing)\n");
  fprintf (stderr,	       
      "  -l --list             list available modes on stdout\n");
  fprintf (stderr,	       
      "  -r --resolution x,y   Resolution\n");
  fprintf (stderr,	       
      "  -o --overscan x,y     Overscan percentage (as float)\n");
  fprintf (stderr,	       
      "  -s --size <size>      Size (for predefined modes)\n");
  fprintf (stderr,	       
      "  -S --system           NTSC,NTSC-J,PAL,PAL-60,PAL-NC,PAL-M,PAL-M60,PAL-X,SECAM\n");
  fprintf (stderr,	       
      "  -C --connector        AUTO,COMPOSITE,SVIDEO,BOTH,CONVERT\n");
  fprintf (stderr,
      "  -T --chip             BROOKTREE,CONEXANT,NVIDIA,CHRONTEL1,CHRONTEL2,\n");
  fprintf (stderr,
      "                        PHILIPS1,PHILIPS2,NVIDIA or <bus>:<addr>\n");
  fprintf (stderr,
      "  -A --card-addr <addr> video card pci address as <bus>:<slot>.<func>\n");
  fprintf (stderr,
      "  -W --win-id id        select window by numerical id\n");
  fprintf (stderr,
      "  -w --win-name name    select window by name\n");
  fprintf (stderr,
      "  -c --center           center selected window\n");
  fprintf (stderr,
      "  -X --switch-mode      switch X mode\n");
  fprintf (stderr,
      "  -1                    use first head\n");
  fprintf (stderr,
      "  -2                    use second head (if available)\n");
  fprintf (stderr,
      "  -q --query            query status of display (monitor, TV, flatpanel)\n");
  fprintf (stderr,
      "  -P --probe            probe and print system information\n");
  fprintf (stderr,
      "  -N --nvdev            enable usage of /dev/nv* devices\n");
  fprintf (stderr,
      "  -g --gui              always use gui\n");
  fprintf (stderr,
      "  -n --no-root          (only for debugging)\n");
  fprintf (stderr,
      "  -d --debug            (only for debugging)\n");
  fprintf (stderr,             
      "     --set name:val     Specify setting\n");
  fprintf (stderr, "\nThe following settings are available:\n");
  for (i = 0, p = opt_set_list; p->name; i++, p++) 
  {
    fprintf (stderr, "  %-20s (%4i to %4i)", p->name, p->min, p->max);
    if (i % 2) {
      fprintf (stderr, "\n");
    } else {
      fprintf (stderr, "  ");
    }
  }
  if (i % 2) fprintf (stderr, "\n");
  for (i = 0, r = opt_flag_list; r->name; i++, r++) 
  {
    fprintf (stderr, "  %-20s (flag) %4s", r->name, "");
    if (i % 2) {
      fprintf (stderr, "\n");
    } else {
      fprintf (stderr, "  ");
    }
  }
  if (i % 2) fprintf (stderr, "\n");
  exit (1);
}

/* -------- -------- */

ChipPtr findDefOptChip (CardPtr card)
{
  ChipPtr c, ca;

  ca = card->chips; 
  if (opt_tv_chip != TV_NO_CHIP) {
    for (c = card->chips; c; c = c->next)
    {
      if (c->type == opt_tv_chip) {
	ca = c; break;
      }
    }
  }
  return ca;
}

void printRegs (TVMode *mode, CardType card, TVChip chip)
{
  printf ("*** Resolution %03i x %03i  Overscan %06.3f x %06.3f\n",
	  mode->spec.res_x, mode->spec.res_y, mode->spec.hoc, mode->spec.voc);
  print_tv_regs (&mode->regs.enc, chip);
  print_crt_regs (&mode->regs.crtc, card);
}

int doQuery (int head)
{
  int dev;

  if (head == -1) head = 1;
  back_card->getHeadDev (head, &dev);
  printf ("%i", dev);
  return dev + 100; /* exit code */
}

int doList (CardPtr card)
{
  ChipPtr chip;
  DataFunc *func;
  TVMode *m;
  static char* sys[] = {
    "NTSC", "NTSC-J", "PAL", "PAL-60", "PAL-N", "PAL-Nc", "PAL-M", 
    "PAL-M60", "PAL-X", "SECAM"
  };

  /* TODO FIXME: Use card->listModes */
  /* Hm. Must set chip for that, which does init ... */

#if 1

  chip = findDefOptChip (card);
  if (!chip) RAISE (MSG_ABORT, "No encoder chip found, and none specified.\n");
  func = data_func (card->type, chip->type); 
  m = func->modes ();
  printf ("Modes for %s on %s\n", chip->name, card->name);
  if (m) {
    while (m->spec.system != TV_SYSTEM_NONE) {
      if (opt_system == TV_SYSTEM_NONE || m->spec.system == opt_system) {
	printf ("%s\t%4i,%4i\t%s\t%05.2f,%05.2f\t%s\n", sys[m->spec.system],
		m->spec.res_x, m->spec.res_y, m->spec.size,
		m->spec.hoc, m->spec.voc, m->spec.aspect);
      }
      m++;
    }
  }
  
#else

  back_card->setChip(, FALSE);
  /* FIXME change listModes to use card->type, chip->type ? */
  n = back_card->listModes (opt_system, &modes);
  printf ("Modes for %s on %s\n", chip->name, card->name);
  for (m = modes, i = 1; i <= n; i++, m++) {
	printf ("%s\t%4i,%4i\t%s\t%05.2f,%05.2f\t%s\n", sys[m->spec.system],
		m->spec.res_x, m->spec.res_y, m->spec.size,
		m->spec.hoc, m->spec.voc, m->spec.aspect);
  }
  xfree (modes);

#endif
  
  return 0; /* exit code */
}

void doDebug (void)
{
  CardPtr card;

#ifdef USE_UNIX_BACKEND
#if 1
  if (! back_root_avail ()) {
    printf ("No root backend\n");
    exit (1);
  }
  card = back_root_init ();
#else
  if (! back_client_avail ()) {
    printf ("No client backend\n");
    exit (1);
  }
  card = back_client_init ();
#endif
#endif /* USE_UNIX_BACKEND */
#ifdef USE_MSWIN_BACKEND
  if (! back_win_avail ()) {
    printf ("No nvidia driver backend\n");
    exit (1);
  }
  card = back_win_init ();
#endif /* USE_MSWIN_BACKEND */
  printf ("Backend is open, card=%s\n", card->name);
  back_access->openCard (card);
  printf ("Card is open\n");
#if 0
  {
    int i, n;
    TVMode *modes, *m;

    n = back_card->listModes (TV_SYSTEM_NTSC, &modes);
    for (m = modes, i = 1; i <= n; i++, m++) {
      printf ("%ix%i_%s ", m->spec.res_x, m->spec.res_y, m->spec.size);
    }
    printf ("\n");
    xfree (modes);
  }
#endif
  {
    TVMode mode;

    if (back_card->findByOverscan (TV_SYSTEM_NTSC, 800, 600, 8.0, 8.0, 
				   &mode))
    {
      printf ("found %03i x %03i (%06.3f x %06.3f) %s\n",
	mode.spec.res_x, mode.spec.res_y, mode.spec.hoc, mode.spec.voc,
	mode.spec.size);
    }
  }
  back_access->closeCard ();
}

void modifySet (TVSettings *s, OptIntVal *m)
{
  int i;

  while (m) {
    i = (char *) m->addr - (char *) s;
    if (i < 0 || i > sizeof(TVSettings)) {
      RAISE (MSG_ABORT, "Wrong settings to modify");
    }
    *(m->addr) = m->val;
    m = m->link;
  }
}

Bool processOptSet (char *opt, OptIntVal **head, OptIntVal **tail)
{
  char *s;
  char *name;
  int val;
  OptIntDecl *p;
  OptIntVal *q;
  OptFlagDecl *r;

  if (!opt) return FALSE;
  name = opt;
  s = strchr(opt, ':');
  if (!s) {
    RAISE (MSG_ERROR, "Illegal setting: %s", opt);
    return FALSE;
  }
  *s++ = '\0';
  RAISE (MSG_DEBUG_NL, "process settings %s", name);
  for (p = opt_set_list; p->name; p++) {
    if (strcasecmp (name, p->name) == 0) break;
  }
  if (p->name) {
    if (sscanf (s, "%i", &val) != 1) {
      RAISE (MSG_ERROR, "Illegal setting: %s", opt);
      return FALSE;
    }
    RAISE (MSG_DEBUG, " = %i", val);
    if (val < p->min || val > p->max) {
      RAISE (MSG_ERROR, "Setting %s out of range (%i to %i)", 
	       p->name, p->min, p->max);
      return FALSE;
    }
    q = (OptIntVal *) xalloc (sizeof (OptIntVal));
    q->addr = p->addr;
    q->link = NULL;
    q->val = val;
    if (*tail) (*tail)->link = q;
    *tail = q;
    if (!(*head)) *head = q;
    return TRUE;
  }
  for (r = opt_flag_list; r->name; r++) {
    if (strcasecmp (name, r->name) == 0) break;
  }
  if (r->name) {
    if (strcasecmp (s, "false") == 0) val = 0; 
    else if (strcasecmp (s, "off") == 0) val = 0; 
    else if (strcasecmp (s, "0") == 0) val = 0; 
    else if (strcasecmp (s, "true") == 0) val = 1; 
    else if (strcasecmp (s, "on") == 0) val = 1; 
    else if (strcasecmp (s, "1") == 0) val = 1; 
    else {
      RAISE (MSG_DEBUG, "");
      RAISE (MSG_ERROR, "Illegal flag value: %s", s);
      return FALSE;
    }
    RAISE (MSG_DEBUG, " = %s", val ? "true" : "false");
    *(r->addr_mask) |= r->mask;
    if (val) {
      *(r->addr_val) |= r->mask;
    } else {
      *(r->addr_val) &= ~r->mask;
    }
    return TRUE;
  }
  RAISE (MSG_ERROR, "Unknown setting %s", name);
  return FALSE;
}

/* 
 * Find card specified by opt_pci_* 
 */

CardPtr findCard (CardPtr list)
{
  CardPtr card;

  for (card = list; card; card = card->next) {
    if ((opt_pci_bus  == -1 || opt_pci_bus  == card->addr_bus) &&
        (opt_pci_slot == -1 || opt_pci_slot == card->addr_slot) &&
        (opt_pci_func == -1 || opt_pci_func == card->addr_func))
    {
      return card;
    }
  }
  return card;
}

#ifdef TEST_HOOKS
#define main xmain
#endif

int main (int argc, char *argv[])
{
  CardPtr main_card_list, main_card = NULL;
  int c = '?';
  ActionMain act_main = ACTION_NONE;
  ActionMain act_next;
  ActionAux act_aux = 0;
#if HAVE_X
  Display *main_dpy = NULL;
  int main_screen = 0;
#endif
  TVMode main_mode;
  TVRegs main_regs;
  int main_head;
  OptIntVal *opt_set_tail = NULL;

  opterr = 0;
  while ((c = getopt_long (argc, argv, short_options, 
                long_options, NULL)) != EOF) 
  {
    act_next = ACTION_NONE;
    switch(c) 
    {
      case 'h': /* Print usage */
      case '?':
        usage(); 
	break;
      case 'D': 
	opt_dpy = optarg; 
	break;
      case '.': /* Gtk/Gdk option, ignore */
        break;
      case 'n':
	opt_no_root = TRUE; 
	break;
      case 'r':
	if (!optarg || sscanf (optarg, "%i,%i", &opt_res_x, &opt_res_y) != 2) 
	  usage ();
	break;
      case 'o':
	if (!optarg || sscanf (optarg, "%f,%f", &opt_hoc, &opt_voc) != 2)
	  usage ();
	break;
      case 'd':
	opt_debug = TRUE;
	break;
      case 'm':
	act_next = ACTION_TVOFF;
	break;
      case 't':
	act_next = ACTION_TVON;
	break;
      case 'b':
	act_next = ACTION_TVBARS;
	break;
      case 'p':
	act_next = ACTION_PRINT;
	break;
      case 'P':
	act_next = ACTION_PROBE;
	break;
      case 'g':
	act_next = ACTION_GUI;
	break;
      case 'q':
	act_next = ACTION_QUERY;
	break;
      case 'l':
        act_next = ACTION_LIST;
	break;
      case 's':
	if (optarg) opt_size = optarg;
	break;
      case 'S':
	if (optarg && strcasecmp (optarg, "ntsc") == 0) 
	  opt_system = TV_SYSTEM_NTSC; 
	else if (optarg && strcasecmp (optarg, "ntsc-j") == 0) 
	  opt_system = TV_SYSTEM_NTSC_J; 
	else if (optarg && strcasecmp (optarg, "pal") == 0) 
	  opt_system = TV_SYSTEM_PAL; 
	else if (optarg && strcasecmp (optarg, "pal-60") == 0) 
	  opt_system = TV_SYSTEM_PAL_60; 
	else if (optarg && strcasecmp (optarg, "pal-n") == 0) 
	  opt_system = TV_SYSTEM_PAL_N; 
	else if (optarg && strcasecmp (optarg, "pal-nc") == 0) 
	  opt_system = TV_SYSTEM_PAL_NC; 
	else if (optarg && strcasecmp (optarg, "pal-m") == 0) 
	  opt_system = TV_SYSTEM_PAL_M; 
	else if (optarg && strcasecmp (optarg, "pal-m60") == 0) 
	  opt_system = TV_SYSTEM_PAL_M60; 
	else if (optarg && strcasecmp (optarg, "pal-x") == 0) 
	  opt_system = TV_SYSTEM_PAL_X; 
	else if (optarg && strcasecmp (optarg, "secam") == 0) 
	  opt_system = TV_SYSTEM_SECAM; 
	else
	  usage ();
	break;
      case 'C':
	if (optarg && strcasecmp (optarg, "auto") == 0) 
	  opt_connect = CONNECT_AUTO; 
	else if (optarg && strcasecmp (optarg, "fbas") == 0) 
	  opt_connect = CONNECT_COMPOSITE; 
	else if (optarg && strcasecmp (optarg, "composite") == 0) 
	  opt_connect = CONNECT_COMPOSITE; 
	else if (optarg && strcasecmp (optarg, "svhs") == 0) 
	  opt_connect = CONNECT_SVIDEO; 
	else if (optarg && strcasecmp (optarg, "svideo") == 0) 
	  opt_connect = CONNECT_SVIDEO; 
	else if (optarg && strcasecmp (optarg, "both") == 0) 
	  opt_connect = CONNECT_BOTH; 
	else if (optarg && strcasecmp (optarg, "convert") == 0) 
	  opt_connect = CONNECT_CONVERT; 
	else
	  usage ();
	break;
      case 'T':
	if (optarg && strcasecmp (optarg, "brooktree") == 0) 
	  opt_tv_chip = TV_BROOKTREE; 
	else if (optarg && strcasecmp (optarg, "conexant") == 0) 
	  opt_tv_chip = TV_CONEXANT; 
	else if (optarg && strcasecmp (optarg, "chrontel1") == 0) 
	  opt_tv_chip = TV_CHRONTEL_MODEL1; 
	else if (optarg && strcasecmp (optarg, "chrontel2") == 0) 
	  opt_tv_chip = TV_CHRONTEL_MODEL2; 
	else if (optarg && strcasecmp (optarg, "chrontel") == 0) 
	  opt_tv_chip = TV_CHRONTEL_MODEL1; 
	else if (optarg && strcasecmp (optarg, "philips1") == 0) 
	  opt_tv_chip = TV_PHILIPS_MODEL1; 
	else if (optarg && strcasecmp (optarg, "philips2") == 0) 
	  opt_tv_chip = TV_PHILIPS_MODEL2; 
	else if (optarg && strcasecmp (optarg, "philips") == 0) 
	  opt_tv_chip = TV_PHILIPS_MODEL1; 
	else if (optarg && strcasecmp (optarg, "nvidia") == 0) 
	  opt_tv_chip = TV_NVIDIA; 
	else if (optarg && 
		 sscanf (optarg, "%i:%x", &opt_tv_bus, &opt_tv_addr) == 2)
	  opt_tv_chip = TV_CHIP_BY_ADDR;
	else
	  usage ();
	break;
      case 'A':
	if (!optarg || 
	    sscanf (optarg, "%x:%x.%x", &opt_pci_bus, &opt_pci_slot, 
		    &opt_pci_func) < 2) 
	  usage ();
	break;
      case 'w':
	opt_window_name = optarg;
	break;
#if HAVE_X
      case 'W':
	if (sscanf (optarg, "0x%lx", &opt_window) != 1) opt_window = None;
	if (opt_window == None) {
	  if (sscanf (optarg, "%ld", &opt_window) != 1) opt_window = None;
	}
	if (opt_window == None)
	  RAISE (MSG_ERROR, "Invalid window id format: %s.", optarg);
	break;
#endif
      case 'c':
        act_aux |= ACTION_CENTER;
        break;
      case 'X':
        act_aux |= ACTION_XMODE;
        break;
      case 'F':
        act_aux |= ACTION_FETCH;
	break;
      case 'N':
        opt_nvdev = TRUE;
        break;
      case '1':
        opt_head = 1;
        break;
      case '2':
        opt_head = 2;
        break;
      case 300:
        if (!processOptSet (optarg, &opt_set_act, &opt_set_tail))
	  usage ();
	break;
    }
    if (act_next != ACTION_NONE) 
    {
      if (act_main != ACTION_NONE) {
	RAISE (MSG_WARNING, "More than one action. Choosing last one.");
      }
      act_main = act_next;
    }
  }

  /* Do the probing right away, without processing any other options. */

  if (act_main == ACTION_PROBE) {
#ifdef DEBUG_PROBE    
#ifdef USE_UNIX_BACKEND
    if (back_root_avail ()) {
      main_card_list = back_root_init ();
    } else if (opt_nvdev && back_nvdev_avail (TRUE)) {
      main_card_list = back_nvdev_init ();
    }
#endif
#ifdef USE_MSWIN_BACKEND
    if (back_win_avail ()) {
      main_card_list = back_win_init ();
    }
#endif
    else {
      main_card_list = NULL;
    }
    if (!main_card_list) {
      RAISE (MSG_ABORT, "Either you are not root, or no NVidia card found.");
    }
    main_card = findCard (main_card_list);
    bdir_probeSystem (main_card);
    return 0;
#else
    RAISE (MSG_ABORT, "No probing enabled in this built.");
#endif
  }

  /* Start to process the options. First, if -s/-o is specified, default
     -r and -S, and assume -g if no main action is present. */

  if (opt_size || opt_hoc >= 0.0 || opt_voc >= 0.0) {
    if (opt_system == TV_SYSTEM_NONE) {
      RAISE (MSG_INFO, "Defaulting to PAL TV system.");
      opt_system = TV_SYSTEM_PAL;
    }
    if (opt_res_x <= 0) {
      opt_res_x = 800;
      RAISE (MSG_INFO, "Defaulting to x resolution %i", opt_res_x);
    }
    if (opt_res_y <= 0) {
      opt_res_y = 600;
      RAISE (MSG_INFO, "Defaulting to y resolution %i", opt_res_y);
    }
    if (act_main == ACTION_NONE) {
      act_main = ACTION_GUI;
    }
  }

  if (act_main == ACTION_NONE && !act_aux) {
    act_main = ACTION_GUI;
  }

  /* Next, choose the backend. If a main action is present, always try
     to get a backend. If -n is specified, force the null backend. For
     the list of modes and printing, we can get away with having the
     null backend, as well, unless we have to fetch the mode. 
  */

  if (act_main != ACTION_NONE) {
    if (opt_no_root) {
      RAISE (MSG_DEBUG, "init null");
      main_card = main_card_list = back_null_init (); 
    } else {
#ifdef USE_UNIX_BACKEND
      if (back_root_avail ()) {
	RAISE (MSG_DEBUG, "init root");
	main_card_list = back_root_init ();
      }
#endif
#ifdef USE_MSWIN_BACKEND
      if (back_win_avail ()) {
	RAISE (MSG_DEBUG, "init root");
	main_card_list = back_win_init ();
      }
#endif
#ifdef USE_CLIENT_BACKEND
      else if (back_client_avail ()) {
	RAISE (MSG_DEBUG, "init client");
	main_card_list = back_client_init ();
      }
#endif
#ifdef USE_UNIX_BACKEND
      else if (opt_nvdev && back_nvdev_avail (TRUE)) {
	RAISE (MSG_DEBUG, "init nvdev");
	main_card_list = back_nvdev_init ();
      }
#endif
      else if (act_main == ACTION_PRINT && ! (act_aux & ACTION_FETCH) ||
	       act_main == ACTION_LIST && (opt_tv_chip != TV_NO_CHIP)) {
	RAISE (MSG_DEBUG, "init null");
	main_card_list = back_null_init ();
      } else {
	RAISE (MSG_ABORT, "%s%s", "Cannot access video cards. Either you "
	       "are not root, or the\nNVidia devices are not accessible.",
	       (act_main == ACTION_LIST) ? "\n(For a list of modes, please "
	       "specify the encoder chip in this case.)" : "");
      }
    }
    if (!main_card_list) {
      RAISE (MSG_ABORT, "No supported video card found.");
    }
    main_card = findCard (main_card_list);
    if (!main_card) {
      RAISE (MSG_ABORT, "No supported video card found at specified address.");
    }
  }

  /* Next, open the card specified by --card-addr (or the first one), unless 
     the main action is none or the gui. Find the right chip, if specified. 
     Note: If printing, we can find all tv chip types in the null backend. */

  if (act_main != ACTION_NONE && act_main != ACTION_GUI && 
      act_main != ACTION_DEBUG) 
  {
    back_access->openCard (main_card);
    if (opt_head != -1 && 
	(act_main == ACTION_TVON || act_main == ACTION_TVOFF)) 
    {
      back_card->getHeads (&main_head, NULL, NULL, NULL);
      back_card->setHeads (-1, opt_head, opt_head);
      if (act_main == ACTION_TVON && opt_head != main_head &&
	  ! back_card->getTwinView (NULL, NULL)) 
      {
	back_card->initSharedView (NULL, NULL);
      }
    }
    if (opt_tv_chip != TV_NO_CHIP) {
      ChipPtr chip;

      for (chip = main_card->chips; chip; chip = chip->next) {
	if (opt_tv_chip == chip->type || (opt_tv_chip == TV_CHIP_BY_ADDR &&
	    opt_tv_bus == chip->bus && opt_tv_addr == chip->addr)) break;
      }
      if (chip) {
	if (opt_tv_chip == TV_CHIP_BY_ADDR) {
	  opt_tv_chip = chip->type;
	}
	back_card->setChip (chip, TRUE);
	/* FIXME: Should set head here */
      } else {
	RAISE (MSG_ABORT, "Cannot find specified chip.");
      }
    }
  }

  /* Next, find the mode for printing -p and tv-on -t. This is required,
     fail if no mode is found. */

  if ((act_aux & ACTION_FETCH) && act_main != ACTION_PRINT) {
    RAISE (MSG_WARNING, "Can only fetch mode for printing. Ignoring option.");
  }
  if (act_main == ACTION_TVON || act_main == ACTION_PRINT) {
    if (opt_size) {
      if (! back_card->findBySize (opt_system, opt_res_x, opt_res_y, opt_size,
				 &main_mode)) 
      {
	RAISE (MSG_ABORT, "Cannot find '%s' mode %i x %i \n"
	       "Please specify as e.g. -r 800,600 -s Large\n", 
	       opt_size, opt_res_x, opt_res_y);
      }
    } else if (opt_hoc >= 0.0 || opt_voc >= 0.0) {
      if (opt_hoc < 0.0) opt_hoc = 0.10;
      if (opt_voc < 0.0) opt_voc = 0.10;
      /* FIXME: Defaulting message */
      /* FIXME: findByOverscan */
      RAISE (MSG_ABORT, "Specifying modes by overscan is not implemented yet.");
    } else if (act_main == ACTION_PRINT && (act_aux & ACTION_FETCH)) {
      back_card->getMode (&main_mode.regs);
      main_mode.spec.system = TV_SYSTEM_NONE;
      main_mode.spec.res_x = 0;
      main_mode.spec.res_y = 0;
      main_mode.spec.size = "Fetch";
      main_mode.spec.aspect = "";
      main_mode.spec.hoc = 0.0;
      main_mode.spec.voc = 0.0;
      main_mode.descFlags = 0;
    } else {
      RAISE (MSG_ABORT, "No mode specified.\n"
	     "Please specify as e.g. -r 800,600 -s Large");
    }
  }
      
  /* For tv-off, the auxiliary actions or the gui, we need a display */

  if (act_main == ACTION_GUI || act_main == ACTION_TVOFF || act_aux) {
#if HAVE_GTK
    /* For some reason, just XOpenDisplay is not enough if GDK is 
       linked in / to be used later. */
    gdk_init (&argc, &argv);
#if HAVE_X
    main_dpy = gdk_display;
    main_screen = my_gdk_screen;
#endif 
#else
#if HAVE_X
    main_dpy = XOpenDisplay (opt_dpy);
    main_screen = DefaultScreen (main_dpy);
#endif
#endif
  }

  /* For tv-off, we need a monitor mode. Get it with the XVidMode extension,
     and default resolution, if necessary. */

  if (act_main == ACTION_TVOFF) {
    main_regs.devFlags = DEV_MONITOR;
#if HAVE_X
    if (opt_res_x > 0 && opt_res_y > 0) {
      find_vidmode (main_dpy, main_screen, opt_res_x, opt_res_y, 
		    &main_regs.crtc, data_card_func(main_card->type)->make);
    } else 
    if (get_vidmode (main_dpy, main_screen, &opt_res_x, &opt_res_y, 
		     &main_regs.crtc, data_card_func(main_card->type)->make)) {
    } else
#endif
    if (opt_size && data_vesa_mode (opt_size, &main_regs.crtc, 
				    data_card_func(main_card->type)->make)) {
    } else {
      RAISE (MSG_ABORT, "No monitor mode to switch tv off. "
	     "You need the XVidMode extension.\n");
    }
    /* NOTE: There must be a valid mode in main_crt here */
  }

  /* If there is still no resolution (i.e., opt_res both -1), then we cannot
     switch the X mode, even if requested, so disable that */

  if ((act_aux & ACTION_XMODE) && ((opt_res_x < 0) || (opt_res_y < 0))) {
    act_aux &= ~ACTION_XMODE;
    RAISE (MSG_WARNING,  "No resolution found. Switching X mode disabled.");
  }

  /* Finally, modify the settings, unless the main action is none, gui,
     query, list, or tv off. */

  if (act_main != ACTION_NONE && act_main != ACTION_GUI &&
      act_main != ACTION_DEBUG && act_main != ACTION_QUERY && 
      act_main != ACTION_TVOFF && act_main != ACTION_LIST && back_card) 
  {
    back_card->getSettings (&opt_set);
    if (opt_connect > CONNECT_NONE) {
      opt_set.connector = opt_connect;
    } else {
      opt_set.connector = CONNECT_BOTH;
    }
    modifySet (&opt_set, opt_set_act);
    opt_set.flags = opt_mode_flags;
    /* FIXME: Should probe if CONNECT_AUTO */
  }

  /* Now we start executing the actions. Printing, color bars and 
     debugging ignore the auxiliary actions, and are handled first. */

  switch (act_main) 
  {
    case ACTION_TVBARS:
      back_card->setTestImage (NULL, &opt_set);
      return 0;
    case ACTION_PRINT:
      printRegs (&main_mode, main_card->type, 
        (opt_tv_chip == TV_NO_CHIP && main_card->chips) ? 
	   main_card->chips->type : opt_tv_chip);
      return 0;
    case ACTION_DEBUG:
      doDebug ();
      return 0;
    case ACTION_QUERY:
      return doQuery (opt_head);
    case ACTION_LIST:
      return doList (main_card);
    default:
      break;
  }
  
  /* If we reached this point, we have to process the auxiliary options.
     If the action is tv-off, we should execute the action first, so
     switching the X mode comes afterwards. There must be valid monitor
     mode in main_crt at this point. */

  if (act_main == ACTION_TVOFF) {
    back_card->setMode (&main_regs);
  }

  /* Now, switch the X mode */

#if HAVE_X
  if (act_aux & ACTION_XMODE) {
    switch_vidmode (main_dpy, main_screen, opt_res_x, opt_res_y);
    XSync (main_dpy, False);
  }
#endif

  /* Next, look for the window. */

#if HAVE_X
  if (opt_window_name) {
    opt_window = Window_With_Name (main_dpy, 
      RootWindow(main_dpy, main_screen), opt_window_name);
    if (opt_window == None) {
      RAISE (MSG_ERROR, "No window with name '%s' exists.", opt_window_name);
    }
  }
#endif

  /* Next, center the window, with respect to the resolution, if given. */

#if HAVE_X
  if (act_aux & ACTION_CENTER) {
    if (opt_window != None) {
      center_window (main_dpy, main_screen, opt_window, opt_res_x, opt_res_y);
    } else {
      RAISE (MSG_ABORT, "No window specified.");
    }
  }
#endif

  /* Next, switch tv mode on. The main_mode must be a valid tv mode at
     this point. */

  if (act_main == ACTION_TVON) {
    main_mode.regs.devFlags = DEV_MONITOR | DEV_TELEVISION;

    if (opt_mode_mask & TV_DESC_DUALVIEW) {
      if (! (opt_mode_flags & TV_DESC_DUALVIEW))
	main_mode.regs.devFlags = DEV_TELEVISION;
    }
    back_card->setModeSettings (&main_mode.regs, &opt_set);
  }

  /* Finally, start the gui. */

  if (act_main == ACTION_GUI) {
#ifdef HAVE_GTK
    gui_main (argc, argv, main_card); 
#else
    RAISE (MSG_ABORT, "Compiled without gui");
#endif
  }

  return 0;
}

/* -------------------------------------------------------------------

Table of options/action:

                                       mode  backend   card    head
none     -r -X          -c             no    no        ignore  no
gui  -g  -r -X -s/-o -S -c -C/-f       opt   yes       ignore  no
on   -t  -r -X -s/-o -S -c -C/-f -1/2  yes   yes       open    yes
off  -m  -r -X          -c       -1/2  no    yes       open    yes
bars -b                    -C/-f       no    yes       open    no
prt  -p  -r -F -s/-o -S 	       yes   yes/none  open    no
qry  -q                          -1/2  no    yes       open    yes
     -P
     -d


Rules:

If only -s/-o is given, -r and -S are defaulted, regardless of action.
If -s/-o is specified and no action given, assume -g.
For -g, only auxiliary options are executed. No mode search, no settings.
Mode search is only done for -t and -p, and aborts if none is found/specified.
If card is opened, --chip will be respected.

----------------------------------------------------------------------


xine --display-ratio 0.979 # for 768x576 Large

Not implemented yet:

-x   --x-options         standard x options follow

GDK Options:

+ --display
+ --sync
+ --no-xshm
+ --name
+ --class
  --gxid_host
  --gxid_port
  --xim-preedit
  --xim-status

GTK Options

  --gtk-module
  --g-fatal-warnings

*/


