/* NVTV MS-Windows card access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: back_mswin.c,v 1.5 2003/10/10 16:16:57 dthierbach Exp $
 *
 * Contents:
 *
 * Access the graphics card under MS-Windows via the driver.
 *
 */

#define INT32 XXX /* Hack to escape redefinition of INT32 */
#include <windows.h>
#undef INT32

#include "local.h" /* before everything else */

#include <stdio.h>

#include "backend.h"
#include "back_direct.h"


/* -------- Fake memory mapping -------- */

/* FIXME: Should probably be in extra file that replaces card_direct */

int openDevMem (CardPtr card)
{
  return 1;
}

void closeDevMem (int fd)
{
}

void *mapDevMem (int fd, unsigned long Base, unsigned long Size)
{
  return (void *) Base;
}

void unmapDevMem (unsigned long Base, unsigned long Size)
{
}

/* -------- Quick'n'dirty routines to access driver -------- */

static BOOLEAN (__stdcall *NvIoControl)(ULONG code, ULONG *params) = NULL;

static ULONG gClientContext = 0, gDeviceContext = 0xDABBAD00L;
static char gDeviceName[128] = "Display";

ULONG Nv01AllocRoot(ULONG *ClientContext)
{
  ULONG params[3];
  params[1] = 0;

  NvIoControl(1, params);

  *ClientContext = params[0];
  return params[2];
}

ULONG Nv01AllocDevice(ULONG ClientContext, ULONG DeviceContext, 
  ULONG DeviceNameLen, char *DeviceName)
{
  ULONG params[5];
  params[0] = ClientContext;
  params[1] = DeviceContext;
  params[2] = DeviceNameLen;
  params[3] = (ULONG) DeviceName;

  NvIoControl(6, params);

  return params[4];
}

ULONG Nv01Free(ULONG int1, ULONG int2, ULONG int3)
{
  ULONG params[4];
  params[0] = int1;
  params[1] = int2;
  params[2] = int3;

  NvIoControl(0, params);

  return params[3];
}

ULONG Nv01ConfigGet(ULONG code)
{
  ULONG params[5];
  params[0] = gClientContext;
  params[1] = gDeviceContext;
  params[2] = code;

  NvIoControl(0x0d, params);

  return params[3];
}

ULONG Nv04ConfigGetEx(ULONG code, void *ioparams, ULONG ioparamssize)
{
  ULONG params[6];
  params[0] = gClientContext;
  params[1] = gDeviceContext;
  params[2] = code;
  params[3] = (ULONG)ioparams;
  params[4] = ioparamssize;

  NvIoControl(0x11, params);

  return params[5];
}

/* -------- MS-Windows backend -------- */

static CardInfo card_win_rec = {
  next: NULL, name: "Card", dev: "Dev", arch: "Arch",
  type: CARD_NVIDIA, reg_base: 0, pio_base: 0,
  addr_bus:0, addr_slot:0, addr_func: 0,
  pci_id: 0, chips: NULL
};

static HINSTANCE card_win_lib;

/* 
 *  Check if driver access is possible.
 */

Bool back_win_avail (void)
{
  /* FIXME for multiple calls */
  card_win_lib = LoadLibrary("NVARCH32.DLL");
  if (!card_win_lib) {
    fprintf (stderr, "Cannot load NVARCH32.DLL. Do you have the NVIDIA driver installed?\n");
    /* FIXME Use GetLastError */
    return FALSE;
  }
  NvIoControl = (BOOLEAN (__stdcall *)(ULONG, ULONG *))
    GetProcAddress(card_win_lib, "NvIoControl");
  if (!NvIoControl) {
    fprintf (stderr, "Cannot get NvIoControl address. Maybe the driver has the wrong version?\n");
    /* FIXME Use GetLastError */
    return FALSE;
  }
  return TRUE;
}

/*
 *  Find all cards in system, and return list.
 */

CardPtr back_win_init (void)
{
  unsigned long reg_phys, reg_virt, pci_id;

  Nv01AllocRoot(&gClientContext);
  Nv01AllocDevice(gClientContext, gDeviceContext, 128, gDeviceName);
  reg_virt = Nv01ConfigGet(0x009);
  reg_phys = Nv01ConfigGet(0x00a);
  pci_id   = Nv01ConfigGet(0x00d);
  Nv01Free(gClientContext, gClientContext, gClientContext);

  printf ("virt=%08lX phys=%08lX\n", reg_virt, reg_phys);
  card_win_rec.reg_base = reg_virt;
  card_win_rec.pci_id = (pci_id >> 16) & 0xffff;

  back_access = &bdir_func;
  return &card_win_rec;
}

/* FIXME  FreeLibrary(card_win_lib); leaks if not used? */

/* FIXME EnumDisplaySettingsA, ChangeDisplaySettingsA, 
   CharNextA, SetWindowLongA
*/

/* Alternative:

Use DPMI (Dos Protected Mode Interface), INT 31h, only available for
16bit code.

In WDM, there are _REGISTER_  _PORT_  macros

For Win NT, there are ...

display/mini/s3v  uses 16bit DPMI and passes MMIOBASE on to 32bit

*/

