---@class Gruvbox
---@field config GruvboxConfig
---@field palette GruvboxPalette
local Gruvbox = {}

---@alias Contrast "hard" | "soft" | ""

---@class ItalicConfig
---@field strings boolean
---@field comments boolean
---@field operators boolean
---@field folds boolean
---@field emphasis boolean

---@class HighlightDefinition
---@field fg string?
---@field bg string?
---@field sp string?
---@field blend integer?
---@field bold boolean?
---@field standout boolean?
---@field underline boolean?
---@field undercurl boolean?
---@field underdouble boolean?
---@field underdotted boolean?
---@field strikethrough boolean?
---@field italic boolean?
---@field reverse boolean?
---@field nocombine boolean?

---@class GruvboxConfig
---@field terminal_colors boolean?
---@field undercurl boolean?
---@field underline boolean?
---@field bold boolean?
---@field italic ItalicConfig?
---@field strikethrough boolean?
---@field contrast Contrast?
---@field invert_selection boolean?
---@field invert_signs boolean?
---@field invert_tabline boolean?
---@field invert_intend_guides boolean?
---@field inverse boolean?
---@field overrides table<string, HighlightDefinition>?
---@field palette_overrides table<string, string>?
Gruvbox.config = {
  terminal_colors = true,
  undercurl = true,
  underline = true,
  bold = true,
  italic = {
    strings = true,
    emphasis = true,
    comments = true,
    operators = false,
    folds = true,
  },
  strikethrough = true,
  invert_selection = false,
  invert_signs = false,
  invert_tabline = false,
  invert_intend_guides = false,
  inverse = true,
  contrast = "",
  palette_overrides = {},
  overrides = {},
  dim_inactive = false,
  transparent_mode = false,
}

-- main gruvbox color palette
---@class GruvboxPalette
Gruvbox.palette = {
  dark0_hard = "#1d2021",
  dark0 = "#282828",
  dark0_soft = "#32302f",
  dark1 = "#3c3836",
  dark2 = "#504945",
  dark3 = "#665c54",
  dark4 = "#7c6f64",
  light0_hard = "#f9f5d7",
  light0 = "#fbf1c7",
  light0_soft = "#f2e5bc",
  light1 = "#ebdbb2",
  light2 = "#d5c4a1",
  light3 = "#bdae93",
  light4 = "#a89984",
  bright_red = "#fb4934",
  bright_green = "#b8bb26",
  bright_yellow = "#fabd2f",
  bright_blue = "#83a598",
  bright_purple = "#d3869b",
  bright_aqua = "#8ec07c",
  bright_orange = "#fe8019",
  neutral_red = "#cc241d",
  neutral_green = "#98971a",
  neutral_yellow = "#d79921",
  neutral_blue = "#458588",
  neutral_purple = "#b16286",
  neutral_aqua = "#689d6a",
  neutral_orange = "#d65d0e",
  faded_red = "#9d0006",
  faded_green = "#79740e",
  faded_yellow = "#b57614",
  faded_blue = "#076678",
  faded_purple = "#8f3f71",
  faded_aqua = "#427b58",
  faded_orange = "#af3a03",
  dark_red_hard = "#792329",
  dark_red = "#722529",
  dark_red_soft = "#7b2c2f",
  light_red_hard = "#fc9690",
  light_red = "#fc9487",
  light_red_soft = "#f78b7f",
  dark_green_hard = "#5a633a",
  dark_green = "#62693e",
  dark_green_soft = "#686d43",
  light_green_hard = "#d3d6a5",
  light_green = "#d5d39b",
  light_green_soft = "#cecb94",
  dark_aqua_hard = "#3e4934",
  dark_aqua = "#49503b",
  dark_aqua_soft = "#525742",
  light_aqua_hard = "#e6e9c1",
  light_aqua = "#e8e5b5",
  light_aqua_soft = "#e1dbac",
  gray = "#928374",
}

-- get a hex list of gruvbox colors based on current bg and constrast config
local function get_colors()
  local p = Gruvbox.palette
  local config = Gruvbox.config

  for color, hex in pairs(config.palette_overrides) do
    p[color] = hex
  end

  local bg = vim.o.background
  local contrast = config.contrast

  local color_groups = {
    dark = {
      bg0 = p.dark0,
      bg1 = p.dark1,
      bg2 = p.dark2,
      bg3 = p.dark3,
      bg4 = p.dark4,
      fg0 = p.light0,
      fg1 = p.light1,
      fg2 = p.light2,
      fg3 = p.light3,
      fg4 = p.light4,
      red = p.bright_red,
      green = p.bright_green,
      yellow = p.bright_yellow,
      blue = p.bright_blue,
      purple = p.bright_purple,
      aqua = p.bright_aqua,
      orange = p.bright_orange,
      neutral_red = p.neutral_red,
      neutral_green = p.neutral_green,
      neutral_yellow = p.neutral_yellow,
      neutral_blue = p.neutral_blue,
      neutral_purple = p.neutral_purple,
      neutral_aqua = p.neutral_aqua,
      dark_red = p.dark_red,
      dark_green = p.dark_green,
      dark_aqua = p.dark_aqua,
      gray = p.gray,
    },
    light = {
      bg0 = p.light0,
      bg1 = p.light1,
      bg2 = p.light2,
      bg3 = p.light3,
      bg4 = p.light4,
      fg0 = p.dark0,
      fg1 = p.dark1,
      fg2 = p.dark2,
      fg3 = p.dark3,
      fg4 = p.dark4,
      red = p.faded_red,
      green = p.faded_green,
      yellow = p.faded_yellow,
      blue = p.faded_blue,
      purple = p.faded_purple,
      aqua = p.faded_aqua,
      orange = p.faded_orange,
      neutral_red = p.neutral_red,
      neutral_green = p.neutral_green,
      neutral_yellow = p.neutral_yellow,
      neutral_blue = p.neutral_blue,
      neutral_purple = p.neutral_purple,
      neutral_aqua = p.neutral_aqua,
      dark_red = p.light_red,
      dark_green = p.light_green,
      dark_aqua = p.light_aqua,
      gray = p.gray,
    },
  }

  if contrast ~= nil and contrast ~= "" then
    color_groups[bg].bg0 = p[bg .. "0_" .. contrast]
    color_groups[bg].dark_red = p[bg .. "_red_" .. contrast]
    color_groups[bg].dark_green = p[bg .. "_green_" .. contrast]
    color_groups[bg].dark_aqua = p[bg .. "_aqua_" .. contrast]
  end

  return color_groups[bg]
end

local function get_groups()
  local colors = get_colors()
  local config = Gruvbox.config

  if config.terminal_colors then
    local term_colors = {
      colors.bg0,
      colors.neutral_red,
      colors.neutral_green,
      colors.neutral_yellow,
      colors.neutral_blue,
      colors.neutral_purple,
      colors.neutral_aqua,
      colors.fg4,
      colors.gray,
      colors.red,
      colors.green,
      colors.yellow,
      colors.blue,
      colors.purple,
      colors.aqua,
      colors.fg1,
    }
    for index, value in ipairs(term_colors) do
      vim.g["terminal_color_" .. index - 1] = value
    end
  end

  local groups = {
    GruvboxFg0 = { fg = colors.fg0 },
    GruvboxFg1 = { fg = colors.fg1 },
    GruvboxFg2 = { fg = colors.fg2 },
    GruvboxFg3 = { fg = colors.fg3 },
    GruvboxFg4 = { fg = colors.fg4 },
    GruvboxGray = { fg = colors.gray },
    GruvboxBg0 = { fg = colors.bg0 },
    GruvboxBg1 = { fg = colors.bg1 },
    GruvboxBg2 = { fg = colors.bg2 },
    GruvboxBg3 = { fg = colors.bg3 },
    GruvboxBg4 = { fg = colors.bg4 },
    GruvboxRed = { fg = colors.red },
    GruvboxRedBold = { fg = colors.red, bold = config.bold },
    GruvboxGreen = { fg = colors.green },
    GruvboxGreenBold = { fg = colors.green, bold = config.bold },
    GruvboxYellow = { fg = colors.yellow },
    GruvboxYellowBold = { fg = colors.yellow, bold = config.bold },
    GruvboxBlue = { fg = colors.blue },
    GruvboxBlueBold = { fg = colors.blue, bold = config.bold },
    GruvboxPurple = { fg = colors.purple },
    GruvboxPurpleBold = { fg = colors.purple, bold = config.bold },
    GruvboxAqua = { fg = colors.aqua },
    GruvboxAquaBold = { fg = colors.aqua, bold = config.bold },
    GruvboxOrange = { fg = colors.orange },
    GruvboxOrangeBold = { fg = colors.orange, bold = config.bold },
    GruvboxRedSign = config.transparent_mode and { fg = colors.red, reverse = config.invert_signs }
      or { fg = colors.red, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxGreenSign = config.transparent_mode and { fg = colors.green, reverse = config.invert_signs }
      or { fg = colors.green, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxYellowSign = config.transparent_mode and { fg = colors.yellow, reverse = config.invert_signs }
      or { fg = colors.yellow, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxBlueSign = config.transparent_mode and { fg = colors.blue, reverse = config.invert_signs }
      or { fg = colors.blue, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxPurpleSign = config.transparent_mode and { fg = colors.purple, reverse = config.invert_signs }
      or { fg = colors.purple, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxAquaSign = config.transparent_mode and { fg = colors.aqua, reverse = config.invert_signs }
      or { fg = colors.aqua, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxOrangeSign = config.transparent_mode and { fg = colors.orange, reverse = config.invert_signs }
      or { fg = colors.orange, bg = colors.bg1, reverse = config.invert_signs },
    GruvboxRedUnderline = { undercurl = config.undercurl, sp = colors.red },
    GruvboxGreenUnderline = { undercurl = config.undercurl, sp = colors.green },
    GruvboxYellowUnderline = { undercurl = config.undercurl, sp = colors.yellow },
    GruvboxBlueUnderline = { undercurl = config.undercurl, sp = colors.blue },
    GruvboxPurpleUnderline = { undercurl = config.undercurl, sp = colors.purple },
    GruvboxAquaUnderline = { undercurl = config.undercurl, sp = colors.aqua },
    GruvboxOrangeUnderline = { undercurl = config.undercurl, sp = colors.orange },
    Normal = config.transparent_mode and { fg = colors.fg1, bg = nil } or { fg = colors.fg1, bg = colors.bg0 },
    NormalFloat = config.transparent_mode and { fg = colors.fg1, bg = nil } or { fg = colors.fg1, bg = colors.bg1 },
    NormalNC = config.dim_inactive and { fg = colors.fg0, bg = colors.bg1 } or { link = "Normal" },
    CursorLine = { bg = colors.bg1 },
    CursorColumn = { link = "CursorLine" },
    TabLineFill = { fg = colors.bg4, bg = colors.bg1, reverse = config.invert_tabline },
    TabLineSel = { fg = colors.green, bg = colors.bg1, reverse = config.invert_tabline },
    TabLine = { link = "TabLineFill" },
    MatchParen = { bg = colors.bg3, bold = config.bold },
    ColorColumn = { bg = colors.bg1 },
    Conceal = { fg = colors.blue },
    CursorLineNr = { fg = colors.yellow, bg = colors.bg1 },
    NonText = { link = "GruvboxBg2" },
    SpecialKey = { link = "GruvboxFg4" },
    Visual = { bg = colors.bg3, reverse = config.invert_selection },
    VisualNOS = { link = "Visual" },
    Search = { fg = colors.yellow, bg = colors.bg0, reverse = config.inverse },
    IncSearch = { fg = colors.orange, bg = colors.bg0, reverse = config.inverse },
    CurSearch = { link = "IncSearch" },
    QuickFixLine = { fg = colors.bg0, bg = colors.yellow, bold = config.bold },
    Underlined = { fg = colors.blue, underline = config.underline },
    StatusLine = { fg = colors.bg2, bg = colors.fg1, reverse = config.inverse },
    StatusLineNC = { fg = colors.bg1, bg = colors.fg4, reverse = config.inverse },
    WinBar = { fg = colors.fg4, bg = colors.bg0 },
    WinBarNC = { fg = colors.fg3, bg = colors.bg1 },
    WinSeparator = config.transparent_mode and { fg = colors.bg3, bg = nil } or { fg = colors.bg3, bg = colors.bg0 },
    WildMenu = { fg = colors.blue, bg = colors.bg2, bold = config.bold },
    Directory = { link = "GruvboxBlueBold" },
    Title = { link = "GruvboxGreenBold" },
    ErrorMsg = { fg = colors.bg0, bg = colors.red, bold = config.bold },
    MoreMsg = { link = "GruvboxYellowBold" },
    ModeMsg = { link = "GruvboxYellowBold" },
    Question = { link = "GruvboxOrangeBold" },
    WarningMsg = { link = "GruvboxRedBold" },
    LineNr = { fg = colors.bg4 },
    SignColumn = config.transparent_mode and { bg = nil } or { bg = colors.bg1 },
    Folded = { fg = colors.gray, bg = colors.bg1, italic = config.italic.folds },
    FoldColumn = config.transparent_mode and { fg = colors.gray, bg = nil } or { fg = colors.gray, bg = colors.bg1 },
    Cursor = { reverse = config.inverse },
    vCursor = { link = "Cursor" },
    iCursor = { link = "Cursor" },
    lCursor = { link = "Cursor" },
    Special = { link = "GruvboxOrange" },
    Comment = { fg = colors.gray, italic = config.italic.comments },
    Todo = { fg = colors.bg0, bg = colors.yellow, bold = config.bold, italic = config.italic.comments },
    Done = { fg = colors.orange, bold = config.bold, italic = config.italic.comments },
    Error = { fg = colors.red, bold = config.bold, reverse = config.inverse },
    Statement = { link = "GruvboxRed" },
    Conditional = { link = "GruvboxRed" },
    Repeat = { link = "GruvboxRed" },
    Label = { link = "GruvboxRed" },
    Exception = { link = "GruvboxRed" },
    Operator = { fg = colors.orange, italic = config.italic.operators },
    Keyword = { link = "GruvboxRed" },
    Identifier = { link = "GruvboxBlue" },
    Function = { link = "GruvboxGreenBold" },
    PreProc = { link = "GruvboxAqua" },
    Include = { link = "GruvboxAqua" },
    Define = { link = "GruvboxAqua" },
    Macro = { link = "GruvboxAqua" },
    PreCondit = { link = "GruvboxAqua" },
    Constant = { link = "GruvboxPurple" },
    Character = { link = "GruvboxPurple" },
    String = { fg = colors.green, italic = config.italic.strings },
    Boolean = { link = "GruvboxPurple" },
    Number = { link = "GruvboxPurple" },
    Float = { link = "GruvboxPurple" },
    Type = { link = "GruvboxYellow" },
    StorageClass = { link = "GruvboxOrange" },
    Structure = { link = "GruvboxAqua" },
    Typedef = { link = "GruvboxYellow" },
    Pmenu = { fg = colors.fg1, bg = colors.bg2 },
    PmenuSel = { fg = colors.bg2, bg = colors.blue, bold = config.bold },
    PmenuSbar = { bg = colors.bg2 },
    PmenuThumb = { bg = colors.bg4 },
    DiffDelete = { bg = colors.dark_red },
    DiffAdd = { bg = colors.dark_green },
    DiffChange = { bg = colors.dark_aqua },
    DiffText = { bg = colors.yellow },
    SpellCap = { link = "GruvboxBlueUnderline" },
    SpellBad = { link = "GruvboxRedUnderline" },
    SpellLocal = { link = "GruvboxAquaUnderline" },
    SpellRare = { link = "GruvboxPurpleUnderline" },
    Whitespace = { fg = colors.bg2 },
    DiagnosticError = { link = "GruvboxRed" },
    DiagnosticSignError = { link = "GruvboxRedSign" },
    DiagnosticUnderlineError = { link = "GruvboxRedUnderline" },
    DiagnosticWarn = { link = "GruvboxYellow" },
    DiagnosticSignWarn = { link = "GruvboxYellowSign" },
    DiagnosticUnderlineWarn = { link = "GruvboxYellowUnderline" },
    DiagnosticInfo = { link = "GruvboxBlue" },
    DiagnosticSignInfo = { link = "GruvboxBlueSign" },
    DiagnosticUnderlineInfo = { link = "GruvboxBlueUnderline" },
    DiagnosticHint = { link = "GruvboxAqua" },
    DiagnosticSignHint = { link = "GruvboxAquaSign" },
    DiagnosticUnderlineHint = { link = "GruvboxAquaUnderline" },
    DiagnosticFloatingError = { link = "GruvboxRed" },
    DiagnosticFloatingWarn = { link = "GruvboxOrange" },
    DiagnosticFloatingInfo = { link = "GruvboxBlue" },
    DiagnosticFloatingHint = { link = "GruvboxAqua" },
    DiagnosticVirtualTextError = { link = "GruvboxRed" },
    DiagnosticVirtualTextWarn = { link = "GruvboxYellow" },
    DiagnosticVirtualTextInfo = { link = "GruvboxBlue" },
    DiagnosticVirtualTextHint = { link = "GruvboxAqua" },
    DiagnosticOk = { link = "GruvboxGreenSign" },
    LspReferenceRead = { link = "GruvboxYellowBold" },
    LspReferenceText = { link = "GruvboxYellowBold" },
    LspReferenceWrite = { link = "GruvboxOrangeBold" },
    LspCodeLens = { link = "GruvboxGray" },
    LspSignatureActiveParameter = { link = "Search" },
    gitcommitSelectedFile = { link = "GruvboxGreen" },
    gitcommitDiscardedFile = { link = "GruvboxRed" },
    GitSignsAdd = { link = "GruvboxGreenSign" },
    GitSignsChange = { link = "GruvboxAquaSign" },
    GitSignsDelete = { link = "GruvboxRedSign" },
    NvimTreeSymlink = { fg = colors.neutral_aqua },
    NvimTreeRootFolder = { fg = colors.neutral_purple, bold = true },
    NvimTreeFolderIcon = { fg = colors.neutral_blue, bold = true },
    NvimTreeFileIcon = { fg = colors.light2 },
    NvimTreeExecFile = { fg = colors.neutral_green, bold = true },
    NvimTreeOpenedFile = { fg = colors.bright_red, bold = true },
    NvimTreeSpecialFile = { fg = colors.neutral_yellow, bold = true, underline = true },
    NvimTreeImageFile = { fg = colors.neutral_purple },
    NvimTreeIndentMarker = { fg = colors.dark3 },
    NvimTreeGitDirty = { fg = colors.neutral_yellow },
    NvimTreeGitStaged = { fg = colors.neutral_yellow },
    NvimTreeGitMerge = { fg = colors.neutral_purple },
    NvimTreeGitRenamed = { fg = colors.neutral_purple },
    NvimTreeGitNew = { fg = colors.neutral_yellow },
    NvimTreeGitDeleted = { fg = colors.neutral_red },
    NvimTreeWindowPicker = { bg = colors.faded_aqua },
    debugPC = { bg = colors.faded_blue },
    debugBreakpoint = { link = "GruvboxRedSign" },
    StartifyBracket = { link = "GruvboxFg3" },
    StartifyFile = { link = "GruvboxFg1" },
    StartifyNumber = { link = "GruvboxBlue" },
    StartifyPath = { link = "GruvboxGray" },
    StartifySlash = { link = "GruvboxGray" },
    StartifySection = { link = "GruvboxYellow" },
    StartifySpecial = { link = "GruvboxBg2" },
    StartifyHeader = { link = "GruvboxOrange" },
    StartifyFooter = { link = "GruvboxBg2" },
    StartifyVar = { link = "StartifyPath" },
    StartifySelect = { link = "Title" },
    DirvishPathTail = { link = "GruvboxAqua" },
    DirvishArg = { link = "GruvboxYellow" },
    netrwDir = { link = "GruvboxAqua" },
    netrwClassify = { link = "GruvboxAqua" },
    netrwLink = { link = "GruvboxGray" },
    netrwSymLink = { link = "GruvboxFg1" },
    netrwExe = { link = "GruvboxYellow" },
    netrwComment = { link = "GruvboxGray" },
    netrwList = { link = "GruvboxBlue" },
    netrwHelpCmd = { link = "GruvboxAqua" },
    netrwCmdSep = { link = "GruvboxFg3" },
    netrwVersion = { link = "GruvboxGreen" },
    NERDTreeDir = { link = "GruvboxAqua" },
    NERDTreeDirSlash = { link = "GruvboxAqua" },
    NERDTreeOpenable = { link = "GruvboxOrange" },
    NERDTreeClosable = { link = "GruvboxOrange" },
    NERDTreeFile = { link = "GruvboxFg1" },
    NERDTreeExecFile = { link = "GruvboxYellow" },
    NERDTreeUp = { link = "GruvboxGray" },
    NERDTreeCWD = { link = "GruvboxGreen" },
    NERDTreeHelp = { link = "GruvboxFg1" },
    NERDTreeToggleOn = { link = "GruvboxGreen" },
    NERDTreeToggleOff = { link = "GruvboxRed" },
    CocErrorSign = { link = "GruvboxRedSign" },
    CocWarningSign = { link = "GruvboxOrangeSign" },
    CocInfoSign = { link = "GruvboxBlueSign" },
    CocHintSign = { link = "GruvboxAquaSign" },
    CocErrorFloat = { link = "GruvboxRed" },
    CocWarningFloat = { link = "GruvboxOrange" },
    CocInfoFloat = { link = "GruvboxBlue" },
    CocHintFloat = { link = "GruvboxAqua" },
    CocDiagnosticsError = { link = "GruvboxRed" },
    CocDiagnosticsWarning = { link = "GruvboxOrange" },
    CocDiagnosticsInfo = { link = "GruvboxBlue" },
    CocDiagnosticsHint = { link = "GruvboxAqua" },
    CocSelectedText = { link = "GruvboxRed" },
    CocMenuSel = { link = "PmenuSel" },
    CocCodeLens = { link = "GruvboxGray" },
    CocErrorHighlight = { link = "GruvboxRedUnderline" },
    CocWarningHighlight = { link = "GruvboxOrangeUnderline" },
    CocInfoHighlight = { link = "GruvboxBlueUnderline" },
    CocHintHighlight = { link = "GruvboxAquaUnderline" },
    TelescopeNormal = { link = "GruvboxFg1" },
    TelescopeSelection = { link = "GruvboxOrangeBold" },
    TelescopeSelectionCaret = { link = "GruvboxRed" },
    TelescopeMultiSelection = { link = "GruvboxGray" },
    TelescopeBorder = { link = "TelescopeNormal" },
    TelescopePromptBorder = { link = "TelescopeNormal" },
    TelescopeResultsBorder = { link = "TelescopeNormal" },
    TelescopePreviewBorder = { link = "TelescopeNormal" },
    TelescopeMatching = { link = "GruvboxBlue" },
    TelescopePromptPrefix = { link = "GruvboxRed" },
    TelescopePrompt = { link = "TelescopeNormal" },
    CmpItemAbbr = { link = "GruvboxFg0" },
    CmpItemAbbrDeprecated = { link = "GruvboxFg1" },
    CmpItemAbbrMatch = { link = "GruvboxBlueBold" },
    CmpItemAbbrMatchFuzzy = { link = "GruvboxBlueUnderline" },
    CmpItemMenu = { link = "GruvboxGray" },
    CmpItemKindText = { link = "GruvboxOrange" },
    CmpItemKindVariable = { link = "GruvboxOrange" },
    CmpItemKindMethod = { link = "GruvboxBlue" },
    CmpItemKindFunction = { link = "GruvboxBlue" },
    CmpItemKindConstructor = { link = "GruvboxYellow" },
    CmpItemKindUnit = { link = "GruvboxBlue" },
    CmpItemKindField = { link = "GruvboxBlue" },
    CmpItemKindClass = { link = "GruvboxYellow" },
    CmpItemKindInterface = { link = "GruvboxYellow" },
    CmpItemKindModule = { link = "GruvboxBlue" },
    CmpItemKindProperty = { link = "GruvboxBlue" },
    CmpItemKindValue = { link = "GruvboxOrange" },
    CmpItemKindEnum = { link = "GruvboxYellow" },
    CmpItemKindOperator = { link = "GruvboxYellow" },
    CmpItemKindKeyword = { link = "GruvboxPurple" },
    CmpItemKindEvent = { link = "GruvboxPurple" },
    CmpItemKindReference = { link = "GruvboxPurple" },
    CmpItemKindColor = { link = "GruvboxPurple" },
    CmpItemKindSnippet = { link = "GruvboxGreen" },
    CmpItemKindFile = { link = "GruvboxBlue" },
    CmpItemKindFolder = { link = "GruvboxBlue" },
    CmpItemKindEnumMember = { link = "GruvboxAqua" },
    CmpItemKindConstant = { link = "GruvboxOrange" },
    CmpItemKindStruct = { link = "GruvboxYellow" },
    CmpItemKindTypeParameter = { link = "GruvboxYellow" },
    diffAdded = { link = "DiffAdd" },
    diffRemoved = { link = "DiffDelete" },
    diffChanged = { link = "DiffChange" },
    diffFile = { link = "GruvboxOrange" },
    diffNewFile = { link = "GruvboxYellow" },
    diffOldFile = { link = "GruvboxOrange" },
    diffLine = { link = "GruvboxBlue" },
    diffIndexLine = { link = "diffChanged" },
    NavicIconsFile = { link = "GruvboxBlue" },
    NavicIconsModule = { link = "GruvboxOrange" },
    NavicIconsNamespace = { link = "GruvboxBlue" },
    NavicIconsPackage = { link = "GruvboxAqua" },
    NavicIconsClass = { link = "GruvboxYellow" },
    NavicIconsMethod = { link = "GruvboxBlue" },
    NavicIconsProperty = { link = "GruvboxAqua" },
    NavicIconsField = { link = "GruvboxPurple" },
    NavicIconsConstructor = { link = "GruvboxBlue" },
    NavicIconsEnum = { link = "GruvboxPurple" },
    NavicIconsInterface = { link = "GruvboxGreen" },
    NavicIconsFunction = { link = "GruvboxBlue" },
    NavicIconsVariable = { link = "GruvboxPurple" },
    NavicIconsConstant = { link = "GruvboxOrange" },
    NavicIconsString = { link = "GruvboxGreen" },
    NavicIconsNumber = { link = "GruvboxOrange" },
    NavicIconsBoolean = { link = "GruvboxOrange" },
    NavicIconsArray = { link = "GruvboxOrange" },
    NavicIconsObject = { link = "GruvboxOrange" },
    NavicIconsKey = { link = "GruvboxAqua" },
    NavicIconsNull = { link = "GruvboxOrange" },
    NavicIconsEnumMember = { link = "GruvboxYellow" },
    NavicIconsStruct = { link = "GruvboxPurple" },
    NavicIconsEvent = { link = "GruvboxYellow" },
    NavicIconsOperator = { link = "GruvboxRed" },
    NavicIconsTypeParameter = { link = "GruvboxRed" },
    NavicText = { link = "GruvboxWhite" },
    NavicSeparator = { link = "GruvboxWhite" },
    htmlTag = { link = "GruvboxAquaBold" },
    htmlEndTag = { link = "GruvboxAquaBold" },
    htmlTagName = { link = "GruvboxBlue" },
    htmlArg = { link = "GruvboxOrange" },
    htmlTagN = { link = "GruvboxFg1" },
    htmlSpecialTagName = { link = "GruvboxBlue" },
    htmlLink = { fg = colors.fg4, underline = config.underline },
    htmlSpecialChar = { link = "GruvboxRed" },
    htmlBold = { fg = colors.fg0, bg = colors.bg0, bold = config.bold },
    htmlBoldUnderline = { fg = colors.fg0, bg = colors.bg0, bold = config.bold, underline = config.underline },
    htmlBoldItalic = { fg = colors.fg0, bg = colors.bg0, bold = config.bold, italic = true },
    htmlBoldUnderlineItalic = {
      fg = colors.fg0,
      bg = colors.bg0,
      bold = config.bold,
      italic = true,
      underline = config.underline,
    },
    htmlUnderline = { fg = colors.fg0, bg = colors.bg0, underline = config.underline },
    htmlUnderlineItalic = {
      fg = colors.fg0,
      bg = colors.bg0,
      italic = true,
      underline = config.underline,
    },
    htmlItalic = { fg = colors.fg0, bg = colors.bg0, italic = true },
    xmlTag = { link = "GruvboxAquaBold" },
    xmlEndTag = { link = "GruvboxAquaBold" },
    xmlTagName = { link = "GruvboxBlue" },
    xmlEqual = { link = "GruvboxBlue" },
    docbkKeyword = { link = "GruvboxAquaBold" },
    xmlDocTypeDecl = { link = "GruvboxGray" },
    xmlDocTypeKeyword = { link = "GruvboxPurple" },
    xmlCdataStart = { link = "GruvboxGray" },
    xmlCdataCdata = { link = "GruvboxPurple" },
    dtdFunction = { link = "GruvboxGray" },
    dtdTagName = { link = "GruvboxPurple" },
    xmlAttrib = { link = "GruvboxOrange" },
    xmlProcessingDelim = { link = "GruvboxGray" },
    dtdParamEntityPunct = { link = "GruvboxGray" },
    dtdParamEntityDPunct = { link = "GruvboxGray" },
    xmlAttribPunct = { link = "GruvboxGray" },
    xmlEntity = { link = "GruvboxRed" },
    xmlEntityPunct = { link = "GruvboxRed" },
    clojureKeyword = { link = "GruvboxBlue" },
    clojureCond = { link = "GruvboxOrange" },
    clojureSpecial = { link = "GruvboxOrange" },
    clojureDefine = { link = "GruvboxOrange" },
    clojureFunc = { link = "GruvboxYellow" },
    clojureRepeat = { link = "GruvboxYellow" },
    clojureCharacter = { link = "GruvboxAqua" },
    clojureStringEscape = { link = "GruvboxAqua" },
    clojureException = { link = "GruvboxRed" },
    clojureRegexp = { link = "GruvboxAqua" },
    clojureRegexpEscape = { link = "GruvboxAqua" },
    clojureRegexpCharClass = { fg = colors.fg3, bold = config.bold },
    clojureRegexpMod = { link = "clojureRegexpCharClass" },
    clojureRegexpQuantifier = { link = "clojureRegexpCharClass" },
    clojureParen = { link = "GruvboxFg3" },
    clojureAnonArg = { link = "GruvboxYellow" },
    clojureVariable = { link = "GruvboxBlue" },
    clojureMacro = { link = "GruvboxOrange" },
    clojureMeta = { link = "GruvboxYellow" },
    clojureDeref = { link = "GruvboxYellow" },
    clojureQuote = { link = "GruvboxYellow" },
    clojureUnquote = { link = "GruvboxYellow" },
    cOperator = { link = "GruvboxPurple" },
    cppOperator = { link = "GruvboxPurple" },
    cStructure = { link = "GruvboxOrange" },
    pythonBuiltin = { link = "GruvboxOrange" },
    pythonBuiltinObj = { link = "GruvboxOrange" },
    pythonBuiltinFunc = { link = "GruvboxOrange" },
    pythonFunction = { link = "GruvboxAqua" },
    pythonDecorator = { link = "GruvboxRed" },
    pythonInclude = { link = "GruvboxBlue" },
    pythonImport = { link = "GruvboxBlue" },
    pythonRun = { link = "GruvboxBlue" },
    pythonCoding = { link = "GruvboxBlue" },
    pythonOperator = { link = "GruvboxRed" },
    pythonException = { link = "GruvboxRed" },
    pythonExceptions = { link = "GruvboxPurple" },
    pythonBoolean = { link = "GruvboxPurple" },
    pythonDot = { link = "GruvboxFg3" },
    pythonConditional = { link = "GruvboxRed" },
    pythonRepeat = { link = "GruvboxRed" },
    pythonDottedName = { link = "GruvboxGreenBold" },
    cssBraces = { link = "GruvboxBlue" },
    cssFunctionName = { link = "GruvboxYellow" },
    cssIdentifier = { link = "GruvboxOrange" },
    cssClassName = { link = "GruvboxGreen" },
    cssColor = { link = "GruvboxBlue" },
    cssSelectorOp = { link = "GruvboxBlue" },
    cssSelectorOp2 = { link = "GruvboxBlue" },
    cssImportant = { link = "GruvboxGreen" },
    cssVendor = { link = "GruvboxFg1" },
    cssTextProp = { link = "GruvboxAqua" },
    cssAnimationProp = { link = "GruvboxAqua" },
    cssUIProp = { link = "GruvboxYellow" },
    cssTransformProp = { link = "GruvboxAqua" },
    cssTransitionProp = { link = "GruvboxAqua" },
    cssPrintProp = { link = "GruvboxAqua" },
    cssPositioningProp = { link = "GruvboxYellow" },
    cssBoxProp = { link = "GruvboxAqua" },
    cssFontDescriptorProp = { link = "GruvboxAqua" },
    cssFlexibleBoxProp = { link = "GruvboxAqua" },
    cssBorderOutlineProp = { link = "GruvboxAqua" },
    cssBackgroundProp = { link = "GruvboxAqua" },
    cssMarginProp = { link = "GruvboxAqua" },
    cssListProp = { link = "GruvboxAqua" },
    cssTableProp = { link = "GruvboxAqua" },
    cssFontProp = { link = "GruvboxAqua" },
    cssPaddingProp = { link = "GruvboxAqua" },
    cssDimensionProp = { link = "GruvboxAqua" },
    cssRenderProp = { link = "GruvboxAqua" },
    cssColorProp = { link = "GruvboxAqua" },
    cssGeneratedContentProp = { link = "GruvboxAqua" },
    javaScriptBraces = { link = "GruvboxFg1" },
    javaScriptFunction = { link = "GruvboxAqua" },
    javaScriptIdentifier = { link = "GruvboxRed" },
    javaScriptMember = { link = "GruvboxBlue" },
    javaScriptNumber = { link = "GruvboxPurple" },
    javaScriptNull = { link = "GruvboxPurple" },
    javaScriptParens = { link = "GruvboxFg3" },
    typescriptReserved = { link = "GruvboxAqua" },
    typescriptLabel = { link = "GruvboxAqua" },
    typescriptFuncKeyword = { link = "GruvboxAqua" },
    typescriptIdentifier = { link = "GruvboxOrange" },
    typescriptBraces = { link = "GruvboxFg1" },
    typescriptEndColons = { link = "GruvboxFg1" },
    typescriptDOMObjects = { link = "GruvboxFg1" },
    typescriptAjaxMethods = { link = "GruvboxFg1" },
    typescriptLogicSymbols = { link = "GruvboxFg1" },
    typescriptDocSeeTag = { link = "Comment" },
    typescriptDocParam = { link = "Comment" },
    typescriptDocTags = { link = "vimCommentTitle" },
    typescriptGlobalObjects = { link = "GruvboxFg1" },
    typescriptParens = { link = "GruvboxFg3" },
    typescriptOpSymbols = { link = "GruvboxFg3" },
    typescriptHtmlElemProperties = { link = "GruvboxFg1" },
    typescriptNull = { link = "GruvboxPurple" },
    typescriptInterpolationDelimiter = { link = "GruvboxAqua" },
    purescriptModuleKeyword = { link = "GruvboxAqua" },
    purescriptModuleName = { link = "GruvboxFg1" },
    purescriptWhere = { link = "GruvboxAqua" },
    purescriptDelimiter = { link = "GruvboxFg4" },
    purescriptType = { link = "GruvboxFg1" },
    purescriptImportKeyword = { link = "GruvboxAqua" },
    purescriptHidingKeyword = { link = "GruvboxAqua" },
    purescriptAsKeyword = { link = "GruvboxAqua" },
    purescriptStructure = { link = "GruvboxAqua" },
    purescriptOperator = { link = "GruvboxBlue" },
    purescriptTypeVar = { link = "GruvboxFg1" },
    purescriptConstructor = { link = "GruvboxFg1" },
    purescriptFunction = { link = "GruvboxFg1" },
    purescriptConditional = { link = "GruvboxOrange" },
    purescriptBacktick = { link = "GruvboxOrange" },
    coffeeExtendedOp = { link = "GruvboxFg3" },
    coffeeSpecialOp = { link = "GruvboxFg3" },
    coffeeCurly = { link = "GruvboxOrange" },
    coffeeParen = { link = "GruvboxFg3" },
    coffeeBracket = { link = "GruvboxOrange" },
    rubyStringDelimiter = { link = "GruvboxGreen" },
    rubyInterpolationDelimiter = { link = "GruvboxAqua" },
    rubyDefinedOperator = { link = "rubyKeyword" },
    objcTypeModifier = { link = "GruvboxRed" },
    objcDirective = { link = "GruvboxBlue" },
    goDirective = { link = "GruvboxAqua" },
    goConstants = { link = "GruvboxPurple" },
    goDeclaration = { link = "GruvboxRed" },
    goDeclType = { link = "GruvboxBlue" },
    goBuiltins = { link = "GruvboxOrange" },
    luaIn = { link = "GruvboxRed" },
    luaFunction = { link = "GruvboxAqua" },
    luaTable = { link = "GruvboxOrange" },
    moonSpecialOp = { link = "GruvboxFg3" },
    moonExtendedOp = { link = "GruvboxFg3" },
    moonFunction = { link = "GruvboxFg3" },
    moonObject = { link = "GruvboxYellow" },
    javaAnnotation = { link = "GruvboxBlue" },
    javaDocTags = { link = "GruvboxAqua" },
    javaCommentTitle = { link = "vimCommentTitle" },
    javaParen = { link = "GruvboxFg3" },
    javaParen1 = { link = "GruvboxFg3" },
    javaParen2 = { link = "GruvboxFg3" },
    javaParen3 = { link = "GruvboxFg3" },
    javaParen4 = { link = "GruvboxFg3" },
    javaParen5 = { link = "GruvboxFg3" },
    javaOperator = { link = "GruvboxOrange" },
    javaVarArg = { link = "GruvboxGreen" },
    elixirDocString = { link = "Comment" },
    elixirStringDelimiter = { link = "GruvboxGreen" },
    elixirInterpolationDelimiter = { link = "GruvboxAqua" },
    elixirModuleDeclaration = { link = "GruvboxYellow" },
    scalaNameDefinition = { link = "GruvboxFg1" },
    scalaCaseFollowing = { link = "GruvboxFg1" },
    scalaCapitalWord = { link = "GruvboxFg1" },
    scalaTypeExtension = { link = "GruvboxFg1" },
    scalaKeyword = { link = "GruvboxRed" },
    scalaKeywordModifier = { link = "GruvboxRed" },
    scalaSpecial = { link = "GruvboxAqua" },
    scalaOperator = { link = "GruvboxFg1" },
    scalaTypeDeclaration = { link = "GruvboxYellow" },
    scalaTypeTypePostDeclaration = { link = "GruvboxYellow" },
    scalaInstanceDeclaration = { link = "GruvboxFg1" },
    scalaInterpolation = { link = "GruvboxAqua" },
    markdownItalic = { fg = colors.fg3, italic = true },
    markdownBold = { fg = colors.fg3, bold = config.bold },
    markdownBoldItalic = { fg = colors.fg3, bold = config.bold, italic = true },
    markdownH1 = { link = "GruvboxGreenBold" },
    markdownH2 = { link = "GruvboxGreenBold" },
    markdownH3 = { link = "GruvboxYellowBold" },
    markdownH4 = { link = "GruvboxYellowBold" },
    markdownH5 = { link = "GruvboxYellow" },
    markdownH6 = { link = "GruvboxYellow" },
    markdownCode = { link = "GruvboxAqua" },
    markdownCodeBlock = { link = "GruvboxAqua" },
    markdownCodeDelimiter = { link = "GruvboxAqua" },
    markdownBlockquote = { link = "GruvboxGray" },
    markdownListMarker = { link = "GruvboxGray" },
    markdownOrderedListMarker = { link = "GruvboxGray" },
    markdownRule = { link = "GruvboxGray" },
    markdownHeadingRule = { link = "GruvboxGray" },
    markdownUrlDelimiter = { link = "GruvboxFg3" },
    markdownLinkDelimiter = { link = "GruvboxFg3" },
    markdownLinkTextDelimiter = { link = "GruvboxFg3" },
    markdownHeadingDelimiter = { link = "GruvboxOrange" },
    markdownUrl = { link = "GruvboxPurple" },
    markdownUrlTitleDelimiter = { link = "GruvboxGreen" },
    markdownLinkText = { fg = colors.gray, underline = config.underline },
    markdownIdDeclaration = { link = "markdownLinkText" },
    haskellType = { link = "GruvboxBlue" },
    haskellIdentifier = { link = "GruvboxAqua" },
    haskellSeparator = { link = "GruvboxFg4" },
    haskellDelimiter = { link = "GruvboxOrange" },
    haskellOperators = { link = "GruvboxPurple" },
    haskellBacktick = { link = "GruvboxOrange" },
    haskellStatement = { link = "GruvboxPurple" },
    haskellConditional = { link = "GruvboxPurple" },
    haskellLet = { link = "GruvboxRed" },
    haskellDefault = { link = "GruvboxRed" },
    haskellWhere = { link = "GruvboxRed" },
    haskellBottom = { link = "GruvboxRedBold" },
    haskellImportKeywords = { link = "GruvboxPurpleBold" },
    haskellDeclKeyword = { link = "GruvboxOrange" },
    haskellDecl = { link = "GruvboxOrange" },
    haskellDeriving = { link = "GruvboxPurple" },
    haskellAssocType = { link = "GruvboxAqua" },
    haskellNumber = { link = "GruvboxAqua" },
    haskellPragma = { link = "GruvboxRedBold" },
    haskellTH = { link = "GruvboxAquaBold" },
    haskellForeignKeywords = { link = "GruvboxGreen" },
    haskellKeyword = { link = "GruvboxRed" },
    haskellFloat = { link = "GruvboxAqua" },
    haskellInfix = { link = "GruvboxPurple" },
    haskellQuote = { link = "GruvboxGreenBold" },
    haskellShebang = { link = "GruvboxYellowBold" },
    haskellLiquid = { link = "GruvboxPurpleBold" },
    haskellQuasiQuoted = { link = "GruvboxBlueBold" },
    haskellRecursiveDo = { link = "GruvboxPurple" },
    haskellQuotedType = { link = "GruvboxRed" },
    haskellPreProc = { link = "GruvboxFg4" },
    haskellTypeRoles = { link = "GruvboxRedBold" },
    haskellTypeForall = { link = "GruvboxRed" },
    haskellPatternKeyword = { link = "GruvboxBlue" },
    jsonKeyword = { link = "GruvboxGreen" },
    jsonQuote = { link = "GruvboxGreen" },
    jsonBraces = { link = "GruvboxFg1" },
    jsonString = { link = "GruvboxFg1" },
    mailQuoted1 = { link = "GruvboxAqua" },
    mailQuoted2 = { link = "GruvboxPurple" },
    mailQuoted3 = { link = "GruvboxYellow" },
    mailQuoted4 = { link = "GruvboxGreen" },
    mailQuoted5 = { link = "GruvboxRed" },
    mailQuoted6 = { link = "GruvboxOrange" },
    mailSignature = { link = "Comment" },
    csBraces = { link = "GruvboxFg1" },
    csEndColon = { link = "GruvboxFg1" },
    csLogicSymbols = { link = "GruvboxFg1" },
    csParens = { link = "GruvboxFg3" },
    csOpSymbols = { link = "GruvboxFg3" },
    csInterpolationDelimiter = { link = "GruvboxFg3" },
    csInterpolationAlignDel = { link = "GruvboxAquaBold" },
    csInterpolationFormat = { link = "GruvboxAqua" },
    csInterpolationFormatDel = { link = "GruvboxAquaBold" },
    rustSigil = { link = "GruvboxOrange" },
    rustEscape = { link = "GruvboxAqua" },
    rustStringContinuation = { link = "GruvboxAqua" },
    rustEnum = { link = "GruvboxAqua" },
    rustStructure = { link = "GruvboxAqua" },
    rustModPathSep = { link = "GruvboxFg2" },
    rustCommentLineDoc = { link = "Comment" },
    rustDefault = { link = "GruvboxAqua" },
    ocamlOperator = { link = "GruvboxFg1" },
    ocamlKeyChar = { link = "GruvboxOrange" },
    ocamlArrow = { link = "GruvboxOrange" },
    ocamlInfixOpKeyword = { link = "GruvboxRed" },
    ocamlConstructor = { link = "GruvboxOrange" },
    LspSagaCodeActionTitle = { link = "Title" },
    LspSagaCodeActionBorder = { link = "GruvboxFg1" },
    LspSagaCodeActionContent = { fg = colors.green, bold = config.bold },
    LspSagaLspFinderBorder = { link = "GruvboxFg1" },
    LspSagaAutoPreview = { link = "GruvboxOrange" },
    TargetWord = { fg = colors.blue, bold = config.bold },
    FinderSeparator = { link = "GruvboxAqua" },
    LspSagaDefPreviewBorder = { link = "GruvboxBlue" },
    LspSagaHoverBorder = { link = "GruvboxOrange" },
    LspSagaRenameBorder = { link = "GruvboxBlue" },
    LspSagaDiagnosticSource = { link = "GruvboxOrange" },
    LspSagaDiagnosticBorder = { link = "GruvboxPurple" },
    LspSagaDiagnosticHeader = { link = "GruvboxGreen" },
    LspSagaSignatureHelpBorder = { link = "GruvboxGreen" },
    SagaShadow = { link = "GruvboxBg0" },
    DashboardShortCut = { link = "GruvboxOrange" },
    DashboardHeader = { link = "GruvboxAqua" },
    DashboardCenter = { link = "GruvboxYellow" },
    DashboardFooter = { fg = colors.purple, italic = true },
    MasonHighlight = { link = "GruvboxAqua" },
    MasonHighlightBlock = { fg = colors.bg0, bg = colors.blue },
    MasonHighlightBlockBold = { fg = colors.bg0, bg = colors.blue, bold = true },
    MasonHighlightSecondary = { fg = colors.yellow },
    MasonHighlightBlockSecondary = { fg = colors.bg0, bg = colors.yellow },
    MasonHighlightBlockBoldSecondary = { fg = colors.bg0, bg = colors.yellow, bold = true },
    MasonHeader = { link = "MasonHighlightBlockBoldSecondary" },
    MasonHeaderSecondary = { link = "MasonHighlightBlockBold" },
    MasonMuted = { fg = colors.fg4 },
    MasonMutedBlock = { fg = colors.bg0, bg = colors.fg4 },
    MasonMutedBlockBold = { fg = colors.bg0, bg = colors.fg4, bold = true },
    LspInlayHint = { link = "comment" },
    CarbonFile = { link = "GruvboxFg1" },
    CarbonExe = { link = "GruvboxYellow" },
    CarbonSymlink = { link = "GruvboxAqua" },
    CarbonBrokenSymlink = { link = "GruvboxRed" },
    CarbonIndicator = { link = "GruvboxGray" },
    CarbonDanger = { link = "GruvboxRed" },
    CarbonPending = { link = "GruvboxYellow" },
    NoiceCursor = { link = "TermCursor" },
    NotifyDEBUGBorder = { link = "GruvboxBlue" },
    NotifyDEBUGIcon = { link = "GruvboxBlue" },
    NotifyDEBUGTitle = { link = "GruvboxBlue" },
    NotifyERRORBorder = { link = "GruvboxRed" },
    NotifyERRORIcon = { link = "GruvboxRed" },
    NotifyERRORTitle = { link = "GruvboxRed" },
    NotifyINFOBorder = { link = "GruvboxAqua" },
    NotifyINFOIcon = { link = "GruvboxAqua" },
    NotifyINFOTitle = { link = "GruvboxAqua" },
    NotifyTRACEBorder = { link = "GruvboxGreen" },
    NotifyTRACEIcon = { link = "GruvboxGreen" },
    NotifyTRACETitle = { link = "GruvboxGreen" },
    NotifyWARNBorder = { link = "GruvboxYellow" },
    NotifyWARNIcon = { link = "GruvboxYellow" },
    NotifyWARNTitle = { link = "GruvboxYellow" },
    IlluminatedWordText = { link = "LspReferenceText" },
    IlluminatedWordRead = { link = "LspReferenceRead" },
    IlluminatedWordWrite = { link = "LspReferenceWrite" },
    TSRainbowRed = { fg = colors.red },
    TSRainbowOrange = { fg = colors.orange },
    TSRainbowYellow = { fg = colors.yellow },
    TSRainbowGreen = { fg = colors.green },
    TSRainbowBlue = { fg = colors.blue },
    TSRainbowViolet = { fg = colors.purple },
    TSRainbowCyan = { fg = colors.cyan },
    DapBreakpointSymbol = { fg = colors.red, bg = colors.bg1 },
    DapStoppedSymbol = { fg = colors.green, bg = colors.bg1 },
    DapUIBreakpointsCurrentLine = { link = "GruvboxYellow" },
    DapUIBreakpointsDisabledLine = { link = "GruvboxGray" },
    DapUIBreakpointsInfo = { link = "GruvboxAqua" },
    DapUIBreakpointsLine = { link = "GruvboxYellow" },
    DapUIBreakpointsPath = { link = "GruvboxBlue" },
    DapUICurrentFrameName = { link = "GruvboxPurple" },
    DapUIDecoration = { link = "GruvboxPurple" },
    DapUIEndofBuffer = { link = "GruvboxBg2" },
    DapUIFloatBorder = { link = "GruvboxAqua" },
    DapUILineNumber = { link = "GruvboxYellow" },
    DapUIModifiedValue = { link = "GruvboxRed" },
    DapUIPlayPause = { fg = colors.green, bg = colors.bg1 },
    DapUIRestart = { fg = colors.green, bg = colors.bg1 },
    DapUIScope = { link = "GruvboxBlue" },
    DapUISource = { link = "GruvboxFg1" },
    DapUIStepBack = { fg = colors.blue, bg = colors.bg1 },
    DapUIStepInto = { fg = colors.blue, bg = colors.bg1 },
    DapUIStepOut = { fg = colors.blue, bg = colors.bg1 },
    DapUIStepOver = { fg = colors.blue, bg = colors.bg1 },
    DapUIStop = { fg = colors.red, bg = colors.bg1 },
    DapUIStoppedThread = { link = "GruvboxBlue" },
    DapUIThread = { link = "GruvboxBlue" },
    DapUIType = { link = "GruvboxOrange" },
    DapUIUnavailable = { link = "GruvboxGray" },
    DapUIWatchesEmpty = { link = "GruvboxGray" },
    DapUIWatchesError = { link = "GruvboxRed" },
    DapUIWatchesValue = { link = "GruvboxYellow" },
    DapUIWinSelect = { link = "GruvboxYellow" },
    NeogitDiffDelete = { link = "DiffDelete" },
    NeogitDiffAdd = { link = "DiffAdd" },
    NeogitHunkHeader = { link = "WinBar" },
    NeogitHunkHeaderHighlight = { link = "WinBarNC" },
    DiffviewStatusModified = { link = "GruvboxGreenBold" },
    DiffviewFilePanelInsertions = { link = "GruvboxGreenBold" },
    DiffviewFilePanelDeletions = { link = "GruvboxRedBold" },
    ["@comment"] = { link = "Comment" },
    ["@none"] = { bg = "NONE", fg = "NONE" },
    ["@preproc"] = { link = "PreProc" },
    ["@define"] = { link = "Define" },
    ["@operator"] = { link = "Operator" },
    ["@punctuation.delimiter"] = { link = "Delimiter" },
    ["@punctuation.bracket"] = { link = "Delimiter" },
    ["@punctuation.special"] = { link = "Delimiter" },
    ["@string"] = { link = "String" },
    ["@string.regex"] = { link = "String" },
    ["@string.escape"] = { link = "SpecialChar" },
    ["@string.special"] = { link = "SpecialChar" },
    ["@character"] = { link = "Character" },
    ["@character.special"] = { link = "SpecialChar" },
    ["@boolean"] = { link = "Boolean" },
    ["@number"] = { link = "Number" },
    ["@float"] = { link = "Float" },
    ["@function"] = { link = "Function" },
    ["@function.builtin"] = { link = "Special" },
    ["@function.call"] = { link = "Function" },
    ["@function.macro"] = { link = "Macro" },
    ["@method"] = { link = "Function" },
    ["@method.call"] = { link = "Function" },
    ["@constructor"] = { link = "Special" },
    ["@parameter"] = { link = "Identifier" },
    ["@keyword"] = { link = "Keyword" },
    ["@keyword.function"] = { link = "Keyword" },
    ["@keyword.operator"] = { link = "GruvboxRed" },
    ["@keyword.return"] = { link = "Keyword" },
    ["@conditional"] = { link = "Conditional" },
    ["@repeat"] = { link = "Repeat" },
    ["@debug"] = { link = "Debug" },
    ["@label"] = { link = "Label" },
    ["@include"] = { link = "Include" },
    ["@exception"] = { link = "Exception" },
    ["@type"] = { link = "Type" },
    ["@type.builtin"] = { link = "Type" },
    ["@type.definition"] = { link = "Typedef" },
    ["@type.qualifier"] = { link = "Type" },
    ["@storageclass"] = { link = "StorageClass" },
    ["@attribute"] = { link = "PreProc" },
    ["@field"] = { link = "Identifier" },
    ["@property"] = { link = "Identifier" },
    ["@variable"] = { link = "GruvboxFg1" },
    ["@variable.builtin"] = { link = "Special" },
    ["@constant"] = { link = "Constant" },
    ["@constant.builtin"] = { link = "Special" },
    ["@constant.macro"] = { link = "Define" },
    ["@namespace"] = { link = "GruvboxFg1" },
    ["@symbol"] = { link = "Identifier" },
    ["@text"] = { link = "GruvboxFg1" },
    ["@text.strong"] = { bold = config.bold },
    ["@text.emphasis"] = { italic = config.italic.emphasis },
    ["@text.underline"] = { underline = config.underline },
    ["@text.strike"] = { strikethrough = config.strikethrough },
    ["@text.title"] = { link = "Title" },
    ["@text.literal"] = { link = "String" },
    ["@text.uri"] = { link = "Underlined" },
    ["@text.math"] = { link = "Special" },
    ["@text.environment"] = { link = "Macro" },
    ["@text.environment.name"] = { link = "Type" },
    ["@text.reference"] = { link = "Constant" },
    ["@text.todo"] = { link = "Todo" },
    ["@text.note"] = { link = "SpecialComment" },
    ["@text.note.comment"] = { fg = colors.purple, bold = config.bold },
    ["@text.warning"] = { link = "WarningMsg" },
    ["@text.danger"] = { link = "ErrorMsg" },
    ["@text.danger.comment"] = { fg = colors.fg0, bg = colors.red, bold = config.bold },
    ["@text.diff.add"] = { link = "diffAdded" },
    ["@text.diff.delete"] = { link = "diffRemoved" },
    ["@tag"] = { link = "Tag" },
    ["@tag.attribute"] = { link = "Identifier" },
    ["@tag.delimiter"] = { link = "Delimiter" },
    ["@punctuation"] = { link = "Delimiter" },
    ["@macro"] = { link = "Macro" },
    ["@structure"] = { link = "Structure" },
    ["@lsp.type.class"] = { link = "@type" },
    ["@lsp.type.comment"] = {}, -- do not overwrite comments
    ["@lsp.type.decorator"] = { link = "@macro" },
    ["@lsp.type.enum"] = { link = "@type" },
    ["@lsp.type.enumMember"] = { link = "@constant" },
    ["@lsp.type.function"] = { link = "@function" },
    ["@lsp.type.interface"] = { link = "@constructor" },
    ["@lsp.type.macro"] = { link = "@macro" },
    ["@lsp.type.method"] = { link = "@method" },
    ["@lsp.type.namespace"] = { link = "@namespace" },
    ["@lsp.type.parameter"] = { link = "@parameter" },
    ["@lsp.type.property"] = { link = "@property" },
    ["@lsp.type.struct"] = { link = "@type" },
    ["@lsp.type.type"] = { link = "@type" },
    ["@lsp.type.typeParameter"] = { link = "@type.definition" },
    ["@lsp.type.variable"] = { link = "@variable" },
  }

  for group, hl in pairs(config.overrides) do
    if groups[group] then
      -- "link" should not mix with other configs (:h hi-link)
      groups[group].link = nil
    end

    groups[group] = vim.tbl_extend("force", groups[group] or {}, hl)
  end

  return groups
end

---@param config GruvboxConfig?
Gruvbox.setup = function(config)
  Gruvbox.config = vim.tbl_deep_extend("force", Gruvbox.config, config or {})
end

--- main load function
Gruvbox.load = function()
  if vim.version().minor < 8 then
    vim.notify_once("gruvbox.nvim: you must use neovim 0.8 or higher")
    return
  end

  -- reset colors
  if vim.g.colors_name then
    vim.cmd.hi("clear")
  end
  vim.g.colors_name = "gruvbox"
  vim.o.termguicolors = true

  local groups = get_groups()

  -- add highlights
  for group, settings in pairs(groups) do
    vim.api.nvim_set_hl(0, group, settings)
  end
end

return Gruvbox
