<?php
/*
# Copyright 2004, Commission scolaire de Laval, Stephane Bond
#
# This file is part of the MILLE-XTERM distribution.
# See the MILLE-XTERM (english) and/or the MILLE (french) project web site
#
# http://www.revolutionlinux.com/mille-xterm/
# http://www.mille.ca/
#
# The MILLE-XTERM framework is covered by the GNU General Public License. See
# the COPYING file in the top-level MILLE-XTERM directory. Software packages
# that are included in the MILLE-XTERM distribution have their own licenses.
#
# -------------------------------------------------------------------------
*/
	require_once 'dbFunctions.php';
	
	/**
	*
	* Find the default node for a subnet 
	* 
	* @package MILLE-CONFIGURATOR
	*/
	class SubnetNodeFinder {

		/**
		* ip address of the computer
		* @access private
		* @var int
		*/
		var $ip;
		/**
		* subnet attribute value of the node
		* @access private
		* @var int
		*/
		var $subnet;
		/**
		* nodeid
		* @access private
		* @var string
		*/
		var $nodeid;

		/**
		* Last Error
		* @access private
		* @var string
		*/
		var $lastError;
		/**
		* Last Error
		* @access private
		* @var string
		*/
		var $isError;
		

		/**
		* The constructor of a SubnetNodeFinder
		*
		* Initialised in by :
		* <ul>
		* <li> IP of the computer
		* </ul>
		* 
		* Example of valid initialisation :
		*
		* <code>
		*   $finder=new SubnetNodeFinder("10.1.0.9");
		* </code>
		*
		* @access public
		*/
		function SubnetNodeFinder($ipaddr) {
			$this->notFound = false;
			$this->ip = $ipaddr;
			
			// Build IN clause content including all possibility
			$inClause="";
			foreach (array(8,16,18,19,20,21,22,23,24,25,26,27,28,29,30,32) as $i) {
				$inClause = $inClause.",'".$this->getSubnetDef($i,$ipaddr)."'";
			}
			$inClause = substr($inClause, 1);
			
			// Retrieve corresponding subnets
			$where="attributesdef_id=(SELECT id FROM attributesdef WHERE name='NODE_SUBNET') ".
				"AND value IN (".$inClause.")";
			$result=$this->getFromDB("nodes_id, value",$where,"attributes",true,true);
			if ($result) {
				// Keep higher value
				$highLine=NULL;
				$highVal=0;
				foreach ($result as $ptr) {
					list($ipVal, $maskVal) = split('[\/]',$ptr['value']);
					$curVal=intval($maskVal);
					if ($curVal >= $highVal) {
						$highLine=$ptr;
						$highVal=$curVal;
					}
				}
			}
			if (!$highLine) {
				// No node found
				$this->notFound = true;
			} else {
				$this->subnet = $highLine['value'];
				$this->nodeid = $highLine['nodes_id'];
			}
		}

		
		/**
		* Compute the subnet for a specific mask width
		*
		* @access private
		* @return int
		*/
		function getSubnetDef($width, $ipaddr) {
			// Split ip addr in 4 parts (0.1.2.3)
			$parts=split('[.]',$ipaddr);
			// Compute mask for the the last part
			$lastPart = (int) floor($width / 8);
			$lastBits = $width % 8;
			$lastMask = 0;
			if ($lastBits > 0) {
				$lastMask = 1;
				for ($i = 0; $i < $lastBits-1; $i++) {
					$lastMask = ($lastMask << 1) + 1;
				}
				for ($i = $lastBits; $i < 8; $i++) {
					$lastMask = ($lastMask << 1);
				}
			}
			// Apply mask to ip
			$parts[$lastPart] = intval($parts[$lastPart]) & $lastMask;
			for ($i = $lastPart+1; $i<4; $i++) {
				$parts[$i]=0;
			}
			return $parts[0].'.'.$parts[1].'.'.$parts[2].'.'.$parts[3].'/'.$width;
		}
		
		/**
		* Return true if not found
		*
		* @access public
		* @return int
		*/
		function notFound() {
			return $this->notFound;
		}
		
		
		/**
		* Get the node ID
		*
		* @access public
		* @return int
		*/
		function getNodeID() {
			return $this->nodeid;
		}
		
		/**
		* Get the subnet string
		*
		* @access public
		* @return int
		*/
		function getSubnet() {
			return $this->subnet;
		}

		/**
		* Return yes if the object exists
		* To be used just after the construction
		*
		* @access public
		* @return float
		*/
		function isError() {
			return $this->isError;
		}

		/**
		* Get the last error
		*
		* @access public
		* @return float
		*/
		function lastError() {
			return $this->lastError;
		}


		
		/**
		* GetFromDB
		*
		* @access private
		* @return array
		*/
		function getFromDB($column,$where,$from="attributes",$isFatal=true,$multiple = false) {
			$query='SELECT '.$column.' FROM '.$from.' WHERE '.$where;
			if ($multiple) {
				$result = select($query.";");
			}
			else {
				$result = singleResultSelect($query);
			}
			if (!$result) {	
				$this->lastError = getMessage('not_found').$query; 
				$this->notFound = true; 
				if ($isFatal) 
					$this->isError = true; return NULL; 
			}
			elseif (!is_array($result)) { 
				$this->lastError = getMessage('db_error').$result." : ".$query; 
				if ($isFatal) $this->isError = true; return NULL;
			}
			return $result;
		}
		
	} // class SubnetNodeFinder
?>
