from juju.lib.testing import TestCase

from juju.errors import ProviderError
from juju.providers.common.instance_type import (
    TypeSolver, InstanceType)
from juju.machine.constraints import ConstraintSet


from twisted.internet.defer import inlineCallbacks


class InstanceTypeSolverTest(TestCase):

    @inlineCallbacks
    def setUp(self):
        self.instance_types = {
            "ExtraSmall": InstanceType("amd64", 0.1, 768),
            "Small": InstanceType("amd64", 1, 1792),
            "Medium": InstanceType("amd64", 2, 3584),
            "Large": InstanceType("amd64", 4, 7168),
            "ExtraLarge": InstanceType("amd64", 8, 14336),
            "ExtraLargeMem": InstanceType("amd64", 8, 28336),
            "Wimpy": InstanceType("arm64", 8, 7168),
            }
        self.solver = TypeSolver(self.instance_types)
        self.cs = ConstraintSet("cloud")
        self.cs.register_generics(self.instance_types.keys())
        yield super(InstanceTypeSolverTest, self).setUp()

    def test_type_solver(self):

        self.assertEqual(
            self.solver.run(self.cs.parse(["cpu=4"])),
            "Large")

        self.assertEqual(
            self.solver.run(self.cs.parse(["cpu=3.5"])),
            "Large")

        self.assertEqual(
            self.solver.run(self.cs.parse(["mem=2G"])),
            "Medium")

        self.assertEqual(
            self.solver.run(self.cs.parse(["cpu=2.5", "mem=2G"])),
            "Large")

        self.assertEqual(
            self.solver.run(self.cs.parse(["cpu=8", "mem=2G"])),
            "ExtraLarge")

        self.assertEqual(
            self.solver.run(self.cs.parse(["arch=arm64", "mem=2G"])),
            "Wimpy")

        constraints = self.cs.parse(["arch=arm64", "mem=32G"])
        e = self.assertRaises(
            ProviderError,
            self.solver.run,
            constraints)
        self.assertIn("No instance type satisfies", str(e))

        # Any auto solves for min 1 cpu, 512mb ram.
        self.assertEqual(
            self.solver.run(self.cs.parse(["instance-type=any"])),
            "Small")

        self.assertEqual(
            self.solver.run(self.cs.parse(
                ["instance-type=ExtraLarge"])),
            "ExtraLarge")
