# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file describes the interface that must be implemented for each
# underlying tool in order for gquilt to provide an interface to for the tool

import os, os.path, shutil, pango, collections

from gquilt_pkg import console, fsdb
from gquilt_pkg import cmd_result
from gquilt_pkg import putils
from gquilt_pkg import ws_event

DEFAULT_NAME_EVARS = ["GIT_AUTHOR_NAME", "GECOS"]
DEFAULT_EMAIL_VARS = ["GIT_AUTHOR_EMAIL", "EMAIL_ADDRESS"]

Deco = collections.namedtuple('Deco', ['style', 'foreground'])

PatchData = collections.namedtuple('PatchData', ['name', 'state', 'guards'])

class Interface:
    def __init__(self, name, cmd=None):
        self._name_envars = DEFAULT_NAME_EVARS
        self._email_envars = DEFAULT_EMAIL_VARS
        self.name = name
        self.cmd_label = cmd if cmd else '%s ' % name
        self.status_deco_map = {
            None: Deco(pango.STYLE_NORMAL, "black"),
        }
    def _map_cmd_result(self, result, ignore_err_re=None):
        assert False, "Must be defined in child"
    def _run_cmd_on_console(self, cmd, input_text=None, ignore_err_re=None):
        result = console.exec_console_cmd(cmd)
        return self._map_cmd_result(result, ignore_err_re=ignore_err_re)
    def count_ok_meld(self, count):
        # override in back end if there are restrictions on its use of meld
        return True
    def display_files_diff_in_viewer(self, viewer, file_name, patch=None):
        assert False, 'display_file_diff_in_viewer() must be provided in child class!!!'
    def do_add_files_to_patch(self, filelist):
        assert False, 'do_add_files_to_patch() must be provided in child class!!!'
    def do_copy_file(self, file_path, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of copy
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file_path))
        if not force and os.path.exists(dest):
            return cmd_result.Result(cmd_result.ERROR_SUGGEST_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, sout = self.get_patch_files(None, False)
        if not dest in patch_files:
            res, sout, serr = self.do_add_files_to_patch([dest])
            if res is not cmd_result.OK:
                return cmd_result.Result(res, sout, serr)
        try:
            shutil.copy(file_path, dest)
            result = (cmd_result.OK, "", "")
        except (IOError, os.error, shutil.Error) as why:
            serr = "Copy %s to %s failed. %s." % (file_path, dest, str(why))
            console.LOG.append_stderr(serr)
            result = (cmd_result.ERROR, "", serr)
        ws_event.notify_events(ws_event.FILE_ADD|ws_event.FILE_DEL)
        return result
    def do_delete_patch(self, patch):
        assert False, 'do_delete_patch() must be provided in child class!!!'
    def do_exec_tool_cmd(self, cmd):
        assert False, 'do_exec_tool_cmd() must be provided in child class!!!'
    def do_finish_patch(self, patch):
        assert False, 'do_merge_patch() must be provided in child class!!!'
    def do_fold_patch(self, patch):
        assert False, 'do_fold_patch() must be provided in child class!!!'
    def do_fold_patch_file(self, filename):
        assert False, 'do_fold_patch_file() must be provided in child class!!!'
    def do_import_patch(self, filename, patchname=None, force=False):
        assert False, 'do_import_patch() must be provided in child class!!!'
    def do_move_file(self, file_path, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of move
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file_path))
        if not force and os.path.exists(dest):
            return cmd_result.Result(cmd_result.ERROR_SUGGEST_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, sout = self.get_patch_files(None, False)
        if file_path not in patch_files:
            res, sout, serr = self.do_add_files_to_patch([file_path])
            if res is not cmd_result.OK:
                return cmd_result.Result(res, sout, serr)
        if dest not in patch_files:
            res, sout, serr = self.do_add_files_to_patch([dest])
            if res is not cmd_result.OK:
                return cmd_result.Result(res, sout, serr)
        try:
            os.rename(file_path, dest)
            result = (cmd_result.OK, "", "")
        except (IOError, os.error, shutil.Error) as why:
            serr = "Copy %s to %s failed. %s." % (file_path, dest, str(why))
            console.LOG.append_stderr(serr)
            result = (cmd_result.ERROR, "", serr)
        ws_event.notify_events(ws_event.FILE_ADD|ws_event.FILE_DEL)
        return result
    def do_new_patch(self, name, force=False):
        assert False, 'do_new_patch() must be provided in child class!!!'
    def do_pop_to(self, patch=None):
        assert False, 'do_pop_to() must be provided in child class!!!'
    def do_push_to(self, patch=None, force=False, merge=False):
        assert False, 'do_push_to() must be provided in child class!!!'
    def do_refresh(self, patch=None, force=False, notify=True):
        assert False, 'do_refresh() must be provided in child class!!!'
    def do_remove_files_from_patch(self, filelist, patch=None):
        assert False, 'do_remove_files_from_patch() must be provided in child class!!!'
    def do_rename_patch(self, patch, newname):
        assert False, 'do_rename_patch() must be provided in child class!!!'
    def do_revert_files_in_patch(self, filelist, patch=None):
        assert False, 'do_revert_files_in_patch() must be provided in child class!!!'
    def do_select_guards(self, guards):
        assert False, 'do_select_guards() must be provided in child class!!!'
    def do_set_patch_guards(self, patch_name, guards):
        assert False, 'do_set_patch_guards() must be provided in child class!!!'
    def do_set_patch_description(self, patch, description):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.get_patch_file_name(patch)
        res = putils.set_patch_descr_lines(pfn, description.splitlines())
        if res:
            res = cmd_result.OK
            serr = ""
        else:
            res = cmd_result.ERROR
            serr = "Error reading patch description\n"
        if console.LOG is not None:
            console.LOG.log_entry('set description for "' + patch + '"')
        return cmd_result.Result(res, "", serr)
    def extdiff_and_full_patch_ok(self):
        return False
    def get_all_patches_data(self):
        assert False, 'get_all_patches_data() must be provided in child class!!!'
    def _get_first_in_envar(self, envar_list):
        for envar in envar_list:
            try:
                value = os.environ[envar]
                if value is not "":
                    return value
            except KeyError:
                continue
        return ""
    def get_applied_patches(self):
        return []
    def get_author_name_and_email(self):
        name = self._get_first_in_envar(self._name_envars)
        if not name:
            name = "UNKNOWN"
        email = self._get_first_in_envar(self._email_envars)
        if not email:
            email = "UNKNOWN"
        return "%s <%s>" % (name, email)
    def get_combined_diff(self, start_patch=None, end_patch=None):
        assert False, 'get_combined_diff() must be provided in child class!!!'
    def get_description_is_finish_ready(self, patch):
        # make sure we don't get into an infinite loop if this isn't defined
        # in the derived back end
        return True
    def get_diff(self, filelist=list(), patch=None):
        assert False, 'get_diff() must be provided in child class!!!'
    def get_in_progress(self):
        assert False, 'get_in_progress() must be provided in child class!!!'
    def get_next_patch(self):
        assert False, 'get_next_patch() must be provided in child class!!!'
    def get_patch_description(self, patch):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.get_patch_file_name(patch)
        if os.path.exists(pfn):
            res, lines = putils.get_patch_descr_lines(pfn)
            if res:
                return cmd_result.Result(cmd_result.OK, os.linesep.join(lines) + os.linesep, "")
            else:
                return cmd_result.Result(cmd_result.ERROR, "", "Error reading patch description\n")
        else:
            return cmd_result.Result(cmd_result.OK, "", "")
    def get_patch_file_name(self, patch):
        assert False, 'get_patch_file_name() must be provided in child class!!!'
    def get_patch_files(self, patch=None, withstatus=True):
        assert False, 'get_patch_files() must be provided in child class!!!'
    def get_patch_guards(self, patch):
        return ''
    def get_playground_root(self, fdir=None):
        if fdir:
            return fdir
        return os.getcwd()
    def get_selected_guards(self):
        return []
    def get_top_patch(self):
        assert False, 'get_top_patch() must be provided in child class!!!'
    def has_add_files(self):
        # override in back end if the tool does not have an "add file" function
        return True
    def has_finish_patch(self):
        # override in back end if the tool has a "qfinish patch" function
        return False
    def has_guards(self):
        # override in back end if the tool supports patch guards
        return False
    def has_refresh_non_top(self):
        return False
    def is_available(self):
        return False
    def is_patch_applied(self, patch):
        assert False, 'is_patch_applied() must be provided in child class!!!'
    def is_playground(self, fdir=None):
        assert False, 'is_playground() must be provided in child class!!!'
    def last_patch_in_series(self):
        assert False, 'last_patch_in_series() must be provided in child class!!!'
    def new_playground(self, fdir=None):
        assert False, 'new_playground() must be provided in child class!!!'
    def requires(self):
        assert False, 'requires() must be provided in child class!!!'

# create a null back end to use when the working directory is not a valid
# playground
class NullInterface(Interface):
    def __init__(self):
        Interface.__init__(self, "null")
    def _map_cmd_result(self, result, ignore_err_re=None):
        return result
    def _message(self):
        return os.getcwd() + ' is not a valid playground\n'
    def do_add_files_to_patch(self, filelist):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_copy_file(self, file_name, dest, force=False):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_new_patch(self, name, force=False):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_exec_tool_cmd(self, cmd):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_import_patch(self, filename, patchname=None, force=False):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_fold_patch_file(self, patch):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_move_file(self, file_name, dest, force=False):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_pop_to(self, patch=None):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_push_to(self, patch=None):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def do_refresh(self, patch=None, force=False):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def get_all_patches_data(self):
        return []
    def get_combined_diff(self, start_patch=None, end_patch=None):
        return cmd_result.Result(cmd_result.ERROR, "", self._message())
    def get_in_progress(self):
        return False
    def get_patch_file_db(self, patch=None):
        return fsdb.NullFileDb()
    def get_patch_files(self, patch=None, withstatus=True):
        return cmd_result.Result(cmd_result.OK, "", "")
    def get_top_patch(self):
        return ""
    def get_ws_file_db(self, patch=None):
        return fsdb.OsFileDb()
    def is_available(self):
        return True
    def last_patch_in_series(self):
        return ""
    def requires(self):
        return "Nothing!"
