"This module provides a simple Python wrapper for the SWIG-generated Mesh class"

# Copyright (C) 2012 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Marie E. Rognes (meg@simula.no)
#
# First added:  2012-01-12
# Last changed: 2013-03-11

__all__ = []

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Replacement constructor for Mesh class. We cannot create a Python
# subclass of cpp.Mesh since then isinstance(UnitSquare, Mesh) would
# not be true.

def __init__(self, *args, **kwargs):
    """Create a mesh from a filename or a geometry.

    A mesh may be created from a given filename, which should contain
    mesh data stored in DOLFIN XML format:

    .. code-block:: python

        mesh = Mesh("mesh.xml")

    An empty mesh may be created as follows:

    .. code-block:: python

        mesh = Mesh()

    A copy of a mesh may be created as follows:

    .. code-block:: python

        mesh_copy = Mesh(mesh)

    A mesh may also be created from a CSG geometry and a resolution:

    .. code-block:: python

        mesh = Mesh(geometry, resolution)

    """

    # Reuse C++ doc-string too
    __doc__ = cpp.Mesh.__doc__

    # Initialize base class
    cpp.Mesh.__cppinit__(self, *args, **kwargs)

# Change mesh constructor
cpp.Mesh.__cppinit__ = cpp.Mesh.__init__
cpp.Mesh.__init__ = __init__


# Attach ipython notebook svg rendering function for 1D and 2D meshes
def mesh2svg(mesh):
    c = mesh.cells()
    # TODO: Abort if c.shape[0] is too large?

    x = mesh.coordinates()
    d = x.shape[1]

    cellname = mesh.ufl_cell().cellname()
    assert (cellname == "triangle" and d == 2 and c.shape[1] == 3)

    mesh_origin = [x[:,k].min() for k in range(d)]
    x_min= [x[:,k].min() for k in range(d)]
    x_max= [x[:,k].max() for k in range(d)]
    mesh_width = x_max[0]-x_min[0]
    mesh_height = x_max[1]-x_min[1]
    display_width = 600.0
    scale = float(display_width / mesh_width)

    polyline = '<polyline points="%s" style="fill:none;stroke:black;stroke-width:2" />'
    polygons = []
    nv = c.shape[1]
    p = [None]*(nv+1)
    for i in range(c.shape[0]):
        for j in range(nv):
            x0, x1 = x[c[i,j],:]
            x0 = x0 - x_min[0]
            x1 = x_max[1] - x1
            p[j] = "%d,%d" % (int(scale*x0), int(scale*x1))
        p[nv] = p[0]
        polygons.append(polyline % " ".join(p))
    return '<svg xmlns="http://www.w3.org/2000/svg" version="1.1">\n%s\n</svg>' % '\n'.join(polygons)
cpp.Mesh._repr_svg_ = mesh2svg

