/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "sine_mixer.h"
#include "viewwin.h"
#include "input.h"
#include "console.h"
#include "dingl.h"
#include <iostream>
#include <math.h>
using namespace std;

extern console cons;

sine_mixer::sine_mixer () : sine_levels ("sine_levels") {
  num_harmonics (sine_levels.nlev);
}

sine_mixer::~sine_mixer () {}

void sine_mixer::num_harmonics (int h) {
  nharmonics = h;
  if (nharmonics < 1) nharmonics = 1;
  prep_harmonics ();
}

void sine_mixer::prep_harmonics () {

  harmonics.clear ();

  for (int i = 0, j = 1; i < nharmonics; ++i, ++j) {
    harmonics.push_back (vector<float>());
    vector<float>& harmonic = harmonics[i];
    harmonic.clear ();
    float dx = 2 * pi * j / (NUM_SINE_SAMPLES - 1);
    float x = 0;
    for (int p = 0; p < NUM_SINE_SAMPLES; ++p) {
      harmonic.push_back (sin(x));
      x += dx;
    }
  }

  nonorm.clear ();
  for (int i = 0; i < NUM_SINE_SAMPLES; ++i) nonorm.push_back(0);

  mix ();

}

void sine_mixer::mix () {

  for (int p = 0, q = nonorm.size (); p < q; ++p) nonorm[p] = 0;

  for (int i = 0; i < nharmonics; ++i) {
    float lev = sine_levels.values[i];
    if (lev != 0) {
      vector<float>& harmonic = harmonics[i];
      for (int p = 0, q = nonorm.size (); p < q; ++p) nonorm[p] += (lev * harmonic[p]);
    }
  }

  normalise ();

}

void sine_mixer::normalise () {
  norm = nonorm;
  int n = norm.size ();
  if (n) {
    float max = fabs (norm [0]);
    for (int i = 0; i < n; ++i) {
      float v = fabs (norm[i]);
      if (v > max) max = v;
    }
    if (max != 0) for (int p = 0; p < n; ++p) norm[p] /= max;
  }
}

bool sine_mixer::handle_input () {
  bool result = sine_levels.handle_input ();
  if (result) {mix (); return true;}
  return false;
}

void sine_mixer::draw () {
  sine_levels.draw ();
}

void sine_mixer::make_curve (multi_curve& c) {
  float dx = 1. / (NUM_SINE_SAMPLES - 1);
  float x = 0;
  c.clear (0);
  for (int i = 0, j = norm.size(); i < j; ++i) {
    float y = norm[i];
    c.add_vertex (x, y);
    c.add_left_tangent (x, y);
    c.add_right_tangent (x, y);
    x += dx;
  }
  c.evaluate ();
}

void sine_mixer::set_sine_samples (int s) {

  NUM_SINE_SAMPLES = s;
  if (NUM_SINE_SAMPLES < MIN_SINE_SAMPLES) NUM_SINE_SAMPLES = MIN_SINE_SAMPLES;

  extern sine_mixer sinemixer;
  sinemixer.prep_harmonics ();

}
