// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __UnitManager__
#define __UnitManager__

// Class UnitManager helps you to have control of all units built.

#include <iostream>
#include <list>
#include <map>
#include <string>
using namespace std;
#include "Puma/ErrorStream.h"

namespace Puma {


class Unit;
class Source;
class Tokenizer;

class UnitManager {
public:
  typedef map<string, Unit*> UMap;

private:
  ErrorStream *_err;        // For error messages.
  list<Unit*> _nonames;     // Noname units list.
  mutable UMap _umap;       // Named units, map supports fast lookup by name
  typedef UMap::value_type UMapPair;
  Tokenizer *_tokenizer;    // File scanner.

public:
  UnitManager (ErrorStream &err) : _err (&err), _tokenizer (0) {}

  // Destructor: Delete all units.
  virtual ~UnitManager ();

  // Get the unit hash table.
  UMap &getTable () const { return _umap; }
  typedef UMap::iterator UMapIter;

  // Get/Set the file scanner that is used by newUnit
  Tokenizer *tokenizer () const { return _tokenizer; }
  void tokenizer (Tokenizer *t) { _tokenizer = t; }

  // Add a unit to manager.
  void addUnit (Unit *unit);

  // Remove (and destroy) a unit from hash table. If the name
  // is NULL remove all (but not the nonames).
  void removeUnit (const char *name = (const char*) 0, bool = false,
                   bool destroy = true) const;

  // Empty a unit. Don't remove from hash table.
  // Preserve the last save location information.
  void closeUnit (const char *name = (const char*) 0, bool = false) const;

  // Build and add a new unit.
  Unit *newUnit (const char *, Source *in = (Source*) 0, bool = false);

  // Get a unit by name.
  Unit *getUnit (const char *name, bool = false) const;

  // Print given unit to the given outstream. If no
  // name is given, print all units.
  void print (const char *name = (const char*) 0, bool = false,
              ostream &out = cout) const;

  // Destroy all noname units.
  void removeNonames ();

  // Initialize the unit manager and the managed
  // units for the next parse process.
  void init ();
};


} // namespace Puma

#endif /* __UnitManager__ */
