/*
 *  acm : an aerial combat simulator for X
 *  Copyright (C) 1991-1998  Riley Rainey
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include "pm.h"
#include "../util/memory.h"
#include "missile.h"
#include "init.h"
#include "inventory.h"
#include "weapon.h"
#include "sounds.h"
#include "gear.h"
#include "../util/units.h"

#define aim120_IMPORT
#include "aim120.h"

static int hasFired[manifest_MAXPLAYERS];
static int count[manifest_MAXPLAYERS];

/*
 *  AIM-120 selection function
 *
 *  A selection function normally determines whether there are any weapons
 *  of this type on-board.  If so, and the weapon system is functional
 *  (in other words, undamaged) then return 1; otherwise return 0.
 */

static int
select_aim120(craft * c)
{
	hasFired[c->pIndex] = 0;
	count[c->pIndex] = weapon_countOrdinance(c, weapon_AIM120);
	return 1;

}


#ifdef FIXME_NOT_USED
static double
Rmax(craft *c)
{
	return 40.0 * units_NmToFeetFactor;
}
#endif


#ifdef FIXME_NOT_USED
static void
computeASECircleParameters(craft *c,
						   double * ASE_diameter_millirad,
						   double * ASE_dot_az_millirad,
						   double * ASE_dot_el_millirad)
{
	double range_feet, rmax_feet = Rmax(c), hs, omegay, omegap, h;
	VPoint v, t, vrel;
	craft *target;
	radarInfo *pr = NULL, *p;
	int i;

	/*
	 * find target information entry in the radar info table
	 */

	for (i = 0, p = c->rinfo; i < c->rtop; ++i, ++p) {
		if (c->curRadarTarget  == p->targetID ) {
			pr = p;
			break;
		}
	}

	/*
	 *  Without a radar lock, display only the ASE circle.
	 *
	 *  We represent this case by passing the ASE diameter as a negative value.
	 */

	if ( c->curRadarTarget == -1 || pr == NULL) {
		*ASE_diameter_millirad = -130.0;
		*ASE_dot_az_millirad = 0.0;
		*ASE_dot_az_millirad = 0.0;
		return;
	}

	target = &ptbl[c->curRadarTarget];

	/*
	 *  Range greater than Rmax? Place aircraft and target on
	 *  a lead collision course.
	 */

	VSub(&target->Cg, &c->Cg, &v);

	/*
	 * t becomes relative position of target wrt to aircraft body axes (feet)
	 */

	t = pr->rel;

	VReverseTransform_( &v, &c->trihedral, &vrel );

	/*
	 *  If range is less than Rmax, compute lead collision based on aircraft
	 *  plus missile velocity, not just our aircraft's velocity.
	 */

	if (range_feet < rmax_feet) {

		vrel.x -= 1000.0;  /* 1,000 fps is just a SWAG */
		
	}

	hs = t.x * t.x + t.y * t.y;

	/*
	 *  Omega values are rates of azimuth and elevation changes (rad/sec)
	 */

	omegay = (vrel.y * t.x - vrel.x * t.y) / hs;

	omegap = (vrel.z * hs - t.z * (vrel.x * t.x + vrel.y * t.y)) /
		(sqrt(hs) * (hs + t.z * t.z));

	/*
	 *  Just SWAGs here ...
	 */

	*ASE_diameter_millirad = 130.0;
	*ASE_dot_az_millirad = units_RADtoDEG(omegay) * 150.0;
	*ASE_dot_el_millirad = units_RADtoDEG(omegap) * 150.0;

	h = sqrt (
		*ASE_dot_az_millirad * *ASE_dot_az_millirad +
		*ASE_dot_el_millirad * *ASE_dot_el_millirad
		);

	/*
	 *  Limit ASE "dot" to position just outside the circle
	 */

	if (h > 70.0) {
		*ASE_dot_az_millirad = *ASE_dot_az_millirad * 70.0 / h;
		*ASE_dot_el_millirad = *ASE_dot_el_millirad * 70.0 / h;
	}

}
#endif


static int
update_aim120(craft * c)
{
	int i;

	if( ! hasFired[c->pIndex] )
		return 1;
	
	hasFired[c->pIndex] = 0;  /* reset fire request */

	/*
	 *  Missile won't fire if we have "Weight on wheels"
	 *  or if we run out of ammunition.
	 */

	if( gear_someWheelGroundContact(c)
	|| count[c->pIndex] <= 0 )
		return 1;

	/* Get station from which to launch the missile: */
	i = weapon_getReadyStation(c, weapon_AIM120);
	if (i < 0){
		fprintf(stderr, "Oops. Can't find an AIM-120\n");
		return 1;
	}

	/*
	 *  Decrement missiles counter.
	 *  In arcade mode, we never run out of ammunition
	 */

	if (arcadeMode == 0) {
		c->station[i].id = -1;
		count[c->pIndex]--;
	}

	/* Launch missile from station i: */
	missile_fire(c, i);
	sounds_playSound(c, sounds_MissileLaunch, FALSE);

	return 1;
}

static int
fire_aim120(craft * c)
{
	hasFired[c->pIndex] = 1;
	return 1;
}


#ifdef FIXME_NOT_USED
static double
missileTimeToImpact (craft * c, craftType * w)
{
	double v, t, root1, root2, r, a1, d, n;

	v = c->VT;
	a1 = (w->maxThrust - 0.5 * c->rho * w->CDOrigin * v * v)
		/ (w->emptyWeight + w->maxFuel) * units_earth_g;

	if (c->curRadarTarget >= 0 && a1 >= 0.0) {

		d = c->targetDistance;
		r = c->targetClosure;

		n = r * r + 2.0 * a1 * d;
		if (n > 0) {
			n = sqrt(n);
			root1 = (-r + n) / a1;
			root2 = (-r - n) / a1;
			if (root1 >= 0.0)
				if (root2 >= 0.0)
					if (root1 < root2)
						t = root1;
					else
						t = root2;
				else
					t = root1;
			else if (root2 >= 0.0)
				t = root2;
			else
				t = -1.0;
		}
		else
			t = -1.0;
	}

	else {
		t = -1.0;
	}

	return t;

}
#endif


/*
 *  AIM-120 display function
 *
 *  Update the HUD display strings associated with this weapon system.
 *  c = the aircraft.
 *  w = the missile's description record.
 *  dummy1, dummy2 = ignored, not set.
 *
 *  This code may be called by drones, so the viewer may be NULL.
 *
 *  Return a nonzero value if have a reasonable chance of scoring a kill.
 */

static int
display_aim120(craft * c, craftType * w, viewer * unused, int dummy1, int dummy2)
{

	char      s[16];
	double    d, a1, v, r, root1, root2, n, t;
	int       target;

	sprintf(s, "%d %s", count[c->pIndex], weapon_idToName(weapon_AIM120));
	strcpy(c->leftHUD[3], s);

/*
 *  Compute time to target t. Gives < 0.0 if no target available or not
 *  reachable.
 */

	target = c->curRadarTarget;
	v = VMagnitude(&c->Cg);
	a1 = (w->maxThrust - 0.5 * c->air.rho * w->CDOrigin * v * v)
		/ (w->emptyWeight + w->maxFuel) * units_earth_g;

	if (target >= 0 && a1 >= 0.0) {

		d = c->targetDistance;
		r = c->targetClosure;

		n = r * r + 2.0 * a1 * d;
		if (n > 0) {
			n = sqrt(n);
			root1 = (-r + n) / a1;
			root2 = (-r - n) / a1;
			if (root1 >= 0.0)
				if (root2 >= 0.0)
					if (root1 < root2)
						t = root1;
					else
						t = root2;
				else
					t = root1;
			else if (root2 >= 0.0)
				t = root2;
			else
				t = -1.0;
		}
		else
			t = -1.0;
	}

	else
		t = -1.0;

/*
 *  Update HUD display strings.
 */

	if (t < 0.0)
		sprintf(s, "ARM      --");
	else if (target >= 0 && t >= (w->armDelay + 0.5) && t <= 90.0)
		sprintf(s, "LOCKED   %d", (int) (t + 0.5));
	else if (t <= 90.0)
		sprintf(s, "IN RANGE %d", (int) (t + 0.5));
	else
		sprintf(s, "ARM      %d", (int) (t + 0.5));
	
	strcpy(c->leftHUD[2], s);

	strcpy(c->leftHUD[4], "");

/*
 *  Return TRUE if we are recommending a missile shot.
 */

	return target >= 0 && t >= (w->armDelay + 0.5) && t <= 90.0;
}


static weapon_Type aim120Desc =
{
	select_aim120,				/* select */
	update_aim120,				/* update */
	display_aim120,				/* display procedure */
	fire_aim120,					/* fire */
	(int (*)(craft *)) NULL,	/* fire button release */
};

weapon_Type *
aim120_new(void)
{

	craftType *c;
	FILE     *f;
	dis_entity_type em1 =
	{2, 1, 225, 1, 2, 1, 0};
	dis_entity_type em2 =
	{0, 0, 0, 0, 0, 0, 0};

	c = inventory_craftTypeNew(NULL);
	c->name = memory_strdup( weapon_idToName(weapon_AIM120) );

	c->entityType = em1;
	c->altEntityType = em2;

	aim120Desc.w = c;

	c->CDOrigin = 0.2;			/* 5" radius of body */
	c->CDFactor = -2.56694;

	c->CDBOrigin = 0.0;
	c->CDBFactor = 0.0;

	VIdentMatrix(&(c->I));
	c->I.m[0][0] = 0.0;
	c->I.m[1][1] = 0.0;
	c->I.m[2][2] = 0.0;
	c->cmSlope = -1.88;

	c->wingS = 1.0;

/*
 *  Assume 150.0 lb of weight is fuel and that it burns for about 4 seconds.
 *  That yields a fuel burn rate of 40 lb/s.
 */

	c->emptyWeight = 100.0;
	c->maxFuel = 234.0;
	c->maxThrust = 2500.0;
	c->spFuelConsump = 16.0;  /* Isp = 220,  SFC = 3600.0 / Isp */

/*
 *  Three seconds arm delay:
 */

	c->armDelay = 3.0;

	f = init_fopen("missiles/aim9.obv", "r");  /* FIXME: missing custom image */
	c->object = VReadObject(f);
	fclose(f);

	return &aim120Desc;
}
