// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program computes a projection of a function
// between two mixed finite element spaces.

#include <dolfin.h>
#include "generated_code/ProjectionMixed.h"

using namespace dolfin;
using namespace ProjectionMixed;

class FScalar: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = 3.14159*x[0]*x[1];
  }
};

class FVector: public Expression 
{
public:
  FVector() : Expression(2) {}
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 2.0 * x[0];
    values[1] = 3.0 * x[1];
  }
};

class FTensor: public Expression 
{
public:
  FTensor() : Expression(2,2) {}
  void eval(Array<double>& values, const Array<double>& x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 0.1 * x[0] * x[1];
    values[1] = 0.2 * x[0] * x[1];
    values[2] = 0.3 * x[0] * x[1];
    values[3] = 0.4 * x[0] * x[1];
  }
};

int main(int argc, char ** argv)
{
  UnitSquare mesh(10, 10);

  BilinearForm::TrialSpace V(mesh);

  CoefficientSpace_ft FT(mesh);
  CoefficientSpace_fv FV(mesh);
  CoefficientSpace_fs FS(mesh);

  FTensor ft;
  FVector fv;
  FScalar fs;

  Function u(V);

  BilinearForm a(V, V);
  LinearForm L(V);

  L.ft = ft;
  L.fv = fv;
  L.fs = fs;

  u = Function(V);
  solve(a==L, u); 

  // Get subfunctions
  Function ut = u[0];
  Function uv = u[1];
  Function us = u[2];
  
  // Write subfunctions to file
  File ut_file("ut.pvd");
  ut_file << ut;
  File uv_file("uv.pvd");
  uv_file << uv;
  File us_file("us.pvd");
  us_file << us;

  return 0;
}

