/*
 * Strawberry Music Player
 * Copyright 2018-2021, Jonas Kvinge <jonas@jkvinge.net>
 *
 * Strawberry is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Strawberry is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Strawberry.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"

#include <cstring>
#include <glib.h>

#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>

#include <QObject>
#include <QMetaObject>
#include <QCoreApplication>
#include <QStandardPaths>
#include <QtConcurrent>
#include <QString>
#include <QDir>
#include <QFile>
#include <QAbstractEventDispatcher>

#include "core/logging.h"
#include "utilities/envutils.h"

#ifdef HAVE_MOODBAR
#  include "ext/gstmoodbar/gstmoodbarplugin.h"
#endif

#include "gststartup.h"

GThread *GstStartup::kGThread = nullptr;

gpointer GstStartup::GLibMainLoopThreadFunc(gpointer) {

  qLog(Info) << "Creating GLib main event loop.";

  GMainLoop *gloop = g_main_loop_new(nullptr, false);
  g_main_loop_run(gloop);
  g_main_loop_unref(gloop);

  return nullptr;

}

GstStartup::GstStartup(QObject *parent) : QObject(parent) {

  initializing_ = QtConcurrent::run(&GstStartup::InitializeGStreamer);

  const QMetaObject *mo = QAbstractEventDispatcher::instance(qApp->thread())->metaObject();
  if (mo && strcmp(mo->className(), "QEventDispatcherGlib") != 0 && strcmp(mo->superClass()->className(), "QEventDispatcherGlib") != 0) {
    kGThread = g_thread_new(nullptr, GstStartup::GLibMainLoopThreadFunc, nullptr);
  }

}

GstStartup::~GstStartup() {
  if (kGThread) g_thread_unref(kGThread);
}

void GstStartup::InitializeGStreamer() {

  SetEnvironment();

  gst_init(nullptr, nullptr);
  gst_pb_utils_init();

#ifdef HAVE_MOODBAR
  gstfastspectrum_register_static();
#endif

#ifdef Q_OS_WIN32
  // Use directsoundsink as the default sink on Windows.
  // wasapisink does not support device switching and wasapi2sink has issues, see #1227.
  GstRegistry *reg = gst_registry_get();
  if (reg) {
    if (GstPluginFeature *directsoundsink = gst_registry_lookup_feature(reg, "directsoundsink")) {
      gst_plugin_feature_set_rank(directsoundsink, GST_RANK_PRIMARY);
      gst_object_unref(directsoundsink);
    }
    if (GstPluginFeature *wasapisink = gst_registry_lookup_feature(reg, "wasapisink")) {
      gst_plugin_feature_set_rank(wasapisink, GST_RANK_SECONDARY);
      gst_object_unref(wasapisink);
    }
    if (GstPluginFeature *wasapi2sink = gst_registry_lookup_feature(reg, "wasapi2sink")) {
      gst_plugin_feature_set_rank(wasapi2sink, GST_RANK_SECONDARY);
      gst_object_unref(wasapi2sink);
    }
  }
#endif

}

void GstStartup::SetEnvironment() {

#ifdef USE_BUNDLE

  const QString app_path = QCoreApplication::applicationDirPath();

  // Set plugin root path
  QString plugin_root_path;
#  if defined(Q_OS_MACOS)
  plugin_root_path = QDir::cleanPath(app_path + "/../PlugIns");
#  elif defined(Q_OS_UNIX)
  plugin_root_path = QDir::cleanPath(app_path + "/../plugins");
#  elif defined(Q_OS_WIN32)
  plugin_root_path = app_path;
#  endif

  // Set GIO module path
  const QString gio_module_path = plugin_root_path + "/gio-modules";

  // Set GStreamer plugin scanner path
  QString gst_plugin_scanner;
#  if defined(Q_OS_UNIX)
  gst_plugin_scanner = plugin_root_path + "/gst-plugin-scanner";
#  endif

  // Set GStreamer plugin path
  QString gst_plugin_path;
#  if defined(Q_OS_WIN32)
  gst_plugin_path = plugin_root_path + "/gstreamer-plugins";
#  else
  gst_plugin_path = plugin_root_path + "/gstreamer";
#  endif

  if (!gio_module_path.isEmpty()) {
    if (QDir(gio_module_path).exists()) {
      qLog(Debug) << "Setting GIO module path to" << gio_module_path;
      Utilities::SetEnv("GIO_EXTRA_MODULES", gio_module_path);
    }
    else {
      qLog(Error) << "GIO module path" << gio_module_path << "does not exist.";
    }
  }

  if (!gst_plugin_scanner.isEmpty()) {
    if (QFile::exists(gst_plugin_scanner)) {
      qLog(Debug) << "Setting GStreamer plugin scanner to" << gst_plugin_scanner;
      Utilities::SetEnv("GST_PLUGIN_SCANNER", gst_plugin_scanner);
    }
    else {
      qLog(Error) << "GStreamer plugin scanner" << gst_plugin_scanner << "does not exist.";
    }
  }

  if (!gst_plugin_path.isEmpty()) {
    if (QDir(gst_plugin_path).exists()) {
      qLog(Debug) << "Setting GStreamer plugin path to" << gst_plugin_path;
      Utilities::SetEnv("GST_PLUGIN_PATH", gst_plugin_path);
      // Never load plugins from anywhere else.
      Utilities::SetEnv("GST_PLUGIN_SYSTEM_PATH", gst_plugin_path);
    }
    else {
      qLog(Error) << "GStreamer plugin path" << gst_plugin_path << "does not exist.";
    }
  }

#endif  // USE_BUNDLE


#if defined(Q_OS_WIN32) || defined(Q_OS_MACOS)
  QString gst_registry_filename = QStandardPaths::writableLocation(QStandardPaths::AppLocalDataLocation) + QString("/gst-registry-%1-bin").arg(QCoreApplication::applicationVersion());
  qLog(Debug) << "Setting GStreamer registry file to" << gst_registry_filename;
  Utilities::SetEnv("GST_REGISTRY", gst_registry_filename);
#endif

}
