use ui;
use graphics;
use core:geometry;
use core:lang;
use lang:bs:macro;

/**
 * A set of global variables.
 */
class Globals extends Data {
	init(GlobalVar[] globals) {
		init() {}

		for (g in globals) {
			var obj = unsafe:RawPtr:fromGlobal(g);
			content << Var(g.name, obj, StackPointer());
		}

		// Sort alphabetically.
		content.sort((a, b) => a.name < b.name);

		invalidateSize();
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		Float width = rect.size.w;
		Point at = pos + compositeBorder;

		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			// Align the text a bit nicer on the y-axis, to line up the baselines...
			g.draw(v.text, dataFg, at + Point(0, dataBorder.h));

			v.value.pos = at + Point(namesWidth + dataWidth - dataSz.w, 0);
			v.value.draw(g, false);

			at.y += max(nameSz.h, dataSz.h) + dataBorder.h;
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.value.drawLinks(g, active);
		}
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		// Note: 'object' will point to this object due to how the setup is done in the Workspace.

		for (v in content) {
			v.update(work);
		}

		invalidateSize();
	}

	// We don't need to deal with writes to any of our content, we only contain "pointers" to the actual memory.
	void clearAccess() : override {}
	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) : override {}

private:
	class Var on Render {
		Str name;
		Text text;
		Data value;
		unsafe:RawPtr obj;

		init(Str name, unsafe:RawPtr obj, Data value) {
			init() {
				name = name;
				text = Text(name, variableFont);
				value = value;
				obj = obj;
			}
		}

		void update(Workspace ws) {
			value.update(ws, obj, 0);
		}
	}

	Var[] content;
	Float namesWidth;
	Float dataWidth;
	Float height;

	Size computeSize() : override {
		// TODO: It would probably be nice to recurse into
		// the values that are themselves variables and render them
		// more compactly.
		namesWidth = 0;
		dataWidth = 0;
		height = 0;
		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			namesWidth = max(namesWidth, nameSz.w + dataBorder.w);
			dataWidth = max(dataWidth, dataSz.w);
			height += max(nameSz.h, dataSz.h) + dataBorder.h;
		}

		height -= dataBorder.h;
		Size(namesWidth + dataWidth, height) + compositeBorder*2;
	}
}
