import io


def nucleotide_complements_table():
    # A nice list of complements can be found at:
    # http://www.reverse-complement.com/ambiguity.html
    complements = dict(
        A="T",
        C="G",
        G="C",
        T="A",
        a="t",
        c="g",
        g="c",
        t="a",
        U="A",
        u="a",
        # R, purine (A, G) vs Y, pyrimidine (C, T)
        R="Y",
        Y="R",
        r="y",
        y="r",
        # K, keto (G, T) vs A, amino (A, C)
        K="M",
        M="K",
        k="m",
        m="k",
        # B, not A, vs V, not T
        B="V",
        V="B",
        b="v",
        v="b",
        # D, not C vs H, not G
        D="H",
        H="D",
        d="h",
        h="d",
        # S, W and N's complements are the same. So they are not explicitly
        # included above
    )

    table = []
    for i in range(256):
        c = chr(i)
        if c in complements:
            table.append(f"'{complements[c]}'")
        else:
            table.append(i)
    return table


def make_table(variable_name, table, columns=16):
    out = io.StringIO()
    out.write(variable_name + " = {")
    for i, literal in enumerate(table):
        if i % columns == 0:
            out.write("\n    ")
        out.write(f"{literal:3}, ")
    out.write("\n")
    out.write("};\n")
    return out.getvalue()


def main():
    with open("src/dnaio/_conversions.h", "wt", encoding="utf-8") as out:
        out.write(
            "// This file is generated by generate_conversion_tables.py\n"
            "// Please do not edit manually.\n\n"
        )
        out.write(
            make_table(
                "static const char NUCLEOTIDE_COMPLEMENTS[256]",
                nucleotide_complements_table(),
            )
        )


if __name__ == "__main__":
    main()
