//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================
#include <vtkm/cont/ColorTable.h>
#include <vtkm/cont/Logging.h>

#include <vtkm/cont/vtkm_cont_export.h>

#include <algorithm>
#include <cctype>
#include <mutex>
#include <set>
#include <sstream>

namespace vtkm
{
namespace cont
{
namespace internal
{

static vtkm::cont::ColorTable::Preset DEFAULT_PRESET = vtkm::cont::ColorTable::Preset::Viridis;

std::string GetColorSpaceString(vtkm::ColorSpace space)
{
  switch (space)
  {
    case vtkm::ColorSpace::RGB:
      return std::string("RGB");
    case vtkm::ColorSpace::HSV:
      return std::string("HSV");
    case vtkm::ColorSpace::HSVWrap:
      return std::string("HSVWrap");
    case vtkm::ColorSpace::Lab:
      return std::string("Lab");
    case vtkm::ColorSpace::Diverging:
      return std::string("Diverging");
  }

  throw vtkm::cont::ErrorBadValue("Encountered invalid color space.");
}

vtkm::ColorSpace GetColorSpaceEnum(const std::string& colorSpaceString)
{
  std::string spaceString = colorSpaceString;

  //convert to lower case
  std::transform(spaceString.begin(), spaceString.end(), spaceString.begin(), [](char c) {
    return static_cast<char>(std::tolower(static_cast<unsigned char>(c)));
  });

  if (spaceString == "rgb")
  {
    return vtkm::ColorSpace::RGB;
  }

  if (spaceString == "hsv")
  {
    return vtkm::ColorSpace::HSV;
  }

  if ((spaceString == "hsv_wrap") || (spaceString == "hsv-wrap") || (spaceString == "hsvwrap"))
  {
    return vtkm::ColorSpace::HSVWrap;
  }

  if (spaceString == "lab")
  {
    return vtkm::ColorSpace::Lab;
  }

  if (spaceString == "diverging")
  {
    return vtkm::ColorSpace::Diverging;
  }

  std::stringstream message;
  message << "Invalid color space name: '" << colorSpaceString << "'";
  throw vtkm::cont::ErrorBadValue(message.str());
}
}
}
} // vtkm::cont::internal

namespace
{

bool IStringEqual(const std::string& str1, const std::string& str2)
{
  if (str1.size() != str2.size())
  {
    return false;
  }

  auto itr1 = str1.begin();
  auto itr2 = str2.begin();
  while ((itr1 != str1.end()) && (itr2 != str2.end()))
  {
    if (std::tolower(*itr1) != std::tolower(*itr2))
    {
      return false;
    }
    ++itr1;
    ++itr2;
  }

  return true;
}

struct ColorTablePreset
{
  vtkm::cont::ColorTable::Preset Preset;
  std::string Name;
  vtkm::ColorSpace ColorSpace;
  vtkm::Vec<double, 3> NanColor;
  std::vector<double> RGBPoints;
  std::vector<double> AlphaPoints;

  VTKM_CONT
  ColorTablePreset(vtkm::cont::ColorTable::Preset preset,
                   std::string&& name,
                   vtkm::ColorSpace colorSpace,
                   vtkm::Vec<double, 3>&& nanColor,
                   std::vector<double>&& rgbPoints,
                   std::vector<double>&& alphaPoints = { 0.0, 1.0, 0.5, 0.0, 1.0, 1.0, 0.5, 0.0 })
    : Preset(preset)
    , Name(std::move(name))
    , ColorSpace(colorSpace)
    , NanColor(std::move(nanColor))
    , RGBPoints(std::move(rgbPoints))
    , AlphaPoints(std::move(alphaPoints))
  {
  }

  VTKM_CONT vtkm::cont::ColorTable MakePreset() const
  {
    return vtkm::cont::ColorTable(
      this->Name, this->ColorSpace, this->NanColor, this->RGBPoints, this->AlphaPoints);
  }
};

VTKM_CONT void BuildColorTablePresetsVector(std::vector<ColorTablePreset>& presets)
{
  // clang-format off
  presets = std::vector<ColorTablePreset>{
    { vtkm::cont::ColorTable::Preset::CoolToWarm,
      "Cool to Warm",
      vtkm::ColorSpace::Diverging,
      { 1, 1, 0 },
      {
        0, 0.23137254902, 0.298039215686, 0.752941176471,
        0.5, 0.865, 0.865, 0.865,
        1, 0.705882352941, 0.0156862745098, 0.149019607843
      }
    },
    { vtkm::cont::ColorTable::Preset::CoolToWarmExtended,
      "Cool to Warm Extended",
      vtkm::ColorSpace::Lab,
      { 0.25, 0, 0 },
      {
        0, 0, 0, 0.34902,
        0.03125, 0.039216, 0.062745, 0.380392,
        0.0625, 0.062745, 0.117647, 0.411765,
        0.09375, 0.090196, 0.184314, 0.45098,
        0.125, 0.12549, 0.262745, 0.501961,
        0.15625, 0.160784, 0.337255, 0.541176,
        0.1875, 0.2, 0.396078, 0.568627,
        0.21875, 0.239216, 0.454902, 0.6,
        0.25, 0.286275, 0.5215689999999999, 0.65098,
        0.28125, 0.337255, 0.592157, 0.7019609999999999,
        0.3125, 0.388235, 0.654902, 0.74902,
        0.34375, 0.466667, 0.737255, 0.819608,
        0.375, 0.572549, 0.819608, 0.878431,
        0.40625, 0.654902, 0.866667, 0.9098039999999999,
        0.4375, 0.752941, 0.917647, 0.941176,
        0.46875, 0.823529, 0.956863, 0.968627,
        0.48, 0.988235, 0.960784, 0.901961,
        0.5, 0.941176, 0.984314, 0.988235,
        0.52, 0.988235, 0.945098, 0.85098,
        0.54, 0.980392, 0.898039, 0.784314,
        0.5625, 0.968627, 0.835294, 0.698039,
        0.59375, 0.94902, 0.733333, 0.588235,
        0.625, 0.929412, 0.65098, 0.509804,
        0.65625, 0.9098039999999999, 0.564706, 0.435294,
        0.6875, 0.878431, 0.458824, 0.352941,
        0.71875, 0.839216, 0.388235, 0.286275,
        0.75, 0.760784, 0.294118, 0.211765,
        0.78125, 0.7019609999999999, 0.211765, 0.168627,
        0.8125, 0.65098, 0.156863, 0.129412,
        0.84375, 0.6, 0.09411799999999999, 0.09411799999999999,
        0.875, 0.54902, 0.066667, 0.098039,
        0.90625, 0.501961, 0.05098, 0.12549,
        0.9375, 0.45098, 0.054902, 0.172549,
        0.96875, 0.4, 0.054902, 0.192157,
        1, 0.34902, 0.070588, 0.211765
      }
    },
    { vtkm::cont::ColorTable::Preset::Viridis,
      "Viridis",
      vtkm::ColorSpace::Lab,
      { 1, 0, 0 },
      {
        0.000000, 0.282365, 0.000000, 0.331010,
        0.003922, 0.284420, 0.000000, 0.337034,
        0.007843, 0.286409, 0.000000, 0.343000,
        0.011765, 0.288328, 0.000000, 0.348901,
        0.015686, 0.290179, 0.002655, 0.354738,
        0.019608, 0.291960, 0.007928, 0.360509,
        0.023529, 0.293673, 0.013524, 0.366211,
        0.027451, 0.295315, 0.019447, 0.371842,
        0.031373, 0.296886, 0.025709, 0.377403,
        0.035294, 0.298386, 0.032320, 0.382889,
        0.039216, 0.299816, 0.039285, 0.388299,
        0.043137, 0.301173, 0.046264, 0.393634,
        0.047059, 0.302457, 0.052924, 0.398888,
        0.050980, 0.303670, 0.059348, 0.404063,
        0.054902, 0.304808, 0.065577, 0.409155,
        0.058824, 0.305873, 0.071647, 0.414162,
        0.062745, 0.306864, 0.077587, 0.419085,
        0.066667, 0.307780, 0.083418, 0.423920,
        0.070588, 0.308622, 0.089155, 0.428668,
        0.074510, 0.309391, 0.094811, 0.433324,
        0.078431, 0.310084, 0.100399, 0.437890,
        0.082353, 0.310702, 0.105925, 0.442363,
        0.086275, 0.311245, 0.111399, 0.446742,
        0.090196, 0.311714, 0.116827, 0.451026,
        0.094118, 0.312107, 0.122211, 0.455214,
        0.098039, 0.312426, 0.127560, 0.459306,
        0.101961, 0.312671, 0.132872, 0.463299,
        0.105882, 0.312843, 0.138154, 0.467195,
        0.109804, 0.312940, 0.143406, 0.470991,
        0.113725, 0.312965, 0.148633, 0.474688,
        0.117647, 0.312917, 0.153833, 0.478285,
        0.121569, 0.312798, 0.159008, 0.481782,
        0.125490, 0.312608, 0.164161, 0.485180,
        0.129412, 0.312348, 0.169292, 0.488476,
        0.133333, 0.312019, 0.174400, 0.491673,
        0.137255, 0.311623, 0.179486, 0.494771,
        0.141176, 0.311160, 0.184552, 0.497771,
        0.145098, 0.310631, 0.189596, 0.500670,
        0.149020, 0.310039, 0.194618, 0.503474,
        0.152941, 0.309384, 0.199620, 0.506180,
        0.156863, 0.308668, 0.204600, 0.508790,
        0.160784, 0.307893, 0.209559, 0.511307,
        0.164706, 0.307059, 0.214495, 0.513729,
        0.168627, 0.306170, 0.219410, 0.516061,
        0.172549, 0.305227, 0.224301, 0.518300,
        0.176471, 0.304231, 0.229170, 0.520452,
        0.180392, 0.303184, 0.234016, 0.522516,
        0.184314, 0.302089, 0.238839, 0.524495,
        0.188235, 0.300947, 0.243638, 0.526391,
        0.192157, 0.299761, 0.248411, 0.528205,
        0.196078, 0.298533, 0.253162, 0.529941,
        0.200000, 0.297265, 0.257887, 0.531597,
        0.203922, 0.295958, 0.262587, 0.533179,
        0.207843, 0.294614, 0.267264, 0.534688,
        0.211765, 0.293235, 0.271914, 0.536125,
        0.215686, 0.291826, 0.276540, 0.537494,
        0.219608, 0.290387, 0.281138, 0.538796,
        0.223529, 0.288920, 0.285713, 0.540035,
        0.227451, 0.287426, 0.290261, 0.541210,
        0.231373, 0.285909, 0.294783, 0.542327,
        0.235294, 0.284368, 0.299280, 0.543386,
        0.239216, 0.282809, 0.303752, 0.544390,
        0.243137, 0.281231, 0.308199, 0.545341,
        0.247059, 0.279638, 0.312619, 0.546241,
        0.250980, 0.278029, 0.317015, 0.547093,
        0.254902, 0.276409, 0.321385, 0.547898,
        0.258824, 0.274776, 0.325732, 0.548659,
        0.262745, 0.273134, 0.330055, 0.549377,
        0.266667, 0.271483, 0.334353, 0.550054,
        0.270588, 0.269825, 0.338628, 0.550693,
        0.274510, 0.268163, 0.342880, 0.551296,
        0.278431, 0.266495, 0.347108, 0.551862,
        0.282353, 0.264825, 0.351315, 0.552396,
        0.286275, 0.263153, 0.355500, 0.552898,
        0.290196, 0.261479, 0.359663, 0.553371,
        0.294118, 0.259807, 0.363805, 0.553816,
        0.298039, 0.258135, 0.367927, 0.554233,
        0.301961, 0.256465, 0.372028, 0.554624,
        0.305882, 0.254797, 0.376110, 0.554992,
        0.309804, 0.253132, 0.380174, 0.555335,
        0.313725, 0.251469, 0.384220, 0.555658,
        0.317647, 0.249809, 0.388247, 0.555959,
        0.321569, 0.248155, 0.392257, 0.556239,
        0.325490, 0.246503, 0.396251, 0.556502,
        0.329412, 0.244855, 0.400228, 0.556746,
        0.333333, 0.243211, 0.404192, 0.556972,
        0.337255, 0.241572, 0.408139, 0.557182,
        0.341176, 0.239935, 0.412074, 0.557375,
        0.345098, 0.238302, 0.415994, 0.557553,
        0.349020, 0.236673, 0.419902, 0.557716,
        0.352941, 0.235045, 0.423797, 0.557863,
        0.356863, 0.233418, 0.427681, 0.557997,
        0.360784, 0.231794, 0.431554, 0.558116,
        0.364706, 0.230171, 0.435415, 0.558221,
        0.368627, 0.228546, 0.439268, 0.558311,
        0.372549, 0.226922, 0.443110, 0.558388,
        0.376471, 0.225295, 0.446944, 0.558451,
        0.380392, 0.223665, 0.450770, 0.558500,
        0.384314, 0.222033, 0.454588, 0.558534,
        0.388235, 0.220396, 0.458399, 0.558554,
        0.392157, 0.218753, 0.462203, 0.558560,
        0.396078, 0.217104, 0.466001, 0.558549,
        0.400000, 0.215446, 0.469793, 0.558524,
        0.403922, 0.213781, 0.473580, 0.558483,
        0.407843, 0.212104, 0.477363, 0.558425,
        0.411765, 0.210417, 0.481140, 0.558349,
        0.415686, 0.208716, 0.484915, 0.558255,
        0.419608, 0.207003, 0.488686, 0.558143,
        0.423529, 0.205274, 0.492454, 0.558012,
        0.427451, 0.203530, 0.496219, 0.557860,
        0.431373, 0.201769, 0.499982, 0.557687,
        0.435294, 0.199991, 0.503743, 0.557493,
        0.439216, 0.198194, 0.507502, 0.557276,
        0.443137, 0.196379, 0.511261, 0.557038,
        0.447059, 0.194544, 0.515016, 0.556774,
        0.450980, 0.192688, 0.518773, 0.556484,
        0.454902, 0.190810, 0.522528, 0.556167,
        0.458824, 0.188910, 0.526283, 0.555823,
        0.462745, 0.186988, 0.530037, 0.555451,
        0.466667, 0.185044, 0.533791, 0.555049,
        0.470588, 0.183077, 0.537546, 0.554615,
        0.474510, 0.181088, 0.541301, 0.554149,
        0.478431, 0.179079, 0.545056, 0.553651,
        0.482353, 0.177049, 0.548812, 0.553119,
        0.486275, 0.174998, 0.552566, 0.552552,
        0.490196, 0.172927, 0.556324, 0.551947,
        0.494118, 0.170840, 0.560080, 0.551306,
        0.498039, 0.168735, 0.563837, 0.550623,
        0.501961, 0.166618, 0.567594, 0.549902,
        0.505882, 0.164487, 0.571352, 0.549138,
        0.509804, 0.162350, 0.575110, 0.548333,
        0.513725, 0.160207, 0.578868, 0.547483,
        0.517647, 0.158063, 0.582626, 0.546589,
        0.521569, 0.155921, 0.586383, 0.545648,
        0.525490, 0.153785, 0.590141, 0.544660,
        0.529412, 0.151663, 0.593898, 0.543623,
        0.533333, 0.149559, 0.597654, 0.542537,
        0.537255, 0.147482, 0.601409, 0.541399,
        0.541176, 0.145437, 0.605164, 0.540211,
        0.545098, 0.143435, 0.608916, 0.538969,
        0.549020, 0.141481, 0.612666, 0.537672,
        0.552941, 0.139587, 0.616416, 0.536320,
        0.556863, 0.137764, 0.620162, 0.534912,
        0.560784, 0.136021, 0.623907, 0.533446,
        0.564706, 0.134373, 0.627646, 0.531922,
        0.568627, 0.132832, 0.631384, 0.530340,
        0.572549, 0.131411, 0.635118, 0.528696,
        0.576471, 0.130126, 0.638847, 0.526991,
        0.580392, 0.128989, 0.642573, 0.525224,
        0.584314, 0.128018, 0.646291, 0.523393,
        0.588235, 0.127226, 0.650007, 0.521498,
        0.592157, 0.126631, 0.653715, 0.519538,
        0.596078, 0.126247, 0.657418, 0.517513,
        0.600000, 0.126090, 0.661113, 0.515421,
        0.603922, 0.126173, 0.664801, 0.513261,
        0.607843, 0.126511, 0.668482, 0.511032,
        0.611765, 0.127115, 0.672154, 0.508735,
        0.615686, 0.127996, 0.675818, 0.506368,
        0.619608, 0.129165, 0.679471, 0.503931,
        0.623529, 0.130627, 0.683115, 0.501422,
        0.627451, 0.132388, 0.686749, 0.498842,
        0.631373, 0.134453, 0.690371, 0.496189,
        0.635294, 0.136823, 0.693984, 0.493462,
        0.639216, 0.139497, 0.697583, 0.490663,
        0.643137, 0.142475, 0.701170, 0.487789,
        0.647059, 0.145750, 0.704743, 0.484839,
        0.650980, 0.149322, 0.708303, 0.481814,
        0.654902, 0.153179, 0.711848, 0.478714,
        0.658824, 0.157319, 0.715379, 0.475537,
        0.662745, 0.161731, 0.718894, 0.472282,
        0.666667, 0.166406, 0.722392, 0.468952,
        0.670588, 0.171339, 0.725874, 0.465543,
        0.674510, 0.176516, 0.729338, 0.462055,
        0.678431, 0.181930, 0.732785, 0.458488,
        0.682353, 0.187571, 0.736212, 0.454843,
        0.686275, 0.193430, 0.739620, 0.451118,
        0.690196, 0.199499, 0.743009, 0.447313,
        0.694118, 0.205768, 0.746376, 0.443429,
        0.698039, 0.212229, 0.749723, 0.439464,
        0.701961, 0.218873, 0.753047, 0.435418,
        0.705882, 0.225697, 0.756350, 0.431288,
        0.709804, 0.232688, 0.759629, 0.427079,
        0.713725, 0.239844, 0.762884, 0.422786,
        0.717647, 0.247154, 0.766113, 0.418413,
        0.721569, 0.254613, 0.769318, 0.413957,
        0.725490, 0.262219, 0.772499, 0.409418,
        0.729412, 0.269963, 0.775651, 0.404798,
        0.733333, 0.277841, 0.778778, 0.400093,
        0.737255, 0.285848, 0.781876, 0.395304,
        0.741176, 0.293982, 0.784946, 0.390430,
        0.745098, 0.302235, 0.787986, 0.385473,
        0.749020, 0.310604, 0.790996, 0.380433,
        0.752941, 0.319087, 0.793977, 0.375306,
        0.756863, 0.327678, 0.796926, 0.370097,
        0.760784, 0.336377, 0.799844, 0.364801,
        0.764706, 0.345176, 0.802729, 0.359421,
        0.768627, 0.354077, 0.805582, 0.353955,
        0.772549, 0.363073, 0.808400, 0.348403,
        0.776471, 0.372164, 0.811185, 0.342764,
        0.780392, 0.381348, 0.813934, 0.337038,
        0.784314, 0.390618, 0.816649, 0.331226,
        0.788235, 0.399977, 0.819328, 0.325327,
        0.792157, 0.409416, 0.821970, 0.319343,
        0.796078, 0.418938, 0.824575, 0.313270,
        0.800000, 0.428536, 0.827143, 0.307111,
        0.803922, 0.438211, 0.829672, 0.300864,
        0.807843, 0.447962, 0.832163, 0.294529,
        0.811765, 0.457781, 0.834615, 0.288107,
        0.815686, 0.467673, 0.837028, 0.281595,
        0.819608, 0.477629, 0.839401, 0.274994,
        0.823529, 0.487652, 0.841734, 0.268302,
        0.827451, 0.497734, 0.844026, 0.261524,
        0.831373, 0.507875, 0.846277, 0.254656,
        0.835294, 0.518075, 0.848489, 0.247699,
        0.839216, 0.528326, 0.850659, 0.240654,
        0.843137, 0.538631, 0.852787, 0.233518,
        0.847059, 0.548980, 0.854874, 0.226296,
        0.850980, 0.559379, 0.856921, 0.218982,
        0.854902, 0.569817, 0.858926, 0.211584,
        0.858824, 0.580295, 0.860890, 0.204098,
        0.862745, 0.590812, 0.862814, 0.196523,
        0.866667, 0.601358, 0.864697, 0.188865,
        0.870588, 0.611937, 0.866540, 0.181121,
        0.874510, 0.622540, 0.868342, 0.173299,
        0.878431, 0.633168, 0.870107, 0.165395,
        0.882353, 0.643813, 0.871833, 0.157419,
        0.886275, 0.654473, 0.873520, 0.149372,
        0.890196, 0.665146, 0.875172, 0.141258,
        0.894118, 0.675826, 0.876787, 0.133092,
        0.898039, 0.686512, 0.878368, 0.124877,
        0.901961, 0.697194, 0.879914, 0.116633,
        0.905882, 0.707875, 0.881428, 0.108374,
        0.909804, 0.718544, 0.882912, 0.100128,
        0.913725, 0.729204, 0.884366, 0.091925,
        0.917647, 0.739843, 0.885791, 0.083809,
        0.921569, 0.750463, 0.887190, 0.075839,
        0.925490, 0.761059, 0.888564, 0.068090,
        0.929412, 0.771623, 0.889915, 0.060670,
        0.933333, 0.782158, 0.891245, 0.053712,
        0.937255, 0.792653, 0.892555, 0.047401,
        0.941176, 0.803110, 0.893849, 0.041954,
        0.945098, 0.813520, 0.895127, 0.037646,
        0.949020, 0.823881, 0.896392, 0.034853,
        0.952941, 0.834195, 0.897645, 0.033643,
        0.956863, 0.844452, 0.898890, 0.034045,
        0.960784, 0.854656, 0.900126, 0.036086,
        0.964706, 0.864796, 0.901358, 0.039789,
        0.968627, 0.874877, 0.902586, 0.044954,
        0.972549, 0.884886, 0.903815, 0.051230,
        0.976471, 0.894827, 0.905045, 0.058381,
        0.980392, 0.904703, 0.906278, 0.066206,
        0.984314, 0.914506, 0.907515, 0.074535,
        0.988235, 0.924240, 0.908760, 0.083251,
        0.992157, 0.933899, 0.910011, 0.092252,
        0.996078, 0.943486, 0.911273, 0.101474,
        1.000000, 0.952999, 0.912545, 0.110859
      }
    },
    { vtkm::cont::ColorTable::Preset::Inferno,
      "Inferno",
      vtkm::ColorSpace::Lab,
      { 0, 1, 0 },
      {
        0.000000, 0.002811, 0.000240, 0.013985,
        0.003922, 0.004020, 0.000978, 0.018723,
        0.007843, 0.005536, 0.001879, 0.024433,
        0.011765, 0.007352, 0.002929, 0.031151,
        0.015686, 0.009463, 0.004122, 0.038857,
        0.019608, 0.011875, 0.005444, 0.047160,
        0.023529, 0.014599, 0.006883, 0.055493,
        0.027451, 0.017636, 0.008432, 0.063834,
        0.031373, 0.021018, 0.010066, 0.072261,
        0.035294, 0.024744, 0.011784, 0.080708,
        0.039216, 0.028838, 0.013567, 0.089220,
        0.043137, 0.033326, 0.015396, 0.097809,
        0.047059, 0.038216, 0.017268, 0.106440,
        0.050980, 0.043431, 0.019155, 0.115162,
        0.054902, 0.048694, 0.021041, 0.123968,
        0.058824, 0.054003, 0.022919, 0.132833,
        0.062745, 0.059374, 0.024768, 0.141775,
        0.066667, 0.064826, 0.026549, 0.150830,
        0.070588, 0.070350, 0.028264, 0.159955,
        0.074510, 0.075950, 0.029888, 0.169149,
        0.078431, 0.081638, 0.031404, 0.178413,
        0.082353, 0.087425, 0.032773, 0.187768,
        0.086275, 0.093316, 0.033979, 0.197196,
        0.090196, 0.099308, 0.035007, 0.206680,
        0.094118, 0.105402, 0.035836, 0.216206,
        0.098039, 0.111606, 0.036445, 0.225770,
        0.101961, 0.117917, 0.036814, 0.235353,
        0.105882, 0.124337, 0.036928, 0.244940,
        0.109804, 0.130862, 0.036774, 0.254505,
        0.113725, 0.137489, 0.036346, 0.264028,
        0.117647, 0.144209, 0.035642, 0.273476,
        0.121569, 0.151015, 0.034667, 0.282818,
        0.125490, 0.157897, 0.033434, 0.292024,
        0.129412, 0.164838, 0.031965, 0.301050,
        0.133333, 0.171827, 0.030286, 0.309867,
        0.137255, 0.178847, 0.028428, 0.318436,
        0.141176, 0.185884, 0.026431, 0.326728,
        0.145098, 0.192914, 0.024350, 0.334702,
        0.149020, 0.199922, 0.022234, 0.342333,
        0.152941, 0.206897, 0.020130, 0.349604,
        0.156863, 0.213821, 0.018086, 0.356495,
        0.160784, 0.220686, 0.016148, 0.363003,
        0.164706, 0.227481, 0.014356, 0.369119,
        0.168627, 0.234204, 0.012742, 0.374850,
        0.172549, 0.240846, 0.011335, 0.380201,
        0.176471, 0.247410, 0.010157, 0.385183,
        0.180392, 0.253892, 0.009227, 0.389814,
        0.184314, 0.260296, 0.008549, 0.394106,
        0.188235, 0.266628, 0.008129, 0.398080,
        0.192157, 0.272888, 0.007968, 0.401753,
        0.196078, 0.279083, 0.008063, 0.405142,
        0.200000, 0.285215, 0.008411, 0.408265,
        0.203922, 0.291291, 0.009007, 0.411141,
        0.207843, 0.297313, 0.009846, 0.413787,
        0.211765, 0.303288, 0.010914, 0.416217,
        0.215686, 0.309221, 0.012204, 0.418447,
        0.219608, 0.315115, 0.013705, 0.420488,
        0.223529, 0.320976, 0.015407, 0.422355,
        0.227451, 0.326807, 0.017302, 0.424057,
        0.231373, 0.332610, 0.019380, 0.425607,
        0.235294, 0.338393, 0.021633, 0.427012,
        0.239216, 0.344154, 0.024053, 0.428283,
        0.243137, 0.349899, 0.026629, 0.429427,
        0.247059, 0.355628, 0.029355, 0.430449,
        0.250980, 0.361347, 0.032224, 0.431359,
        0.254902, 0.367053, 0.035231, 0.432159,
        0.258824, 0.372753, 0.038366, 0.432856,
        0.262745, 0.378447, 0.041596, 0.433456,
        0.266667, 0.384136, 0.044789, 0.433960,
        0.270588, 0.389823, 0.047945, 0.434374,
        0.274510, 0.395506, 0.051063, 0.434700,
        0.278431, 0.401190, 0.054146, 0.434944,
        0.282353, 0.406873, 0.057192, 0.435105,
        0.286275, 0.412556, 0.060200, 0.435186,
        0.290196, 0.418244, 0.063171, 0.435189,
        0.294118, 0.423933, 0.066107, 0.435115,
        0.298039, 0.429627, 0.069009, 0.434966,
        0.301961, 0.435324, 0.071876, 0.434744,
        0.305882, 0.441027, 0.074713, 0.434448,
        0.309804, 0.446734, 0.077518, 0.434082,
        0.313725, 0.452446, 0.080294, 0.433643,
        0.317647, 0.458163, 0.083041, 0.433136,
        0.321569, 0.463886, 0.085764, 0.432561,
        0.325490, 0.469615, 0.088461, 0.431915,
        0.329412, 0.475348, 0.091135, 0.431200,
        0.333333, 0.481090, 0.093787, 0.430416,
        0.337255, 0.486835, 0.096419, 0.429563,
        0.341176, 0.492588, 0.099031, 0.428643,
        0.345098, 0.498345, 0.101627, 0.427653,
        0.349020, 0.504107, 0.104208, 0.426596,
        0.352941, 0.509877, 0.106775, 0.425470,
        0.356863, 0.515649, 0.109331, 0.424277,
        0.360784, 0.521427, 0.111876, 0.423015,
        0.364706, 0.527209, 0.114411, 0.421684,
        0.368627, 0.532995, 0.116940, 0.420283,
        0.372549, 0.538784, 0.119463, 0.418815,
        0.376471, 0.544575, 0.121981, 0.417278,
        0.380392, 0.550371, 0.124499, 0.415673,
        0.384314, 0.556168, 0.127015, 0.413999,
        0.388235, 0.561966, 0.129533, 0.412255,
        0.392157, 0.567765, 0.132054, 0.410442,
        0.396078, 0.573565, 0.134579, 0.408560,
        0.400000, 0.579364, 0.137111, 0.406610,
        0.403922, 0.585160, 0.139652, 0.404592,
        0.407843, 0.590957, 0.142202, 0.402505,
        0.411765, 0.596750, 0.144764, 0.400348,
        0.415686, 0.602540, 0.147340, 0.398122,
        0.419608, 0.608325, 0.149932, 0.395828,
        0.423529, 0.614106, 0.152540, 0.393466,
        0.427451, 0.619879, 0.155168, 0.391037,
        0.431373, 0.625645, 0.157816, 0.388541,
        0.435294, 0.631405, 0.160488, 0.385978,
        0.439216, 0.637154, 0.163184, 0.383348,
        0.443137, 0.642895, 0.165908, 0.380651,
        0.447059, 0.648623, 0.168659, 0.377888,
        0.450980, 0.654340, 0.171441, 0.375059,
        0.454902, 0.660043, 0.174256, 0.372167,
        0.458824, 0.665731, 0.177104, 0.369210,
        0.462745, 0.671404, 0.179990, 0.366190,
        0.466667, 0.677059, 0.182912, 0.363108,
        0.470588, 0.682698, 0.185876, 0.359964,
        0.474510, 0.688315, 0.188880, 0.356760,
        0.478431, 0.693913, 0.191930, 0.353495,
        0.482353, 0.699488, 0.195023, 0.350170,
        0.486275, 0.705039, 0.198164, 0.346788,
        0.490196, 0.710566, 0.201354, 0.343346,
        0.494118, 0.716067, 0.204595, 0.339851,
        0.498039, 0.721540, 0.207889, 0.336298,
        0.501961, 0.726982, 0.211236, 0.332694,
        0.505882, 0.732396, 0.214641, 0.329035,
        0.509804, 0.737776, 0.218102, 0.325327,
        0.513725, 0.743124, 0.221622, 0.321566,
        0.517647, 0.748435, 0.225202, 0.317759,
        0.521569, 0.753709, 0.228845, 0.313903,
        0.525490, 0.758947, 0.232551, 0.309999,
        0.529412, 0.764143, 0.236321, 0.306051,
        0.533333, 0.769299, 0.240159, 0.302058,
        0.537255, 0.774410, 0.244061, 0.298022,
        0.541176, 0.779479, 0.248033, 0.293944,
        0.545098, 0.784499, 0.252073, 0.289827,
        0.549020, 0.789472, 0.256184, 0.285671,
        0.552941, 0.794397, 0.260365, 0.281477,
        0.556863, 0.799270, 0.264618, 0.277247,
        0.560784, 0.804091, 0.268943, 0.272981,
        0.564706, 0.808858, 0.273340, 0.268682,
        0.568627, 0.813569, 0.277812, 0.264350,
        0.572549, 0.818223, 0.282356, 0.259986,
        0.576471, 0.822819, 0.286973, 0.255592,
        0.580392, 0.827356, 0.291665, 0.251167,
        0.584314, 0.831831, 0.296431, 0.246714,
        0.588235, 0.836244, 0.301271, 0.242232,
        0.592157, 0.840592, 0.306184, 0.237723,
        0.596078, 0.844875, 0.311172, 0.233186,
        0.600000, 0.849092, 0.316232, 0.228623,
        0.603922, 0.853241, 0.321365, 0.224036,
        0.607843, 0.857322, 0.326571, 0.219421,
        0.611765, 0.861332, 0.331849, 0.214781,
        0.615686, 0.865271, 0.337198, 0.210114,
        0.619608, 0.869139, 0.342617, 0.205424,
        0.623529, 0.872933, 0.348106, 0.200706,
        0.627451, 0.876653, 0.353664, 0.195963,
        0.631373, 0.880298, 0.359289, 0.191193,
        0.635294, 0.883868, 0.364983, 0.186395,
        0.639216, 0.887360, 0.370741, 0.181568,
        0.643137, 0.890777, 0.376566, 0.176712,
        0.647059, 0.894115, 0.382453, 0.171827,
        0.650980, 0.897374, 0.388404, 0.166908,
        0.654902, 0.900554, 0.394415, 0.161957,
        0.658824, 0.903654, 0.400487, 0.156971,
        0.662745, 0.906675, 0.406619, 0.151946,
        0.666667, 0.909613, 0.412807, 0.146884,
        0.670588, 0.912472, 0.419054, 0.141780,
        0.674510, 0.915247, 0.425355, 0.136631,
        0.678431, 0.917942, 0.431711, 0.131436,
        0.682353, 0.920553, 0.438119, 0.126193,
        0.686275, 0.923082, 0.444578, 0.120898,
        0.690196, 0.925528, 0.451090, 0.115546,
        0.694118, 0.927890, 0.457650, 0.110136,
        0.698039, 0.930168, 0.464258, 0.104662,
        0.701961, 0.932363, 0.470913, 0.099123,
        0.705882, 0.934474, 0.477614, 0.093511,
        0.709804, 0.936499, 0.484358, 0.087827,
        0.713725, 0.938441, 0.491147, 0.082061,
        0.717647, 0.940296, 0.497977, 0.076213,
        0.721569, 0.942067, 0.504847, 0.070277,
        0.725490, 0.943753, 0.511761, 0.064247,
        0.729412, 0.945353, 0.518711, 0.058121,
        0.733333, 0.946868, 0.525700, 0.051891,
        0.737255, 0.948296, 0.532725, 0.045561,
        0.741176, 0.949638, 0.539788, 0.039112,
        0.745098, 0.950894, 0.546885, 0.032904,
        0.749020, 0.952063, 0.554014, 0.027232,
        0.752941, 0.953146, 0.561180, 0.022116,
        0.756863, 0.954141, 0.568376, 0.017577,
        0.760784, 0.955050, 0.575606, 0.013637,
        0.764706, 0.955870, 0.582863, 0.010319,
        0.768627, 0.956603, 0.590153, 0.007648,
        0.772549, 0.957248, 0.597469, 0.005654,
        0.776471, 0.957805, 0.604815, 0.004365,
        0.780392, 0.958273, 0.612188, 0.003815,
        0.784314, 0.958652, 0.619587, 0.004038,
        0.788235, 0.958943, 0.627014, 0.005073,
        0.792157, 0.959145, 0.634463, 0.006958,
        0.796078, 0.959257, 0.641939, 0.009737,
        0.800000, 0.959280, 0.649436, 0.013457,
        0.803922, 0.959213, 0.656956, 0.018167,
        0.807843, 0.959056, 0.664501, 0.023921,
        0.811765, 0.958809, 0.672065, 0.030778,
        0.815686, 0.958472, 0.679650, 0.038798,
        0.819608, 0.958045, 0.687253, 0.047546,
        0.823529, 0.957528, 0.694877, 0.056390,
        0.827451, 0.956920, 0.702516, 0.065327,
        0.831373, 0.956222, 0.710173, 0.074363,
        0.835294, 0.955435, 0.717847, 0.083501,
        0.839216, 0.954559, 0.725534, 0.092743,
        0.843137, 0.953590, 0.733239, 0.102098,
        0.847059, 0.952534, 0.740955, 0.111570,
        0.850980, 0.951389, 0.748682, 0.121171,
        0.854902, 0.950158, 0.756420, 0.130905,
        0.858824, 0.948842, 0.764165, 0.140781,
        0.862745, 0.947441, 0.771920, 0.150814,
        0.866667, 0.945960, 0.779677, 0.161006,
        0.870588, 0.944398, 0.787439, 0.171373,
        0.874510, 0.942762, 0.795201, 0.181922,
        0.878431, 0.941050, 0.802964, 0.192670,
        0.882353, 0.939264, 0.810723, 0.203628,
        0.886275, 0.937416, 0.818476, 0.214805,
        0.890196, 0.935511, 0.826218, 0.226217,
        0.894118, 0.933559, 0.833943, 0.237872,
        0.898039, 0.931563, 0.841653, 0.249795,
        0.901961, 0.929530, 0.849339, 0.262002,
        0.905882, 0.927483, 0.856995, 0.274501,
        0.909804, 0.925441, 0.864610, 0.287300,
        0.913725, 0.923403, 0.872186, 0.300443,
        0.917647, 0.921409, 0.879706, 0.313922,
        0.921569, 0.919491, 0.887156, 0.327744,
        0.925490, 0.917657, 0.894536, 0.341961,
        0.929412, 0.915970, 0.901821, 0.356535,
        0.933333, 0.914460, 0.909000, 0.371498,
        0.937255, 0.913186, 0.916052, 0.386828,
        0.941176, 0.912202, 0.922962, 0.402519,
        0.945098, 0.911571, 0.929705, 0.418535,
        0.949020, 0.911359, 0.936266, 0.434835,
        0.952941, 0.911627, 0.942627, 0.451365,
        0.956863, 0.912435, 0.948769, 0.468039,
        0.960784, 0.913828, 0.954687, 0.484788,
        0.964706, 0.915840, 0.960372, 0.501502,
        0.968627, 0.918481, 0.965832, 0.518121,
        0.972549, 0.921750, 0.971071, 0.534539,
        0.976471, 0.925625, 0.976103, 0.550703,
        0.980392, 0.930072, 0.980946, 0.566595,
        0.984314, 0.935053, 0.985619, 0.582103,
        0.988235, 0.940521, 0.990140, 0.597270,
        0.992157, 0.946429, 0.994526, 0.612080,
        0.996078, 0.952737, 0.998796, 0.626535,
        1.000000, 0.959400, 1.002963, 0.640626
      }
    },
    { vtkm::cont::ColorTable::Preset::Plasma,
      "Plasma",
      vtkm::ColorSpace::Lab,
      { 0, 1, 0 },
      {
        0.000000, 0.185001, 0.000000, 0.530073,
        0.003922, 0.191119, 0.000000, 0.535244,
        0.007843, 0.197129, 0.000000, 0.540150,
        0.011765, 0.203042, 0.000000, 0.544821,
        0.015686, 0.208873, 0.000000, 0.549286,
        0.019608, 0.214631, 0.000000, 0.553570,
        0.023529, 0.220322, 0.000000, 0.557689,
        0.027451, 0.225958, 0.000000, 0.561661,
        0.031373, 0.231550, 0.000000, 0.565505,
        0.035294, 0.237106, 0.000000, 0.569231,
        0.039216, 0.242629, 0.000000, 0.572851,
        0.043137, 0.248125, 0.000000, 0.576370,
        0.047059, 0.253596, 0.000000, 0.579799,
        0.050980, 0.259047, 0.000000, 0.583143,
        0.054902, 0.264480, 0.000000, 0.586406,
        0.058824, 0.269898, 0.000000, 0.589595,
        0.062745, 0.275304, 0.000000, 0.592712,
        0.066667, 0.280699, 0.000000, 0.595761,
        0.070588, 0.286086, 0.000000, 0.598745,
        0.074510, 0.291464, 0.000000, 0.601665,
        0.078431, 0.296835, 0.000000, 0.604524,
        0.082353, 0.302199, 0.000000, 0.607322,
        0.086275, 0.307559, 0.000000, 0.610060,
        0.090196, 0.312917, 0.000000, 0.612742,
        0.094118, 0.318270, 0.000000, 0.615364,
        0.098039, 0.323623, 0.000000, 0.617929,
        0.101961, 0.328971, 0.000000, 0.620434,
        0.105882, 0.334319, 0.000000, 0.622883,
        0.109804, 0.339664, 0.000000, 0.625272,
        0.113725, 0.345008, 0.000000, 0.627601,
        0.117647, 0.350348, 0.000000, 0.629871,
        0.121569, 0.355687, 0.000000, 0.632079,
        0.125490, 0.361026, 0.000000, 0.634226,
        0.129412, 0.366362, 0.000000, 0.636308,
        0.133333, 0.371696, 0.000000, 0.638327,
        0.137255, 0.377028, 0.000000, 0.640279,
        0.141176, 0.382358, 0.000000, 0.642163,
        0.145098, 0.387684, 0.000000, 0.643979,
        0.149020, 0.393006, 0.000000, 0.645723,
        0.152941, 0.398327, 0.000000, 0.647396,
        0.156863, 0.403643, 0.000000, 0.648995,
        0.160784, 0.408954, 0.000000, 0.650518,
        0.164706, 0.414260, 0.000000, 0.651964,
        0.168627, 0.419562, 0.000000, 0.653331,
        0.172549, 0.424856, 0.000000, 0.654617,
        0.176471, 0.430144, 0.000000, 0.655819,
        0.180392, 0.435427, 0.000000, 0.656938,
        0.184314, 0.440700, 0.000000, 0.657970,
        0.188235, 0.445966, 0.000000, 0.658913,
        0.192157, 0.451222, 0.000000, 0.659767,
        0.196078, 0.456470, 0.000000, 0.660529,
        0.200000, 0.461706, 0.000000, 0.661198,
        0.203922, 0.466931, 0.000000, 0.661772,
        0.207843, 0.472145, 0.000000, 0.662249,
        0.211765, 0.477346, 0.000000, 0.662629,
        0.215686, 0.482535, 0.000000, 0.662909,
        0.219608, 0.487709, 0.000000, 0.663088,
        0.223529, 0.492869, 0.000000, 0.663165,
        0.227451, 0.498013, 0.000000, 0.663138,
        0.231373, 0.503141, 0.000000, 0.663007,
        0.235294, 0.508253, 0.000000, 0.662771,
        0.239216, 0.513346, 0.000000, 0.662429,
        0.243137, 0.518422, 0.000000, 0.661980,
        0.247059, 0.523478, 0.000000, 0.661424,
        0.250980, 0.528514, 0.000000, 0.660760,
        0.254902, 0.533531, 0.000000, 0.659987,
        0.258824, 0.538524, 0.000000, 0.659106,
        0.262745, 0.543498, 0.000000, 0.658116,
        0.266667, 0.548448, 0.000000, 0.657020,
        0.270588, 0.553375, 0.000000, 0.655814,
        0.274510, 0.558277, 0.000000, 0.654502,
        0.278431, 0.563155, 0.000000, 0.653083,
        0.282353, 0.568008, 0.000000, 0.651560,
        0.286275, 0.572833, 0.000000, 0.649932,
        0.290196, 0.577634, 0.000000, 0.648200,
        0.294118, 0.582408, 0.000000, 0.646367,
        0.298039, 0.587153, 0.000000, 0.644433,
        0.301961, 0.591870, 0.000000, 0.642402,
        0.305882, 0.596560, 0.000000, 0.640273,
        0.309804, 0.601220, 0.000000, 0.638050,
        0.313725, 0.605851, 0.000000, 0.635735,
        0.317647, 0.610452, 0.000039, 0.633330,
        0.321569, 0.615023, 0.007472, 0.630837,
        0.325490, 0.619563, 0.015271, 0.628260,
        0.329412, 0.624072, 0.023435, 0.625602,
        0.333333, 0.628551, 0.031974, 0.622864,
        0.337255, 0.632999, 0.040877, 0.620051,
        0.341176, 0.637416, 0.049419, 0.617164,
        0.345098, 0.641800, 0.057377, 0.614209,
        0.349020, 0.646154, 0.064892, 0.611187,
        0.352941, 0.650476, 0.072057, 0.608100,
        0.356863, 0.654765, 0.078935, 0.604956,
        0.360784, 0.659025, 0.085579, 0.601753,
        0.364706, 0.663252, 0.092022, 0.598498,
        0.368627, 0.667448, 0.098296, 0.595192,
        0.372549, 0.671611, 0.104421, 0.591841,
        0.376471, 0.675743, 0.110416, 0.588447,
        0.380392, 0.679845, 0.116300, 0.585012,
        0.384314, 0.683916, 0.122080, 0.581540,
        0.388235, 0.687957, 0.127771, 0.578033,
        0.392157, 0.691965, 0.133378, 0.574497,
        0.396078, 0.695945, 0.138913, 0.570933,
        0.400000, 0.699894, 0.144379, 0.567344,
        0.403922, 0.703814, 0.149783, 0.563734,
        0.407843, 0.707704, 0.155133, 0.560102,
        0.411765, 0.711565, 0.160429, 0.556455,
        0.415686, 0.715397, 0.165678, 0.552792,
        0.419608, 0.719200, 0.170881, 0.549117,
        0.423529, 0.722976, 0.176044, 0.545432,
        0.427451, 0.726724, 0.181168, 0.541741,
        0.431373, 0.730444, 0.186255, 0.538043,
        0.435294, 0.734138, 0.191310, 0.534340,
        0.439216, 0.737804, 0.196333, 0.530637,
        0.443137, 0.741445, 0.201328, 0.526933,
        0.447059, 0.745059, 0.206294, 0.523230,
        0.450980, 0.748648, 0.211235, 0.519527,
        0.454902, 0.752210, 0.216152, 0.515831,
        0.458824, 0.755749, 0.221047, 0.512137,
        0.462745, 0.759263, 0.225921, 0.508450,
        0.466667, 0.762752, 0.230774, 0.504769,
        0.470588, 0.766217, 0.235612, 0.501093,
        0.474510, 0.769658, 0.240430, 0.497427,
        0.478431, 0.773077, 0.245235, 0.493770,
        0.482353, 0.776471, 0.250022, 0.490123,
        0.486275, 0.779843, 0.254797, 0.486487,
        0.490196, 0.783192, 0.259560, 0.482859,
        0.494118, 0.786518, 0.264310, 0.479242,
        0.498039, 0.789823, 0.269051, 0.475635,
        0.501961, 0.793105, 0.273782, 0.472039,
        0.505882, 0.796366, 0.278505, 0.468453,
        0.509804, 0.799603, 0.283218, 0.464878,
        0.513725, 0.802820, 0.287926, 0.461313,
        0.517647, 0.806016, 0.292627, 0.457761,
        0.521569, 0.809188, 0.297322, 0.454218,
        0.525490, 0.812341, 0.302013, 0.450686,
        0.529412, 0.815472, 0.306701, 0.447163,
        0.533333, 0.818583, 0.311387, 0.443650,
        0.537255, 0.821671, 0.316069, 0.440146,
        0.541176, 0.824737, 0.320750, 0.436650,
        0.545098, 0.827782, 0.325431, 0.433164,
        0.549020, 0.830806, 0.330110, 0.429688,
        0.552941, 0.833809, 0.334792, 0.426217,
        0.556863, 0.836790, 0.339474, 0.422755,
        0.560784, 0.839749, 0.344160, 0.419299,
        0.564706, 0.842685, 0.348848, 0.415849,
        0.568627, 0.845601, 0.353540, 0.412405,
        0.572549, 0.848493, 0.358235, 0.408967,
        0.576471, 0.851363, 0.362935, 0.405536,
        0.580392, 0.854210, 0.367641, 0.402107,
        0.584314, 0.857033, 0.372354, 0.398682,
        0.588235, 0.859835, 0.377073, 0.395260,
        0.592157, 0.862611, 0.381799, 0.391842,
        0.596078, 0.865364, 0.386535, 0.388425,
        0.600000, 0.868092, 0.391277, 0.385011,
        0.603922, 0.870795, 0.396030, 0.381598,
        0.607843, 0.873474, 0.400794, 0.378186,
        0.611765, 0.876127, 0.405566, 0.374775,
        0.615686, 0.878754, 0.410351, 0.371363,
        0.619608, 0.881353, 0.415146, 0.367950,
        0.623529, 0.883926, 0.419955, 0.364537,
        0.627451, 0.886471, 0.424776, 0.361122,
        0.631373, 0.888988, 0.429609, 0.357706,
        0.635294, 0.891477, 0.434458, 0.354287,
        0.639216, 0.893936, 0.439321, 0.350865,
        0.643137, 0.896366, 0.444200, 0.347440,
        0.647059, 0.898766, 0.449092, 0.344012,
        0.650980, 0.901135, 0.454002, 0.340580,
        0.654902, 0.903471, 0.458927, 0.337145,
        0.658824, 0.905776, 0.463869, 0.333704,
        0.662745, 0.908048, 0.468830, 0.330259,
        0.666667, 0.910286, 0.473807, 0.326809,
        0.670588, 0.912491, 0.478804, 0.323353,
        0.674510, 0.914661, 0.483819, 0.319892,
        0.678431, 0.916796, 0.488853, 0.316425,
        0.682353, 0.918893, 0.493907, 0.312952,
        0.686275, 0.920954, 0.498980, 0.309474,
        0.690196, 0.922978, 0.504074, 0.305987,
        0.694118, 0.924963, 0.509188, 0.302495,
        0.698039, 0.926910, 0.514324, 0.298996,
        0.701961, 0.928816, 0.519481, 0.295491,
        0.705882, 0.930683, 0.524661, 0.291979,
        0.709804, 0.932507, 0.529861, 0.288458,
        0.713725, 0.934290, 0.535084, 0.284930,
        0.717647, 0.936029, 0.540329, 0.281395,
        0.721569, 0.937724, 0.545597, 0.277854,
        0.725490, 0.939377, 0.550890, 0.274304,
        0.729412, 0.940984, 0.556204, 0.270748,
        0.733333, 0.942546, 0.561543, 0.267185,
        0.737255, 0.944059, 0.566904, 0.263615,
        0.741176, 0.945527, 0.572292, 0.260036,
        0.745098, 0.946945, 0.577702, 0.256450,
        0.749020, 0.948314, 0.583137, 0.252857,
        0.752941, 0.949633, 0.588597, 0.249258,
        0.756863, 0.950901, 0.594080, 0.245652,
        0.760784, 0.952119, 0.599591, 0.242040,
        0.764706, 0.953283, 0.605124, 0.238423,
        0.768627, 0.954396, 0.610684, 0.234800,
        0.772549, 0.955453, 0.616268, 0.231174,
        0.776471, 0.956456, 0.621877, 0.227544,
        0.780392, 0.957404, 0.627513, 0.223910,
        0.784314, 0.958293, 0.633175, 0.220271,
        0.788235, 0.959125, 0.638862, 0.216629,
        0.792157, 0.959898, 0.644575, 0.212987,
        0.796078, 0.960612, 0.650314, 0.209347,
        0.800000, 0.961266, 0.656078, 0.205707,
        0.803922, 0.961859, 0.661869, 0.202071,
        0.807843, 0.962389, 0.667686, 0.198439,
        0.811765, 0.962856, 0.673528, 0.194815,
        0.815686, 0.963259, 0.679397, 0.191199,
        0.819608, 0.963597, 0.685291, 0.187595,
        0.823529, 0.963869, 0.691214, 0.184003,
        0.827451, 0.964073, 0.697160, 0.180430,
        0.831373, 0.964209, 0.703132, 0.176878,
        0.835294, 0.964276, 0.709134, 0.173348,
        0.839216, 0.964272, 0.715159, 0.169847,
        0.843137, 0.964198, 0.721211, 0.166379,
        0.847059, 0.964050, 0.727289, 0.162950,
        0.850980, 0.963829, 0.733394, 0.159563,
        0.854902, 0.963533, 0.739524, 0.156228,
        0.858824, 0.963161, 0.745680, 0.152949,
        0.862745, 0.962713, 0.751862, 0.149735,
        0.866667, 0.962187, 0.758070, 0.146594,
        0.870588, 0.961582, 0.764304, 0.143534,
        0.874510, 0.960894, 0.770564, 0.140564,
        0.878431, 0.960123, 0.776851, 0.137691,
        0.882353, 0.959270, 0.783163, 0.134930,
        0.886275, 0.958333, 0.789499, 0.132294,
        0.890196, 0.957313, 0.795862, 0.129792,
        0.894118, 0.956206, 0.802248, 0.127440,
        0.898039, 0.955014, 0.808659, 0.125248,
        0.901961, 0.953730, 0.815096, 0.123223,
        0.905882, 0.952353, 0.821559, 0.121379,
        0.909804, 0.950887, 0.828044, 0.119731,
        0.913725, 0.949334, 0.834553, 0.118292,
        0.917647, 0.947687, 0.841084, 0.117066,
        0.921569, 0.945941, 0.847643, 0.116054,
        0.925490, 0.944103, 0.854223, 0.115271,
        0.929412, 0.942178, 0.860821, 0.114719,
        0.933333, 0.940143, 0.867449, 0.114377,
        0.937255, 0.938021, 0.874094, 0.114262,
        0.941176, 0.935800, 0.880762, 0.114347,
        0.945098, 0.933482, 0.887450, 0.114618,
        0.949020, 0.931072, 0.894156, 0.115054,
        0.952941, 0.928561, 0.900886, 0.115611,
        0.956863, 0.925959, 0.907631, 0.116256,
        0.960784, 0.923271, 0.914394, 0.116929,
        0.964706, 0.920482, 0.921176, 0.117546,
        0.968627, 0.917616, 0.927973, 0.118027,
        0.972549, 0.914677, 0.934779, 0.118242,
        0.976471, 0.911670, 0.941597, 0.118021,
        0.980392, 0.908607, 0.948424, 0.117125,
        0.984314, 0.905512, 0.955251, 0.115218,
        0.988235, 0.902422, 0.962073, 0.111795,
        0.992157, 0.899408, 0.968866, 0.106073,
        0.996078, 0.896560, 0.975612, 0.096681,
        1.000000, 0.894058, 0.982254, 0.081069
      }
    },
    { vtkm::cont::ColorTable::Preset::BlackBodyRadiation,
      "Black-Body Radiation",
      vtkm::ColorSpace::RGB,
      { 0, 0.498039215686, 1 },
      {
        0, 0, 0, 0,
        0.4, 0.901960784314, 0, 0,
        0.8, 0.901960784314, 0.901960784314, 0,
        1, 1, 1, 1
      }
    },
    { vtkm::cont::ColorTable::Preset::XRay,
      "X Ray",
      vtkm::ColorSpace::RGB,
      { 1, 0, 0 },
      { 0, 1, 1, 1, 1, 0, 0, 0 }
    },
    { vtkm::cont::ColorTable::Preset::Green,
      "Green",
      vtkm::ColorSpace::Lab,
      { 0.25, 0, 0 },
      {
        0.00, 0.054902, 0.109804, 0.121569,
        0.05, 0.074510, 0.172549, 0.180392,
        0.10, 0.086275, 0.231373, 0.219608,
        0.15, 0.094118, 0.278431, 0.250980,
        0.20, 0.109804, 0.349020, 0.278431,
        0.25, 0.113725, 0.400000, 0.278431,
        0.30, 0.117647, 0.451000, 0.270588,
        0.35, 0.117647, 0.490196, 0.243137,
        0.40, 0.113725, 0.521569, 0.203922,
        0.45, 0.109804, 0.549020, 0.152941,
        0.50, 0.082353, 0.588235, 0.082353,
        0.55, 0.109804, 0.631373, 0.050980,
        0.60, 0.211765, 0.678431, 0.082353,
        0.65, 0.317647, 0.721569, 0.113725,
        0.70, 0.431373, 0.760784, 0.160784,
        0.75, 0.556863, 0.800000, 0.239216,
        0.80, 0.666667, 0.839216, 0.294118,
        0.85, 0.784314, 0.878431, 0.396078,
        0.90, 0.886275, 0.921569, 0.533333,
        0.95, 0.960784, 0.949020, 0.670588,
        1.00, 1.000000, 0.984314, 0.901961
      }
    },
    { vtkm::cont::ColorTable::Preset::BlackBlueWhite,
      "Black - Blue - White",
      vtkm::ColorSpace::RGB,
      { 1, 1, 0 },
      {
        0, 0, 0, 0,
        0.333, 0, 0, 0.501960784314,
        0.666, 0, 0.501960784314, 1,
        1, 1, 1, 1
      }
    },
    { vtkm::cont::ColorTable::Preset::BlueToOrange,
      "Blue to Orange",
      vtkm::ColorSpace::Lab,
      { 0.25, 0, 0 },
      {
        0.000000, 0.086275, 0.003922, 0.298039,
        0.030334, 0.113725, 0.023529, 0.450980,
        0.055527, 0.105882, 0.050980, 0.509804,
        0.073008, 0.039216, 0.039216, 0.560784,
        0.089974, 0.031372, 0.098039, 0.600000,
        0.106427, 0.043137, 0.164706, 0.639216,
        0.130077, 0.054902, 0.243137, 0.678431,
        0.161440, 0.054902, 0.317647, 0.709804,
        0.200000, 0.050980, 0.396078, 0.741176,
        0.225000, 0.039216, 0.466667, 0.768627,
        0.250000, 0.031372, 0.537255, 0.788235,
        0.276093, 0.031372, 0.615686, 0.811765,
        0.302828, 0.023529, 0.709804, 0.831373,
        0.329563, 0.050980, 0.800000, 0.850980,
        0.351671, 0.070588, 0.854902, 0.870588,
        0.372237, 0.262745, 0.901961, 0.862745,
        0.390231, 0.423529, 0.941176, 0.874510,
        0.417995, 0.572549, 0.964706, 0.835294,
        0.436504, 0.658824, 0.980392, 0.843137,
        0.449871, 0.764706, 0.980392, 0.866667,
        0.464267, 0.827451, 0.980392, 0.886275,
        0.492545, 0.913725, 0.988235, 0.937255,
        0.501285, 1.000000, 1.000000, 0.972549,
        0.510026, 0.988235, 0.980392, 0.870588,
        0.522879, 0.992157, 0.972549, 0.803922,
        0.532648, 0.992157, 0.964706, 0.713725,
        0.549100, 0.988235, 0.956863, 0.643137,
        0.575321, 0.980392, 0.917647, 0.509804,
        0.597429, 0.968627, 0.874510, 0.407843,
        0.620051, 0.949020, 0.823529, 0.321569,
        0.636504, 0.929412, 0.776471, 0.278431,
        0.660668, 0.909804, 0.717647, 0.235294,
        0.682262, 0.890196, 0.658824, 0.196078,
        0.700000, 0.878431, 0.619608, 0.168627,
        0.725000, 0.870588, 0.549020, 0.156863,
        0.750000, 0.850980, 0.474510, 0.145098,
        0.775000, 0.831373, 0.411765, 0.133333,
        0.800000, 0.811765, 0.345098, 0.113725,
        0.825000, 0.788235, 0.266667, 0.094118,
        0.850000, 0.741176, 0.184314, 0.074510,
        0.875000, 0.690196, 0.125490, 0.062745,
        0.900000, 0.619608, 0.062745, 0.043137,
        0.923393, 0.549020, 0.027451, 0.070588,
        0.943959, 0.470588, 0.015686, 0.090196,
        0.967095, 0.400000, 0.003922, 0.101961,
        1.000000, 0.188235, 0.000000, 0.070588
      }
    },
    {vtkm::cont::ColorTable::Preset::GrayToRed,
     "Gray to Red",
     vtkm::ColorSpace::Lab,
     { 0, 0.498039215686, 1 },
     {
       0.000000, 0.101961, 0.101961, 0.101961,
       0.062745, 0.227451, 0.227451, 0.227451,
       0.125490, 0.359939, 0.359939, 0.359939,
       0.188236, 0.502653, 0.502653, 0.502653,
       0.250981, 0.631373, 0.631373, 0.631373,
       0.313726, 0.749865, 0.749865, 0.749865,
       0.376471, 0.843368, 0.843368, 0.843368,
       0.439216, 0.926105, 0.926105, 0.926105,
       0.501961, 0.999846, 0.997232, 0.995694,
       0.564706, 0.994925, 0.908651, 0.857901,
       0.627451, 0.982468, 0.800692, 0.706113,
       0.690196, 0.960323, 0.667820, 0.536332,
       0.752941, 0.894579, 0.503806, 0.399769,
       0.815687, 0.817070, 0.332180, 0.281046,
       0.878432, 0.728489, 0.155017, 0.197386,
       0.941177, 0.576932, 0.055363, 0.149250,
       1.000000, 0.403922, 0.000000, 0.121569
     }
    },
    { vtkm::cont::ColorTable::Preset::ColdAndHot,
      "Cold and Hot",
      vtkm::ColorSpace::RGB,
      { 1, 1, 0 },
      {
        0, 0, 1, 1,
        0.45, 0, 0, 1,
        0.5, 0, 0, 0.501960784314,
        0.55, 1, 0, 0,
        1, 1, 1, 0
      }
    },
    { vtkm::cont::ColorTable::Preset::BlueGreenOrange,
      "Blue - Green - Orange",
      vtkm::ColorSpace::Lab,
      { 0.25, 0.0, 0.0 },
      {
        0.0,0.831373,0.909804,0.980392,
        0.0125,0.74902,0.862745,0.960784,
        0.025,0.694118,0.827451,0.941176,
        0.05,0.568627,0.760784,0.921569,
        0.075,0.45098,0.705882,0.901961,
        0.1,0.345098,0.643137,0.858824,
        0.125,0.247059,0.572549,0.819608,
        0.15,0.180392,0.521569,0.780392,
        0.16,0.14902,0.490196,0.74902,
        0.18,0.129412,0.447059,0.709804,
        0.2,0.101961,0.427451,0.690196,
        0.21,0.094118,0.403922,0.658824,
        0.22,0.090196,0.392157,0.639216,
        0.23,0.082353,0.368627,0.619608,
        0.24,0.070588,0.352941,0.6,
        0.25,0.066667,0.329412,0.568627,
        0.26,0.07451,0.313725,0.541176,
        0.27,0.086275,0.305882,0.509804,
        0.28,0.094118,0.286275,0.478431,
        0.29,0.101961,0.278431,0.45098,
        0.3,0.109804,0.266667,0.411765,
        0.31,0.113725,0.258824,0.380392,
        0.32,0.113725,0.25098,0.34902,
        0.33,0.109804,0.266667,0.321569,
        0.34,0.105882,0.301961,0.262745,
        0.35,0.094118,0.309804,0.243137,
        0.36,0.082353,0.321569,0.227451,
        0.37,0.07451,0.341176,0.219608,
        0.38,0.070588,0.360784,0.211765,
        0.39,0.066667,0.380392,0.215686,
        0.4,0.062745,0.4,0.176471,
        0.425,0.07451,0.419608,0.145098,
        0.45,0.086275,0.439216,0.117647,
        0.475,0.121569,0.470588,0.117647,
        0.5,0.184314,0.501961,0.14902,
        0.525,0.254902,0.541176,0.188235,
        0.55,0.32549,0.580392,0.231373,
        0.575,0.403922,0.619608,0.278431,
        0.6,0.501961,0.670588,0.333333,
        0.63,0.592157,0.729412,0.4,
        0.65,0.741176,0.788235,0.490196,
        0.67,0.858824,0.858824,0.603922,
        0.7,0.921569,0.835294,0.580392,
        0.75,0.901961,0.729412,0.494118,
        0.8,0.858824,0.584314,0.388235,
        0.85,0.8,0.439216,0.321569,
        0.9,0.678431,0.298039,0.203922,
        0.95,0.54902,0.168627,0.109804,
        0.975,0.478431,0.082353,0.047059,
        1.0,0.45098,0.007843,0.0
      }
    },
    { vtkm::cont::ColorTable::Preset::YellowGrayBlue,
      "Yellow - Gray - Blue",
      vtkm::ColorSpace::Lab,
      { 0.25, 0, 0 },
      {
        0.000000, 0.301961, 0.047059, 0.090196,
        0.016390, 0.396078, 0.039216, 0.058824,
        0.032780, 0.494118, 0.054902, 0.035294,
        0.049171, 0.588235, 0.113725, 0.023529,
        0.065561, 0.662745, 0.168627, 0.015686,
        0.081951, 0.741176, 0.227451, 0.003922,
        0.098341, 0.788235, 0.290196, 0.000000,
        0.114732, 0.862745, 0.380392, 0.011765,
        0.131122, 0.901961, 0.458824, 0.027451,
        0.147512, 0.917647, 0.521569, 0.047059,
        0.163902, 0.925490, 0.580392, 0.078431,
        0.180293, 0.937255, 0.643137, 0.121569,
        0.196683, 0.945098, 0.709804, 0.184314,
        0.213073, 0.952941, 0.768627, 0.247059,
        0.229463, 0.964706, 0.827451, 0.325490,
        0.245854, 0.968627, 0.878431, 0.423529,
        0.262244, 0.972549, 0.917647, 0.513725,
        0.278634, 0.980392, 0.949020, 0.596078,
        0.295024, 0.980392, 0.972549, 0.670588,
        0.311415, 0.988235, 0.988235, 0.756863,
        0.327149, 0.984314, 0.988235, 0.854902,
        0.327805, 0.988235, 0.988235, 0.858824,
        0.327815, 0.952941, 0.952941, 0.894118,
        0.327815, 0.952941, 0.952941, 0.894118,
        0.344836, 0.890196, 0.890196, 0.807843,
        0.361856, 0.827451, 0.823529, 0.737255,
        0.378877, 0.776471, 0.764706, 0.678431,
        0.395898, 0.725490, 0.713725, 0.627451,
        0.412919, 0.678431, 0.662745, 0.580392,
        0.429940, 0.631373, 0.607843, 0.533333,
        0.446960, 0.580392, 0.556863, 0.486275,
        0.463981, 0.537255, 0.505882, 0.443137,
        0.481002, 0.498039, 0.458824, 0.407843,
        0.498023, 0.462745, 0.419608, 0.372549,
        0.515043, 0.431373, 0.388235, 0.345098,
        0.532064, 0.403922, 0.356863, 0.317647,
        0.549085, 0.372549, 0.321569, 0.294118,
        0.566106, 0.345098, 0.294118, 0.266667,
        0.583127, 0.317647, 0.262745, 0.239216,
        0.600147, 0.286275, 0.231373, 0.211765,
        0.617168, 0.254902, 0.200000, 0.184314,
        0.634189, 0.231373, 0.172549, 0.164706,
        0.651210, 0.200000, 0.145098, 0.137255,
        0.668240, 0.149020, 0.196078, 0.278431,
        0.701416, 0.200000, 0.254902, 0.345098,
        0.734592, 0.247059, 0.317647, 0.415686,
        0.767768, 0.305882, 0.388235, 0.494118,
        0.800944, 0.372549, 0.458824, 0.568627,
        0.834120, 0.443137, 0.533333, 0.643137,
        0.867296, 0.517647, 0.615686, 0.725490,
        0.900472, 0.600000, 0.698039, 0.800000,
        0.933648, 0.686275, 0.784314, 0.870588,
        0.966824, 0.760784, 0.858824, 0.929412,
        0.983412, 0.807843, 0.901961, 0.960784,
        1.000000, 0.890196, 0.956863, 0.984314
      }
    },
    { vtkm::cont::ColorTable::Preset::RainbowUniform,
      "Rainbow Uniform",
      vtkm::ColorSpace::RGB,
      { 1, 0, 0 },
      {
        0.000000,0.020000,0.381300,0.998100,
        0.023810,0.020000,0.424268,0.969070,
        0.047619,0.020000,0.467234,0.940033,
        0.071429,0.020000,0.510200,0.911000,
        0.095238,0.020000,0.546401,0.872669,
        0.119048,0.020000,0.582600,0.834333,
        0.142857,0.020000,0.618800,0.796000,
        0.166667,0.020000,0.652535,0.749802,
        0.190476,0.020000,0.686267,0.703600,
        0.214286,0.020000,0.720000,0.657400,
        0.238095,0.020000,0.757035,0.603735,
        0.261905,0.020000,0.794067,0.550066,
        0.285714,0.020000,0.831100,0.496400,
        0.309524,0.021354,0.864537,0.428558,
        0.333333,0.023313,0.897999,0.360739,
        0.357143,0.015976,0.931048,0.292563,
        0.380952,0.274211,0.952563,0.153568,
        0.404762,0.493355,0.961904,0.111195,
        0.428571,0.643900,0.977300,0.046900,
        0.452381,0.762402,0.984670,0.034600,
        0.476190,0.880901,0.992033,0.022300,
        0.500000,0.999529,0.999519,0.013488,
        0.523810,0.999403,0.955036,0.079067,
        0.547619,0.999400,0.910666,0.148134,
        0.571429,0.999400,0.866300,0.217200,
        0.595238,0.999270,0.818036,0.217201,
        0.619048,0.999133,0.769766,0.217200,
        0.642857,0.999000,0.721500,0.217200,
        0.666667,0.999136,0.673436,0.217201,
        0.690476,0.999267,0.625366,0.217200,
        0.714286,0.999400,0.577300,0.217200,
        0.738095,0.999403,0.521068,0.217201,
        0.761905,0.999400,0.464833,0.217200,
        0.785714,0.999400,0.408600,0.217200,
        0.809524,0.994760,0.331773,0.211231,
        0.833333,0.986713,0.259518,0.190122,
        0.857143,0.991246,0.147994,0.210789,
        0.880952,0.949903,0.116867,0.252901,
        0.904762,0.903200,0.078433,0.291800,
        0.928571,0.856500,0.040000,0.330700,
        0.952381,0.798903,0.043333,0.358434,
        0.976190,0.741299,0.046667,0.386167,
        1.000000,0.683700,0.050000,0.413900
      }
    },
    { vtkm::cont::ColorTable::Preset::Jet,
      "Jet",
      vtkm::ColorSpace::RGB,
      { 0.25, 0, 0 },
      {
        0, 0, 0, 0.5625,
        0.111111, 0, 0, 1,
        0.3650795, 0, 1, 1,
        0.4920635, 0.5, 1, 0.5,
        0.6190475, 1, 1, 0,
        0.873016, 1, 0, 0,
        1, 0.5, 0, 0
      }
    },
    { vtkm::cont::ColorTable::Preset::RainbowDesaturated,
      "Rainbow Desaturated",
      vtkm::ColorSpace::RGB,
      { 1, 1, 0 },
      {
        0, 0.278431372549, 0.278431372549, 0.858823529412,
        0.143, 0, 0, 0.360784313725,
        0.285, 0, 1, 1,
        0.429, 0, 0.501960784314, 0,
        0.571, 1, 1, 0,
        0.714, 1, 0.380392156863, 0,
        0.857, 0.419607843137, 0, 0,
        1, 0.878431372549, 0.301960784314, 0.301960784314
      }
    }
  };
  // clang-format on
}

VTKM_CONT const std::vector<ColorTablePreset> GetColorTablePresetsVector()
{
  static std::once_flag calledPresets;
  static std::vector<ColorTablePreset> presets;

  std::call_once(calledPresets, BuildColorTablePresetsVector, std::ref(presets));

  return presets;
}
} // anonymous namespace

namespace vtkm
{
namespace cont
{
namespace internal
{

VTKM_CONT_EXPORT
bool LoadColorTablePreset(vtkm::cont::ColorTable::Preset preset, vtkm::cont::ColorTable& table)
{
  if (preset == vtkm::cont::ColorTable::Preset::Default)
  {
    preset = DEFAULT_PRESET;
  }
  for (auto&& ctPreset : GetColorTablePresetsVector())
  {
    if (ctPreset.Preset == preset)
    {
      table = ctPreset.MakePreset();
      return true;
    }
  }

  VTKM_DEPRECATED_SUPPRESS_BEGIN
  // Handle deprecated names
  switch (preset)
  {
    case vtkm::cont::ColorTable::Preset::DEFAULT:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Default, table);
    case vtkm::cont::ColorTable::Preset::COOL_TO_WARM:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::CoolToWarm, table);
    case vtkm::cont::ColorTable::Preset::COOL_TO_WARM_EXTENDED:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::CoolToWarmExtended, table);
    case vtkm::cont::ColorTable::Preset::VIRIDIS:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Viridis, table);
    case vtkm::cont::ColorTable::Preset::INFERNO:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Inferno, table);
    case vtkm::cont::ColorTable::Preset::PLASMA:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Plasma, table);
    case vtkm::cont::ColorTable::Preset::BLACK_BODY_RADIATION:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::BlackBodyRadiation, table);
    case vtkm::cont::ColorTable::Preset::X_RAY:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::XRay, table);
    case vtkm::cont::ColorTable::Preset::GREEN:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Green, table);
    case vtkm::cont::ColorTable::Preset::BLACK_BLUE_WHITE:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::BlackBlueWhite, table);
    case vtkm::cont::ColorTable::Preset::BLUE_TO_ORANGE:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::BlueToOrange, table);
    case vtkm::cont::ColorTable::Preset::GRAY_TO_RED:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::GrayToRed, table);
    case vtkm::cont::ColorTable::Preset::COLD_AND_HOT:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::ColdAndHot, table);
    case vtkm::cont::ColorTable::Preset::BLUE_GREEN_ORANGE:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::BlueGreenOrange, table);
    case vtkm::cont::ColorTable::Preset::YELLOW_GRAY_BLUE:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::YellowGrayBlue, table);
    case vtkm::cont::ColorTable::Preset::RAINBOW_UNIFORM:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::RainbowUniform, table);
    case vtkm::cont::ColorTable::Preset::JET:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::Jet, table);
    case vtkm::cont::ColorTable::Preset::RAINBOW_DESATURATED:
      return LoadColorTablePreset(vtkm::cont::ColorTable::Preset::RainbowDesaturated, table);
    default:
      // Should not get here.
      return false;
  }
  VTKM_DEPRECATED_SUPPRESS_END
}

VTKM_CONT_EXPORT std::set<std::string> GetPresetNames()
{
  std::set<std::string> names;
  for (auto&& ctPreset : GetColorTablePresetsVector())
  {
    names.insert(ctPreset.Name);
  }
  names.insert("Default");
  return names;
}

VTKM_CONT_EXPORT bool LoadColorTablePreset(std::string name, vtkm::cont::ColorTable& table)
{
  if (IStringEqual(name, "Default"))
  {
    return LoadColorTablePreset(DEFAULT_PRESET, table);
  }
  for (auto&& ctPreset : GetColorTablePresetsVector())
  {
    if (IStringEqual(name, ctPreset.Name))
    {
      table = ctPreset.MakePreset();
      return true;
    }
  }
  VTKM_LOG_S(vtkm::cont::LogLevel::Warn,
             "Attempted to load color table preset '" << name << "' but no such preset exists.");
  return false;
}
}
}
} // namespace vtkm::cont::internal
