"""A generic, unlimited undo mechanism."""

# undo.py - Undo-related code.
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

class TagUndo:
    """An undo item for a change to a file's tag.
    
    The file given to the constructor must be a \"real\" underlying
    file, not a file proxy."""

    # "file" really should be an underlying file, or this might not
    # work, for two reasons:
    #
    # (a) file proxies are discarded when the file moves in the
    # hierarchy, and accessing a "dead" one is not guaranteed to work
    # except in special circumstances, and
    #
    # (b) accessing a file proxy could trigger adding an undo!
    def __init__(self, file, fromdict, todict):
        self.file=file
        # is there a way to avoid this copy?
        self.fromdict=fromdict.copy()
        self.todict=todict.copy()

    def undo(self):
        """Set the tags of the enclosed file based on \"fromdict\"."""
        self.file.set_tags(self.fromdict)

    def redo(self):
        """Set the tags of the enclosed file based on \"todict\"."""
        self.file.set_tags(self.todict)

class UndoManager:
    """A class that encapsulates the logic for undoing and redoing actions.

    In addition to being able to store a stacks of actions-to-undo and
    a stack of actions-to-redo, an undo manager is capable of
    temporarily suppressing the generation of new undos, and of
    grouping multiple undo events which are related to the same user
    action.

    Actions that can be undone are represented as objects with undo()
    and redo() methods."""

    # The two hooks are called immediately after undoing or redoing an
    # action.
    def __init__(self, post_undo_hook=lambda:None, post_redo_hook=lambda:None,
                 post_add_undo_hook=lambda:None):
        """Create an UndoManager.

        post_undo_hook is called immediately after undoing an action,
        post_redo_hook is called immediately after redoing an action,
        and post_add_undo_hook is called immediately after adding a
        new undo item."""

        # "undo" holds the stack of things to undo; "redo" holds a
        # stack of things to redo.

        self.undo_stack=[]
        self.redo_stack=[]
        self.active_undo_groups=0
        self.active_undo_group=None
        self.undos_suppressed=0

        self.post_undo_hook=post_undo_hook
        self.post_redo_hook=post_redo_hook
        self.post_add_undo_hook=post_add_undo_hook

    def suppress_undos(self):
        """Suppress adding new undos.

        This is meant to be used to guard undo() and redo(), to
        prevent extraneous items from being added to the undo stack.
        However, it may be useful for other purposes.

        This method may be called multiple times; each call must be
        balanced by a call to unsuppress_undos."""
        
        assert(self.active_undo_groups==0)

        self.undos_suppressed+=1

    def unsuppress_undos(self):
        """Cancel a single call to suppress_undos."""
        assert(self.undos_suppressed>0)

        self.undos_suppressed-=1

    def add_undo(self, to_undo):
        """Add a new action to the top of the undo stack.

        If new undos are suppressed, this is a no-op.  If there is an
        open group, the addition of the undo item is deferred until
        the group is closed.

        This implicitly empties the redo stack."""
        if not self.undos_suppressed:
            if self.active_undo_groups>0:
                self.active_undo_group.append(to_undo)
            else:
                self.undo_stack.append(to_undo)
                # kill the stuff to redo.
                self.redo_stack=[]
                self.post_add_undo_hook()

    def undo(self, *dummy):
        """Activates the top object on the undo stack.

        This calls the undo() method of the object on top of the undo
        stack, then moves that object to the top of the redo stack."""
        self.suppress_undos()

        if len(self.undo_stack) <> 0:
            self.undo_stack[-1].undo()
            self.redo_stack.append(self.undo_stack[-1])
            del self.undo_stack[-1]

        self.unsuppress_undos()

        self.post_undo_hook()

    def redo(self, *dummy):
        """Activates the top object on the redo stack.

        This calls the redo() method of the object on top of the redo
        stack, then moves that object to the top of the undo stack."""
        self.suppress_undos()

        if len(self.redo_stack) <> 0:
            self.redo_stack[-1].redo()
            self.undo_stack.append(self.redo_stack[-1])
            del self.redo_stack[-1]

        self.unsuppress_undos()

        self.post_redo_hook()

    def open_undo_group(self):
        """Begin a group of undo items.

        Any items added between a call to this method and a matching
        call to close_undo_group will be collected into a single
        action.  (ie, calling undo() once will undo them all) This is
        meant to indicate the beginning of a computation which may
        produce many undo items, but was triggered by a single user
        action.

        Calls to open_undo_group/close_undo_group may be nested."""
        if not self.undos_suppressed:
            if self.active_undo_groups==0:
                assert(self.active_undo_group==None)
                self.active_undo_group=[]
                self.active_undo_groups=1
            else:
                self.active_undo_groups+=1

    def close_undo_group(self):
        """End a group of undo items.

        This balances a single active call to open_undo_group.  If
        there are no more outstanding open groups AND at least one
        undo item was added while the group was open, a new undo item
        will be created for the group."""

        if not self.undos_suppressed:
            assert(self.active_undo_groups>0)

            self.active_undo_groups-=1
            if self.active_undo_groups==0:
                class GroupUndo:
                    def __init__(self, items):
                        self.__contents=items

                    def undo(self):
                        for item in self.__contents:
                            item.undo()

                    def redo(self):
                        for item in self.__contents:
                            item.redo()

                if len(self.active_undo_group)>0:
                    self.add_undo(GroupUndo(self.active_undo_group))

                self.active_undo_group=None

    def has_undos(self):
        """Returns true if and only if there is at least one item in
        the undo stack."""

        return len(self.undo_stack)>0

    def has_redos(self):
        """Returns true if and only if there is at least one item in
        the redo stack."""
        return len(self.redo_stack)>0
