/*

  Launcher for external processes


  Copyright © 2011-6, 2018 F.Hroch (hroch@physics.muni.cz)

  This file is part of Munipack.

  Munipack is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Munipack is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "mprocess.h"
#include <wx/wx.h>
#include <wx/app.h>
#include <wx/stream.h>
#include <wx/txtstrm.h>
#include <wx/log.h>
#include <wx/event.h>
#include <wx/process.h>
#include <wx/regex.h>
#include <wx/stopwatch.h>
#include <wx/datetime.h>
#include <iostream>
#include <stdio.h>


using namespace std;

MuniProcess::MuniProcess(wxEvtHandler *h, const wxString& c,
			 const wxArrayString& args):
  wxProcess(h),command(c),ArgsBuffer(args),exitcode(1),handler(h),
  timer(this),tick(250),MaxBuffer(10000),InputIndex(0), killing(false)
{
  SetFitsKeys();

  argv = static_cast<wchar_t **>(malloc((args.GetCount()+2)*sizeof(wchar_t *)));
  argv[0] = wxStrdup(command.wc_str());
  size_t l = 1;
  for(size_t i = 0; i < args.GetCount(); i++)
    argv[l++] = wxStrdup(args[i].wc_str());
  argv[l] = 0;

  Bind(wxEVT_TIMER,&MuniProcess::OnTimer,this);
  Bind(wxEVT_END_PROCESS,&MuniProcess::OnFinish,this);
}

MuniProcess::~MuniProcess()
{
  Unbind(wxEVT_TIMER,&MuniProcess::OnTimer,this);
  Unbind(wxEVT_END_PROCESS,&MuniProcess::OnFinish,this);

  if( argv ) {
    for(size_t i = 0; argv[i] != 0; i++)
      free(argv[i]);
    free(argv);
  }
}

void MuniProcess::OnStart()
{
  SetRuntimeEnvironment();

  wxLogDebug("Launching `" + command + "' ...");

  Redirect();
  long pid = wxExecute(argv,wxEXEC_ASYNC,this);

  if( pid <= 0 ) {
    wxLogError("Failed to launch the external command `" + command + "'.");
    return;
  }

  timer.Start(tick);
  stopwatch.Start();

  wxASSERT(wxProcess::Exists(pid));
}

void MuniProcess::SetRuntimeEnvironment()
{
  // Switch-off buffering of gfortran's stdout and stderr.
  // We need this setting for on-the-fly parsing of outputs.
  wxSetEnv("GFORTRAN_UNBUFFERED_PRECONNECTED","Y");

  // set path for libexec, generally unportable (!)
  wxString xpath;
  wxGetEnv("PATH",&xpath);

#ifdef MUNIPACK_LIBEXEC_DIR
  xpath = wxString(MUNIPACK_LIBEXEC_DIR ":") + xpath;
#endif

  wxString libexecpath;
  if( wxGetEnv("MUNIPACK_LIBEXEC_PATH",&libexecpath) )
    xpath = libexecpath + wxString(":") + xpath;

  wxSetEnv("PATH",xpath.c_str());
}

void MuniProcess::SetFitsKeys()
{
  wxASSERT(InputBuffer.IsEmpty());

  // redefine FITS keywords by environment variables

  const char *keys[] = {
    "FITS_KEY_FILTER",
    "FITS_KEY_TEMPERATURE",
    "FITS_KEY_DATEOBS",
    "FITS_KEY_EXPTIME",
    "FITS_KEY_OBJECT",
    "FITS_KEY_SATURATE",
    "FITS_KEY_READNOISE",
    "FITS_KEY_GAIN",
    "FITS_KEY_AREA",
    "FITS_KEY_EPOCH",
    "FITS_KEY_LATITUDE",
    "FITS_KEY_LONGITUDE",
    "FITS_KEY_ALTITUDE",
    "FITS_KEY_AIRMASS",
    "FITS_KEY_TIME",
    0
  };

  for(size_t i = 0; keys[i] != 0; i++) {
    wxString var, key(keys[i]);
    if( wxGetEnv(key,&var) )
      InputBuffer.Add(key + " = '" + var + "'" );
  }
}


void MuniProcess::AddInput(const wxArrayString& i)
{
  for(size_t l = 0; l < i.GetCount(); l++)
    InputBuffer.Add(i[l]);
}

void MuniProcess::Write(const char *line)
{
  InputBuffer.Add(wxString(line));
}

void MuniProcess::Write(const wxString& fmt, ...)
{
  wxString line;

  va_list par;
  va_start(par, fmt);
  line.PrintfV(fmt,par);
  va_end(par);

  InputBuffer.Add(line);
}

wxKillError MuniProcess::Kill(wxSignal sig, int flags)
{
  killing = true;
  return wxProcess::Kill(GetPid(),sig,flags);
}

void MuniProcess::OnTimer(wxTimerEvent& event)
{
  //  wxLogDebug("MuniProcess::OnTimer");
  Flush();
}

void MuniProcess::OnIdle(wxIdleEvent& event)
{
  wxLogDebug("MuniProcess::OnIdle");
  Flush();

  event.RequestMore();
  event.Skip();
}

bool MuniProcess::StopLine(const wxString& line, int& stopcode)
{
  // Processing of possible 'STOP <NUMBER>',

  wxRegEx re("^[[:space:]]*STOP[[:space:]]+([[:digit:]])[[:space:]]*$",
	     wxRE_DEFAULT|wxRE_ICASE);
  wxASSERT(re.IsValid());

  if( re.Matches(line) ) {

    wxString a(re.GetMatch(line,1));
    long s;
    if( a.ToLong(&s) ) {
      stopcode = s;
      return true;
    }
  }

  return false;
}


void MuniProcess::OnFinish(wxProcessEvent& event)
{
  exitcode = 1; // default is error

  wxTimeSpan ts(wxTimeSpan::Milliseconds(stopwatch.Time()));
  wxLogDebug(ts.Format("MuniProcess elapsed time: %Hh %Mm %S.%ls"));
  wxLogDebug("MuniProcess::OnFinish: %d %d.",event.GetPid(),event.GetExitCode());

  timer.Stop();
  Flush();

  /*

    On finish, we must test both the exit code and some error
    output. The testing of just exit code isn't adequate because
    the implementation of wxExecute returns -1 in case when waitpid
    gives 0 (no changes).

    The outline is little bit complicated by using of some
    Fortran utilities which prints STOP <NUMBER> to indicate
    their return statuses.

    By the way, this code looks for 'STOP 0' string as the last
    error output line of a subprocces. We are ensure that
    the process correctly finished.

    There is also another exception. When user requested killing
    of a process, we always returns non-zero. ???

    Shortly:

      Just when both event.GetExitCode() and STOP indicates 0 (zero)
      the exit 0 is returned. All others combinations generates
      non-zero status.

  */

  if( event.GetExitCode() == 0 ) {

    for(int i = ErrorBuffer.GetCount() - 1; i >= 0; i--) {
      int code;
      if( StopLine(ErrorBuffer.Item(i),code) ) {
	exitcode = code;
	break;
      }
    }
  }
  else
    exitcode = event.GetExitCode();


  if( exitcode != 0 ) {

    wxLogVerbose("Executing of external utility `" + command +
		 "' finished with doubts:");

    if( ArgsBuffer.GetCount() > 0 ) {
      wxLogVerbose("=> Arguments:");
      for(size_t i = 0; i < ArgsBuffer.GetCount(); i++)
	wxLogVerbose("[%d]: "+ArgsBuffer[i],(int)i);
    }

    if( InputBuffer.GetCount() > 0 ) {
      wxLogVerbose("=> Standard input:");
      for(size_t i = 0; i < InputBuffer.GetCount(); i++)
	wxLogVerbose(InputBuffer[i]);
    }

    if( OutputBuffer.GetCount() > 0 ) {
      wxLogVerbose("=> Standard output:");
      for(size_t i = 0; i < OutputBuffer.GetCount(); i++)
	wxLogVerbose(OutputBuffer[i]);
    }

    if( ErrorBuffer.GetCount() > 0 ) {
      wxLogVerbose("=> Error output:");
      for(size_t i = 0; i < ErrorBuffer.GetCount(); i++)
	wxLogVerbose(ErrorBuffer[i]);
    }

    /*

       There is also idea to save output in a structured document
       like XML to additional parsing.

     */
  }

  wxQueueEvent(handler,new wxProcessEvent(event.GetId(),event.GetPid(),exitcode));
}

void MuniProcess::Flush()
{
  //  wxLogDebug("MuniProcess::Flush");

  wxStopWatch sw;
  long tlimit = tick/4;  // because we have 3 pieces blocking of 3/4 tick
  bool debug = wxLog::GetLogLevel() <= wxLOG_Debug;

  /*
    To give a chance of GUI/CLI to be updated, we must periodically
     interrupt this flush. The limit for each stream is derived
     from ticks and one should be short (about 10-50ms).
  */

  // Input stream
  if( InputIndex != InputBuffer.GetCount() ) {
    wxOutputStream *i = GetOutputStream();
    if( i ) {
      wxTextOutputStream out(*i);

      sw.Start();
      while( InputIndex < InputBuffer.GetCount() && i->IsOk() &&
	     sw.Time() < tlimit ) {
	wxLogDebug(InputBuffer[InputIndex]);
	out << InputBuffer[InputIndex] << endl;
	InputIndex++;
      }
      if( i->GetLastError() != wxSTREAM_NO_ERROR )
	wxLogError("Input stream last error: %d",i->GetLastError());
    }
  }

  if( InputIndex == InputBuffer.GetCount() )
    CloseOutput();

  // Output stream
  if( IsInputAvailable() ) {
    wxInputStream *i = GetInputStream();
    if( i ) {
      wxTextInputStream out(*i);
      size_t n = 0;
      sw.Start();
      while( i->IsOk() && ! i->Eof() && i->CanRead() && sw.Time() < tlimit ) {
	wxString line = out.ReadLine();
	if( line != "" ) {
	  if( debug )
	    fprintf(stdout,"%s\n",static_cast<const char *>(line.char_str()));
	  OutputBuffer.Add(line);
	  n++;
	}
      }
      if( OutputBuffer.GetCount() > MaxBuffer )
	OutputBuffer.RemoveAt(0,n);
    }
  }

  // Error stream
  if( IsErrorAvailable() ) {

    wxInputStream *e = GetErrorStream();
    if( e ) {
      wxTextInputStream err(*e);
      size_t n = 0;
      sw.Start();
      while( e->IsOk() && ! e->Eof() && e->CanRead() && sw.Time() < tlimit ) {
	wxString line = err.ReadLine();

	if( line != "" ) {

	  // filter lines with STOP 0 to be quite
	  int code;
	  if( ! (StopLine(line,code) && code == 0 && debug ) )
	    fprintf(stderr,"%s\n",static_cast<const char *>(line.char_str()));

	  ErrorBuffer.Add(line);
	  n++;
	}
      }
      if( ErrorBuffer.GetCount() > MaxBuffer )
	ErrorBuffer.RemoveAt(0,n);
    }
  }
}
