/*
 * Copyright (c) 2001 Dmitry Dicky diwil@mail.ru
 * All rights reserved.
 *
 * Modifications Copyright (c) 2010 Diane Gagne diane@hartmantech.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS `AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Modified 2010 by Diane Gagne at Hartman Technica to include support for 
 * denormalized numbers
 *
 * $Id: addsf.c,v 1.3 2004/01/31 13:09:57 diwil Exp $
 */

#include "mathsf.h"

long __addsf3(long a1,long a2)
{
  register long mant1, mant2;
  register int exp1, exp2;
  register int sign;

  if(!(a1 & NO_SIGN))
    return a2;

  if(!(a2 & NO_SIGN)) 
    return a1;

  exp1 = (a1>>SHORT_SHIFT)&SHORT_EXPONENT_MASK;
  exp2 = (a2>>SHORT_SHIFT)&SHORT_EXPONENT_MASK;


  exp1 >>= EXPONENT_SHIFT;
  exp2 >>= EXPONENT_SHIFT;

  //If there is a large difference between the two values, return the
  //larger one, and ignore the addition.
  if (exp1 > exp2 + 25) return a1;
  if (exp2 > exp1 + 25) return a2;

#if __MSP430LIBC_IEEE754_ERRORS__
  //Check for NANs or Infinities - with the shift 0x00FF is a full exponent mask
  if(exp1 == 0x00FF)
    {
      //if a1 is a NAN, return a1
      mant1 = a1 & MANTISSA_MASK;
      if(mant1 != 0)
	return a1;
    }
  if(exp2 == 0x00FF)
    {
      //if a2 is a NAN, return a2
      mant2 = a2 & MANTISSA_MASK;
      if(mant2 != 0)
	return a2;

      //check if a1 is also infinity, then return infinity or NAN
      if(exp1 == 0x00FF)
	{
	  //if the are the same sign return infinity of that sign
	  if(((a1>>SHORT_SHIFT)&SHORT_SIGNBIT_MASK) == ((a2>>SHORT_SHIFT)&SHORT_SIGNBIT_MASK))
	    return(a1);
	  else
	    return NAN;
	}
    }

  //normalize any denormalized numbers
  if(exp1 == 0)
    {
      //get only the mantissa part of the first number
      mant1 = a1 & MANTISSA_MASK;
      //otherwise denormalise the number
      //do one shift to start
      mant1 = mant1<<1;
      //while the 24th bit is not filled repeatedly shift the 
      //mantissa right by 1 and subtract 1 from the exponent
      while((mant1 & MANTISSA_BIT) != MANTISSA_BIT)
	{
	  mant1 = mant1<<1;
	  exp1 = exp1-1;
	}
    }
  else
    mant1 = MANT(a1);
 
  //do the same thing for the other operand
  if(exp2 == 0)
    {
      mant2 = a2 & MANTISSA_MASK;
      mant2 = mant2<<1;
      while((mant2 & MANTISSA_BIT) != MANTISSA_BIT)
	{
	  mant2 = mant2<<1;
	  exp2 = exp2-1;
	}
    }
  else
    mant2 = MANT(a2);
#else
  mant1 = MANT(a1);
  mant2 = MANT(a2);
#endif

  sign = (a1>>SHORT_SHIFT)&SHORT_SIGNBIT_MASK;
  mant1 <<= 6;
  if(sign)
    mant1 = -mant1;


  sign = (a2>>SHORT_SHIFT)&SHORT_SIGNBIT_MASK;
   mant2 <<= 6;
  if(sign)
    mant2 = -mant2;

  if (exp1 > exp2)
    {
    mant2 >>= exp1 - exp2;
    }
  else
  {
    mant1 >>= exp2 - exp1;
    exp1 = exp2;
  }

  mant1 += mant2;

  /* from this point we do not need 'mant2' anymore.*/
  if (mant1 < 0)
  {
    mant1 = -mant1;
    sign = SHORT_SIGNBIT_MASK;
  }
  else if (!mant1)
    return 0l;
  else
    sign=0;

  while (!(mant1 & 0xE0000000))
  {
    mant1 <<= 1;
    exp1--;
  }


  if (mant1 & (1l << 30))
  {
    mant1 >>= 1;
    exp1++;
  }

  mant1 += (mant1 & 0x40) ? 0x20 : 0x1F;
  if (mant1 & (1l << 30))
  {
    mant1 >>= 1;
    exp1++;
  }  

  mant1 >>= 6;
  mant1 &= ~HIDDEN;
  exp1 <<= EXPONENT_SHIFT;
  mant1 |= ((long)(sign | exp1) << SHORT_SHIFT);
  return mant1;
}
