/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "Censord IMSpector filter plugin"
#define PLUGIN_SHORT_NAME "Censord"

#define CENSORD_SOCKET "/tmp/.censord.sock"

bool localdebugmode;

extern "C"
{
	bool initfilterplugin(struct filterplugininfo &filterplugininfo,
		class Options &options, bool debugmode);
	void closefilterplugin(void);
	bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent);
};

int getheaders(Socket &sock, std::map<std::string, std::string> &headers);

bool initfilterplugin(struct filterplugininfo &filterplugininfo,
	class Options &options, bool debugmode)
{
	if (options["censord"] != "on") return false;

	localdebugmode = debugmode;
	
	filterplugininfo.pluginname = PLUGIN_NAME;
	
	return true;
}

void closefilterplugin(void)
{
	return;
}

/* The main plugin function. See filterplugin.cpp. */
bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent)
{
	class Socket censordsock(AF_UNIX, SOCK_STREAM);
	std::string request;
	int modifiedbufferlength = strlen(modifiedbuffer);
	bool filtered = false;
	char replybuffer[BUFFER_SIZE];
	
	memset(replybuffer, 0, BUFFER_SIZE);
	
	/* Ignore stuff that isn't a message event. */
	if (imevent.type != TYPE_MSG) return false;

	request = stringprintf( \
		"imspector-%s\r\n" \
		"protocol %s\r\n" \
		"localid %s\r\n" \
		"remoteid %s\r\n" \
		"charset UTF-8\r\n" \
		"length %d\r\n" \
		"\r\n" \
		"%s",
		imevent.outgoing ? "outgoing" : "incoming",
		imevent.protocolname.c_str(),
		imevent.localid.c_str(),
		imevent.remoteid.c_str(),
		modifiedbufferlength,
		modifiedbuffer);		
	
	/* Complete the connection. */
	if (!(censordsock.connectsocket(CENSORD_SOCKET, "")))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't connect to censord");
		return false;
	}
	
	if (!censordsock.sendalldata(request.c_str(), request.length()))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't send request to censord");
		return false;
	}
	
	if (censordsock.recvline(replybuffer, BUFFER_SIZE) < 0)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get response from censord");
		return false;
	}
	
	std::map<std::string, std::string> headers;
	
	if (getheaders(censordsock, headers) < 0)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get response from censord for headers");
		return false;
	}
	
	stripnewline(replybuffer);
	
	if (strncmp(replybuffer, "BLCK", 4) == 0)
	{
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Censord requests we block");
		filtered = true;
	}
	else if (strncmp(replybuffer, "PASS", 4) == 0)
	{
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Censord requests we pass");
	}
	else if (strncmp(replybuffer, "ERR!", 4) == 0)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Censord returned an error: %s", replybuffer);
	}
	else if (strncmp(replybuffer, "MDFY", 4) == 0)
	{
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Censord wants to modify something");

		if (headers["length"].empty())
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": No length field specified");
			return false;
		}
		
		int replybufferlength = atol(headers["length"].c_str());
		
		if (replybufferlength != modifiedbufferlength)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unmatched lengths are not supported yet (%d != %d)",
				replybufferlength, modifiedbufferlength);
			return false;
		}
		
		memset(replybuffer, 0, BUFFER_SIZE);
		
		if (!(censordsock.recvalldata(replybuffer, replybufferlength)))
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get manipulated text");
			return false;
		}
		
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Content after: %s\n",
			replybuffer);
		
		/* Finally copy the censor deamons modified data. */
		memcpy(modifiedbuffer, replybuffer, replybufferlength);
	}
	else
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unknown censord response");
	}
	
	censordsock.closesocket();
	
	/* Append the category information. */
	if (!headers["result"].empty())
		imevent.categories += headers["result"];
	
	return filtered;
}

/* Uses the socket to read in the headers from censord into a hash. Headers
 * are "key value\r\n" formatted. Processing ends on an empty line. */
int getheaders(Socket &sock, std::map<std::string, std::string> &headers)
{
	char buffer[BUFFER_SIZE];
	int lines = 0;
		
	while (true)
	{
		memset(buffer, 0, BUFFER_SIZE);

		if (sock.recvline(buffer, BUFFER_SIZE) < 0)	return -1;
	
		stripnewline(buffer);
		
		if (!strlen(buffer)) break;
	
		char *s = buffer;
		std::string header, value;
		
		while (*s && *s != ' ')
		{
			header += *s;
			s++;
		}
		while (*s && *s == ' ') s++;
		while (*s)
		{
			value += *s;
			s++;
		}
		
		headers[header] = value;
		
		lines++;
		
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": header: %s value: %s",
			header.c_str(), value.c_str());
	}
	
	return lines;
}
