/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file filter.c
 * \brief Contains all function needed for filtering caller list
 */

#include <ffgtk.h>
#include <filter.h>

/** Global filter list */
static GList *psFilters = NULL;
/** current options list */
static GList *psCurrentOptions = NULL;
/** filter tree view */
static GtkWidget *psSelectionTreeView = NULL;
/** current y position of filter row */
static int nTableY = 0;

static void addButton( GtkWidget *psWidget, gpointer pTable );

/**
 * \brief Compare with pattern matching
 * \param pnA first string
 * \param pnB second string
 * \return 0 = no match, 1 = match
 */
static gint filterCompare( gchar *pnA, gchar *pnB ) {
	GPatternSpec *psSpec = NULL;
	gint nResult = 0;

	psSpec = g_pattern_spec_new( pnA );
	nResult = g_pattern_match_string( psSpec, pnB );
	g_pattern_spec_free( psSpec );

	return nResult;
}

/**
 * \brief Check caller entry for filter match
 * \param psCaller caller structure
 * \param psFilter filter to match with
 * \return 0 = no match, 1 = match
 */
gint checkFilter( struct sCaller *psCaller, struct sFilter *psFilter ) {
	struct sFilterOptions *psOption = NULL;
	GList *psList = NULL;
	gint nResult = 0;
	gint nDates = 0;
	gint nDatesValid = 0;
	gboolean bCallType = FALSE;
	gboolean bCallTypeValid = FALSE;
	gboolean bName = FALSE;
	gboolean bNameValid = FALSE;
	gboolean bNumber = FALSE;
	gboolean bNumberValid = FALSE;
	gboolean bLocalNumber = FALSE;
	gboolean bLocalNumberValid = FALSE;
	GPatternSpec *psSpec = NULL;

	if ( psFilter == NULL ) {
		return 0;
	}

	for ( psList = psFilter -> psOptions; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psOption = psList -> data;

		switch ( psOption -> nType ) {
			case 0:
				/* Calltype */
				bCallType = TRUE;
				if ( psOption -> nSubType == 0 || psOption -> nSubType == psCaller -> nType ) {
					bCallTypeValid = TRUE;
				}
				break;
			case 1: {
				/* Date */
				char *pnPartA1, *pnPartA2, *pnPartA3, *pnPartA4;
				char *pnPartB1, *pnPartB2, *pnPartB3, *pnPartB4;
				int nMonthStart1, nMonthStart2;
				int nYearStart1, nYearStart2;
				int nTimeStart1, nTimeStart2;
				int nRet;

				if ( psOption -> pnEntry == NULL ) {
					break;
				}

				nDates++;

				nMonthStart1 = findString( psOption -> pnEntry, 0, "." );
				nYearStart1 = findString( psOption -> pnEntry, nMonthStart1 + 1, "." );
				nTimeStart1 = findString( psOption -> pnEntry, nYearStart1 + 1, " " );
				pnPartA1 = getSubString( psOption -> pnEntry, 0, nMonthStart1 );
				pnPartA2 = getSubString( psOption -> pnEntry, nMonthStart1 + 1, nYearStart1 - nMonthStart1 - 1 );
				pnPartA3 = getSubString( psOption -> pnEntry, nYearStart1 + 1, nTimeStart1 - nYearStart1 - 1 );
				pnPartA4 = getSubString( psOption -> pnEntry, nTimeStart1 + 1, strlen( psOption -> pnEntry ) - nTimeStart1 - 1 );

				nMonthStart2 = findString( psCaller -> pnDateTime, 0, "." );
				nYearStart2 = findString( psCaller -> pnDateTime, nMonthStart2 + 1, "." );
				nTimeStart2 = findString( psCaller -> pnDateTime, nYearStart2 + 1, " " );
				pnPartB1 = getSubString( psCaller -> pnDateTime, 0, nMonthStart2 );
				pnPartB2 = getSubString( psCaller -> pnDateTime, nMonthStart2 + 1, nYearStart2 - nMonthStart2 - 1 );
				pnPartB3 = getSubString( psCaller -> pnDateTime, nYearStart2 + 1, nTimeStart2 - nYearStart2 - 1 );
				pnPartB4 = getSubString( psCaller -> pnDateTime, nTimeStart2 + 1, strlen( psCaller -> pnDateTime ) - nTimeStart2 - 1 );

				switch ( psOption -> nSubType ) {
					case 0:
						nRet = filterCompare( pnPartA3, pnPartB3 );
						if ( nRet == 1 ) {
							nRet = filterCompare( pnPartA2, pnPartB2 );
							if ( nRet == 1 ) {
								nRet = filterCompare( pnPartA1, pnPartB1 );
								if ( nRet == 1 ) {
									nDatesValid++;
								}
							}
						}
						break;
					case 1:
						nRet = filterCompare( pnPartA3, pnPartB3 );
						if ( nRet == 1 ) {
							nRet = filterCompare( pnPartA2, pnPartB2 );
							if ( nRet == 1 ) {
								nRet = filterCompare( pnPartA1, pnPartB1 );
								if ( nRet != 1 ) {
									nDatesValid++;
								}
							} else {
								nDatesValid++;
							}
						} else {
							nDatesValid++;
						}
						break;
					case 2:
						nRet = strcmp( pnPartA3, pnPartB3 );
						if ( nRet >= 0 ) {
							nRet = strcmp( pnPartA2, pnPartB2 );
							if ( nRet >= 0 ) {
								nRet = strcmp( pnPartA1, pnPartB1 );
								if ( nRet < 0 ) {
									nDatesValid++;
								}
							} else {
								nDatesValid++;
							}
						} else {
							nDatesValid++;
						}
						break;
					case 3:
						nRet = strcmp( pnPartA3, pnPartB3 );
						if ( nRet <= 0 ) {
							nRet = strcmp( pnPartA2, pnPartB2 );
							if ( nRet <= 0 ) {
								nRet = strcmp( pnPartA1, pnPartB1 );
								if ( nRet > 0 ) {
									nDatesValid++;
								}
							} else {
								nDatesValid++;
							}
						} else {
							nDatesValid++;
						}
						break;
				}

				g_free( pnPartA1 );
				g_free( pnPartA2 );
				g_free( pnPartA3 );
				g_free( pnPartA4 );
				g_free( pnPartB1 );
				g_free( pnPartB2 );
				g_free( pnPartB3 );
				g_free( pnPartB4 );
				break;
			}
			/* Name */
			case 2:
				bName = TRUE;
				if ( psOption -> pnEntry != NULL && psCaller -> pnName != NULL ) {
					switch ( psOption -> nSubType ) {
						/* is */
						case 0:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnName ) ) {
								bNameValid = TRUE;
							}
							break;
						/* is not */
						case 1:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( !g_pattern_match_string( psSpec, psCaller -> pnName ) ) {
								bNameValid = TRUE;
							}
							break;
						/* starts with */
						case 2:
							psSpec = g_pattern_spec_new( g_strconcat( psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnName ) ) {
								bNameValid = TRUE;
							}
							break;
						/* contains */
						case 3:
							psSpec = g_pattern_spec_new( g_strconcat( "*", psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnName ) ) {
								bNameValid = TRUE;
							}
							break;
					}
					g_pattern_spec_free( psSpec );
					psSpec = NULL;
				}
				break;
			/* Number */
			case 3:
				bNumber = TRUE;
				if ( psOption -> pnEntry != NULL && psCaller -> pnNumber != NULL ) {
					switch ( psOption -> nSubType ) {
						/* is */
						case 0:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnNumber ) ) {
								bNumberValid = TRUE;
							}
							break;
						/* is not */
						case 1:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( !g_pattern_match_string( psSpec, psCaller -> pnNumber ) ) {
								bNumberValid = TRUE;
							}
							break;
						/* starts with */
						case 2:
							psSpec = g_pattern_spec_new( g_strconcat( psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnNumber ) ) {
								bNumberValid = TRUE;
							}
							break;
						/* contains */
						case 3:
							psSpec = g_pattern_spec_new( g_strconcat( "*", psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnNumber ) ) {
								bNumberValid = TRUE;
							}
							break;
					}
					g_pattern_spec_free( psSpec );
					psSpec = NULL;
				}
				break;
			case 4:
				/* Local Number */
				bLocalNumber = TRUE;
				if ( psOption -> pnEntry != NULL && psCaller -> pnLocalNumber != NULL ) {
					switch ( psOption -> nSubType ) {
						/* is */
						case 0:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnLocalNumber ) ) {
								bLocalNumberValid = TRUE;
							}
							break;
						/* is not */
						case 1:
							psSpec = g_pattern_spec_new( psOption -> pnEntry );
	
							if ( !g_pattern_match_string( psSpec, psCaller -> pnLocalNumber ) ) {
								bLocalNumberValid = TRUE;
							}
							break;
						/* starts with */
						case 2:
							psSpec = g_pattern_spec_new( g_strconcat( psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnLocalNumber ) ) {
								bLocalNumberValid = TRUE;
							}
							break;
						/* contains */
						case 3:
							psSpec = g_pattern_spec_new( g_strconcat( "*", psOption -> pnEntry, "*", NULL ) );
	
							if ( g_pattern_match_string( psSpec, psCaller -> pnLocalNumber ) ) {
								bLocalNumberValid = TRUE;
							}
							break;
					}

					g_pattern_spec_free( psSpec );
					psSpec = NULL;
				}
				break;
		}
	}

	if ( ( bCallType == TRUE && bCallTypeValid == TRUE ) || bCallType == FALSE ) {
		nResult |= 0x01;
	}

	if ( ( nDates != 0 && nDates == nDatesValid ) || nDates == 0 ) {
		nResult |= 0x02;
	}

	if ( ( bName == TRUE && bNameValid == TRUE ) || bName == FALSE ) {
		nResult |= 0x04;
	}
	if ( ( bNumber == TRUE && bNumberValid == TRUE ) || bNumber == FALSE ) {
		nResult |= 0x08;
	}
	if ( ( bLocalNumber == TRUE && bLocalNumberValid == TRUE ) || bLocalNumber == FALSE ) {
		nResult |= 0x10;
	}

	return !( nResult == 0x1F );
}

/**
 * \brief Find filter by name
 * \param pnName filter name
 * \return filter structure or NULL
 */
struct sFilter *findFilter( const gchar *pnName ) {
	GList *psList = NULL;
	struct sFilter *psFilter = NULL;

	for ( psList = psFilters; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psFilter = psList -> data;

		if( !strcmp( psFilter -> pnName, pnName ) ) {
			return psFilter;
		}
	}

	return NULL;
}

/**
 * \brief Update filter view information
 * \param psStore filter liststore
 * \param psTreeView filter treeview
 */
void updateFilter( GtkListStore *psStore, GtkWidget *psTreeView ) {
	GtkTreeIter sIter;
	struct sFilter *psFilter = NULL;
	struct sFilterOptions *psOption = NULL;
	GList *psOptions = NULL;
	GList *psList = NULL;

	psSelectionTreeView = psTreeView;

	if ( psFilters == NULL ) {
		/* Add standard filters */

		/* All calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "All calls" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 0;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}

		/* Incoming calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "Incoming calls" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 1;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}

		/* Outgoing calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "Outgoing calls" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 3;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}

		/* Missed calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "Missed calls" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 2;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}

		/* VoiceBox calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "Voice box" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 4;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}

		/* FaxBox calls */
		psFilter = g_malloc0( sizeof( struct sFilter ) );
		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( _( "Fax" ) );
			psOptions = NULL;
			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			if ( psOption != NULL ) {
				psOption -> nType = 0;
				psOption -> nSubType = 5;
				psOption -> pnEntry = NULL;
				psOptions = g_list_append( psOptions, psOption );
				psFilter -> psOptions = psOptions;
			}

			psFilters = g_list_append( psFilters, psFilter );
		}
	}

	for ( psList = psFilters; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		struct sFilterInfo sInfo;
		gchar *pnCount = NULL;

		psFilter = psList -> data;
		getFilterInfo( psFilter, &sInfo );
		gtk_list_store_append( psStore, &sIter );
		gtk_list_store_set( psStore, &sIter, COL_FILTER_TYPE, psFilter -> pnName, -1 );
		pnCount = g_strdup_printf( "%d", sInfo.nCount );
		gtk_list_store_set( psStore, &sIter, COL_FILTER_COUNT, pnCount, -1 );
		g_free( pnCount );
		pnCount = g_strdup_printf( "%d:%2.2d", sInfo.nDuration / 60, sInfo.nDuration % 60 );
		gtk_list_store_set( psStore, &sIter, COL_FILTER_DURATION, pnCount, -1 );
		g_free( pnCount );
	}

	gtk_tree_view_expand_all( GTK_TREE_VIEW( psTreeView ) )	;
}

/**
 * \brief Update option combobox according filter type
 * \param psWidget combobox widget
 * \param pNext option combobox pointer
 */
static void filterTypeChanged( GtkWidget *psWidget, gpointer pNext ) {
	GtkWidget *psComboBox = GTK_WIDGET( pNext );
	GtkListStore *psStore;
	gint nActive = gtk_combo_box_get_active( GTK_COMBO_BOX( psWidget ) );
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( pNext ), "entry" );

	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( psComboBox ) ) );
	gtk_list_store_clear( psStore );

	if ( psEntry != NULL ) {
		gtk_widget_set_sensitive( psEntry, TRUE );
	}

	switch ( nActive ) {
		case 0:
			/* Type */
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "All" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Incoming" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Missed" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Outgoing" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Voice box" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Fax box" ) );
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			if ( psEntry != NULL ) {
				gtk_widget_set_sensitive( psEntry, FALSE );
			}
			break;
		case 1:
			/* Date */
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is not" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is after" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is before" ) );
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			break;
		case 2:
			/* Name */
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is not" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Starts with" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Contains" ) );
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			break;
		case 3:
			/* Number */
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is not" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Starts with" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Contains" ) );
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			break;
		case 4:
			/* Local number */
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Is not" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Starts with" ) );
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Contains" ) );
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			break;
	}
}

/**
 * \brief Filter type change, store it
 * \param psWidget combobox widget
 * \param pData filter option pointer
 */
static void typeChanged( GtkWidget *psWidget, gpointer pData ) {
	struct sFilterOptions *psOption = pData;

	psOption -> nType = gtk_combo_box_get_active( GTK_COMBO_BOX( psWidget ) );
}

/**
 * \brief Filter subtype change, store it
 * \param psWidget subtype combobox widget
 * \param pData filter option pointer
 */
static void subTypeChanged( GtkWidget *psWidget, gpointer pData ) {
	struct sFilterOptions *psOption = pData;

	psOption -> nSubType = gtk_combo_box_get_active( GTK_COMBO_BOX( psWidget ) );
}

/**
 * \brief Entry changed
 * \param psWidget entry widget
 * \param pData filter option pointer
 */
static void entryChanged( GtkWidget *psWidget, gpointer pData ) {
	struct sFilterOptions *psOption = pData;

	if ( psOption -> pnEntry != NULL ) {
		g_free( psOption -> pnEntry );
		psOption -> pnEntry = NULL;
	}

	psOption -> pnEntry = g_strdup( gtk_entry_get_text( GTK_ENTRY( psWidget ) ) );
}

/**
 * \brief Delete button callback
 * \param psWidget button widget
 * \param pBox box widget pointer
 */
static void delButton( GtkWidget *psWidget, gpointer pBox ) {
	GtkWidget *psBox = GTK_WIDGET( pBox );
	struct sFilterOptions *psOption = g_object_get_data( G_OBJECT( psBox ), "option" );

	if ( g_list_length( psCurrentOptions ) <= 1 ) {
		return;
	}

	psCurrentOptions = g_list_remove( psCurrentOptions, psOption );
	g_free( psOption );

	gtk_widget_destroy( psBox );
}

/**
 * \brief Add new option to table
 * \param pTable table widget pointer
 * \param psOption filter option pointer
 */
static void addOption( gpointer pTable, struct sFilterOptions *psOption ) {
	GtkWidget *psTable = pTable;
	GtkWidget *psTypeBox;
	GtkWidget *psSubTypeBox;
	GtkWidget *psEntry;
	GtkWidget *psDelButton;
	GtkWidget *psAddButton;
	GtkWidget *psBox;
	GtkWidget *psBox2;

	if ( psOption == NULL ) {
		psOption = g_malloc0( sizeof( struct sFilterOptions ) );
	}

	psCurrentOptions = g_list_append( psCurrentOptions, psOption );

	psBox = gtk_hbox_new( FALSE, 5 );
	g_object_set_data( G_OBJECT( psBox ), "option", psOption );

	psBox2 = gtk_hbox_new( TRUE, 5 );

	psTypeBox = gtk_combo_box_new_text();
	gtk_combo_box_append_text( GTK_COMBO_BOX( psTypeBox ), _( "Call type" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psTypeBox ), _( "Date" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psTypeBox ), _( "Name" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psTypeBox ), _( "Number" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psTypeBox ), _( "Local number" ) );
	if ( psOption != NULL ) {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psTypeBox ), psOption -> nType );
	} else {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psTypeBox ), 0 );
	}
	gtk_widget_show( psTypeBox );
	g_signal_connect( G_OBJECT( psTypeBox ), "changed", G_CALLBACK( typeChanged ), psOption );
	gtk_box_pack_start( GTK_BOX( psBox2 ), psTypeBox, FALSE, TRUE, 0 );

	psSubTypeBox = gtk_combo_box_new_text();
	filterTypeChanged( psTypeBox, psSubTypeBox );
	if ( psOption != NULL ) {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psSubTypeBox ), psOption -> nSubType );
	} else {
		gtk_combo_box_set_active( GTK_COMBO_BOX( psSubTypeBox ), 0 );
	}
	gtk_widget_show( psSubTypeBox );
	g_signal_connect( G_OBJECT( psSubTypeBox ), "changed", G_CALLBACK( subTypeChanged ), psOption );
	gtk_box_pack_start( GTK_BOX( psBox2 ), psSubTypeBox, FALSE, TRUE, 0 );

	gtk_box_pack_start( GTK_BOX( psBox ), psBox2, FALSE, TRUE, 0 );

	psEntry = gtk_entry_new();
	if ( psOption != NULL && psOption -> pnEntry != NULL) {
		gtk_entry_set_text( GTK_ENTRY( psEntry ), psOption -> pnEntry );
	}
	gtk_widget_show( psEntry );
	g_signal_connect( G_OBJECT( psEntry ), "changed", G_CALLBACK( entryChanged ), psOption );
	gtk_box_pack_start( GTK_BOX( psBox ), psEntry, TRUE, TRUE, 0 );
	g_object_set_data( G_OBJECT( psSubTypeBox ), "entry", psEntry );

	psAddButton = gtk_button_new();
	GtkWidget *psAddImage = gtk_image_new_from_stock( GTK_STOCK_ADD, GTK_ICON_SIZE_BUTTON );
	gtk_button_set_image( GTK_BUTTON( psAddButton ), psAddImage );
	g_signal_connect( G_OBJECT( psAddButton ), "clicked", G_CALLBACK( addButton ), psTable );
	gtk_widget_show( psAddButton );
	gtk_box_pack_start( GTK_BOX( psBox ), psAddButton, FALSE, TRUE, 0 );

	psDelButton = gtk_button_new();
	GtkWidget *psDelImage = gtk_image_new_from_stock( GTK_STOCK_REMOVE, GTK_ICON_SIZE_BUTTON );
	gtk_button_set_image( GTK_BUTTON( psDelButton ), psDelImage );
	g_signal_connect( G_OBJECT( psDelButton ), "clicked", G_CALLBACK( delButton ), psBox );
	gtk_widget_show( psDelButton );
	gtk_box_pack_start( GTK_BOX( psBox ), psDelButton, FALSE, TRUE, 0 );

	gtk_widget_show_all( GTK_WIDGET( psBox ) );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psBox ), 0, 5, nTableY, nTableY + 1, GTK_EXPAND | GTK_FILL, 0, 0, 0 );
	g_signal_connect( G_OBJECT( psTypeBox ), "changed", G_CALLBACK( filterTypeChanged ), psSubTypeBox );
	nTableY++;
}

/**
 * \brief Add button callback
 * \param psWidget add button widget
 * \param pTable current table widget
 */
static void addButton( GtkWidget *psWidget, gpointer pTable ) {
	addOption( pTable, NULL );
}

/**
 * \brief Add new filter dialog response
 * \param psWidget window
 * \param nResponse response id
 * \param pUserData telephone number
 */
static void filterAddResponse( GtkWidget *psWidget, gint nResponse, gpointer pUserData ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psWidget ), "entry" );
	GtkListStore *psFilterStore = g_object_get_data( G_OBJECT( psWidget ), "filter_store" );

	if ( nResponse == GTK_RESPONSE_OK ) {
		const gchar *pnName = gtk_entry_get_text( GTK_ENTRY( psEntry ) );
		struct sFilter *psNewFilter = g_malloc0( sizeof( struct sFilter ) );

		psNewFilter -> pnName = g_strdup( pnName );
		psNewFilter -> psOptions = psCurrentOptions;
		psFilters = g_list_append( psFilters, psNewFilter );

		gtk_list_store_clear( psFilterStore );
		updateFilter( psFilterStore, psSelectionTreeView );
		SaveFilters();
	}

	gtk_widget_destroy( psWidget );
}

/**
 * \brief Add new filter dialog
 * \param psWidget add filter button
 * \param pData filter list store
 */
void filterAdd( GtkWidget *psWidget, gpointer pData ) {
	GtkWidget *psAddDialog;
	GtkWidget *psTable;
	GtkWidget *psLabel;
	GtkWidget *psEntry;
	GtkListStore *psFilterStore = pData;
	GtkWidget *psBox = NULL;

	psCurrentOptions = NULL;
	psAddDialog = gtk_dialog_new_with_buttons( _( "Add new filter" ), NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );
	psTable = gtk_table_new( 3, 5, FALSE );
	psLabel = gtk_label_new( _( "Enter filter name:" ) );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psLabel ), 0, 1, nTableY, nTableY + 1, GTK_FILL, 0, 5, 0 );
	psEntry = gtk_entry_new();
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psEntry ), 1, 5, nTableY, nTableY + 1, GTK_EXPAND | GTK_FILL, 0, 5, 0 );
	nTableY++;
	addButton( NULL, psTable );

	gtk_widget_show_all( psTable );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psAddDialog ) );
	if ( psBox != NULL ) {
		gtk_box_pack_start( GTK_BOX( psBox ), psTable, TRUE, TRUE, 5 );
	}

	g_object_set_data( G_OBJECT( psAddDialog ), "entry", psEntry );
	g_object_set_data( G_OBJECT( psAddDialog ), "filter_store", psFilterStore );
	g_signal_connect( psAddDialog, "response", G_CALLBACK( filterAddResponse ), NULL );
	
	gtk_widget_show_all( GTK_WIDGET( psAddDialog ) );
}

/**
 * \brief Delete filter dialog
 * \param psWidget add filter button
 * \param pData filter list store
 */
void filterDel( GtkWidget *psWidget, gpointer pData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psSelectionTreeView ) );
	struct sFilter *psFilter = NULL;
	GtkListStore *psFilterStore = pData;

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, COL_FILTER_TYPE, &sName );

		GtkWidget *psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
																	GTK_BUTTONS_OK_CANCEL, _( "Do you want to delete the filter '%s'?" ), g_value_get_string( &sName ) );
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Delete filter" ) );
		gint nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( psDialog );

		if ( nResult != GTK_RESPONSE_OK ) {
			return;
		}

		psFilter = findFilter( g_value_get_string( &sName ) );
		if ( psFilter != NULL ) {
			psFilters = g_list_remove( psFilters, psFilter );
			Debug( KERN_WARNING, "Free data!!\n" );
		}

		g_value_unset( &sName );
		gtk_list_store_clear( psFilterStore );
		updateFilter( psFilterStore, psSelectionTreeView );
		SaveFilters();
	}
}

/**
 * \brief Edit filter dialog
 * \param psWidget add filter button
 * \param pData filter list store
 */
void filterEdit( GtkWidget *psWidget, gpointer pData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psSelectionTreeView ) );
	struct sFilter *psFilter = NULL;
	GtkWidget *psEditDialog;
	GtkWidget *psTable;
	GtkWidget *psLabel;
	GtkWidget *psEntry;
	GList *psList = NULL;
	GtkListStore *psFilterStore = pData;
	struct sFilterOptions *psOption = NULL;
	GtkWidget *psBox = NULL;

	if ( !gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		return;
	}

	GValue sName = { 0 };

	gtk_tree_model_get_value( psModel, &sSelectedIter, COL_FILTER_TYPE, &sName );

	psFilter = findFilter( g_value_get_string( &sName ) );
	g_value_unset( &sName );
	if ( psFilter == NULL ) {
		return;
	}

	psCurrentOptions = NULL;
	psEditDialog = gtk_dialog_new_with_buttons( _( "Edit filter" ), NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psTable = gtk_table_new( 3, 5, FALSE );
	psLabel = gtk_label_new( _( "Enter filter name:" ) );
	nTableY = 0;
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psLabel ), 0, 1, nTableY, nTableY + 1, GTK_FILL, 0, 0, 5 );
	psEntry = gtk_entry_new();
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psEntry ), 1, 5, nTableY, nTableY + 1, GTK_EXPAND | GTK_FILL, 0, 5, 5 );
	gtk_entry_set_text( GTK_ENTRY( psEntry ), psFilter -> pnName );
	nTableY++;

	for ( psList = psFilter -> psOptions; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psOption = psList -> data;

		addOption( psTable, psOption );
	}
	
	psCurrentOptions = psFilter -> psOptions;
 
	gtk_widget_show_all( psTable );

	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psEditDialog ) );
	if ( psBox != NULL ) {
		gtk_box_pack_start( GTK_BOX( psBox ), psTable, TRUE, TRUE, 5 );
	}

	gtk_dialog_run( GTK_DIALOG( psEditDialog ) );

	if ( psFilter -> pnName != NULL ) {
		g_free( psFilter -> pnName );
	}
	psFilter -> pnName = g_strdup( gtk_entry_get_text( GTK_ENTRY( psEntry ) ) );

	psFilter -> psOptions = psCurrentOptions;

	gtk_widget_destroy( psEditDialog );
	gtk_list_store_clear( psFilterStore );
	updateFilter( psFilterStore, psSelectionTreeView );
	SaveFilters();
}

/**
 * \brief Add filter from xml node
 * \param psNode xml node
 */
static void addFilter( xmlnode *psNode ) {
	struct sFilter *psFilter = NULL;
	GList *psOptions = NULL;
	struct sFilterOptions *psOption = NULL;
	xmlnode *psChild = NULL;
	xmlnode *psChild2 = NULL;
	gchar *pnName = NULL;
	gchar *pnType;
	gchar *pnSubType;
	gchar *pnEntry;

	psChild = xmlnode_get_child( psNode, "name" );
	if ( psChild != NULL ) {
		pnName = xmlnode_get_data( psChild );

		for ( psChild = xmlnode_get_child( psNode, "option" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
			psChild2 = xmlnode_get_child( psChild, "type" );
			if ( psChild2 == NULL ) {
				continue;
			}
			pnType = xmlnode_get_data( psChild2 );

			psChild2 = xmlnode_get_child( psChild, "subtype" );
			if ( psChild2 == NULL ) {
				g_free( pnType );
			}
			pnSubType = xmlnode_get_data( psChild2 );

			psChild2 = xmlnode_get_child( psChild, "entry" );
			if ( psChild2 == NULL ) {
				g_free( pnSubType );
				g_free( pnType );
			}

			pnEntry = xmlnode_get_data( psChild2 );

			psOption = g_malloc0( sizeof( struct sFilterOptions ) );
			psOption -> nType = atoi( pnType );
			psOption -> nSubType = atoi( pnSubType );
			psOption -> pnEntry = g_strdup( pnEntry );

			psOptions = g_list_append( psOptions, psOption );

			g_free( pnEntry );
			g_free( pnSubType );
			g_free( pnType );
		}

		psFilter = g_malloc0( sizeof( struct sFilter ) );

		if ( psFilter != NULL ) {
			psFilter -> pnName = g_strdup( pnName );
			psFilter -> psOptions = psOptions;
			psFilters = g_list_append( psFilters, psFilter );
		}

		g_free( pnName );
	}
}

/**
 * \brief Load filters
 */
void LoadFilters( void ) {
	xmlnode *psNode = NULL, *psChild;
	DIR *psDir = NULL;

	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

	psNode = readXmlFromFile( "filters.xml", _( "filters" ) );
	if ( psNode == NULL ) {
		Debug( KERN_DEBUG, "Could not read filters.xml\n" );
		return;
	}

	for ( psChild = xmlnode_get_child( psNode, "filter" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addFilter( psChild );
	}

	xmlnode_free( psNode );
}

/**
 * \brief Convert filter to xml node
 * \param psFilter filter structure
 * \return create xmlnode
 */
static xmlnode *filterToXmlnode( struct sFilter *psFilter ) {
	xmlnode *node, *option, *child;
	struct sFilterOptions *psOption;
	GList *psList = NULL;

	node = xmlnode_new( "filter" );

	child = xmlnode_new_child( node, "name" );
	xmlnode_insert_data(child, psFilter -> pnName, -1 );

	for ( psList = psFilter -> psOptions; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psOption = psList -> data;
		gchar *pnTmp;

		option = xmlnode_new_child( node, "option" );

		child = xmlnode_new_child( option, "type" );
		pnTmp = g_strdup_printf( "%d", psOption -> nType );
		xmlnode_insert_data(child, pnTmp, -1 );
		g_free( pnTmp );

		child = xmlnode_new_child( option, "subtype" );
		pnTmp = g_strdup_printf( "%d", psOption -> nSubType );
		xmlnode_insert_data(child, pnTmp, -1 );
		g_free( pnTmp );

		child = xmlnode_new_child( option, "entry" );
		if ( psOption -> pnEntry != NULL ) {
			xmlnode_insert_data(child, psOption -> pnEntry, -1 );
		} else {
			xmlnode_insert_data(child, "", -1 );
		}
	}

	return node;
}

/**
 * \brief Convert filter list to xml node
 * \return created xml node
 */
static xmlnode *filtersToXmlnode(void){
	xmlnode *node, *child;
	GList *cur;

	node = xmlnode_new( "filter" );
	xmlnode_set_attrib( node, "version", "1.0" );

	for ( cur = psFilters; cur != NULL; cur = cur->next ) {
		child = filterToXmlnode( cur->data );
		xmlnode_insert_child( node, child );
	}

	return node;
}

/**
 * \brief Save filters
 */
void SaveFilters( void ) {
	xmlnode *node;
	char *pnData;

	node = filtersToXmlnode();
	pnData = xmlnode_to_formatted_str( node, NULL );
	if ( pnData != NULL ) {
		int nFile, nResult;
		char *pnFile = g_build_filename( getUserDir(), "filters.xml", NULL );

		nFile = open( pnFile, O_RDWR | O_CREAT | O_TRUNC, 0600 );
		if ( nFile > 0 ) {
			nResult = write( nFile, pnData, strlen( pnData ) );
			if ( nResult != strlen( pnData ) ) {
				Debug( KERN_WARNING, "Could not save file %s\n", pnFile );
			}
			close( nFile );
		}
	}
	g_free( pnData );
	xmlnode_free( node );
}
