// SPDX-License-Identifier: LicenseRef-AGPL-3.0-only-OpenSSL

#include <chiaki/rpcrypt.h>

#ifdef CHIAKI_LIB_ENABLE_MBEDTLS
#include "mbedtls/aes.h"
#include "mbedtls/md.h"
#else
#include <openssl/hmac.h>
#include <openssl/evp.h>
#endif

#include <string.h>
#include <stdbool.h>
#include <assert.h>

static const uint8_t echo_b[] = { 0xe1, 0xec, 0x9c, 0x3a, 0xdd, 0xbd, 0x08, 0x85, 0xfc, 0x0e, 0x1d, 0x78, 0x90, 0x32, 0xc0, 0x04 };

static void bright_ambassador_ps4_pre10(uint8_t *bright, uint8_t *ambassador, const uint8_t *nonce, const uint8_t *morning)
{
	static const uint8_t echo_a[] = { 0x01, 0x49, 0x87, 0x9b, 0x65, 0x39, 0x8b, 0x39, 0x4b, 0x3a, 0x8d, 0x48, 0xc3, 0x0a, 0xef, 0x51 };

	for(uint8_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
	{
		uint8_t v = nonce[i];
		v -= i;
		v -= 0x27;
		v ^= echo_a[i];
		ambassador[i] = v;
	}

	for(uint8_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
	{
		uint8_t v = morning[i];
		v -= i;
		v += 0x34;
		v ^= echo_b[i];
		v ^= nonce[i];
		bright[i] = v;
	}
}

static ChiakiErrorCode bright_ambassador(ChiakiTarget target, uint8_t *bright, uint8_t *ambassador, const uint8_t *nonce, const uint8_t *morning)
{
	static const uint8_t keys_a_ps4[0x70 * 0x20] = {
		0xdf, 0x40, 0x82, 0xa6, 0x2e, 0xd8, 0x6b, 0x2b, 0xf5, 0x6c,
		0x5f, 0xf5, 0xfb, 0x21, 0x69, 0xb4, 0x00, 0x27, 0x5e, 0x87,
		0xfa, 0x1a, 0xa7, 0x95, 0x16, 0xa1, 0xb7, 0xb7, 0xca, 0xf2,
		0x3d, 0x7a, 0x04, 0x97, 0xef, 0x49, 0xeb, 0x51, 0xbc, 0xe9,
		0x2a, 0x47, 0x84, 0xba, 0xfb, 0x08, 0x78, 0x4a, 0xa7, 0x82,
		0x17, 0xad, 0x4b, 0x26, 0x34, 0x97, 0xb2, 0xde, 0x41, 0x7d,
		0xc6, 0x08, 0xc4, 0xe9, 0xe7, 0x73, 0xa1, 0x22, 0xa2, 0xf3,
		0x2a, 0x6f, 0x8c, 0xc0, 0x45, 0xf6, 0x84, 0xb4, 0xd7, 0x54,
		0x07, 0xd5, 0x76, 0x33, 0xd3, 0x8b, 0x61, 0xe7, 0xea, 0xd6,
		0x98, 0x32, 0xf5, 0xeb, 0x9e, 0x1b, 0xbd, 0xe2, 0x01, 0x31,
		0x45, 0x01, 0x0d, 0x51, 0x1f, 0x77, 0xaf, 0x0c, 0x34, 0xb7,
		0x23, 0x40, 0x0d, 0xc0, 0xac, 0x7e, 0x04, 0xb5, 0xf0, 0x75,
		0xde, 0x5e, 0xba, 0xfa, 0xa6, 0x25, 0x56, 0xde, 0x53, 0x83,
		0x94, 0x4e, 0xa4, 0xd9, 0x4e, 0xf4, 0x73, 0xab, 0xd6, 0x96,
		0xb6, 0xfc, 0x09, 0x46, 0xe2, 0x1a, 0x9d, 0x4f, 0xf1, 0x89,
		0x90, 0xa0, 0x34, 0xf4, 0x55, 0x60, 0x29, 0xf9, 0x39, 0xdb,
		0x20, 0xaa, 0xdf, 0x8b, 0xff, 0x6c, 0xc0, 0xa9, 0x96, 0xad,
		0x71, 0x67, 0xae, 0xb6, 0x1a, 0x6e, 0xd4, 0x92, 0x7a, 0xc7,
		0xac, 0x31, 0xc8, 0x21, 0x24, 0x3b, 0xb3, 0x6a, 0x1d, 0x4d,
		0xc0, 0x82, 0x5f, 0x53, 0x9f, 0xf9, 0xeb, 0x4a, 0x41, 0x47,
		0x27, 0x5a, 0xbc, 0x3b, 0xc8, 0x50, 0x20, 0xeb, 0x11, 0x02,
		0xf8, 0xc5, 0x7e, 0xa4, 0xd3, 0x85, 0x52, 0x90, 0x61, 0x4e,
		0x88, 0xff, 0x81, 0xcf, 0x75, 0x58, 0x21, 0xfe, 0x58, 0x8f,
		0xd0, 0x75, 0xf6, 0x20, 0xaa, 0x54, 0x3a, 0x45, 0x54, 0xf3,
		0xe4, 0x44, 0xda, 0x98, 0x7f, 0x09, 0x9e, 0xa6, 0x72, 0x63,
		0xbc, 0x9f, 0x46, 0x77, 0x7e, 0x24, 0xf8, 0xda, 0xc4, 0x94,
		0x04, 0xa9, 0x23, 0xa7, 0xea, 0x67, 0xe3, 0x85, 0x41, 0xa4,
		0x4b, 0x8e, 0xc1, 0x92, 0xde, 0x96, 0xc8, 0x09, 0xb7, 0xd4,
		0x41, 0x60, 0x89, 0xf0, 0xdb, 0x09, 0x56, 0x01, 0xc1, 0x6c,
		0x33, 0xb1, 0x37, 0xbd, 0x79, 0x6f, 0xab, 0x1c, 0x61, 0xe9,
		0x18, 0xfc, 0xb3, 0x6c, 0xbc, 0xfe, 0xea, 0xb1, 0x9e, 0xce,
		0xba, 0x83, 0xda, 0xe5, 0x81, 0xd7, 0x07, 0xb7, 0x8a, 0xf0,
		0x89, 0x39, 0xdc, 0x30, 0x6b, 0x0d, 0xa7, 0x2b, 0xc9, 0x51,
		0xdc, 0x4f, 0x99, 0x98, 0x3c, 0xcf, 0x62, 0x1c, 0x56, 0xf1,
		0x86, 0x88, 0xdb, 0xce, 0x10, 0xbf, 0x3d, 0xde, 0xe6, 0xf5,
		0x02, 0x51, 0xf7, 0xa5, 0xe3, 0x0a, 0xdb, 0xeb, 0xae, 0xb0,
		0x76, 0x89, 0x62, 0x4b, 0xa7, 0xbd, 0xd6, 0xfe, 0x8d, 0x01,
		0x28, 0x26, 0xf3, 0x76, 0x65, 0xba, 0x0f, 0x44, 0x77, 0x65,
		0x79, 0x47, 0x06, 0x43, 0xae, 0xa8, 0xd1, 0x5e, 0x20, 0x0e,
		0x72, 0x2a, 0x15, 0xd9, 0x42, 0x96, 0xe7, 0xdb, 0x40, 0xae,
		0x4e, 0x2b, 0x1c, 0x6d, 0x71, 0xa2, 0x8e, 0xa6, 0xe1, 0x4d,
		0xe8, 0x84, 0xa3, 0xa5, 0x59, 0x41, 0x0c, 0x9e, 0xe9, 0x3e,
		0x5a, 0xc8, 0x02, 0x3f, 0xbb, 0xf3, 0xe1, 0x0a, 0xf7, 0x3d,
		0xd7, 0xaa, 0xe5, 0x64, 0xc6, 0x4e, 0xc3, 0xc9, 0xbb, 0x32,
		0x30, 0xa9, 0x18, 0xec, 0xb8, 0xae, 0x61, 0xc9, 0x1a, 0x62,
		0xeb, 0x47, 0x92, 0x0c, 0xa7, 0xe0, 0x6c, 0x33, 0xf5, 0x84,
		0xec, 0x69, 0x31, 0xdb, 0xad, 0xdf, 0x3d, 0xbb, 0x4c, 0x4b,
		0xa1, 0x8c, 0xf0, 0x46, 0x52, 0x65, 0x72, 0xaf, 0x6a, 0xc8,
		0xce, 0xb3, 0x27, 0xd3, 0x32, 0x78, 0xb1, 0x39, 0xd6, 0xd1,
		0x5e, 0x09, 0x06, 0xff, 0xdc, 0x8f, 0xca, 0x16, 0x9b, 0x5e,
		0xa2, 0x18, 0x64, 0xee, 0x8c, 0x0b, 0x9b, 0xd7, 0x41, 0xf8,
		0x35, 0x98, 0xd3, 0x5c, 0x78, 0xe2, 0xbb, 0xa5, 0xf7, 0x68,
		0x02, 0x8a, 0xd2, 0x14, 0x85, 0x50, 0x8b, 0x98, 0x2a, 0x09,
		0x15, 0xc0, 0x9e, 0x9c, 0x47, 0x1f, 0x92, 0xc2, 0xa5, 0x2a,
		0x3b, 0xaa, 0x7b, 0xde, 0xb8, 0xe7, 0xee, 0xa7, 0xe0, 0x4a,
		0xb4, 0x81, 0x28, 0x7f, 0x15, 0x38, 0xe8, 0x78, 0xff, 0xc1,
		0xed, 0x98, 0xd8, 0x4f, 0x23, 0x3f, 0x77, 0x29, 0xac, 0xa5,
		0xbc, 0xd0, 0x8f, 0x2b, 0x89, 0xb9, 0x99, 0xbe, 0xc1, 0xf1,
		0x3e, 0x4e, 0xbd, 0x30, 0xd7, 0x6c, 0x1c, 0x09, 0x4e, 0x7c,
		0x13, 0x28, 0xf5, 0xc3, 0xff, 0xc1, 0xcf, 0x8a, 0x2a, 0x41,
		0xd0, 0x73, 0x24, 0x23, 0x31, 0x13, 0x77, 0xa4, 0x33, 0xc0,
		0x7e, 0x7b, 0xc0, 0x81, 0xb5, 0xf5, 0x25, 0x80, 0x45, 0xc5,
		0x94, 0x65, 0xa4, 0x8a, 0x94, 0xc1, 0xb6, 0xf4, 0x69, 0x61,
		0x11, 0xd5, 0x43, 0xea, 0x4b, 0x38, 0x4a, 0x56, 0x14, 0xdd,
		0x3b, 0xa6, 0x99, 0xdf, 0xe5, 0x87, 0x2a, 0xf9, 0x12, 0xbf,
		0x45, 0xb1, 0xd9, 0x74, 0x67, 0x8b, 0x2b, 0x57, 0xc3, 0xd0,
		0xb1, 0x9a, 0x4a, 0xcc, 0x4f, 0x63, 0x79, 0x5a, 0x4f, 0x1d,
		0xa6, 0xff, 0x65, 0xee, 0xbe, 0xd9, 0x16, 0xea, 0xa9, 0x66,
		0x07, 0x97, 0x71, 0x43, 0xdd, 0x64, 0xe3, 0x00, 0x03, 0xbc,
		0xba, 0x1b, 0xf3, 0xa6, 0xa6, 0xee, 0xef, 0xf8, 0xa7, 0xf2,
		0x15, 0xf9, 0xcf, 0x8c, 0x62, 0x2a, 0x1e, 0x9b, 0xbe, 0xfb,
		0x5a, 0xb7, 0xed, 0xd5, 0xd2, 0x71, 0x2a, 0xc9, 0xeb, 0x65,
		0x68, 0x77, 0x34, 0xa4, 0xbb, 0x44, 0xab, 0x01, 0xe5, 0x79,
		0x19, 0x37, 0xb2, 0x07, 0x01, 0xc6, 0xca, 0xb3, 0x07, 0xb8,
		0x55, 0xac, 0x03, 0xae, 0xf6, 0x3f, 0x29, 0x12, 0xa1, 0xba,
		0x1d, 0x94, 0x3a, 0xa3, 0xb5, 0x6c, 0x6e, 0xf4, 0x31, 0x3f,
		0x38, 0xa7, 0xe0, 0x34, 0x31, 0xf2, 0x2a, 0x7a, 0xe6, 0xcf,
		0x3f, 0xc4, 0xe9, 0xb9, 0x0f, 0x05, 0x9f, 0xec, 0x9a, 0xdc,
		0xb3, 0x59, 0xc4, 0x61, 0xaa, 0x10, 0xb6, 0xce, 0x73, 0xe3,
		0xe3, 0x8c, 0xf8, 0x5d, 0x0e, 0x95, 0x3e, 0x1a, 0x4f, 0xbe,
		0x77, 0x87, 0xce, 0x8a, 0x17, 0xf5, 0x20, 0xa7, 0x01, 0xf0,
		0xd7, 0xe6, 0x34, 0x63, 0x04, 0x76, 0xbe, 0x74, 0xb3, 0x44,
		0x7d, 0xdf, 0x11, 0x32, 0xb9, 0x13, 0x32, 0x15, 0x1a, 0xed,
		0xcf, 0x1f, 0x85, 0x91, 0xdb, 0xa3, 0xc4, 0xf2, 0x0f, 0x2c,
		0x8f, 0x77, 0x0a, 0xc4, 0x09, 0x7d, 0x19, 0x12, 0x7c, 0x46,
		0xf7, 0x9d, 0x3a, 0x94, 0x86, 0x4b, 0x22, 0x67, 0x6f, 0x90,
		0x57, 0x99, 0x70, 0x28, 0x3d, 0x9c, 0x53, 0xd2, 0xd3, 0x0b,
		0xc1, 0x29, 0xb7, 0x39, 0x8c, 0xb7, 0x62, 0xc8, 0xbb, 0x64,
		0x4d, 0xcf, 0xe0, 0x5f, 0x64, 0xfc, 0x47, 0xb2, 0xef, 0x0a,
		0x68, 0x8d, 0x9a, 0xcd, 0x9f, 0x49, 0x77, 0x6b, 0x50, 0x47,
		0x7c, 0xfc, 0xa4, 0xd4, 0x2c, 0x4d, 0x53, 0xc1, 0x26, 0x26,
		0x92, 0x38, 0x0d, 0xba, 0x1c, 0x71, 0x81, 0x17, 0x4a, 0x59,
		0x0f, 0x80, 0xfd, 0xad, 0x69, 0x5c, 0x58, 0xa7, 0xff, 0xef,
		0x1a, 0xca, 0xd6, 0xfb, 0x7e, 0x09, 0xdc, 0x32, 0x17, 0x34,
		0x31, 0x9c, 0xfa, 0x78, 0xf9, 0x88, 0x1a, 0x50, 0x6b, 0x09,
		0x24, 0xfe, 0x75, 0xcd, 0x22, 0xbd, 0xf0, 0x12, 0x10, 0x9d,
		0xc5, 0x7b, 0xbe, 0x9f, 0xa4, 0xc8, 0x7a, 0x7c, 0xb1, 0xa7,
		0x46, 0x2d, 0x48, 0xa1, 0x08, 0xd8, 0x75, 0x3c, 0x40, 0x53,
		0x59, 0x5a, 0x04, 0xf6, 0xcb, 0xd6, 0x2f, 0x51, 0x9e, 0x4a,
		0xa8, 0xef, 0x02, 0x3c, 0x61, 0xa8, 0x45, 0x5a, 0x99, 0x9a,
		0x73, 0x09, 0xc0, 0x71, 0x6f, 0x4d, 0xb7, 0x2c, 0x69, 0x6c,
		0x1e, 0x85, 0x48, 0xea, 0x33, 0x5b, 0xb1, 0x28, 0x09, 0xe3,
		0x24, 0x47, 0x7a, 0x12, 0xd8, 0x77, 0xa0, 0xfd, 0x3c, 0xcd,
		0x23, 0x53, 0x61, 0xf0, 0x9e, 0x49, 0x44, 0x19, 0x9c, 0xe1,
		0x0a, 0xb6, 0xc5, 0x93, 0x35, 0x4d, 0x06, 0xde, 0xb5, 0xac,
		0x15, 0xe2, 0x8d, 0x47, 0x6a, 0x8a, 0x29, 0x58, 0x8f, 0xfc,
		0x2b, 0x32, 0xc4, 0x0e, 0x4e, 0x75, 0xb2, 0x71, 0x44, 0x91,
		0xb6, 0xfb, 0x50, 0x60, 0xc4, 0x50, 0x4d, 0xd9, 0x1c, 0xd7,
		0x28, 0xb5, 0x2e, 0x08, 0x82, 0xfd, 0xbd, 0x7c, 0x48, 0x62,
		0x00, 0x01, 0x30, 0xa5, 0x47, 0x2a, 0x10, 0x4b, 0xfe, 0x3e,
		0xb5, 0xd7, 0xe2, 0x1e, 0x5a, 0xd0, 0xe3, 0x6b, 0xa0, 0x9c,
		0xe3, 0x12, 0x05, 0xb9, 0x1c, 0x7b, 0x3f, 0x18, 0x88, 0x11,
		0xc3, 0x4d, 0xe8, 0x83, 0x78, 0xf1, 0x8c, 0x6a, 0xb3, 0x87,
		0x7f, 0x67, 0xdf, 0x47, 0xc4, 0x2f, 0xea, 0x9c, 0x79, 0x98,
		0x35, 0x2d, 0x7b, 0x2d, 0xc7, 0x3e, 0x31, 0x13, 0x6e, 0xf3,
		0xfd, 0xa8, 0x16, 0x25, 0x03, 0x3f, 0xb1, 0x14, 0x77, 0xff,
		0xa1, 0xf2, 0xe9, 0x98, 0x32, 0xe4, 0x2f, 0x3e, 0xb0, 0x6e,
		0x29, 0x84, 0x0f, 0xdd, 0x85, 0xbb, 0xda, 0x0a, 0xc6, 0x19,
		0x78, 0x9a, 0x6b, 0xdd, 0x8b, 0x40, 0x39, 0x1f, 0x19, 0xce,
		0xad, 0x79, 0xc9, 0xda, 0x59, 0xbf, 0x6e, 0xbc, 0xb0, 0xc8,
		0x6e, 0x61, 0xeb, 0x2e, 0xba, 0xa6, 0x50, 0x06, 0x8c, 0x74,
		0xab, 0x23, 0x4f, 0x9d, 0x9e, 0x20, 0x67, 0x9a, 0x0c, 0xea,
		0x60, 0x14, 0x85, 0x42, 0xc4, 0x7d, 0xc0, 0x1c, 0x62, 0xa5,
		0x7e, 0x65, 0x1d, 0xee, 0xa1, 0xfd, 0xed, 0x82, 0x19, 0x1e,
		0x90, 0x87, 0xab, 0xdb, 0x80, 0xb7, 0xb2, 0xab, 0xa0, 0xbc,
		0xdb, 0x99, 0x9b, 0x2e, 0x8e, 0xc4, 0x96, 0xfe, 0x31, 0x82,
		0x9e, 0xe9, 0xc9, 0xa3, 0xd0, 0xe3, 0x52, 0x0f, 0x73, 0x24,
		0xd7, 0xb6, 0x41, 0xcc, 0xfc, 0xae, 0x86, 0xb5, 0x79, 0xe5,
		0x25, 0xc8, 0x76, 0x60, 0x48, 0x3c, 0xa4, 0x74, 0x6a, 0x7b,
		0x7e, 0xa8, 0x5b, 0xf8, 0x5b, 0x79, 0x0f, 0x19, 0x95, 0x7d,
		0x7f, 0x56, 0x31, 0xcc, 0x34, 0xd0, 0xb8, 0x0c, 0x29, 0x3f,
		0x83, 0x54, 0x02, 0x75, 0xe3, 0x79, 0xb6, 0x64, 0xf9, 0x66,
		0x7f, 0x90, 0xe1, 0xa7, 0x7b, 0xfa, 0x17, 0x68, 0x07, 0xd1,
		0xef, 0x35, 0x52, 0xf3, 0x56, 0xde, 0x35, 0x35, 0x14, 0x1b,
		0xcd, 0x97, 0x1a, 0xe5, 0xf4, 0xb5, 0xae, 0xfc, 0x53, 0x0c,
		0xec, 0x36, 0xf0, 0xb4, 0x30, 0xea, 0x9b, 0xde, 0x60, 0x31,
		0x72, 0x7e, 0x99, 0x41, 0x78, 0x90, 0x81, 0x73, 0x0a, 0x81,
		0xe2, 0xaa, 0x70, 0x68, 0x67, 0x7a, 0x52, 0x4a, 0x4d, 0xd4,
		0x9f, 0x90, 0x0d, 0xd9, 0x3a, 0x1f, 0x8d, 0x80, 0xfa, 0x68,
		0x62, 0x54, 0xae, 0xb7, 0x0f, 0x3a, 0xfa, 0xed, 0xc9, 0xce,
		0x20, 0x8d, 0x88, 0x48, 0x79, 0x82, 0x8a, 0xf6, 0xc5, 0x4d,
		0x6e, 0xed, 0x38, 0x81, 0x75, 0x65, 0xac, 0xf6, 0x72, 0xee,
		0xf0, 0x2e, 0x6d, 0x3f, 0x2e, 0x2a, 0x54, 0xf2, 0xf1, 0x5a,
		0xb9, 0x81, 0xce, 0x56, 0xbf, 0xed, 0xab, 0xd5, 0x32, 0xf1,
		0x95, 0x86, 0x81, 0xf9, 0xfd, 0x38, 0x1d, 0xc7, 0x9f, 0x55,
		0xe1, 0x2c, 0xd1, 0x28, 0x66, 0x26, 0x81, 0xf5, 0x12, 0x72,
		0x02, 0x35, 0x87, 0x65, 0x01, 0xea, 0x4f, 0xf7, 0x95, 0xba,
		0xb1, 0x09, 0x91, 0x9c, 0x89, 0xb4, 0x48, 0x1d, 0x7b, 0xb4,
		0x37, 0x76, 0x6c, 0xce, 0x83, 0x0e, 0x2c, 0xb8, 0xd5, 0xe0,
		0x43, 0x36, 0x3c, 0x2d, 0xb1, 0x0f, 0x29, 0x4e, 0x1d, 0x36,
		0x58, 0x9a, 0xcd, 0xf4, 0xa7, 0xad, 0xae, 0xdb, 0x8f, 0x59,
		0xd0, 0x78, 0xb4, 0xcb, 0xfd, 0xe6, 0x1b, 0xa9, 0x24, 0x26,
		0xb7, 0xa2, 0xc0, 0xbd, 0x08, 0x53, 0xd2, 0x4e, 0xb2, 0x19,
		0x90, 0xb1, 0xb0, 0xa7, 0x57, 0xaa, 0xeb, 0x11, 0x71, 0xdd,
		0x3b, 0xae, 0x04, 0xf4, 0x44, 0xa5, 0x27, 0xeb, 0xb3, 0x2e,
		0xdd, 0x70, 0x7e, 0x0d, 0x2b, 0x2f, 0xc4, 0x4c, 0xee, 0xff,
		0x4d, 0x94, 0xde, 0x6f, 0x48, 0xa1, 0x7b, 0xa2, 0xa2, 0xef,
		0x3a, 0xa3, 0x0c, 0x99, 0x4e, 0x6e, 0xd4, 0x64, 0xbb, 0xd2,
		0xa0, 0x38, 0x39, 0xe8, 0x16, 0xa3, 0x4b, 0x77, 0xb7, 0xa4,
		0xf6, 0xc0, 0x12, 0xec, 0xfe, 0x97, 0xc0, 0xa4, 0xfc, 0x90,
		0x8c, 0x07, 0xfb, 0x0a, 0xc9, 0x4f, 0xe8, 0xf7, 0x6e, 0x45,
		0xf6, 0xab, 0x8b, 0xb1, 0x58, 0x56, 0x5b, 0xda, 0x12, 0xc0,
		0x4c, 0xe1, 0x74, 0x1c, 0xb9, 0x14, 0xdf, 0xe4, 0x17, 0x53,
		0xb3, 0x0c, 0xe2, 0xe1, 0xbc, 0xfc, 0xe6, 0x9c, 0xed, 0xed,
		0x82, 0x20, 0x2a, 0x4d, 0xdf, 0xd8, 0x78, 0x6e, 0xf5, 0x16,
		0x57, 0x48, 0x58, 0x9a, 0xb6, 0x57, 0x20, 0xf0, 0x74, 0xcc,
		0xcf, 0xdd, 0x5c, 0xf4, 0xd6, 0x7e, 0xc4, 0x53, 0x13, 0xa4,
		0xf1, 0x68, 0xc5, 0xa4, 0xe1, 0x91, 0x58, 0xcf, 0x5c, 0x9f,
		0x36, 0xbf, 0x20, 0x21, 0x1b, 0x34, 0x5c, 0xbd, 0x2c, 0x64,
		0xc5, 0x6f, 0x7f, 0xb9, 0x91, 0x65, 0xcf, 0x25, 0x44, 0x93,
		0x99, 0x1f, 0xb9, 0x82, 0xeb, 0xc5, 0x3b, 0x4f, 0xd2, 0xad,
		0x5e, 0xaf, 0x69, 0xfc, 0x2f, 0x54, 0xe4, 0xd0, 0x34, 0xdb,
		0x47, 0xef, 0x8b, 0x16, 0x54, 0x2b, 0x67, 0x27, 0x3c, 0x50,
		0x9f, 0xf0, 0x6c, 0xe6, 0xe2, 0x08, 0xd0, 0x0a, 0x36, 0x76,
		0xc1, 0x65, 0x62, 0x01, 0x4d, 0x70, 0xb8, 0x42, 0xe4, 0x88,
		0xee, 0x63, 0x0b, 0x74, 0x64, 0x2b, 0x15, 0xf1, 0x90, 0x6c,
		0xa3, 0x3f, 0xf4, 0x94, 0xca, 0x3f, 0x48, 0x64, 0x5a, 0x09,
		0x36, 0x5c, 0x7c, 0x4f, 0x75, 0xf0, 0x4e, 0xe3, 0x3d, 0xf3,
		0x6d, 0x9b, 0x4f, 0x99, 0x29, 0xf1, 0xe2, 0xff, 0x92, 0x38,
		0x5d, 0xf6, 0xf3, 0xd2, 0xc5, 0x5d, 0x1d, 0x89, 0x78, 0x09,
		0xbb, 0x6a, 0x3a, 0x0a, 0x9c, 0x01, 0x4c, 0xd9, 0xfd, 0x55,
		0x48, 0xa1, 0xf7, 0xa1, 0x9c, 0x55, 0xc1, 0x31, 0xaf, 0xa5,
		0x03, 0xbb, 0xba, 0x8c, 0x76, 0x0a, 0xf0, 0xd6, 0x03, 0x57,
		0x71, 0xf5, 0xfe, 0xad, 0x7f, 0x36, 0x23, 0x57, 0xba, 0x9b,
		0x20, 0x92, 0x14, 0x9e, 0xcf, 0x76, 0x59, 0xcd, 0xe2, 0x47,
		0x12, 0xb4, 0x1e, 0xb5, 0x52, 0xc3, 0x7f, 0xda, 0xa0, 0x35,
		0xc2, 0x55, 0x56, 0x97, 0xd0, 0x57, 0xa0, 0xcc, 0xcd, 0x27,
		0xcc, 0x08, 0x1c, 0x9c, 0x06, 0xfd, 0xbe, 0x47, 0xdf, 0x78,
		0x08, 0x17, 0xb6, 0x7f, 0x31, 0x1e, 0x0a, 0xdc, 0x54, 0xcc,
		0x06, 0xf7, 0xbf, 0xee, 0x74, 0x48, 0x21, 0x39, 0x6c, 0xf0,
		0x1f, 0x29, 0x19, 0x76, 0x19, 0x7f, 0xc2, 0x38, 0x94, 0x8a,
		0xe4, 0x2d, 0x7f, 0xff, 0xf7, 0x45, 0x57, 0x21, 0x98, 0xaa,
		0x65, 0x35, 0xba, 0x3c, 0x56, 0xff, 0x62, 0x88, 0xb9, 0xe8,
		0xe9, 0xdb, 0xd1, 0xf8, 0x59, 0x0f, 0x50, 0x13, 0x8b, 0xc2,
		0xc2, 0xbd, 0xba, 0x44, 0x44, 0x36, 0xe1, 0xb0, 0x13, 0x32,
		0x3a, 0x68, 0x4d, 0x62, 0xa8, 0x50, 0x87, 0x44, 0xbb, 0x31,
		0xfb, 0xe3, 0x25, 0x73, 0xa0, 0xf1, 0x2c, 0xf5, 0x34, 0x19,
		0x09, 0x66, 0xa9, 0x46, 0x21, 0xc7, 0xa2, 0x91, 0x7d, 0xe9,
		0xf4, 0x9d, 0xa9, 0x82, 0x28, 0xf4, 0x1f, 0x28, 0xa8, 0xa9,
		0x01, 0xf1, 0x97, 0x70, 0x9d, 0x61, 0x72, 0xd5, 0x8b, 0x84,
		0xcf, 0xf1, 0x7c, 0xc2, 0x7d, 0xa7, 0xa3, 0x25, 0x4f, 0x68,
		0x3f, 0x5e, 0x8e, 0xb6, 0x16, 0x18, 0x78, 0xcf, 0x63, 0x53,
		0x57, 0x08, 0x11, 0x55, 0x65, 0x1b, 0x55, 0x5e, 0x8d, 0xce,
		0x68, 0x2d, 0x77, 0x16, 0x1e, 0x87, 0xa6, 0xd0, 0x11, 0x44,
		0x37, 0x8f, 0xa2, 0x8b, 0xb0, 0xdb, 0x4b, 0x01, 0x2f, 0xd7,
		0x1e, 0x16, 0x93, 0xb5, 0x19, 0xad, 0xb7, 0xde, 0x13, 0x0d,
		0xac, 0xcd, 0x77, 0x6a, 0x15, 0x61, 0xeb, 0xe0, 0xd1, 0xbf,
		0xcf, 0x69, 0x10, 0x28, 0x72, 0x58, 0xfe, 0xc7, 0xce, 0xbc,
		0x92, 0xfd, 0x5f, 0x9c, 0xd6, 0x6d, 0x56, 0xbd, 0x39, 0x35,
		0xe0, 0xa6, 0x73, 0x19, 0x45, 0x6a, 0xf2, 0x36, 0x8a, 0x0a,
		0x58, 0xe1, 0x23, 0x07, 0x28, 0xa4, 0x89, 0x7a, 0x51, 0xbb,
		0xba, 0xd6, 0x42, 0x78, 0xbb, 0x5f, 0x42, 0x0b, 0xc0, 0x8f,
		0xc9, 0x3c, 0xc3, 0x7e, 0x41, 0x22, 0x94, 0x3c, 0x02, 0xd3,
		0x2c, 0xec, 0xf8, 0x8f, 0x14, 0x62, 0x7c, 0x78, 0x47, 0x73,
		0xe8, 0xd5, 0x40, 0x2d, 0x26, 0x82, 0x72, 0x16, 0xd7, 0xdd,
		0x9e, 0x8d, 0x8e, 0xfb, 0xaf, 0x83, 0x73, 0xd8, 0x4d, 0x85,
		0x28, 0x53, 0x72, 0xc1, 0xb0, 0x08, 0x5a, 0xc6, 0x6e, 0xbe,
		0xf9, 0x21, 0xdd, 0x8c, 0x30, 0xf5, 0x5e, 0xbe, 0xec, 0xcc,
		0xd2, 0x8a, 0x1a, 0xc3, 0xf4, 0x83, 0x98, 0x86, 0x3b, 0x4e,
		0x6d, 0x3e, 0x9e, 0xbe, 0x84, 0x80, 0x2f, 0x27, 0xbb, 0x6a,
		0x3e, 0x51, 0x18, 0xb0, 0x7a, 0x82, 0x82, 0x02, 0x53, 0x6d,
		0x2d, 0x2a, 0xab, 0x4a, 0x2e, 0x3b, 0xa4, 0x88, 0x90, 0x99,
		0x74, 0x10, 0x27, 0xa7, 0xe9, 0x5c, 0x52, 0x5b, 0xf5, 0xb0,
		0x00, 0xc7, 0x52, 0x91, 0x04, 0xc7, 0xff, 0x01, 0xb7, 0x56,
		0x40, 0x9f, 0x56, 0x54, 0xde, 0x96, 0x65, 0x61, 0x76, 0x06,
		0xb6, 0xdb, 0x16, 0xc8, 0x1c, 0x0b, 0xff, 0xf0, 0x73, 0xd0,
		0x89, 0x22, 0x86, 0xc6, 0xac, 0xc5, 0xb1, 0x98, 0x21, 0xd0,
		0x6c, 0x30, 0x92, 0x43, 0xb4, 0xa9, 0x80, 0xc8, 0xdd, 0x6f,
		0xf0, 0x4b, 0x57, 0x80, 0x6d, 0x53, 0xfb, 0xd4, 0x95, 0x39,
		0xde, 0x6e, 0xf5, 0x28, 0x9e, 0x92, 0x94, 0x42, 0x99, 0x42,
		0x4d, 0x02, 0x0a, 0x7b, 0x9b, 0x8f, 0xc6, 0x32, 0xb0, 0xd9,
		0x5d, 0x81, 0x74, 0xd2, 0x76, 0xdb, 0x64, 0x21, 0xdf, 0xc2,
		0xee, 0x5f, 0x77, 0x88, 0x13, 0x60, 0x32, 0xd8, 0x97, 0xf4,
		0x1c, 0x77, 0xe0, 0x49, 0xce, 0x1d, 0xa2, 0xbb, 0x66, 0xca,
		0x27, 0xfd, 0xc1, 0x96, 0x3a, 0x50, 0x4f, 0x1f, 0xbb, 0x56,
		0x24, 0x85, 0x76, 0x77, 0xfd, 0x84, 0x7b, 0xb4, 0x3c, 0x87,
		0x01, 0x09, 0x89, 0x4c, 0x0f, 0x8e, 0x44, 0xc0, 0x49, 0x39,
		0x49, 0x8b, 0x93, 0x09, 0xa6, 0x8d, 0x93, 0xf1, 0x5a, 0x3f,
		0x1f, 0x64, 0x2f, 0xd2, 0xe2, 0xbe, 0x99, 0x38, 0x38, 0xf1,
		0xec, 0x15, 0x46, 0xe3, 0x8a, 0x95, 0xe6, 0x3c, 0xf8, 0xa3,
		0x38, 0xb8, 0xc0, 0x20, 0x5f, 0xcb, 0x36, 0xe4, 0x90, 0x79,
		0x0c, 0x8e, 0xb6, 0xe5, 0x48, 0x23, 0x2d, 0xcb, 0x3c, 0x88,
		0x31, 0x35, 0xe7, 0x8e, 0xfa, 0xf5, 0x81, 0x36, 0x35, 0x96,
		0x98, 0x39, 0xb0, 0x2d, 0xa3, 0xd7, 0x84, 0x3a, 0x92, 0x30,
		0x07, 0x9b, 0x70, 0x61, 0xda, 0xd6, 0x80, 0x94, 0x7b, 0x93,
		0x01, 0x10, 0xe2, 0x03, 0x23, 0x83, 0xc2, 0xef, 0x2f, 0x4e,
		0x31, 0x01, 0x87, 0x0c, 0x0b, 0x0c, 0x1f, 0x20, 0xaa, 0x75,
		0x52, 0xb5, 0xbc, 0xa2, 0x55, 0xd6, 0x06, 0x3f, 0xf1, 0x6a,
		0x1e, 0xa4, 0x00, 0x43, 0xdb, 0x0f, 0x9a, 0xd4, 0xc4, 0x45,
		0x2f, 0x20, 0x94, 0x57, 0xbc, 0x6e, 0xeb, 0x63, 0xda, 0x90,
		0xb2, 0x56, 0x9b, 0x6c, 0x81, 0xb3, 0x15, 0xd3, 0x1d, 0xe1,
		0xe4, 0x75, 0xaa, 0x3b, 0xbf, 0x45, 0x46, 0xd7, 0x0d, 0x31,
		0xb1, 0xf5, 0x8b, 0xfb, 0x5b, 0x3e, 0x48, 0x4b, 0xea, 0x09,
		0x0c, 0x82, 0xd3, 0x7a, 0x45, 0x1c, 0xc2, 0x4d, 0xda, 0x4d,
		0x08, 0xd1, 0x5d, 0x7c, 0xb5, 0x50, 0x24, 0xbf, 0x3b, 0x5a,
		0xb7, 0xca, 0x76, 0xc4, 0xee, 0x64, 0xf7, 0xc7, 0x77, 0x2d,
		0x06, 0x8d, 0x9b, 0xee, 0x7c, 0x6b, 0xf5, 0x6b, 0x8f, 0x62,
		0x4d, 0x37, 0x66, 0x31, 0x5f, 0xde, 0x2b, 0xb2, 0xe4, 0x70,
		0xda, 0xeb, 0x49, 0xe4, 0x1b, 0x1f, 0xdf, 0x45, 0xee, 0x5a,
		0x69, 0x68, 0x59, 0xe5, 0x71, 0x14, 0xcc, 0x4e, 0x73, 0xa4,
		0x20, 0x38, 0x5c, 0xc7, 0xa7, 0x28, 0xd5, 0xa0, 0xbf, 0xfd,
		0x6a, 0x0f, 0x92, 0x6f, 0x02, 0x64, 0x49, 0x19, 0xe4, 0x18,
		0x23, 0xa2, 0xda, 0x9b, 0x75, 0x7b, 0xc8, 0xdb, 0x8f, 0x78,
		0x68, 0x4c, 0x67, 0x47, 0x82, 0xf6, 0x10, 0x44, 0x23, 0x36,
		0x74, 0x02, 0x58, 0xad, 0x75, 0xd4, 0xad, 0x59, 0x9e, 0x2e,
		0x99, 0x67, 0xc2, 0x5b, 0x89, 0x91, 0x01, 0x37, 0x2e, 0xfe,
		0x50, 0xc4, 0x6d, 0xe7, 0x8b, 0x34, 0x53, 0x22, 0xd0, 0xc3,
		0x1f, 0x16, 0xc9, 0x14, 0x2f, 0x0b, 0x76, 0xdf, 0x29, 0xb9,
		0x32, 0xd4, 0xcd, 0x5e, 0x04, 0xdb, 0xf5, 0x47, 0x3f, 0x99,
		0x26, 0xcb, 0x6b, 0xac, 0xae, 0x2b, 0xbc, 0x75, 0xf2, 0x8d,
		0x64, 0xff, 0xd2, 0x2b, 0xfb, 0x8c, 0x81, 0x79, 0xed, 0x8d,
		0x86, 0x89, 0xc7, 0xfd, 0xe8, 0xde, 0x1a, 0xe1, 0x96, 0x30,
		0xba, 0xe3, 0x8d, 0xc8, 0x43, 0xd5, 0x97, 0x52, 0xe5, 0x19,
		0x0b, 0xd4, 0xba, 0x41, 0xea, 0x9a, 0xad, 0xcc, 0x94, 0x24,
		0x6e, 0x9c, 0x8f, 0xd1, 0x69, 0x6b, 0x4a, 0x50, 0x92, 0xba,
		0x73, 0x41, 0xb1, 0xd1, 0x82, 0x10, 0xbb, 0xcf, 0x3c, 0x4b,
		0xdf, 0x49, 0xdc, 0xdf, 0xd3, 0xbb, 0xa6, 0xac, 0x63, 0xbc,
		0x88, 0x6a, 0x4f, 0x3b, 0xfa, 0xd4, 0xdd, 0xd4, 0x3c, 0x3a,
		0xe5, 0x68, 0xd7, 0x86, 0xbd, 0xcd, 0xd3, 0x6d, 0xa8, 0x22,
		0xca, 0x71, 0xd6, 0xd0, 0xa7, 0xa7, 0x75, 0xd2, 0x70, 0xd4,
		0xb4, 0x27, 0x78, 0x2b, 0xcd, 0x16, 0x10, 0x81, 0xb1, 0x59,
		0xfe, 0xe4, 0x1b, 0x62, 0xd1, 0xcf, 0x63, 0x7c, 0x72, 0x78,
		0xc2, 0x96, 0xaf, 0x88, 0x76, 0x28, 0x0a, 0x47, 0xeb, 0xa8,
		0xd9, 0xd1, 0xef, 0xea, 0x01, 0x20, 0x15, 0x63, 0x89, 0xb4,
		0xe7, 0x99, 0xa0, 0x3b, 0x2f, 0xdf, 0x99, 0x0b, 0xc3, 0xef,
		0xa6, 0xf3, 0xe0, 0x90, 0xc9, 0x82, 0x4c, 0x3e, 0x51, 0xe9,
		0x2c, 0x86, 0x7d, 0x31, 0x7e, 0x8f, 0x5d, 0x7a, 0x24, 0xff,
		0x65, 0xa9, 0xb6, 0xbc, 0x18, 0xc8, 0x28, 0x4c, 0x8d, 0xda,
		0xc7, 0x98, 0x6a, 0x54, 0xad, 0x22, 0x19, 0xca, 0x3d, 0x4d,
		0x88, 0xcc, 0x4e, 0x5c, 0x2a, 0xef, 0xd0, 0x60, 0x3c, 0xe3,
		0x49, 0xd2, 0xf6, 0x59, 0x38, 0x25, 0xed, 0x31, 0x52, 0x18,
		0xb4, 0x31, 0x31, 0x43, 0x8f, 0x85, 0x7d, 0x59, 0xa8, 0xd2,
		0x26, 0x4c, 0xa0, 0xfb, 0xf8, 0xdb, 0x6a, 0x12, 0xad, 0x40,
		0x73, 0xb1, 0x5d, 0x61, 0xf5, 0x49, 0x65, 0x7b, 0x44, 0x30,
		0x49, 0x7a, 0xd7, 0xe9, 0xce, 0x34, 0x9c, 0x64, 0x6e, 0x31,
		0xae, 0x48, 0xad, 0x4c, 0x69, 0xc4, 0xd9, 0xf8, 0x2a, 0xb0,
		0x7e, 0xb3, 0xba, 0xec, 0xea, 0x2b, 0x49, 0x09, 0x21, 0x53,
		0x63, 0xb7, 0xc6, 0xd6, 0x80, 0x9c, 0x77, 0x27, 0xe9, 0xf6,
		0xc7, 0x06, 0x2c, 0x8e, 0xc1, 0xb8, 0x31, 0x15, 0xa9, 0x87,
		0x13, 0xfd, 0xfd, 0x32, 0x18, 0xdd, 0x5c, 0xba, 0xf3, 0x70,
		0x94, 0x09, 0x9d, 0xb6, 0xbc, 0x2c, 0x91, 0xef, 0x42, 0xb7,
		0x13, 0x25, 0x90, 0x77, 0x10, 0xdd, 0x80, 0xa9, 0xd9, 0xab,
		0xef, 0xcf, 0xbe, 0x75, 0x93, 0xe2, 0xbc, 0xe7, 0x26, 0xdf,
		0x77, 0x88, 0xe0, 0x90, 0xc0, 0x2d, 0x20, 0xc9, 0x8c, 0xfa,
		0x82, 0x91, 0x57, 0xfb, 0xb1, 0xec, 0xdf, 0x2b, 0x2f, 0x9b,
		0x6a, 0x23, 0xa0, 0x3e, 0x2a, 0xa3, 0x39, 0x8c, 0xea, 0x09,
		0x06, 0x03, 0x7b, 0xa8, 0xe7, 0x77, 0x32, 0x21, 0x1a, 0x29,
		0xcd, 0x22, 0x60, 0x9b, 0xc0, 0xfe, 0xc2, 0xf3, 0x9c, 0x83,
		0x56, 0x8f, 0xf0, 0x22, 0xad, 0xfd, 0xe3, 0xa9, 0xe3, 0xc5,
		0xbf, 0xd1, 0xf3, 0xbd, 0xc6, 0x43, 0xc3, 0x84, 0xf2, 0x46,
		0x91, 0x5b, 0x37, 0x43, 0xb9, 0xcf, 0xe4, 0x47, 0x7a, 0x5a,
		0xfd, 0xd2, 0xc0, 0xb6, 0x7e, 0x99, 0xea, 0x15, 0x1b, 0x31,
		0x3c, 0x14, 0x6b, 0xbf, 0x26, 0x4f, 0xfa, 0x38, 0x39, 0xef,
		0x75, 0x4c, 0xda, 0x4c, 0x4f, 0x00, 0xef, 0x0f, 0xc6, 0x3b,
		0x5b, 0x4c, 0xf1, 0x1b, 0x35, 0x92, 0xb9, 0x74, 0x9e, 0x0f,
		0x16, 0x2a, 0x2b, 0x3c, 0xa0, 0x80, 0xf0, 0xe5, 0x56, 0xaa,
		0xdb, 0x0e, 0x4f, 0xfc, 0x5f, 0xb2, 0xd1, 0x0b, 0xe2, 0x04,
		0x1e, 0xe8, 0x6f, 0x4f, 0x81, 0xdb, 0xc7, 0x1a, 0x0c, 0xb0,
		0x00, 0xed, 0x8a, 0x7b, 0x1d, 0x44, 0xde, 0x30, 0xfa, 0xb6,
		0x88, 0x3b, 0x73, 0x81, 0xb3, 0x57, 0x31, 0x38, 0x0e, 0x13,
		0x77, 0x9e, 0x5a, 0x4d, 0xa1, 0x93, 0x73, 0x30, 0x0d, 0x44,
		0x56, 0x97, 0x88, 0xbf, 0xc4, 0x36, 0xe9, 0xec, 0xe2, 0x58,
		0x11, 0x39, 0x33, 0x71, 0x6b, 0xd4, 0xb7, 0x4a, 0x9a, 0x2c,
		0xc0, 0xac, 0x88, 0xeb, 0x08, 0x8c, 0x43, 0x48, 0xa2, 0xba,
		0xfa, 0x98, 0x95, 0x92, 0x2d, 0xa2, 0xee, 0x72, 0xb3, 0x7e,
		0xd9, 0x11, 0x1a, 0xa3, 0x41, 0xc4, 0x65, 0x19, 0x1d, 0xdb,
		0x92, 0xf4, 0x3e, 0xf4, 0xaa, 0x49, 0xad, 0x77, 0xc6, 0x03,
		0xdd, 0x19, 0x7a, 0x8d, 0x62, 0x09, 0x19, 0xd7, 0xf7, 0xd3,
		0x84, 0xf1, 0xcb, 0x45, 0x54, 0xd1, 0x74, 0x1e, 0xf8, 0xe6,
		0xa4, 0xfd, 0xaa, 0x30, 0xf8, 0xf9, 0xe1, 0x2b, 0xcf, 0x65,
		0x22, 0xf9, 0x73, 0xcf, 0x5d, 0xe3, 0xde, 0x93, 0x94, 0x84,
		0x8f, 0xe1, 0x66, 0x99, 0x6a, 0x71, 0x76, 0xae, 0xdb, 0x16,
		0x2b, 0x73, 0xab, 0x4e, 0x61, 0x08, 0xa0, 0xdc, 0x3f, 0xd4,
		0xe3, 0xac, 0x89, 0x7a, 0x31, 0xc1, 0xb9, 0x20, 0x92, 0x10,
		0x58, 0x1b, 0x87, 0xfd, 0xe3, 0x7c, 0x42, 0xc8, 0x26, 0x06,
		0x50, 0xce, 0xbb, 0x00, 0x0b, 0x9f, 0x09, 0xdc, 0xac, 0x31,
		0xe0, 0xcd, 0xa6, 0xee, 0x23, 0xef, 0x64, 0x87, 0x65, 0x5f,
		0xfb, 0x5f, 0x6c, 0x29, 0x4a, 0xf9, 0xd3, 0xaf, 0x4e, 0xaa,
		0x39, 0x30, 0x82, 0x05, 0xc6, 0x1c, 0x1e, 0x83, 0x8f, 0xe4,
		0xf7, 0x1c, 0xca, 0x3b, 0x03, 0x01, 0x80, 0xe4, 0x6a, 0x84,
		0x71, 0xd4, 0xe9, 0xa6, 0xba, 0x72, 0x89, 0x69, 0xfa, 0x56,
		0x6d, 0x26, 0xe3, 0x1d, 0xfa, 0x91, 0x8f, 0x4c, 0xcc, 0x50,
		0x74, 0x75, 0x2f, 0xe3, 0x97, 0x16, 0x07, 0x78, 0x40, 0xde,
		0x58, 0x22, 0xca, 0x41, 0xde, 0xb9, 0x92, 0x56, 0x55, 0x89,
		0x85, 0x20, 0x44, 0x5a, 0xd0, 0xfc, 0x88, 0x4e, 0x49, 0x81,
		0x32, 0xcf, 0x44, 0xb6, 0xee, 0x58, 0x0e, 0x22, 0xf8, 0x8f,
		0x05, 0x2a, 0x78, 0x30, 0x34, 0x11, 0xb0, 0x8f, 0x21, 0x10,
		0xb7, 0xe2, 0x80, 0xa2, 0xe0, 0x69, 0xbc, 0x29, 0xe3, 0x48,
		0x08, 0xa6, 0xdc, 0xbc, 0x68, 0xd7, 0xd7, 0x84, 0x23, 0x3f,
		0xb0, 0x0a, 0x69, 0xa1, 0x87, 0x13, 0x29, 0x0d, 0x71, 0x3d,
		0xad, 0xad, 0xf0, 0x5e, 0x53, 0x04, 0x03, 0x9e, 0x4b, 0xb2,
		0x24, 0xc2, 0x82, 0x8c, 0xef, 0x4c, 0x70, 0xcd, 0x58, 0xb3,
		0x19, 0xa4, 0xe8, 0x0e, 0xc0, 0xfd, 0x57, 0x2b, 0x65, 0xee,
		0xa5, 0x43, 0x02, 0xf2, 0xbe, 0xe7, 0x66, 0xc9, 0xb5, 0xfd,
		0x70, 0xc0, 0x93, 0x9b, 0xe8, 0x60, 0x74, 0xd4, 0xcc, 0xa5,
		0xa6, 0xa2, 0x66, 0x35, 0xdc, 0x74, 0xf3, 0x02, 0x16, 0x74,
		0x9a, 0x91, 0x0a, 0x3d, 0xa4, 0x8b, 0x98, 0xd1, 0xfd, 0x29,
		0x34, 0xea, 0x52, 0x89, 0x16, 0x62, 0xdd, 0xed, 0xb5, 0x8b,
		0xed, 0xe4, 0x03, 0xd9
	};

	static const uint8_t keys_a_ps5[0x70 * 0x20] = {
		0x7f, 0xae, 0x83, 0x20, 0x47, 0x39, 0xb6, 0x5b, 0x23, 0x9b,
		0x5c, 0xa5, 0x37, 0xc4, 0xdf, 0x59, 0x26, 0x81, 0x0a, 0x53,
		0xa1, 0xae, 0xa3, 0xa3, 0xf9, 0xbe, 0xaa, 0x9f, 0xca, 0xe0,
		0x9e, 0x45, 0x8d, 0x43, 0x7f, 0x53, 0x3e, 0xc0, 0x17, 0x90,
		0xd2, 0x03, 0x2d, 0x92, 0xa8, 0x1b, 0x3e, 0x17, 0x9a, 0x9d,
		0xc9, 0x45, 0x39, 0x9e, 0x6c, 0x5f, 0xb6, 0xb2, 0x42, 0x57,
		0x6a, 0x1a, 0xa0, 0x35, 0x89, 0xdd, 0x6a, 0xc6, 0xa9, 0x2d,
		0x06, 0xa2, 0xd3, 0x70, 0xd5, 0xa7, 0x91, 0x32, 0x42, 0x5a,
		0x44, 0x0f, 0x35, 0x86, 0x18, 0xc0, 0x75, 0x9f, 0x8e, 0x07,
		0xb6, 0x80, 0x46, 0xc1, 0xbc, 0xe3, 0x4f, 0xc0, 0xdc, 0xa1,
		0xaf, 0xfb, 0x1b, 0x2b, 0x4d, 0xb6, 0x14, 0x8f, 0xaf, 0xeb,
		0x49, 0xdd, 0x2a, 0xb3, 0xfd, 0xb2, 0x32, 0x86, 0x3c, 0xe7,
		0x1a, 0x96, 0x14, 0x9d, 0xb1, 0xd9, 0x90, 0x72, 0xb3, 0xb3,
		0x3d, 0x6f, 0x3d, 0xcc, 0x26, 0xbc, 0x51, 0xda, 0x91, 0x6e,
		0xc6, 0xa9, 0xf9, 0x9f, 0x86, 0x07, 0x13, 0x70, 0xdc, 0xe2,
		0x21, 0xd7, 0x4c, 0xb3, 0xc3, 0x98, 0x02, 0x94, 0x9a, 0x63,
		0x6d, 0x34, 0x73, 0x2d, 0xa1, 0x23, 0x83, 0xa8, 0x58, 0x34,
		0x3f, 0x30, 0x1b, 0x9b, 0x01, 0xfb, 0xfa, 0xe4, 0x02, 0xa3,
		0x52, 0x4d, 0xfa, 0xc8, 0x16, 0xb8, 0xfa, 0xc5, 0x67, 0xe1,
		0x89, 0x28, 0xe1, 0xe5, 0xf4, 0xd7, 0xa2, 0x9f, 0xcc, 0xd4,
		0x39, 0x91, 0x41, 0xe9, 0xa2, 0x7a, 0x44, 0xb4, 0xb6, 0x4b,
		0xe9, 0xd7, 0x2c, 0x0b, 0x13, 0xc5, 0xd9, 0xa4, 0x50, 0x0c,
		0x15, 0x94, 0xc8, 0x65, 0x88, 0xac, 0xe4, 0x14, 0x77, 0x23,
		0x04, 0x9e, 0x30, 0xec, 0x3f, 0x27, 0xe7, 0xc2, 0xb1, 0x76,
		0x15, 0xbb, 0x45, 0x44, 0x6f, 0x1b, 0xcd, 0x2b, 0xae, 0x07,
		0x77, 0x63, 0x1a, 0xde, 0x10, 0xad, 0xe6, 0xde, 0x8c, 0x54,
		0xe5, 0x98, 0xde, 0xcc, 0x3b, 0x8b, 0xe1, 0x52, 0xb2, 0x78,
		0xa4, 0xa6, 0xa8, 0x43, 0x0a, 0xdb, 0x6a, 0xd6, 0x97, 0xe4,
		0x1a, 0x51, 0x8a, 0x81, 0x9c, 0x7d, 0xb9, 0x35, 0x67, 0x7a,
		0xfa, 0xbf, 0x59, 0xce, 0x6c, 0x30, 0xd8, 0x00, 0x95, 0x49,
		0x73, 0x75, 0xc6, 0xfc, 0xb7, 0xf4, 0x93, 0x16, 0x18, 0xc9,
		0x1b, 0x6e, 0x71, 0x84, 0x30, 0x85, 0xe3, 0x00, 0x19, 0x43,
		0xcb, 0x04, 0x61, 0x58, 0x83, 0x91, 0x8b, 0xf8, 0x0a, 0x3d,
		0x4c, 0x47, 0x71, 0x3c, 0x34, 0xf0, 0xa1, 0xee, 0x5d, 0xea,
		0xc0, 0xb9, 0x68, 0xee, 0xa2, 0x61, 0xb9, 0xb4, 0x69, 0x12,
		0x09, 0x90, 0x74, 0x4b, 0xc4, 0xbc, 0xaf, 0x32, 0x6b, 0x9c,
		0x69, 0x72, 0x15, 0x3b, 0xb2, 0x5d, 0x2c, 0x59, 0x95, 0x2b,
		0x72, 0xc3, 0xae, 0x45, 0x4c, 0xda, 0xfa, 0x0c, 0x66, 0x9c,
		0xf8, 0x23, 0xf3, 0x74, 0x9d, 0xf9, 0x0e, 0x5c, 0x18, 0x22,
		0x02, 0x0f, 0x96, 0x0c, 0xe9, 0x9b, 0xb9, 0x05, 0xf5, 0xf9,
		0x38, 0xb7, 0xcd, 0x36, 0x03, 0xb0, 0x89, 0x48, 0x3e, 0x34,
		0xb5, 0xe6, 0x8f, 0xf0, 0xa7, 0x20, 0xad, 0x42, 0x0f, 0x55,
		0xbc, 0x16, 0x8c, 0x81, 0xab, 0xb7, 0x6d, 0x8b, 0x88, 0x88,
		0xde, 0x14, 0x59, 0x7b, 0xf4, 0x28, 0x8a, 0x58, 0x26, 0xbc,
		0xed, 0x03, 0x27, 0x7a, 0x59, 0x35, 0xdd, 0x0f, 0xff, 0x23,
		0xbb, 0x18, 0x93, 0x56, 0xff, 0x48, 0xd8, 0xd3, 0x71, 0xa1,
		0xc3, 0x61, 0x30, 0x83, 0xd4, 0x06, 0xcf, 0x41, 0xde, 0x81,
		0x7a, 0x10, 0x16, 0xce, 0xe7, 0x9f, 0xdc, 0x97, 0x24, 0xc3,
		0x62, 0x04, 0x42, 0xbb, 0xb7, 0xad, 0x13, 0x4e, 0x2b, 0x0c,
		0xf1, 0x36, 0x53, 0xb5, 0xa4, 0x14, 0xec, 0xdf, 0x8a, 0xa1,
		0x9e, 0xd3, 0x62, 0xd1, 0xe8, 0xc1, 0xa3, 0x9a, 0x87, 0x76,
		0x06, 0x0a, 0x84, 0x76, 0x7b, 0x88, 0xdc, 0xeb, 0x4a, 0x68,
		0xa4, 0x0f, 0x83, 0x62, 0xfd, 0xf5, 0x01, 0x50, 0x8b, 0xbc,
		0x6c, 0xd0, 0xe9, 0x60, 0x2c, 0xe5, 0xef, 0x88, 0xce, 0xc6,
		0x83, 0x4f, 0x7a, 0x32, 0xa3, 0xc0, 0x0e, 0x05, 0x83, 0x26,
		0x49, 0x3b, 0x88, 0xe5, 0xb9, 0x26, 0xc5, 0x8f, 0xbb, 0xe3,
		0x54, 0x05, 0x6b, 0x8d, 0x42, 0x56, 0x04, 0x01, 0xd8, 0x0b,
		0xe7, 0x83, 0x0a, 0x66, 0xb2, 0x94, 0xf1, 0x8c, 0x92, 0xb2,
		0xc3, 0xa4, 0x77, 0x1c, 0xb1, 0x5d, 0x24, 0x43, 0x70, 0x2e,
		0x1d, 0x43, 0xa5, 0x03, 0x27, 0x94, 0xf6, 0xaa, 0x06, 0xba,
		0xa0, 0x58, 0xf0, 0xc5, 0xc8, 0xe9, 0x80, 0x09, 0x98, 0xc9,
		0xc1, 0x95, 0xd2, 0x13, 0x33, 0xc8, 0xfb, 0x03, 0x5a, 0x37,
		0xa4, 0xdc, 0x44, 0xd7, 0x7e, 0x4b, 0x41, 0xcb, 0xb0, 0xac,
		0xda, 0x78, 0x80, 0x80, 0x6c, 0xb2, 0xdf, 0xe6, 0x5c, 0x4c,
		0x1e, 0x82, 0xd6, 0x76, 0x56, 0x21, 0xef, 0xd0, 0x6c, 0x0e,
		0xe1, 0x6a, 0xbb, 0x07, 0xfc, 0xbd, 0xa2, 0x34, 0x86, 0x5f,
		0xb0, 0x98, 0xb9, 0xa4, 0xc9, 0x85, 0xd4, 0xb0, 0xb1, 0x45,
		0xfc, 0x3c, 0x52, 0x17, 0x28, 0x5c, 0xe5, 0x10, 0xc7, 0x77,
		0x03, 0xba, 0x51, 0x89, 0x73, 0x6a, 0x28, 0x68, 0xb4, 0x01,
		0xb2, 0xf3, 0x6d, 0x8f, 0x41, 0xf2, 0x88, 0x19, 0x1a, 0x2a,
		0x86, 0xe2, 0xd6, 0xb5, 0x40, 0xbc, 0xf7, 0xd8, 0x6b, 0xe9,
		0xe2, 0x4f, 0x03, 0xae, 0x14, 0x7e, 0x5c, 0xfd, 0xa1, 0xf6,
		0xda, 0xa7, 0x6c, 0xeb, 0xe4, 0x01, 0xa0, 0x64, 0x9e, 0xb6,
		0xad, 0x31, 0xf6, 0xba, 0x1d, 0x88, 0x85, 0xfd, 0x62, 0x6c,
		0x4b, 0x10, 0xc3, 0x3d, 0x97, 0x10, 0xfa, 0x2f, 0x83, 0x23,
		0x55, 0xb0, 0x7d, 0xe6, 0x20, 0xe9, 0xae, 0x89, 0x10, 0xe4,
		0xfc, 0xfd, 0x82, 0xe4, 0x2b, 0x5a, 0x4e, 0xea, 0x17, 0xca,
		0xf8, 0x05, 0xba, 0x03, 0x9f, 0x06, 0x02, 0x3d, 0xb4, 0x9c,
		0x35, 0x27, 0xab, 0x47, 0xb1, 0x67, 0x78, 0x79, 0x4c, 0x6e,
		0xb1, 0x97, 0xb3, 0xb7, 0x7f, 0x54, 0xa7, 0x02, 0x55, 0x35,
		0x4d, 0x0f, 0x62, 0x64, 0x9b, 0xe6, 0x7b, 0xbb, 0x65, 0x5d,
		0x53, 0x21, 0x00, 0x7c, 0xe3, 0x9b, 0xf8, 0x68, 0xae, 0xbc,
		0x0d, 0x5f, 0xdc, 0xed, 0x4d, 0xfe, 0xd0, 0x2e, 0x6c, 0x93,
		0x0c, 0xef, 0x3d, 0x98, 0x74, 0x7b, 0x30, 0xdb, 0x6c, 0xa1,
		0x73, 0x20, 0x08, 0xf8, 0x79, 0xf9, 0x04, 0xe4, 0xa3, 0xaa,
		0xf8, 0x84, 0x50, 0x47, 0x9f, 0x05, 0x34, 0xb6, 0x18, 0x05,
		0xc5, 0x93, 0xb4, 0xb6, 0x07, 0x54, 0xbf, 0xff, 0x6b, 0x1f,
		0x4b, 0xd6, 0x5f, 0x90, 0x69, 0x5e, 0x12, 0xd9, 0x84, 0x48,
		0xf3, 0xb8, 0xe2, 0x95, 0x97, 0xc0, 0xaf, 0x4d, 0x68, 0x88,
		0x1d, 0xa2, 0xa1, 0x96, 0x0a, 0x32, 0xe7, 0x56, 0x6b, 0xc3,
		0x3c, 0xee, 0x89, 0x31, 0xd5, 0xf1, 0x23, 0x5c, 0xec, 0x45,
		0x3c, 0x55, 0x8f, 0xa6, 0x39, 0x08, 0xa8, 0xf2, 0x01, 0x28,
		0xd4, 0x97, 0x7f, 0x91, 0x63, 0x49, 0xe2, 0xc7, 0xf8, 0x7e,
		0xcb, 0x01, 0x1a, 0xdd, 0xe0, 0x57, 0xf8, 0x7f, 0x71, 0x3b,
		0x17, 0xd4, 0xd9, 0x57, 0x20, 0xed, 0x49, 0x4d, 0x8f, 0x57,
		0xc5, 0xaa, 0x65, 0x70, 0x9f, 0xf1, 0x2c, 0x74, 0x0d, 0x57,
		0x07, 0x16, 0x26, 0x74, 0x8e, 0x7a, 0x5e, 0x4f, 0x44, 0xab,
		0x77, 0xe9, 0x79, 0xe2, 0x98, 0xe5, 0xca, 0x64, 0x45, 0x7c,
		0xbf, 0x8c, 0x7c, 0x0d, 0xba, 0x71, 0xec, 0xbf, 0x92, 0x46,
		0x96, 0x75, 0x5e, 0x61, 0xe2, 0x7c, 0x49, 0x4f, 0xcf, 0xa1,
		0x9a, 0xc4, 0x3b, 0x99, 0x0b, 0x8a, 0xd2, 0x12, 0x33, 0x17,
		0x20, 0xa0, 0x76, 0x17, 0x23, 0x3f, 0x76, 0xac, 0xb4, 0xcc,
		0x07, 0x5a, 0x62, 0x22, 0x0b, 0x21, 0x27, 0xf6, 0x80, 0x3d,
		0x30, 0xdc, 0x8c, 0x2e, 0xf7, 0xf3, 0x96, 0xc0, 0xbf, 0x03,
		0x10, 0xd6, 0xbe, 0xa3, 0xe4, 0x19, 0x21, 0x28, 0x49, 0xe4,
		0xa7, 0xe6, 0x4e, 0x69, 0x79, 0x8f, 0x09, 0x55, 0x77, 0x5d,
		0x90, 0x5f, 0xfa, 0x83, 0x35, 0xc8, 0xab, 0xe3, 0xd8, 0x18,
		0x7c, 0x14, 0x8a, 0x73, 0x68, 0x37, 0xc2, 0xf2, 0xb6, 0x89,
		0xc0, 0x4b, 0xbf, 0x5c, 0xb3, 0x9e, 0x50, 0x84, 0x9a, 0x95,
		0xbf, 0x14, 0x95, 0x61, 0x9f, 0xb9, 0x40, 0xa9, 0x9c, 0xb2,
		0xf1, 0x3d, 0x1b, 0x34, 0xc9, 0x26, 0xbf, 0xb3, 0x0d, 0x2f,
		0x85, 0x6c, 0x48, 0xc0, 0x66, 0x56, 0xc3, 0x1a, 0x30, 0xea,
		0x58, 0x96, 0x15, 0xc9, 0xf8, 0x19, 0x7b, 0xa0, 0x07, 0xb3,
		0x01, 0xc8, 0xfc, 0xe0, 0xb6, 0xf4, 0x01, 0x7e, 0x58, 0x7e,
		0x08, 0x09, 0x48, 0x33, 0x86, 0x9a, 0xea, 0xb4, 0x48, 0x51,
		0xe8, 0x5e, 0x69, 0x20, 0x6a, 0x38, 0x77, 0x1b, 0x8a, 0xa5,
		0x43, 0xe4, 0x67, 0x14, 0x96, 0x18, 0xc4, 0x7e, 0x4b, 0xbd,
		0x90, 0xd6, 0x03, 0x43, 0xce, 0xdc, 0x85, 0x23, 0x68, 0x90,
		0x09, 0x72, 0xc9, 0x8d, 0x4d, 0x89, 0xe2, 0x48, 0x6d, 0x03,
		0x2f, 0x42, 0xe0, 0x1a, 0x5b, 0x96, 0x15, 0x16, 0x3a, 0x56,
		0xa9, 0xa9, 0xb1, 0xac, 0xf3, 0x96, 0x58, 0x6d, 0x61, 0x13,
		0x92, 0x79, 0xf1, 0x58, 0x0b, 0x56, 0x5b, 0x2f, 0xe5, 0x43,
		0x35, 0x82, 0x78, 0xce, 0x3c, 0x71, 0x6c, 0x91, 0xc7, 0xc9,
		0xd2, 0xbf, 0x30, 0xb6, 0x68, 0x87, 0x50, 0x06, 0x98, 0x54,
		0x51, 0x2b, 0x2e, 0x57, 0x15, 0x01, 0x1f, 0x85, 0x06, 0x5f,
		0xf0, 0xbe, 0x22, 0x09, 0x14, 0x2a, 0xbf, 0x34, 0x11, 0x9e,
		0x31, 0xa8, 0x7c, 0x9e, 0x17, 0x03, 0x3d, 0x0c, 0xe8, 0x21,
		0x0b, 0x8b, 0x61, 0x03, 0x33, 0x2e, 0xc2, 0x06, 0xc0, 0xc6,
		0x0c, 0xa4, 0xf1, 0x79, 0x18, 0x02, 0xde, 0xcf, 0x3d, 0x6c,
		0xf7, 0x5d, 0x3c, 0x81, 0x81, 0x38, 0x1a, 0x63, 0x90, 0x47,
		0xaa, 0x73, 0x25, 0xd9, 0x18, 0x3b, 0x88, 0xcd, 0xde, 0x06,
		0xd6, 0xbe, 0xcf, 0x2e, 0x20, 0x6a, 0x32, 0xbb, 0xf0, 0x12,
		0xfd, 0x3e, 0xa1, 0xbf, 0x58, 0x63, 0x8d, 0x0c, 0xaf, 0x1f,
		0x81, 0xc2, 0xb7, 0x03, 0xc2, 0xf7, 0x54, 0x81, 0x5b, 0x04,
		0x6e, 0x3e, 0x49, 0x5a, 0xba, 0xfa, 0x39, 0xec, 0xcf, 0xe8,
		0x12, 0x88, 0x30, 0x83, 0xa6, 0x2a, 0xd2, 0x08, 0xd5, 0xae,
		0x3c, 0x63, 0xb4, 0xd2, 0xd9, 0x09, 0x00, 0x47, 0x2a, 0xa3,
		0xae, 0xa6, 0x4e, 0x9e, 0xab, 0x65, 0x40, 0x32, 0xa4, 0x49,
		0xc7, 0xcc, 0x28, 0x0c, 0xe6, 0x60, 0xf8, 0x23, 0xd4, 0x8b,
		0xde, 0xd9, 0xbf, 0x10, 0x18, 0xf4, 0x39, 0x9f, 0x7e, 0x64,
		0x9e, 0xd4, 0x8c, 0x04, 0xe6, 0xec, 0xc6, 0x09, 0xc8, 0x2c,
		0xa4, 0xd4, 0xe2, 0x4b, 0x84, 0x1b, 0xc4, 0x04, 0x90, 0xdf,
		0xf4, 0x4c, 0x68, 0xe8, 0x32, 0xf6, 0xcc, 0xa0, 0x51, 0x48,
		0x92, 0x6c, 0x01, 0x9e, 0xb2, 0x24, 0x0b, 0x76, 0x76, 0x7d,
		0x2a, 0xb4, 0x68, 0x26, 0xe3, 0xae, 0x0d, 0xe5, 0x33, 0xf8,
		0x19, 0x86, 0x19, 0x4f, 0x5a, 0x14, 0xfb, 0x2f, 0x13, 0x83,
		0xce, 0x0a, 0x24, 0xcb, 0x42, 0x7a, 0xa6, 0x26, 0x6d, 0x52,
		0xd6, 0x9b, 0xab, 0x15, 0x36, 0x08, 0xbf, 0x75, 0x41, 0x16,
		0x75, 0x46, 0xb2, 0x59, 0xe0, 0xd2, 0x58, 0xf1, 0xc0, 0x25,
		0xf1, 0x11, 0x4a, 0x6c, 0x39, 0x3c, 0xce, 0xda, 0xec, 0x94,
		0xdc, 0xcb, 0x0a, 0xef, 0x5a, 0x5f, 0x68, 0x13, 0xc7, 0x75,
		0x0f, 0xdd, 0x7d, 0x80, 0x21, 0x6b, 0x32, 0xdc, 0x83, 0x8a,
		0xa4, 0xab, 0x1c, 0xc7, 0x03, 0x29, 0x3e, 0xee, 0xf3, 0x8b,
		0xc8, 0x5a, 0x9a, 0x30, 0x61, 0xc8, 0x1d, 0x51, 0xa8, 0x8b,
		0xd7, 0xa1, 0xf0, 0x6f, 0xe3, 0xec, 0x88, 0x11, 0x3b, 0x9d,
		0xbf, 0x64, 0xda, 0xed, 0x3e, 0x6d, 0xc6, 0xea, 0x36, 0xe1,
		0xbc, 0x54, 0x41, 0x2e, 0x4b, 0x38, 0xa5, 0x29, 0xab, 0xca,
		0x48, 0x53, 0x9e, 0x31, 0x05, 0x5d, 0x50, 0x78, 0xe5, 0x0d,
		0x2d, 0x87, 0x7c, 0x83, 0xf1, 0xe7, 0x18, 0xb2, 0x87, 0xf7,
		0x7f, 0x7f, 0x29, 0xd9, 0xe0, 0x1f, 0xc8, 0xa3, 0xc3, 0x3c,
		0x29, 0x7e, 0x02, 0x5d, 0x3e, 0x0c, 0xa3, 0xa2, 0x90, 0x6b,
		0xe9, 0x19, 0x57, 0x0a, 0x25, 0x6a, 0x19, 0xef, 0x5c, 0x38,
		0xab, 0x44, 0xed, 0x04, 0xaa, 0x58, 0x8e, 0xb7, 0xf6, 0x41,
		0x52, 0x52, 0x42, 0xc8, 0xe3, 0x3f, 0x3a, 0x07, 0xca, 0xa5,
		0xdb, 0x22, 0x83, 0x29, 0xe1, 0x99, 0xa1, 0x89, 0x16, 0x27,
		0xb2, 0x4e, 0x19, 0x66, 0xd8, 0xdd, 0x98, 0x70, 0xab, 0x83,
		0x11, 0x4e, 0xe8, 0x2d, 0x3d, 0x9c, 0x58, 0x8f, 0xd0, 0xeb,
		0xf7, 0x9a, 0xba, 0x1c, 0x16, 0xd0, 0xcd, 0xa8, 0xff, 0xdb,
		0x3f, 0xbc, 0x19, 0xae, 0x90, 0x1b, 0xce, 0x02, 0x1d, 0x75,
		0xd3, 0xd1, 0x6f, 0xd1, 0xc2, 0xf0, 0xeb, 0xab, 0xb1, 0xe2,
		0x47, 0xc1, 0x4b, 0x73, 0x46, 0x7b, 0xbc, 0x3b, 0xef, 0xfb,
		0xe0, 0x63, 0xcf, 0xaa, 0xcf, 0x4c, 0x06, 0x91, 0xd4, 0x00,
		0xbd, 0x87, 0x20, 0x0d, 0x4e, 0x2d, 0x09, 0x7c, 0x13, 0xb4,
		0xbb, 0x6f, 0x21, 0x6a, 0x30, 0xa8, 0x4d, 0x91, 0x46, 0xc0,
		0x4b, 0xfb, 0x6e, 0x74, 0xf6, 0x4d, 0xbb, 0x32, 0x5b, 0xf1,
		0x27, 0x4b, 0xa3, 0x6a, 0x1d, 0xf8, 0x1e, 0x9a, 0xc0, 0x6c,
		0x16, 0x04, 0x4b, 0xca, 0xb6, 0xd4, 0xae, 0x79, 0xa9, 0x6c,
		0x3c, 0x2e, 0xb2, 0xb4, 0xef, 0x73, 0xd2, 0xe9, 0xf4, 0x71,
		0x2f, 0x34, 0xe3, 0x7c, 0x1b, 0xf1, 0x52, 0x0d, 0xa8, 0x5f,
		0xac, 0x7c, 0x48, 0xf0, 0xb3, 0x37, 0x05, 0x4d, 0xbe, 0x69,
		0x27, 0xda, 0x04, 0xc2, 0x33, 0xa3, 0x45, 0xb1, 0x3d, 0x3f,
		0xe1, 0x12, 0x94, 0x2e, 0xd6, 0xf1, 0xfa, 0xaf, 0x1b, 0x91,
		0xe3, 0x3f, 0x59, 0x40, 0x25, 0xd0, 0xba, 0xda, 0xa0, 0x46,
		0x14, 0x59, 0x1d, 0x29, 0xd3, 0xdc, 0xfb, 0xbe, 0x25, 0xb5,
		0x1d, 0x9d, 0xd5, 0x93, 0x03, 0x8f, 0xb1, 0x72, 0x45, 0xf7,
		0x6b, 0x89, 0xbb, 0x04, 0xbf, 0x2c, 0x17, 0x7a, 0x2a, 0x6c,
		0xa1, 0x85, 0xac, 0x88, 0x90, 0x8a, 0xee, 0x7e, 0x25, 0x39,
		0x29, 0xe8, 0x6a, 0x8f, 0x68, 0x24, 0x97, 0xb0, 0x4d, 0x8e,
		0x9c, 0x9b, 0xb5, 0xd4, 0x0b, 0x6c, 0xd4, 0x14, 0xb5, 0xa1,
		0x47, 0x30, 0x1d, 0xd8, 0x84, 0xf6, 0xb5, 0xba, 0xb6, 0xf2,
		0x90, 0xce, 0xd6, 0x9c, 0x71, 0x71, 0xe0, 0xf3, 0xcc, 0x8b,
		0xa6, 0x18, 0x88, 0x6d, 0x1e, 0x2f, 0x3c, 0x6b, 0x69, 0x89,
		0xe1, 0x33, 0xc5, 0xfb, 0x4c, 0x53, 0x55, 0x55, 0x10, 0xb7,
		0xcf, 0x35, 0x84, 0xf6, 0x07, 0xd2, 0xae, 0xc2, 0xc4, 0xf5,
		0x9c, 0xd8, 0xeb, 0x7d, 0xf7, 0xdc, 0xb4, 0xbb, 0xf0, 0xbe,
		0xdd, 0x3d, 0x3f, 0xcb, 0xd2, 0xa1, 0x52, 0x9e, 0xc9, 0x5c,
		0x05, 0x03, 0xf9, 0x9c, 0xf6, 0xdf, 0xb3, 0xd4, 0xe2, 0x54,
		0x34, 0x62, 0x66, 0xf8, 0xa6, 0xed, 0x6d, 0xef, 0x9f, 0xa6,
		0x49, 0x61, 0x87, 0x2f, 0x54, 0x48, 0x99, 0xdd, 0xd5, 0xa3,
		0xe2, 0xc7, 0xbb, 0x69, 0x17, 0x3a, 0xdc, 0x6f, 0xbd, 0xb6,
		0x84, 0xf3, 0xa2, 0x8c, 0x3f, 0x16, 0x12, 0x50, 0x62, 0xa0,
		0xc8, 0x47, 0xd3, 0xa6, 0x9a, 0x67, 0xe6, 0xc5, 0xf9, 0x00,
		0xca, 0x32, 0x9f, 0x20, 0x87, 0xd7, 0xea, 0x15, 0x18, 0xbb,
		0xe0, 0x24, 0x4f, 0x5b, 0x02, 0xb4, 0x2e, 0xc9, 0x1c, 0x66,
		0x45, 0x54, 0x01, 0xe3, 0xe0, 0xd0, 0xc2, 0x5c, 0x8c, 0x69,
		0x1c, 0xa7, 0xf7, 0x15, 0x25, 0x7d, 0x76, 0xb4, 0xba, 0x8d,
		0xa0, 0x5f, 0x1e, 0x46, 0x8e, 0xe1, 0x40, 0xd2, 0xb7, 0x1f,
		0x2f, 0x78, 0xec, 0xb5, 0x81, 0xc4, 0xa7, 0xb1, 0x07, 0x4d,
		0x7f, 0xa1, 0x15, 0x85, 0x7b, 0xdb, 0xaa, 0x84, 0x80, 0xc6,
		0x5a, 0x32, 0xcb, 0xa3, 0x0f, 0x5f, 0xb9, 0x71, 0xae, 0xfa,
		0x03, 0x2a, 0xb5, 0xf8, 0x61, 0x96, 0x80, 0xcb, 0xdf, 0x69,
		0xc6, 0xb0, 0x05, 0x4d, 0x1a, 0x72, 0x6b, 0xb0, 0xed, 0x3e,
		0x52, 0x6f, 0x60, 0xf5, 0x64, 0x4e, 0x18, 0xbb, 0x45, 0x0c,
		0xe7, 0xc2, 0x1a, 0xbc, 0xcc, 0xff, 0x6d, 0x3f, 0x15, 0x57,
		0x94, 0x1c, 0xa2, 0x25, 0x3d, 0xb5, 0x46, 0xe6, 0xc1, 0x26,
		0x39, 0xe1, 0x3d, 0xa4, 0x44, 0x74, 0xf3, 0x0e, 0x92, 0x3c,
		0xd6, 0xa1, 0x93, 0xe0, 0xd6, 0x7b, 0x5d, 0xd3, 0x60, 0x60,
		0x6a, 0xa4, 0xa0, 0x20, 0x02, 0x6e, 0x89, 0x54, 0xf7, 0x00,
		0x8a, 0xc1, 0x3c, 0x77, 0xfe, 0x60, 0xb6, 0xa0, 0x12, 0x1e,
		0xd1, 0xfd, 0x9f, 0x3d, 0xc6, 0x16, 0x1f, 0x09, 0x01, 0x97,
		0x10, 0x3a, 0xa1, 0x59, 0xa4, 0xf8, 0xde, 0xe3, 0x15, 0xfe,
		0x95, 0x83, 0x0f, 0x9c, 0xc2, 0xcb, 0xe6, 0x91, 0x3d, 0x49,
		0x36, 0xc9, 0xfc, 0x02, 0x40, 0x1f, 0x3a, 0x91, 0xf9, 0x6e,
		0x08, 0xa7, 0xb3, 0x8b, 0x9d, 0x6c, 0x7e, 0x71, 0xab, 0xe6,
		0x61, 0xa2, 0xb1, 0xef, 0x31, 0xa8, 0xa9, 0x02, 0x1a, 0xa2,
		0x89, 0xf8, 0xec, 0xee, 0x97, 0xf6, 0xa3, 0xc0, 0xa7, 0x7f,
		0xca, 0xb9, 0x75, 0x52, 0x18, 0xf3, 0x8d, 0x02, 0xb8, 0x59,
		0x7a, 0x20, 0xf6, 0xf3, 0x14, 0x33, 0x68, 0x70, 0xfc, 0x02,
		0xe1, 0x74, 0x4f, 0x6e, 0xc9, 0x9c, 0x5d, 0xc7, 0xc0, 0xb3,
		0xe4, 0x5b, 0x68, 0x5d, 0x56, 0x41, 0x8d, 0x88, 0x40, 0x96,
		0x55, 0xfe, 0x71, 0x90, 0x50, 0x61, 0x34, 0xc1, 0x60, 0xa9,
		0xf0, 0x92, 0x07, 0x70, 0x72, 0x71, 0xfc, 0x53, 0x35, 0x9c,
		0x25, 0x35, 0xc9, 0x01, 0xde, 0xa0, 0x6e, 0x1e, 0x3e, 0x05,
		0x95, 0x57, 0xa3, 0xb5, 0xbe, 0xc0, 0xc5, 0x9e, 0x4c, 0xd0,
		0x09, 0x50, 0xfa, 0xd3, 0x1b, 0x6b, 0x90, 0xce, 0x85, 0x48,
		0x12, 0xbe, 0x9d, 0x95, 0xf3, 0xc7, 0x5d, 0xcb, 0x12, 0x08,
		0x2c, 0x49, 0xe1, 0xd5, 0x93, 0x67, 0x52, 0xa3, 0xac, 0x8f,
		0xdc, 0x8f, 0x64, 0xaa, 0x0c, 0x58, 0x52, 0x30, 0x4e, 0xef,
		0xf8, 0xfe, 0x2b, 0xbe, 0xd3, 0x6c, 0x7b, 0x30, 0xd6, 0xff,
		0x42, 0x94, 0xdc, 0xbf, 0x4e, 0x6b, 0x0d, 0xef, 0x5c, 0xd9,
		0x84, 0x43, 0x10, 0x22, 0x85, 0x24, 0xe9, 0x9a, 0x4b, 0xaf,
		0xb6, 0x9c, 0xce, 0x7b, 0x59, 0x0f, 0x3c, 0x51, 0x16, 0xc8,
		0x2b, 0x18, 0xd9, 0xce, 0x9f, 0x82, 0x86, 0xef, 0xc8, 0xaa,
		0x56, 0xfa, 0xbd, 0x0f, 0xd0, 0x8f, 0x66, 0x76, 0xed, 0x66,
		0x48, 0x8b, 0xa4, 0x67, 0xd1, 0xca, 0xd2, 0x62, 0x13, 0x7c,
		0xde, 0xdf, 0xfe, 0xfc, 0xf9, 0x1a, 0x16, 0x4a, 0xfc, 0xb2,
		0xc1, 0xdd, 0x92, 0x44, 0x9e, 0xe5, 0x44, 0x62, 0xee, 0xeb,
		0xbe, 0xa4, 0x9a, 0x95, 0x3d, 0xb3, 0x69, 0x3f, 0xda, 0x9a,
		0x24, 0xdc, 0x90, 0x56, 0xe0, 0x46, 0x3e, 0x90, 0xf5, 0xea,
		0x98, 0x57, 0x89, 0x9b, 0xf1, 0xd2, 0x67, 0xd7, 0x46, 0x43,
		0xf5, 0x7d, 0x7d, 0x2c, 0xe1, 0x6a, 0xd2, 0x79, 0x77, 0xef,
		0xd1, 0xa1, 0x0f, 0x8c, 0x1a, 0xd7, 0x19, 0x55, 0x86, 0xf2,
		0xc7, 0xa8, 0x9b, 0x32, 0xf9, 0x5e, 0x45, 0x23, 0x52, 0x2d,
		0xb6, 0xdc, 0x65, 0xbf, 0x98, 0xa4, 0x41, 0xff, 0xbb, 0x89,
		0x79, 0xec, 0x16, 0xa2, 0xf8, 0x05, 0x69, 0x64, 0xdf, 0x95,
		0x6b, 0x7d, 0xfc, 0xce, 0x3d, 0xa1, 0xd8, 0xbf, 0x83, 0xe2,
		0x89, 0xc9, 0x1a, 0x36, 0xb9, 0xff, 0xc3, 0x6f, 0x04, 0x1a,
		0xfc, 0x5c, 0x6c, 0x3f, 0x78, 0x7c, 0xf3, 0xe2, 0x0b, 0x72,
		0x55, 0x13, 0x8b, 0x97, 0x12, 0x58, 0xf4, 0x4a, 0x52, 0x36,
		0xc6, 0x42, 0xca, 0xf5, 0xf0, 0x31, 0x9d, 0xf3, 0x60, 0x56,
		0x26, 0xff, 0x81, 0xf8, 0xd3, 0x82, 0x76, 0x1d, 0xb8, 0x9e,
		0xc8, 0x53, 0xab, 0x54, 0x22, 0xf8, 0xef, 0x37, 0x8e, 0xc3,
		0x7b, 0xe9, 0xbf, 0x25, 0x2e, 0x56, 0x2a, 0x18, 0xa9, 0xec,
		0x39, 0x28, 0xa5, 0x9c, 0x8e, 0xc5, 0xaa, 0x7d, 0xa6, 0x97,
		0x8f, 0x44, 0xcf, 0x62, 0xb3, 0xa1, 0x6f, 0x26, 0xcf, 0x5c,
		0x3d, 0x5c, 0x47, 0x40, 0x8f, 0x5d, 0x70, 0x68, 0x01, 0x69,
		0x26, 0x5d, 0x45, 0xa0, 0x60, 0xb1, 0x49, 0xa9, 0xe3, 0x6c,
		0xe7, 0x2a, 0x64, 0x02, 0x9e, 0x43, 0x37, 0x0f, 0x57, 0xd3,
		0xa5, 0xeb, 0x11, 0xe6, 0x69, 0x55, 0xd1, 0x85, 0x6c, 0x78,
		0xc9, 0x8b, 0x44, 0xbc, 0x85, 0x80, 0x93, 0xba, 0x6c, 0xe3,
		0xd2, 0x39, 0x95, 0xc6, 0x49, 0xd1, 0xa9, 0x46, 0x50, 0xd6,
		0xb2, 0x7c, 0x12, 0x5c, 0xa7, 0x7f, 0x2c, 0xb3, 0xaa, 0x01,
		0x4a, 0x06, 0x58, 0xef, 0xe3, 0xad, 0x8c, 0xdc, 0xc6, 0x45,
		0xe6, 0x18, 0x33, 0xcb, 0x19, 0xba, 0x8f, 0x76, 0x72, 0x41,
		0xd5, 0xe3, 0xa9, 0x8f, 0x93, 0x50, 0xdc, 0x7a, 0xcd, 0x4e,
		0x94, 0xc2, 0xaa, 0xb4, 0xc0, 0xf4, 0xad, 0xec, 0x46, 0xff,
		0x65, 0x73, 0x54, 0xb5, 0xfc, 0xbb, 0xd2, 0x61, 0x14, 0xae,
		0xab, 0xf7, 0x52, 0x95, 0xcd, 0x1c, 0x3f, 0x25, 0xc9, 0xc9,
		0xe3, 0x19, 0x79, 0x62, 0xac, 0x02, 0x85, 0x8a, 0xc9, 0x3a,
		0xf1, 0xc5, 0xd2, 0xbd, 0xdc, 0x8f, 0xd7, 0xce, 0xfe, 0xa2,
		0x58, 0x14, 0xc5, 0xa2, 0xd9, 0xef, 0x2d, 0x9d, 0x97, 0xf7,
		0xec, 0x55, 0xf7, 0x2e, 0x4c, 0x0f, 0x2f, 0x3b, 0xde, 0xc0,
		0x41, 0x7f, 0x7d, 0x2a, 0xb9, 0x56, 0xab, 0xe6, 0xf4, 0x11,
		0x52, 0xb6, 0xe6, 0xc9, 0xe6, 0xc6, 0xef, 0x02, 0x6e, 0x4a,
		0xd1, 0x6d, 0x3c, 0xa6, 0x0c, 0xb8, 0xd6, 0x7e, 0x31, 0x3a,
		0xa0, 0xce, 0xcc, 0x0c, 0x31, 0xab, 0xc8, 0xe8, 0x37, 0xfd,
		0x0e, 0x5b, 0xc3, 0x62, 0xf8, 0xdb, 0xcb, 0xf5, 0x00, 0xa0,
		0x40, 0xb3, 0x48, 0xeb, 0x82, 0xae, 0xd2, 0xd3, 0x4a, 0xea,
		0x12, 0x6f, 0x39, 0x85, 0xbb, 0x98, 0xb7, 0x39, 0x3f, 0xf3,
		0x0b, 0x0e, 0x6d, 0xca, 0x84, 0xf8, 0xf1, 0x35, 0xdc, 0xe7,
		0xbd, 0x1c, 0xdd, 0x21, 0x4b, 0xdf, 0xde, 0xb5, 0xcc, 0x16,
		0x87, 0x0f, 0xe8, 0xb8, 0x34, 0x86, 0x45, 0xfc, 0xf4, 0x61,
		0x93, 0x4a, 0x7f, 0x2c, 0x3f, 0xeb, 0xd1, 0xce, 0xc8, 0x61,
		0x0b, 0x72, 0x49, 0x3a, 0xd3, 0x3f, 0x06, 0x05, 0x4c, 0xe0,
		0x23, 0x4d, 0x8a, 0x20, 0xae, 0x87, 0xda, 0x4c, 0x6c, 0xb8,
		0xb9, 0x79, 0xda, 0x1c, 0x73, 0x5e, 0xb6, 0xf2, 0x03, 0x61,
		0x68, 0xff, 0xd5, 0xbf, 0x9a, 0x04, 0xb4, 0x73, 0xc8, 0xda,
		0xb5, 0x82, 0x34, 0x0c, 0x95, 0xfb, 0xda, 0x70, 0xcb, 0xb2,
		0xde, 0xa8, 0x08, 0xdf, 0xe4, 0x24, 0xff, 0x77, 0x5c, 0x2f,
		0x18, 0x0c, 0x82, 0x1f, 0xe6, 0x29, 0x09, 0x8f, 0xfe, 0x67,
		0x96, 0xb2, 0x47, 0x9c, 0x43, 0x3a, 0x58, 0xe8, 0x5f, 0xc9,
		0x73, 0xa4, 0x65, 0x3c, 0xe3, 0x7f, 0xdc, 0xde, 0x0d, 0xdf,
		0x1c, 0x4d, 0x0e, 0x3e, 0xc3, 0xd2, 0x1c, 0xc1, 0x0e, 0x4a,
		0x44, 0xf7, 0x74, 0xce, 0x00, 0x69, 0x43, 0x99, 0xbd, 0xb7,
		0xa1, 0x36, 0xff, 0x0a, 0xc8, 0x6b, 0xdd, 0xd8, 0x62, 0x2f,
		0xf5, 0x37, 0x95, 0x8d, 0x91, 0xd8, 0xa0, 0x5b, 0xc9, 0xa7,
		0xfe, 0x6c, 0xc0, 0x45, 0x7c, 0x00, 0x0d, 0x4a, 0xbf, 0x3e,
		0x00, 0x50, 0xe6, 0xce, 0x69, 0x26, 0xa6, 0x1d, 0xa4, 0xb0,
		0x4b, 0x4e, 0x02, 0x5b, 0x01, 0x0a, 0x1c, 0xcb, 0x11, 0x6e,
		0x74, 0xfe, 0xcc, 0xe1, 0x62, 0xd7, 0xf1, 0x56, 0xf9, 0x3b,
		0x69, 0x46, 0x8c, 0x1c, 0x52, 0x37, 0x25, 0x33, 0x35, 0x03,
		0xea, 0x51, 0x7a, 0x29, 0x85, 0x9a, 0x58, 0x4f, 0x38, 0x9f,
		0xc6, 0x57, 0x54, 0xba, 0x8e, 0x92, 0x59, 0xcc, 0x4f, 0x88,
		0xd5, 0x4a, 0x33, 0x82, 0x9f, 0x2c, 0x7d, 0x2b, 0x99, 0xce,
		0x22, 0x3b, 0x2d, 0xc9, 0x6f, 0x4d, 0x38, 0x87, 0x45, 0x20,
		0xc5, 0x67, 0x41, 0xf2, 0xce, 0xd9, 0x3b, 0x77, 0x41, 0xad,
		0x35, 0x93, 0x0a, 0x53, 0x1a, 0xb2, 0x15, 0xc9, 0x91, 0xb9,
		0xc7, 0x68, 0x5c, 0x1b, 0x29, 0xd2, 0x99, 0xb5, 0x38, 0xa2,
		0x60, 0x5d, 0xc2, 0x2c, 0x36, 0xef, 0xf7, 0xed, 0xfa, 0xe2,
		0x33, 0xdb, 0xb4, 0x35, 0x23, 0xd4, 0x18, 0x90, 0x4e, 0xe3,
		0xb4, 0xec, 0xcd, 0x42, 0xbf, 0x35, 0x28, 0x24, 0x88, 0x18,
		0x3d, 0xfc, 0x97, 0xa6, 0x12, 0x96, 0x68, 0x5b, 0xd4, 0x5b,
		0xf0, 0xe9, 0xda, 0xc7, 0xbd, 0xe1, 0xca, 0x3f, 0x86, 0xff,
		0x8e, 0x16, 0xaa, 0x28, 0x5a, 0x9d, 0xbd, 0x6a, 0xd3, 0xa7,
		0x7b, 0xf8, 0x32, 0x39, 0x87, 0xa2, 0x21, 0x03, 0x11, 0x13,
		0xe6, 0xe7, 0x86, 0x68, 0x92, 0x93, 0x0a, 0x71, 0xa5, 0x5b,
		0x95, 0x01, 0xae, 0x09, 0xb5, 0x68, 0xee, 0x0d, 0xd7, 0x2f,
		0x24, 0x7a, 0x44, 0xf4, 0x00, 0x58, 0x23, 0x07, 0xd3, 0x25,
		0x7c, 0x1a, 0x48, 0x4a, 0x20, 0x06, 0x1d, 0x1f, 0xc4, 0xf8,
		0x5b, 0xa1, 0x88, 0xb3, 0x53, 0x2b, 0x65, 0x89, 0xd4, 0x26,
		0x0b, 0xc1, 0xc4, 0x0f, 0xc5, 0x2f, 0xab, 0xf6, 0xc8, 0xc6,
		0x98, 0x73, 0x81, 0x91, 0x87, 0x7c, 0x9f, 0x50, 0x66, 0x50,
		0xc3, 0xcc, 0xff, 0x7a, 0x9c, 0x25, 0x14, 0x73, 0x94, 0x4b,
		0x8a, 0x51, 0x98, 0x23, 0x52, 0x96, 0x20, 0xa8, 0xe9, 0x0f,
		0x38, 0x36, 0xc4, 0x51, 0xec, 0xea, 0xe4, 0xea, 0xe9, 0x8e,
		0x44, 0x47, 0x13, 0x75, 0xea, 0xb1, 0x90, 0x0a, 0x66, 0xd1,
		0x17, 0xd5, 0xd7, 0x2f, 0x9e, 0x56, 0x86, 0xd1, 0x98, 0x9c,
		0x24, 0xa2, 0x82, 0x84, 0xad, 0xee, 0x50, 0xd8, 0xc1, 0x52,
		0xf1, 0xa7, 0xa2, 0x96, 0xfe, 0x34, 0x6f, 0xf2, 0xf7, 0x37,
		0x53, 0x7f, 0xa5, 0x4f, 0x4a, 0x5e, 0xe5, 0x65, 0x2c, 0xb1,
		0xc5, 0x22, 0x7b, 0xbe, 0xb0, 0x99, 0x61, 0xc8, 0xf3, 0x54,
		0x20, 0x9c, 0x6e, 0x45, 0x1a, 0xdf, 0x68, 0x5c, 0x80, 0x32,
		0xad, 0xd3, 0xa4, 0xac, 0x28, 0xeb, 0x02, 0xc8, 0x31, 0xc0,
		0x74, 0x5b, 0x9b, 0xe0, 0x20, 0x72, 0x12, 0x93, 0xc2, 0x93,
		0x2b, 0x77, 0xcb, 0x2c, 0x08, 0xaa, 0x48, 0x07, 0x05, 0x77,
		0x27, 0x6d, 0x43, 0x1b, 0x75, 0xc5, 0xda, 0x1d, 0x6b, 0x41,
		0x6b, 0x56, 0x5c, 0xf3, 0x0e, 0x8b, 0x17, 0xa2, 0xe8, 0x15,
		0x63, 0xbd, 0x07, 0xfa, 0xcf, 0xea, 0x1b, 0xe7, 0x6d, 0xc1,
		0xf3, 0x8a, 0x0b, 0x21, 0x4e, 0x66, 0x47, 0xa4, 0xd4, 0xed,
		0x3b, 0xfa, 0x7c, 0x65, 0xac, 0x4d, 0x38, 0xda, 0x4f, 0x2a,
		0xc9, 0x52, 0xb1, 0x6e, 0x1f, 0xae, 0xba, 0xde, 0x8c, 0x1d,
		0x73, 0x49, 0xdc, 0x7b, 0x04, 0xdf, 0xda, 0xdc, 0x24, 0xd6,
		0xe1, 0x0a, 0x1a, 0x30, 0x85, 0x00, 0xea, 0xe1, 0xc8, 0xa9,
		0x98, 0x84, 0xdd, 0xa6, 0x8b, 0xe6, 0xbf, 0x96, 0x2b, 0x9f,
		0x27, 0xce, 0x71, 0x4d, 0x4b, 0xdc, 0x21, 0xcb, 0x2f, 0x17,
		0xa8, 0xa5, 0x98, 0xa3, 0x7a, 0xb1, 0x24, 0x79, 0x92, 0x1d,
		0xbd, 0x8d, 0x22, 0x8d, 0xb8, 0x1a, 0x5e, 0x0c, 0x3f, 0x00,
		0xe7, 0xe6, 0x9b, 0x43, 0xcc, 0xe1, 0x9f, 0x1c, 0xde, 0x17,
		0x59, 0xfd, 0xf4, 0xfc
	};

	static const uint8_t keys_b_ps4[0x70 * 0x20] = {
		0x76, 0x65, 0x9d, 0xe4, 0xf7, 0x6c, 0x45, 0x1f, 0x73, 0x42,
		0x47, 0xc3, 0x92, 0x32, 0xf4, 0x38, 0x6f, 0x28, 0xcf, 0xfa,
		0x41, 0x84, 0xbe, 0x1b, 0x5a, 0x23, 0x99, 0xfa, 0xcd, 0x74,
		0xf0, 0x3b, 0x43, 0xc9, 0xd1, 0x30, 0x66, 0x77, 0x5e, 0xac,
		0x39, 0x8e, 0x8a, 0x6c, 0x5c, 0xfe, 0xdd, 0x80, 0xf3, 0x03,
		0x04, 0x30, 0x84, 0x9d, 0x12, 0xee, 0x4a, 0xe3, 0x0f, 0x5e,
		0x5e, 0x97, 0xd0, 0xf9, 0x1b, 0x57, 0x05, 0x74, 0x03, 0x5a,
		0x5d, 0x41, 0xda, 0x6e, 0xfd, 0x4b, 0x1e, 0xc4, 0x4b, 0x76,
		0xe4, 0xd3, 0x39, 0xeb, 0x3c, 0x2c, 0xae, 0x27, 0xe2, 0x98,
		0xe4, 0x08, 0xa3, 0xbe, 0xb0, 0x31, 0x69, 0x14, 0xa3, 0x6c,
		0xdf, 0xfc, 0x1f, 0x56, 0x0d, 0xfe, 0x55, 0x12, 0x97, 0x5e,
		0xad, 0x3d, 0x9b, 0xee, 0xb5, 0xad, 0xa3, 0x56, 0x85, 0xc3,
		0xbd, 0x9b, 0xef, 0xac, 0x55, 0x98, 0x1f, 0xaf, 0xb8, 0x0b,
		0xae, 0xed, 0xf3, 0x87, 0x34, 0xc9, 0x5b, 0x69, 0xf0, 0xed,
		0x4f, 0x03, 0x40, 0xa5, 0x71, 0x24, 0xa2, 0x23, 0x93, 0x52,
		0xe9, 0x87, 0xc4, 0x79, 0x4f, 0xe9, 0x20, 0xbc, 0x04, 0x58,
		0x70, 0x96, 0xb1, 0xd7, 0x37, 0xc4, 0x29, 0x4d, 0x3d, 0x52,
		0xd3, 0xcf, 0x7c, 0x08, 0xe9, 0x82, 0xc6, 0x99, 0xf5, 0xdb,
		0x60, 0x11, 0x9b, 0x9d, 0x25, 0x86, 0x08, 0xb3, 0x77, 0x27,
		0xa8, 0x2c, 0xf9, 0x28, 0x2f, 0x2b, 0x70, 0x6c, 0x4b, 0xbb,
		0x1e, 0x49, 0x3e, 0xd6, 0x69, 0xd8, 0xe8, 0x09, 0xf4, 0x9c,
		0x09, 0x32, 0xdf, 0xdd, 0xe1, 0x0b, 0x41, 0x0e, 0x26, 0x3e,
		0x53, 0xcc, 0x7a, 0xd1, 0x15, 0x16, 0xfb, 0xc3, 0x09, 0xd3,
		0xad, 0xa4, 0xea, 0xa3, 0x4a, 0xbe, 0xf2, 0x88, 0x9a, 0x34,
		0x7b, 0x09, 0x41, 0x44, 0xf1, 0x5e, 0x3c, 0x4f, 0x0e, 0x9f,
		0x5a, 0x95, 0x57, 0x7a, 0xfe, 0xbe, 0xc8, 0x24, 0x1f, 0x98,
		0x85, 0x1e, 0x25, 0x74, 0x90, 0xdd, 0x5a, 0xd1, 0x7c, 0x02,
		0xbe, 0x63, 0xe5, 0xc7, 0xa2, 0x0b, 0x26, 0xdf, 0x20, 0x84,
		0x61, 0xe3, 0xff, 0x2c, 0x69, 0x32, 0x91, 0x62, 0xf8, 0x11,
		0x7c, 0xf2, 0x1e, 0xe5, 0x1e, 0xc1, 0xb2, 0x2a, 0x84, 0x8e,
		0x3a, 0x78, 0xf6, 0xf6, 0x76, 0x76, 0x4e, 0x34, 0x4f, 0x24,
		0xf0, 0x8c, 0x34, 0xaf, 0x95, 0x7a, 0x35, 0x26, 0x5e, 0x28,
		0xe3, 0x2c, 0x66, 0xf9, 0x7f, 0x35, 0xfc, 0xad, 0xfe, 0x6e,
		0x4b, 0xe1, 0x5c, 0x48, 0xec, 0x1a, 0xba, 0xbd, 0xac, 0x73,
		0x7a, 0xe8, 0xbc, 0xad, 0xba, 0x5e, 0xa6, 0xf0, 0xb8, 0x5a,
		0x51, 0x2a, 0x97, 0xe4, 0x34, 0x8a, 0x3b, 0x64, 0x61, 0x97,
		0x57, 0x59, 0x29, 0x87, 0x76, 0x80, 0x78, 0x05, 0x91, 0xc0,
		0xa5, 0x5e, 0x7b, 0x5a, 0xa5, 0x7b, 0x2f, 0x8d, 0x07, 0x7d,
		0xfa, 0x75, 0x32, 0xe8, 0x48, 0x0c, 0x68, 0xbf, 0x70, 0xb3,
		0x79, 0x88, 0xc0, 0xbd, 0xbc, 0x49, 0x9a, 0x9a, 0x07, 0xb8,
		0xa3, 0xc2, 0xb4, 0x1d, 0xca, 0x24, 0x90, 0xe8, 0x2f, 0xe8,
		0x9f, 0x28, 0xba, 0x40, 0x17, 0x50, 0x56, 0xcf, 0x4e, 0xc8,
		0x82, 0xcb, 0x77, 0xf1, 0x74, 0x8d, 0xef, 0x0b, 0x4c, 0x1f,
		0x3d, 0x4d, 0xd2, 0x4e, 0x66, 0x2f, 0x13, 0xe7, 0x8a, 0xb6,
		0xea, 0xd1, 0xd7, 0x36, 0xf3, 0x51, 0x65, 0x07, 0xc1, 0x88,
		0x7d, 0xae, 0xf4, 0xb8, 0x89, 0xb5, 0xce, 0xdd, 0x27, 0xbe,
		0x6f, 0x6c, 0x81, 0x8f, 0x56, 0x02, 0x0d, 0x49, 0xe4, 0x3f,
		0x60, 0x93, 0x55, 0xa3, 0x68, 0x75, 0x5d, 0xc5, 0x5a, 0x09,
		0x90, 0xdc, 0x8f, 0xaa, 0x9f, 0x47, 0xf6, 0xa9, 0xad, 0xdb,
		0x5b, 0xe2, 0x64, 0xce, 0xc1, 0x26, 0xeb, 0xeb, 0xa0, 0x03,
		0x23, 0x61, 0x14, 0x9e, 0x72, 0x48, 0xc2, 0x10, 0x3d, 0x73,
		0xc3, 0xc9, 0x22, 0x01, 0x68, 0x2c, 0x3c, 0x98, 0xa5, 0x85,
		0x0d, 0xa4, 0xec, 0xc6, 0xfc, 0x7d, 0xff, 0x2c, 0xae, 0x60,
		0xc4, 0xb0, 0x71, 0xb9, 0xbc, 0xdb, 0x29, 0xc6, 0x78, 0x0b,
		0xf6, 0x82, 0x38, 0xed, 0x35, 0xb7, 0x92, 0x2f, 0xeb, 0xa9,
		0xe7, 0xc3, 0x1a, 0xa2, 0xae, 0x44, 0x51, 0x71, 0x6e, 0x8c,
		0x21, 0x9f, 0x9c, 0x5d, 0x98, 0x72, 0xa3, 0x2e, 0x74, 0x49,
		0x70, 0xaf, 0x85, 0x7b, 0x14, 0x66, 0x56, 0x99, 0x89, 0x07,
		0xc4, 0xa9, 0xc2, 0xc2, 0xb3, 0x39, 0xa6, 0xf5, 0x5b, 0x3f,
		0x30, 0xcc, 0x2d, 0x03, 0x22, 0xb4, 0x2a, 0x27, 0x1a, 0x46,
		0xbc, 0xe5, 0xf6, 0x19, 0x34, 0xc1, 0x06, 0x6f, 0x28, 0x84,
		0xe1, 0xe3, 0xb0, 0x21, 0x21, 0x84, 0x34, 0x33, 0x8d, 0x01,
		0x26, 0x44, 0x5f, 0xa3, 0x56, 0x19, 0x46, 0xc1, 0x78, 0x29,
		0x38, 0x0b, 0x9f, 0xe1, 0xf1, 0x2b, 0xfc, 0x01, 0x91, 0x71,
		0x1c, 0xa5, 0x9c, 0xb6, 0xe1, 0x5e, 0xfa, 0x26, 0x56, 0x3f,
		0xa8, 0xb2, 0xd1, 0xe4, 0xf3, 0x69, 0x08, 0x36, 0xac, 0xd6,
		0x67, 0xb6, 0xe6, 0xb8, 0xc7, 0xa6, 0x7a, 0x10, 0x49, 0xe7,
		0xba, 0x6f, 0xbf, 0xe8, 0x25, 0xaf, 0xc2, 0xdf, 0xf3, 0xa1,
		0x76, 0xf4, 0x2b, 0x8c, 0x6f, 0x01, 0xce, 0x96, 0x30, 0xd2,
		0x2d, 0xf8, 0x6c, 0xad, 0xed, 0x1e, 0xe5, 0x7a, 0xdf, 0x37,
		0xe6, 0xff, 0xed, 0x9d, 0x5f, 0xa9, 0xfb, 0xf1, 0xbc, 0xc7,
		0x47, 0x5a, 0x6f, 0x77, 0x83, 0x18, 0x0e, 0xba, 0x18, 0x2b,
		0x38, 0xb4, 0x36, 0x32, 0x36, 0xdc, 0x98, 0xec, 0x16, 0xa1,
		0x68, 0x42, 0x95, 0xee, 0xad, 0x43, 0x4c, 0x3d, 0x31, 0xcf,
		0xdb, 0x57, 0xae, 0x0f, 0x78, 0x52, 0xd8, 0xa6, 0x47, 0x4b,
		0xba, 0x94, 0xc2, 0xd2, 0x97, 0x56, 0x2b, 0xb0, 0xbd, 0x43,
		0x30, 0x05, 0x4b, 0x30, 0x7b, 0x8b, 0xe4, 0xee, 0x61, 0xc4,
		0xcd, 0xcb, 0x08, 0xb1, 0x16, 0x2e, 0xa3, 0xe4, 0x20, 0x0e,
		0xd5, 0x02, 0x17, 0x81, 0xba, 0x43, 0xf7, 0x02, 0x7a, 0x0e,
		0x77, 0x15, 0x45, 0x03, 0xaa, 0x84, 0x88, 0x56, 0xb1, 0xee,
		0x3a, 0x1f, 0xe6, 0x28, 0xfd, 0x9f, 0x05, 0x81, 0x8d, 0xbf,
		0x53, 0xa1, 0x5b, 0xbd, 0x9c, 0x79, 0x53, 0x28, 0x73, 0xba,
		0x0d, 0x72, 0xe7, 0x14, 0xed, 0x5d, 0xb7, 0x95, 0x09, 0xa0,
		0xbf, 0xef, 0x75, 0xa1, 0x20, 0xa4, 0x2d, 0x16, 0xa6, 0x58,
		0x61, 0x93, 0x3a, 0x62, 0x40, 0xca, 0x4e, 0xef, 0xe4, 0x31,
		0x18, 0xd1, 0x39, 0x35, 0x7c, 0x78, 0xe3, 0x6c, 0x76, 0x8a,
		0x37, 0xdd, 0xc3, 0x8e, 0x60, 0xe2, 0x0d, 0x67, 0xfc, 0xe7,
		0x37, 0x88, 0x48, 0xb6, 0xe4, 0x4b, 0x87, 0xe6, 0x04, 0x8c,
		0x06, 0xd8, 0xc7, 0x4e, 0x9a, 0x97, 0x9d, 0x9d, 0xa1, 0xfa,
		0x11, 0x27, 0x8c, 0x34, 0xe7, 0x02, 0x55, 0x0f, 0x6e, 0xc6,
		0xcd, 0x6b, 0xc4, 0xf0, 0xa1, 0x05, 0x6b, 0x3f, 0xc4, 0xc9,
		0x47, 0x95, 0x85, 0x1f, 0x80, 0x6a, 0x23, 0xe7, 0x24, 0x9f,
		0xc3, 0xd0, 0x9d, 0x17, 0xea, 0x2f, 0xfa, 0xfa, 0xe4, 0xa6,
		0x88, 0x9a, 0x57, 0x0c, 0x5b, 0xfe, 0xb4, 0x67, 0x3e, 0xf0,
		0x0a, 0x58, 0xa0, 0xc8, 0xc7, 0x16, 0xd6, 0x0b, 0x41, 0x9e,
		0xa9, 0xde, 0xbf, 0x17, 0x49, 0x08, 0x82, 0xc4, 0x4e, 0x56,
		0x53, 0xb0, 0x4e, 0x23, 0x1d, 0xa8, 0x07, 0xde, 0x98, 0xf8,
		0xe9, 0xec, 0x45, 0x3b, 0x50, 0xf6, 0xb6, 0xa3, 0xdc, 0xfa,
		0xa1, 0x86, 0xd3, 0xfa, 0xda, 0xb6, 0x1d, 0x27, 0xed, 0x3d,
		0x17, 0xc9, 0xcc, 0xf6, 0x2f, 0x7d, 0xea, 0x3b, 0x3d, 0x58,
		0xb7, 0xf0, 0x4a, 0xae, 0xf9, 0x2f, 0xa0, 0x80, 0xe9, 0xde,
		0x2e, 0x30, 0x3c, 0xd6, 0x46, 0xca, 0x50, 0x25, 0xf7, 0xa2,
		0x66, 0xec, 0xa5, 0x6e, 0x06, 0x6e, 0x00, 0x3e, 0x08, 0x6c,
		0xe6, 0x3b, 0xfb, 0xc8, 0xa7, 0xd0, 0x1e, 0xc8, 0x25, 0x24,
		0x11, 0x0b, 0x39, 0x0f, 0x70, 0xd9, 0xf8, 0xff, 0x54, 0x54,
		0x9d, 0x02, 0xab, 0x4c, 0x8c, 0x90, 0x94, 0xe5, 0x0e, 0xd5,
		0x62, 0x78, 0xab, 0xa5, 0x9c, 0x11, 0x1c, 0x70, 0x07, 0x9b,
		0x2a, 0xe4, 0xfb, 0xed, 0xfb, 0xcd, 0x92, 0x0d, 0xbc, 0x21,
		0x25, 0x33, 0x0b, 0xbf, 0xf3, 0x18, 0x93, 0x73, 0xed, 0xf3,
		0x88, 0x3b, 0x43, 0x97, 0xce, 0x19, 0x25, 0xf5, 0xbb, 0x71,
		0x26, 0x98, 0x2c, 0xd6, 0xff, 0x32, 0x4f, 0x75, 0xcc, 0x92,
		0x4e, 0xe3, 0xbe, 0xb8, 0xbf, 0x33, 0x89, 0xe5, 0x09, 0x4c,
		0x41, 0x8c, 0xb0, 0x51, 0x74, 0x32, 0xea, 0x96, 0xce, 0x70,
		0x1c, 0x5c, 0x26, 0x34, 0x66, 0x3a, 0x3a, 0xe8, 0x92, 0x4e,
		0xd2, 0x61, 0x70, 0x12, 0x3c, 0x33, 0x9d, 0x09, 0x57, 0x23,
		0x48, 0x81, 0x29, 0x7f, 0xc9, 0x4d, 0x06, 0xe8, 0xc5, 0x6e,
		0x50, 0x7c, 0x8f, 0xfa, 0xf0, 0xb4, 0xe6, 0x9c, 0x7d, 0x5e,
		0x09, 0x36, 0xb3, 0x24, 0xc4, 0xf8, 0xe3, 0x19, 0xa8, 0x65,
		0x3f, 0x2a, 0x84, 0x98, 0x79, 0x18, 0x58, 0x26, 0x29, 0x5e,
		0xbc, 0x3c, 0x7e, 0x80, 0xaa, 0x04, 0x43, 0x1b, 0xbd, 0x36,
		0xef, 0x5c, 0xc6, 0x78, 0xfb, 0xa6, 0xfd, 0x26, 0x6a, 0xa4,
		0xb6, 0x2e, 0x92, 0xb7, 0x7a, 0x02, 0xa7, 0xc8, 0xb0, 0x47,
		0xa8, 0x6e, 0x6a, 0x53, 0x15, 0x81, 0x7d, 0x1c, 0xb0, 0x55,
		0xbe, 0xa9, 0x11, 0x1e, 0xd0, 0xd8, 0x99, 0x37, 0xae, 0x35,
		0xfa, 0xe4, 0x0b, 0x4c, 0x52, 0x0c, 0x12, 0x3a, 0x5f, 0x09,
		0x4e, 0x21, 0xc6, 0x0d, 0x56, 0xe4, 0x23, 0xd2, 0x5e, 0x86,
		0x7f, 0x04, 0xb0, 0x1c, 0xeb, 0x9f, 0x50, 0x17, 0xcc, 0x8e,
		0xf9, 0x8f, 0x00, 0x91, 0xee, 0xa7, 0xa4, 0x91, 0x59, 0xb7,
		0x11, 0x9e, 0x87, 0x40, 0xe2, 0x41, 0x08, 0x0f, 0xec, 0x40,
		0xbc, 0x8c, 0x30, 0x14, 0xcc, 0x97, 0xc9, 0xf0, 0x50, 0x7b,
		0x5c, 0x18, 0xe2, 0x1b, 0x02, 0x7f, 0xda, 0xc0, 0xe6, 0x23,
		0x4b, 0x48, 0xbb, 0xe8, 0x01, 0xe9, 0xd4, 0xb8, 0x52, 0xfa,
		0x36, 0x99, 0xdd, 0x03, 0x8d, 0x21, 0xd3, 0x16, 0x17, 0x2c,
		0x25, 0x20, 0x70, 0x92, 0xd5, 0x57, 0xcb, 0xff, 0x70, 0x16,
		0x33, 0x18, 0xf3, 0x31, 0x1a, 0xb0, 0x07, 0x27, 0x82, 0xc5,
		0x13, 0x86, 0x3d, 0xb5, 0x22, 0xfb, 0x7d, 0x2a, 0x4d, 0x45,
		0xaa, 0x2f, 0x46, 0x67, 0x1e, 0x9f, 0xc4, 0x8b, 0x3a, 0xb8,
		0x8c, 0x56, 0x3c, 0xa4, 0xfe, 0xe7, 0x93, 0xb2, 0xed, 0xc8,
		0x19, 0x3c, 0xc0, 0x5d, 0x38, 0xb0, 0xa2, 0x54, 0xe8, 0x71,
		0xd1, 0x69, 0xc7, 0xb8, 0xdc, 0x33, 0x0c, 0x3b, 0x8f, 0xf4,
		0x37, 0x23, 0x3d, 0x1e, 0x7b, 0x50, 0xc5, 0x4c, 0x3a, 0x6a,
		0x8d, 0xa8, 0x2b, 0x3d, 0x8c, 0x63, 0x4e, 0x52, 0x79, 0xea,
		0x7f, 0xa2, 0x0c, 0xc0, 0xdc, 0xba, 0x60, 0x04, 0x86, 0xa5,
		0xd6, 0x2b, 0xab, 0x2a, 0x17, 0xbe, 0x95, 0x11, 0x2d, 0x14,
		0x4e, 0xc1, 0x42, 0xa3, 0x8e, 0xaf, 0x57, 0x4d, 0x84, 0xd9,
		0x3c, 0x91, 0xb9, 0x45, 0x09, 0x6a, 0xf5, 0xb3, 0x24, 0x4a,
		0x2d, 0x69, 0x46, 0x51, 0x65, 0x9a, 0x37, 0x11, 0x7d, 0xde,
		0x9c, 0x7c, 0x11, 0xaf, 0x81, 0xfa, 0xdf, 0x1e, 0x5d, 0xff,
		0xb2, 0x64, 0x33, 0x75, 0x23, 0x7f, 0x99, 0xc2, 0xa6, 0xb8,
		0x30, 0xdf, 0xd2, 0xb6, 0x5a, 0x35, 0x6b, 0xc5, 0x28, 0xb0,
		0x60, 0x1a, 0x85, 0x59, 0x2e, 0x2b, 0x8b, 0x8f, 0x9a, 0x4c,
		0xd2, 0x9b, 0x90, 0x1e, 0x95, 0x5a, 0xf3, 0xac, 0x34, 0x78,
		0x8c, 0xeb, 0x67, 0x09, 0x6b, 0x95, 0x73, 0x5e, 0x54, 0x1f,
		0x05, 0xd6, 0xe4, 0xf4, 0x3b, 0x4d, 0x93, 0x13, 0x8a, 0xa2,
		0x36, 0x62, 0xb9, 0x76, 0xdc, 0x77, 0xf7, 0x10, 0xf6, 0x73,
		0xb7, 0x49, 0x90, 0x53, 0x9a, 0xbe, 0x96, 0x9d, 0x4c, 0xd4,
		0xf5, 0x09, 0x15, 0x88, 0xe8, 0xc7, 0xbd, 0x04, 0x1e, 0xf8,
		0x70, 0x88, 0x7d, 0xf9, 0x2c, 0xf4, 0x80, 0x5c, 0x6f, 0x55,
		0x73, 0xf2, 0xa0, 0x90, 0xb4, 0xb4, 0xb9, 0x80, 0x49, 0x3a,
		0xc5, 0xda, 0x84, 0x8c, 0x26, 0x84, 0x90, 0xc1, 0x2b, 0xa6,
		0xac, 0x48, 0x2d, 0x37, 0x61, 0x27, 0xd2, 0x18, 0xb8, 0x51,
		0x28, 0x0b, 0x13, 0xbb, 0x35, 0x58, 0xe6, 0xf0, 0x24, 0x18,
		0x28, 0x42, 0x6d, 0xdd, 0xba, 0xe8, 0x64, 0xf2, 0xff, 0x47,
		0x27, 0x59, 0x16, 0x55, 0xcc, 0x5b, 0x54, 0xa3, 0xe2, 0xfb,
		0xef, 0x14, 0x68, 0x8d, 0xbf, 0x2e, 0x9a, 0xbc, 0xfa, 0xb8,
		0x8e, 0x5e, 0x37, 0x8a, 0x1b, 0x67, 0x09, 0xe8, 0x0b, 0xed,
		0x44, 0x71, 0x16, 0x05, 0xbc, 0x5b, 0x2a, 0xf8, 0x8a, 0xdd,
		0xd9, 0x74, 0x53, 0x73, 0xb0, 0xd9, 0xb9, 0xfd, 0xd6, 0x67,
		0x52, 0x7e, 0x2c, 0x50, 0x36, 0xd0, 0x3b, 0xf7, 0x1f, 0xd8,
		0x4a, 0x06, 0x11, 0xf3, 0x21, 0x1f, 0x6e, 0x8f, 0x24, 0x44,
		0xb8, 0xa1, 0xba, 0x0a, 0xbb, 0x6c, 0x18, 0x02, 0x8e, 0xe7,
		0x5b, 0xed, 0xaa, 0x39, 0xbb, 0xb5, 0x99, 0x00, 0xd9, 0x62,
		0x26, 0x5f, 0x26, 0x3e, 0x8e, 0x49, 0x11, 0xd4, 0x2e, 0x70,
		0x5e, 0x79, 0x08, 0x63, 0xfb, 0x8d, 0x70, 0x8f, 0x4f, 0x1d,
		0x4f, 0x33, 0x72, 0xcd, 0xb1, 0xbd, 0x20, 0x4c, 0x74, 0x7f,
		0xde, 0x54, 0xe3, 0x1e, 0xb1, 0x0d, 0xb1, 0xeb, 0x5c, 0x63,
		0xd6, 0xc9, 0xaf, 0x82, 0xa6, 0xc5, 0x19, 0xe7, 0x96, 0x2f,
		0xaf, 0x19, 0x79, 0x4a, 0xe4, 0x78, 0xd0, 0xb3, 0xfc, 0xff,
		0xd7, 0x8f, 0x35, 0x93, 0x41, 0x2d, 0x59, 0xf2, 0xd8, 0xcc,
		0x76, 0xb8, 0x3a, 0xf7, 0xdf, 0x1c, 0xa8, 0x85, 0xd5, 0xf8,
		0x54, 0x63, 0x75, 0x36, 0xea, 0x30, 0x5a, 0x13, 0x74, 0xf9,
		0x6d, 0x55, 0x27, 0x04, 0x0b, 0xa0, 0xa0, 0x6d, 0x9e, 0x3a,
		0x93, 0xbf, 0x4e, 0x92, 0x28, 0x85, 0xe4, 0xcf, 0xc5, 0x06,
		0xa7, 0x17, 0x5a, 0xec, 0xc2, 0x60, 0x6a, 0xab, 0x1b, 0x50,
		0xc9, 0x0b, 0x84, 0x9a, 0x82, 0x45, 0x1e, 0x29, 0xcc, 0x85,
		0x5a, 0xa8, 0xd4, 0x90, 0xfb, 0xa7, 0x2f, 0x89, 0xa7, 0x80,
		0xd0, 0x69, 0xb6, 0xca, 0xe0, 0x00, 0x4e, 0xe8, 0xea, 0x12,
		0x74, 0x85, 0xc8, 0x35, 0xea, 0x1d, 0x23, 0x75, 0x97, 0xcb,
		0x1d, 0x94, 0x6f, 0x2a, 0xa9, 0x7a, 0x32, 0xc5, 0x92, 0xf0,
		0x0a, 0x28, 0xbd, 0x77, 0xbc, 0xbe, 0xae, 0xa7, 0xa8, 0x34,
		0x96, 0x26, 0xaf, 0x83, 0x2a, 0x8e, 0xe9, 0xda, 0xad, 0xb2,
		0x52, 0x94, 0x74, 0x69, 0x70, 0x19, 0xc4, 0xc8, 0x36, 0xf3,
		0x0f, 0x32, 0x0f, 0x6f, 0x2b, 0x75, 0xac, 0x0e, 0x9d, 0x91,
		0x1f, 0x06, 0xba, 0xe3, 0x9c, 0x7f, 0xde, 0x16, 0x3a, 0x3f,
		0xfd, 0x7d, 0x88, 0x61, 0x90, 0xca, 0x98, 0xc1, 0xcf, 0xa5,
		0x50, 0x1f, 0xb3, 0x8c, 0xf9, 0x8c, 0xe9, 0x9f, 0xa3, 0xd3,
		0xd7, 0xce, 0x7f, 0xdf, 0xd2, 0x8e, 0x84, 0x9c, 0x24, 0xc3,
		0x85, 0xb1, 0xc6, 0x94, 0x8f, 0xbb, 0x55, 0x83, 0x8e, 0xa3,
		0xc7, 0xf5, 0x8e, 0x7e, 0xe6, 0x30, 0x13, 0x06, 0x61, 0x9e,
		0x63, 0x24, 0x27, 0x03, 0x01, 0xef, 0xec, 0x6d, 0xb1, 0x56,
		0x86, 0xd9, 0xf3, 0x2e, 0xfc, 0xe2, 0xbb, 0xa5, 0xa7, 0xaa,
		0x9c, 0x06, 0xf3, 0x36, 0x04, 0xae, 0xbd, 0x0f, 0x6f, 0xb9,
		0xc7, 0x42, 0x79, 0xfd, 0xac, 0x8f, 0x1c, 0xcc, 0x01, 0x9f,
		0x45, 0x3a, 0x52, 0xa2, 0x20, 0x31, 0xf6, 0x5d, 0xea, 0x4e,
		0xe7, 0x13, 0x9d, 0x79, 0xe8, 0x81, 0x62, 0xcd, 0xaf, 0x2e,
		0x6b, 0xb5, 0x1b, 0x75, 0xfe, 0x61, 0xf7, 0xcf, 0x4d, 0x90,
		0xdf, 0xe8, 0x56, 0x23, 0x96, 0xe4, 0xab, 0x5d, 0xa5, 0xd8,
		0x2e, 0x9e, 0x0c, 0x30, 0x27, 0xa3, 0x2c, 0x10, 0xda, 0x79,
		0xaf, 0xb9, 0x16, 0x6a, 0x89, 0x33, 0xe7, 0x46, 0xef, 0x43,
		0xe5, 0x5e, 0xb6, 0x38, 0xaa, 0x07, 0xb6, 0x5d, 0xd8, 0x57,
		0x4b, 0x94, 0xbd, 0xd0, 0x73, 0x9c, 0xf9, 0xa9, 0x31, 0x8a,
		0x71, 0xe5, 0xf0, 0xe6, 0xfb, 0x4e, 0xc4, 0xc7, 0x4b, 0xfe,
		0xe2, 0x1e, 0xbc, 0x77, 0xba, 0x3b, 0x2c, 0xa2, 0x71, 0x2e,
		0xf1, 0xd7, 0x33, 0xa2, 0xae, 0x2d, 0x89, 0x6a, 0xcb, 0x63,
		0xed, 0x8d, 0x83, 0x10, 0x96, 0xb4, 0x43, 0xef, 0xc2, 0xe6,
		0x49, 0x13, 0x2f, 0x0a, 0x09, 0xae, 0xca, 0xa8, 0x01, 0xff,
		0xc9, 0x65, 0x87, 0x8d, 0x20, 0x98, 0x5f, 0xfb, 0x25, 0xe0,
		0x7a, 0x3a, 0xcb, 0xf8, 0x48, 0x7e, 0x18, 0xcc, 0x4c, 0x66,
		0xd3, 0x76, 0x10, 0x93, 0xa2, 0x7d, 0x3b, 0x0d, 0x49, 0xca,
		0xa9, 0x18, 0x2a, 0x50, 0xe2, 0xb3, 0xdc, 0x77, 0x3c, 0x23,
		0x84, 0xfe, 0xe7, 0x86, 0x39, 0x7f, 0xc7, 0x18, 0xca, 0xae,
		0x79, 0xfc, 0x44, 0x27, 0xf9, 0xbc, 0xce, 0xdb, 0xda, 0xe9,
		0xd5, 0x83, 0x90, 0xfb, 0x8c, 0x79, 0x4b, 0xc9, 0x1b, 0x6b,
		0x57, 0x0a, 0x88, 0xfc, 0x80, 0x7d, 0x17, 0x10, 0x32, 0x7d,
		0xf5, 0x87, 0xf7, 0x18, 0xd6, 0x7a, 0x60, 0x5c, 0x2b, 0x4a,
		0x19, 0x0a, 0xea, 0x3f, 0xd1, 0x5b, 0x5e, 0x93, 0x38, 0x9f,
		0xc9, 0xfc, 0x6c, 0x67, 0x91, 0xd6, 0x0c, 0x3b, 0x1e, 0xa6,
		0x2d, 0x1a, 0x71, 0x74, 0x57, 0x67, 0x46, 0xa2, 0x93, 0xaf,
		0x9d, 0x85, 0x0e, 0x0b, 0x1d, 0x39, 0x84, 0xe1, 0x7c, 0x8a,
		0x11, 0xed, 0x7b, 0xfe, 0x98, 0xaf, 0x2b, 0x7a, 0xe6, 0xc6,
		0x70, 0xf7, 0xce, 0x13, 0xe6, 0x5a, 0x96, 0x7e, 0x5d, 0x76,
		0x30, 0xfc, 0x22, 0xdc, 0x4e, 0xb2, 0xd5, 0x59, 0x66, 0x54,
		0x70, 0x25, 0x83, 0x1f, 0x16, 0xf0, 0x0a, 0x72, 0xd5, 0xb1,
		0x69, 0x86, 0x13, 0xb8, 0x6c, 0x0f, 0xe0, 0x5a, 0xdd, 0x5e,
		0x26, 0x9a, 0x9e, 0xbc, 0x94, 0xdf, 0xd0, 0x08, 0x4f, 0xc9,
		0x87, 0xf3, 0xd5, 0x6d, 0xad, 0x65, 0xbd, 0x52, 0xd2, 0x10,
		0x71, 0xca, 0x4b, 0xe5, 0x72, 0xcd, 0xae, 0xa7, 0x9c, 0x51,
		0x52, 0x04, 0xe5, 0xcb, 0x80, 0xfa, 0x23, 0x1b, 0x4a, 0x0d,
		0x83, 0x8a, 0xa6, 0x91, 0xef, 0x10, 0x08, 0x05, 0x2a, 0x39,
		0x14, 0x2f, 0xdb, 0x1f, 0x96, 0x8d, 0xe1, 0x58, 0x08, 0xd0,
		0x65, 0x6d, 0x83, 0x8f, 0x71, 0xe0, 0x87, 0x90, 0x96, 0xce,
		0xa4, 0xa0, 0x30, 0xf7, 0x56, 0xd4, 0x88, 0x61, 0x08, 0x61,
		0xda, 0xd1, 0x36, 0x8a, 0x92, 0x3f, 0xb1, 0xd4, 0xd0, 0x6f,
		0x57, 0x9b, 0xb4, 0xe2, 0x96, 0x12, 0xad, 0x93, 0x4b, 0xd6,
		0x59, 0x37, 0x08, 0x4f, 0x7a, 0xc8, 0x7f, 0x37, 0x8e, 0x11,
		0x5f, 0x07, 0xd2, 0x5e, 0xbe, 0x54, 0xcb, 0xa8, 0xc6, 0x2e,
		0xfc, 0x23, 0x7e, 0xab, 0x40, 0x24, 0x19, 0x4c, 0x51, 0x19,
		0x52, 0xcc, 0xc2, 0x70, 0x63, 0xc6, 0x34, 0x1c, 0x97, 0x14,
		0xb1, 0xfe, 0xd3, 0xe6, 0x98, 0xf2, 0xd7, 0x77, 0x5a, 0x69,
		0xc6, 0xf6, 0x0f, 0xb9, 0x62, 0xae, 0x4a, 0xb0, 0x33, 0x3a,
		0x43, 0x16, 0x2b, 0x66, 0xbf, 0xc6, 0x8c, 0x41, 0x8d, 0x0f,
		0x94, 0x79, 0x77, 0x81, 0x01, 0x23, 0xd9, 0x5e, 0x81, 0x74,
		0x1d, 0x87, 0x11, 0x76, 0xc6, 0x0f, 0x40, 0x91, 0x72, 0xe9,
		0x7d, 0x51, 0xb8, 0x72, 0xa9, 0x6f, 0x23, 0x1d, 0x32, 0x33,
		0xde, 0xd8, 0xf7, 0x9c, 0x52, 0xa4, 0x6d, 0x4f, 0x3f, 0x9e,
		0xe4, 0xfa, 0x25, 0x8f, 0x27, 0x78, 0x8d, 0xde, 0x39, 0xa2,
		0xfd, 0xd1, 0x39, 0x01, 0xeb, 0xce, 0xac, 0x60, 0xe4, 0x9f,
		0x69, 0x8b, 0xf8, 0xd1, 0x85, 0x70, 0x74, 0xad, 0x1e, 0x30,
		0x89, 0x9f, 0x84, 0x58, 0xff, 0xd5, 0x84, 0x41, 0xa6, 0x68,
		0x16, 0xc8, 0x07, 0x9b, 0x7a, 0x33, 0x29, 0x05, 0x5c, 0x0e,
		0x08, 0x94, 0x4d, 0x18, 0x80, 0xba, 0x10, 0x82, 0x4e, 0x92,
		0x60, 0xa8, 0xff, 0x1a, 0x0e, 0x46, 0x57, 0x1e, 0x65, 0x67,
		0x25, 0x8d, 0x28, 0xb1, 0xdb, 0x66, 0xca, 0xc0, 0xbc, 0xa9,
		0x90, 0x8a, 0x74, 0x52, 0x97, 0x68, 0x32, 0x34, 0x60, 0xb0,
		0x30, 0x60, 0xd5, 0x89, 0x15, 0x09, 0x1f, 0x95, 0xc5, 0x41,
		0xe2, 0x02, 0x7a, 0x81, 0x7d, 0x49, 0xdd, 0x8c, 0x26, 0x05,
		0xfe, 0xed, 0xa8, 0x12, 0xa4, 0x3d, 0x82, 0x65, 0x3d, 0x87,
		0xed, 0x95, 0x86, 0x80, 0x3f, 0xc6, 0xcd, 0x27, 0x58, 0x56,
		0xca, 0x23, 0xca, 0xa4, 0xd8, 0x06, 0x11, 0xb4, 0x24, 0xc8,
		0x8a, 0xc5, 0x36, 0x42, 0xe3, 0xd4, 0xf7, 0xaf, 0x74, 0xac,
		0x89, 0xee, 0x47, 0x2e, 0x8b, 0xde, 0xd0, 0x85, 0xd4, 0x2e,
		0xb4, 0xae, 0x99, 0x96, 0x88, 0xe2, 0x60, 0x99, 0xfb, 0x62,
		0x5f, 0x8b, 0xa7, 0x24, 0xe2, 0xc3, 0x72, 0x4c, 0x6f, 0x1d,
		0xcd, 0xc1, 0x3a, 0x32, 0x06, 0x98, 0xb5, 0x92, 0x82, 0x98,
		0xb1, 0x7d, 0xca, 0xa1, 0xe1, 0x4d, 0xe1, 0xe5, 0xaf, 0x3a,
		0x08, 0x0d, 0x68, 0x33, 0xb8, 0xc0, 0xc3, 0xfd, 0xb9, 0x90,
		0x47, 0x58, 0xca, 0xb2, 0x8d, 0x2d, 0x75, 0x7d, 0xbb, 0xd3,
		0xc3, 0x0f, 0x7d, 0xa7, 0x5a, 0x5a, 0x25, 0x6f, 0x12, 0xd8,
		0x69, 0xd4, 0xc1, 0x20, 0x19, 0x65, 0x4d, 0x36, 0x7c, 0x7f,
		0xc4, 0x47, 0x00, 0xcb, 0x2c, 0x64, 0x13, 0x06, 0x0a, 0x37,
		0x1d, 0x1d, 0x7d, 0x89, 0x9c, 0x21, 0xd0, 0xaa, 0xa9, 0x01,
		0xee, 0xb2, 0x70, 0xc7, 0xc3, 0x11, 0x60, 0xd9, 0x73, 0xc9,
		0xea, 0x6d, 0x8a, 0x1b, 0x0f, 0x69, 0x1d, 0xb3, 0x95, 0x82,
		0x52, 0xf4, 0xa9, 0x4e, 0x25, 0x85, 0x75, 0xde, 0x12, 0x3c,
		0x11, 0x71, 0x06, 0x88, 0xf0, 0x39, 0x93, 0x29, 0x3a, 0x94,
		0x7c, 0x73, 0x42, 0xea, 0x59, 0x7c, 0x1d, 0x93, 0x1a, 0x14,
		0x10, 0x4c, 0x5e, 0x92, 0x1a, 0x9e, 0x71, 0xf1, 0x42, 0xb9,
		0x72, 0x13, 0x7d, 0x99, 0x4a, 0x28, 0x8b, 0x92, 0x48, 0x1d,
		0x78, 0x19, 0x04, 0xf8, 0xc4, 0xf3, 0xb1, 0x1c, 0x2f, 0x56,
		0xb6, 0xf6, 0x87, 0x00, 0xbd, 0x8c, 0x8c, 0x56, 0xe1, 0x38,
		0xf8, 0x6d, 0x19, 0xe7, 0x7e, 0x30, 0x52, 0xc6, 0x47, 0x72,
		0xc9, 0xa1, 0x30, 0xef, 0x8c, 0xf6, 0x23, 0xbf, 0xfb, 0xd0,
		0xbb, 0x51, 0xa9, 0xf6, 0x1d, 0xa7, 0xd2, 0xc7, 0xaa, 0xa6,
		0xc8, 0xa6, 0xa6, 0x61, 0x8f, 0x8c, 0x22, 0xee, 0x6e, 0xb5,
		0x21, 0x89, 0x72, 0x0e, 0xbe, 0x00, 0xcf, 0x08, 0x80, 0x09,
		0x06, 0x4d, 0xd4, 0xd4, 0xea, 0xb6, 0xa0, 0x8e, 0x71, 0xc8,
		0x73, 0x16, 0xdb, 0x55, 0x0f, 0xed, 0xbf, 0xb7, 0x40, 0x41,
		0x9a, 0x47, 0xfc, 0xad, 0x5b, 0xf4, 0x86, 0x3c, 0x98, 0xd1,
		0xaa, 0x51, 0x15, 0xc3, 0x10, 0x55, 0x8e, 0x5d, 0xf7, 0x9d,
		0xd4, 0x11, 0x35, 0xe7, 0x05, 0x11, 0x7d, 0x0f, 0x2b, 0x44,
		0xfe, 0x12, 0x7f, 0x39, 0xa6, 0x19, 0x91, 0x8b, 0xd0, 0xac,
		0x2c, 0x18, 0x63, 0x6a, 0x33, 0xc2, 0x75, 0xd2, 0xcd, 0xf7,
		0x22, 0x46, 0xca, 0xd1, 0xf9, 0xc0, 0x15, 0x08, 0xc7, 0x10,
		0x67, 0xd1, 0x42, 0x13, 0x27, 0x22, 0xc3, 0xe3, 0x94, 0x66,
		0x00, 0xbf, 0x2f, 0x7d, 0xfb, 0xc5, 0x6f, 0xb9, 0x03, 0x24,
		0xce, 0xfb, 0xf4, 0x6b, 0xad, 0xe0, 0x47, 0xda, 0x5f, 0xea,
		0x61, 0xfd, 0x6a, 0x76, 0x98, 0x15, 0x76, 0x72, 0xef, 0x95,
		0x77, 0x93, 0x3b, 0x23, 0xde, 0x09, 0x38, 0x1c, 0x7b, 0xd5,
		0xd1, 0x5c, 0xc5, 0xc4, 0x34, 0x0c, 0xd8, 0xc2, 0x05, 0xb7,
		0xe5, 0xb5, 0xc9, 0x7b, 0xc1, 0x28, 0xea, 0xc8, 0x07, 0x98,
		0xe2, 0xab, 0x73, 0x24, 0xfb, 0xef, 0xce, 0x0a, 0xa7, 0xb9,
		0x8d, 0xeb, 0x31, 0x09, 0x82, 0xf2, 0x9c, 0x4d, 0xf1, 0x1a,
		0x4a, 0x79, 0x35, 0xa7, 0x7b, 0x7b, 0xf0, 0xa9, 0xe7, 0xb9,
		0xf9, 0x52, 0x63, 0x73, 0x29, 0xe4, 0xd4, 0x13, 0x19, 0x2a,
		0xf2, 0x9d, 0x4e, 0xcc, 0x2a, 0xb3, 0xdf, 0x95, 0xc5, 0xb6,
		0x91, 0x34, 0xf4, 0x10, 0x2e, 0xe7, 0x0d, 0xf8, 0x3a, 0x6d,
		0xdd, 0x30, 0x57, 0xec, 0x05, 0x70, 0xef, 0x80, 0x8c, 0x8a,
		0xe1, 0x2e, 0x9e, 0xd4, 0xd1, 0xb4, 0xdc, 0x14, 0x87, 0xf5,
		0x97, 0x4b, 0x2c, 0xd9, 0x93, 0x71, 0x3a, 0x15, 0x25, 0xa0,
		0x8a, 0xb2, 0x93, 0xb5, 0x62, 0xf7, 0x81, 0x09, 0x9c, 0x8d,
		0xe5, 0x09, 0xb2, 0xda, 0x07, 0xf2, 0x33, 0xf6, 0x5c, 0xd2,
		0x71, 0xb1, 0xaf, 0xea, 0x62, 0x05, 0xf3, 0x5f, 0xd9, 0xb1,
		0x8e, 0xd1, 0x5d, 0xf9, 0x89, 0x85, 0x78, 0xd5, 0x54, 0x56,
		0x6c, 0xd4, 0x20, 0x45, 0xe7, 0x03, 0xcc, 0x25, 0x7e, 0xde,
		0x40, 0x3d, 0xb2, 0x46, 0x90, 0x59, 0xc2, 0xfc, 0xdc, 0x15,
		0xd4, 0x80, 0x5b, 0xd8, 0xec, 0x2e, 0x49, 0xd7, 0xd8, 0x45,
		0x0e, 0x7d, 0xba, 0x09, 0x32, 0x60, 0xdb, 0x64, 0xd6, 0x44,
		0xd7, 0xea, 0xb4, 0xb1, 0xb6, 0xfc, 0x1b, 0xde, 0x4a, 0x48,
		0xa1, 0x6c, 0x5b, 0x5a, 0x1f, 0x35, 0x55, 0x18, 0xd2, 0xae,
		0x88, 0x46, 0x5f, 0x2d, 0x25, 0xb0, 0xd9, 0x4f, 0x40, 0xbb,
		0x23, 0x40, 0x9f, 0x85, 0x19, 0x4e, 0xd4, 0xfc, 0xcd, 0xc4,
		0x59, 0x98, 0x27, 0xe2, 0x7e, 0xab, 0x36, 0x52, 0x51, 0x30,
		0xf4, 0xb4, 0xf3, 0xa1, 0x8d, 0x87, 0x01, 0xe5, 0x2d, 0x42,
		0x4b, 0x87, 0x70, 0x95, 0xde, 0x9d, 0x80, 0x36, 0x4e, 0x8a,
		0x14, 0x40, 0x1c, 0x27, 0x89, 0xe1, 0x42, 0xee, 0x77, 0x87,
		0x4f, 0x4d, 0xb0, 0x38, 0x3f, 0x05, 0x67, 0xe3, 0x9c, 0x88,
		0xd8, 0x68, 0x75, 0xf3, 0xf9, 0x92, 0x8e, 0x78, 0xbe, 0x43,
		0xb9, 0x28, 0xa9, 0x2e, 0xde, 0xc9, 0xbd, 0x48, 0x99, 0x4d,
		0x7d, 0xb6, 0xba, 0x3b, 0x31, 0x20, 0x9f, 0x0c, 0xc4, 0x90,
		0xae, 0x4a, 0x33, 0x17, 0x58, 0x13, 0xb5, 0xfd, 0xec, 0xef,
		0x87, 0x21, 0x0c, 0xce, 0xbd, 0xbd, 0x85, 0x61, 0x07, 0xb8,
		0x7f, 0x93, 0xaa, 0xbd, 0xc6, 0xae, 0x63, 0x2c, 0x12, 0x6a,
		0xaa, 0xe2, 0x9d, 0xf8, 0x55, 0xac, 0xc3, 0xa3, 0x40, 0xf8,
		0xad, 0xfb, 0x21, 0x89, 0x5c, 0x90, 0x21, 0x40, 0x57, 0xfb,
		0x08, 0x5d, 0x28, 0x83, 0xcc, 0x46, 0x76, 0x44, 0x2e, 0xe4,
		0x5b, 0xe7, 0x37, 0x79, 0xc7, 0xc8, 0x0a, 0x0b, 0xc2, 0xb6,
		0x17, 0x48, 0x8c, 0x2b, 0x67, 0xd6, 0xf9, 0x33, 0xe0, 0xbd,
		0x6b, 0xea, 0x79, 0x7d, 0x6d, 0x5b, 0x23, 0x7e, 0x5e, 0x5d,
		0x53, 0xcf, 0x01, 0xea, 0x86, 0x5f, 0x8a, 0x4b, 0x1f, 0x32,
		0xcd, 0xdb, 0x30, 0xe3, 0x27, 0x3f, 0x10, 0x4b, 0xba, 0x3c,
		0xf9, 0x42, 0x78, 0x52, 0x01, 0x87, 0xed, 0xf4, 0x15, 0xa4,
		0x8d, 0xa0, 0x1d, 0xc7, 0x89, 0xf5, 0x4f, 0x99, 0x28, 0x04,
		0xa7, 0x87, 0xf8, 0x9b, 0x77, 0x18, 0x0a, 0xb8, 0x59, 0xf3,
		0xcd, 0x74, 0xf1, 0xd5, 0x13, 0x61, 0xca, 0x02, 0x95, 0xe6,
		0x50, 0x30, 0x96, 0x83, 0x35, 0xf8, 0xd2, 0xea, 0xd7, 0xd6,
		0x5f, 0xbf, 0xab, 0x9b, 0x91, 0xc9, 0xc4, 0x93, 0x57, 0xb5,
		0x55, 0xb6, 0x83, 0xb0, 0x83, 0x17, 0x78, 0xc9, 0xb6, 0xa5,
		0x55, 0x9b, 0x2d, 0x08, 0x1f, 0x09, 0x37, 0x1c, 0x0a, 0x3a,
		0xdc, 0x32, 0xeb, 0x20
	};

	static const uint8_t keys_b_ps5[0x70 * 0x20] = {
		0xe1, 0x34, 0x17, 0x1d, 0xf5, 0x17, 0xda, 0xe1, 0x1b, 0xe6,
		0x0c, 0x73, 0x35, 0xd4, 0x6b, 0x42, 0x28, 0xcd, 0x47, 0x06,
		0x4e, 0x96, 0x87, 0xd5, 0x9d, 0x09, 0xe7, 0x24, 0xa0, 0x1b,
		0xad, 0x32, 0x6d, 0x06, 0x75, 0xdf, 0x77, 0x69, 0x89, 0xeb,
		0x03, 0x24, 0xd6, 0x31, 0x23, 0xc0, 0xf6, 0x0d, 0x60, 0x0c,
		0xb6, 0xad, 0x4b, 0xba, 0x9e, 0xb2, 0x66, 0xfd, 0x8d, 0xb5,
		0xdd, 0xc9, 0xf7, 0x67, 0xc3, 0x93, 0x3f, 0x1e, 0xc1, 0x87,
		0x5c, 0x34, 0xae, 0x24, 0x44, 0xc0, 0x34, 0xf3, 0xd9, 0x1e,
		0x3b, 0x62, 0x0b, 0xa0, 0x00, 0xa2, 0x1e, 0x13, 0x9b, 0xa6,
		0xe8, 0x8b, 0x91, 0xb1, 0x42, 0xb1, 0x2b, 0xa7, 0xce, 0xc6,
		0x85, 0x6e, 0xe9, 0x22, 0xfc, 0x6d, 0x77, 0xb2, 0x9b, 0x71,
		0x5d, 0x1a, 0xf4, 0xb6, 0x93, 0x72, 0xae, 0x46, 0x40, 0x5d,
		0x26, 0x30, 0x98, 0xf8, 0xc7, 0xbe, 0xb7, 0x2c, 0x6e, 0x1b,
		0x13, 0x32, 0x7a, 0xb2, 0x81, 0xa6, 0x96, 0xc4, 0xe8, 0x15,
		0x82, 0xc4, 0xea, 0xc3, 0x28, 0x70, 0xaf, 0x3f, 0xf6, 0x7f,
		0x13, 0xa6, 0x73, 0x27, 0x29, 0xdf, 0x17, 0x38, 0xd4, 0xb0,
		0x72, 0xed, 0x91, 0xb5, 0xb1, 0xe6, 0xd7, 0x3a, 0x34, 0xa9,
		0x67, 0x45, 0x67, 0x02, 0xc2, 0xc6, 0x3c, 0x22, 0xf0, 0x22,
		0xec, 0xff, 0x7d, 0xda, 0x2c, 0xe4, 0x67, 0x47, 0x6b, 0x31,
		0x63, 0xf4, 0xb3, 0x5d, 0xdb, 0x47, 0x33, 0xa6, 0x1c, 0x21,
		0xa3, 0x48, 0x0f, 0x7f, 0xc9, 0x84, 0x1a, 0x57, 0x0d, 0xd5,
		0x72, 0xd6, 0x20, 0x34, 0x36, 0x41, 0x0e, 0x43, 0x11, 0xe5,
		0x43, 0x91, 0x05, 0x69, 0xcf, 0x79, 0x70, 0x90, 0xc5, 0x7a,
		0xf3, 0xb1, 0x3d, 0x59, 0x08, 0x59, 0xe1, 0x41, 0xae, 0x65,
		0x2e, 0x5c, 0xcb, 0x04, 0x8d, 0x4c, 0xac, 0x50, 0x21, 0x93,
		0x06, 0x36, 0xb6, 0xe0, 0xb1, 0xdc, 0x4c, 0x98, 0x48, 0x8f,
		0x95, 0x12, 0x64, 0x40, 0x87, 0x6b, 0xdd, 0xf7, 0x79, 0x0a,
		0xc1, 0x1d, 0x67, 0xf9, 0xf5, 0xf8, 0xce, 0x75, 0x94, 0x0a,
		0x66, 0x17, 0xec, 0x17, 0xee, 0xfa, 0xf6, 0x88, 0x8b, 0xeb,
		0x2e, 0x06, 0xf1, 0x77, 0x1d, 0x22, 0xd3, 0x4c, 0x4f, 0x8d,
		0xa8, 0x6a, 0x53, 0xc7, 0xd4, 0x72, 0xb4, 0xec, 0x97, 0x9d,
		0x31, 0x48, 0xed, 0x62, 0xd2, 0x40, 0x94, 0x23, 0x4c, 0x10,
		0x37, 0x20, 0x3b, 0xb6, 0x9f, 0x1c, 0x99, 0xcf, 0xce, 0x9d,
		0xda, 0xe7, 0xc9, 0x32, 0x88, 0xe9, 0x1d, 0x37, 0xb5, 0xa7,
		0x8b, 0x19, 0x8a, 0x50, 0x50, 0x2d, 0xa9, 0x89, 0x4e, 0x6e,
		0x7d, 0xbf, 0xd2, 0x7e, 0xe1, 0xa2, 0x6b, 0x3e, 0xf1, 0x33,
		0x0e, 0x74, 0x88, 0x96, 0xd8, 0xa3, 0x00, 0x2d, 0x9b, 0xaf,
		0x71, 0xac, 0x68, 0x63, 0x2f, 0xa9, 0x46, 0x6c, 0x73, 0x5d,
		0xb0, 0xd6, 0x92, 0xa3, 0x03, 0xc4, 0x93, 0xb6, 0x48, 0x7f,
		0x3c, 0x15, 0x1b, 0xb1, 0xe0, 0x40, 0x56, 0x8f, 0x63, 0xa6,
		0x94, 0xc9, 0x8b, 0xde, 0x66, 0x2a, 0x5f, 0x32, 0x4a, 0x8d,
		0xa6, 0x50, 0xdf, 0xdf, 0xaa, 0x20, 0xe1, 0xeb, 0x71, 0x05,
		0xd7, 0xca, 0xe1, 0xa8, 0x79, 0xbc, 0x83, 0x94, 0x74, 0x85,
		0x58, 0x89, 0xc8, 0x59, 0x97, 0xaa, 0xf0, 0xfa, 0x5a, 0x37,
		0xa7, 0x4a, 0x13, 0xe6, 0x36, 0x0f, 0xe6, 0x45, 0x29, 0x89,
		0xb6, 0x11, 0x32, 0xdb, 0x0d, 0x75, 0x58, 0xe2, 0x9a, 0x1a,
		0x1b, 0x08, 0x66, 0x2a, 0x3d, 0x66, 0xe1, 0x24, 0x82, 0xe6,
		0x0b, 0xc5, 0xe6, 0xd7, 0x9c, 0x18, 0xa3, 0x4d, 0x2f, 0x4e,
		0x46, 0xd6, 0x21, 0x76, 0xaf, 0xad, 0xa9, 0xd8, 0x37, 0xd1,
		0x3d, 0xfe, 0x32, 0x24, 0xdc, 0xc4, 0xbe, 0x69, 0x2e, 0x97,
		0x8f, 0x09, 0xb1, 0x55, 0x9b, 0x3d, 0xe2, 0x46, 0x1c, 0x99,
		0x27, 0x4d, 0x06, 0xff, 0x72, 0x7a, 0xef, 0xe9, 0x9a, 0xc4,
		0xe9, 0x93, 0xc4, 0xf0, 0xb6, 0x64, 0xa9, 0xea, 0xf2, 0x09,
		0xd6, 0xa3, 0x20, 0x6c, 0xd1, 0x1b, 0xd7, 0x7f, 0x74, 0x00,
		0xf5, 0x60, 0xf8, 0xf4, 0x99, 0x93, 0xa5, 0xc2, 0x2b, 0xb1,
		0xe9, 0x83, 0x39, 0x33, 0x4d, 0x16, 0x0b, 0xb8, 0xa6, 0x6d,
		0x30, 0x9b, 0x97, 0xbe, 0x9f, 0x0c, 0x1e, 0x4e, 0xff, 0x3a,
		0xe2, 0x06, 0x15, 0xba, 0x57, 0x9d, 0x60, 0x76, 0x4d, 0x50,
		0x88, 0x6d, 0x8b, 0x9d, 0xf9, 0x70, 0x3d, 0x66, 0x87, 0x11,
		0x1e, 0x9a, 0x24, 0xaa, 0xa5, 0xe8, 0xf8, 0x38, 0x6c, 0x3e,
		0x6d, 0xda, 0xe5, 0xfc, 0xde, 0x9e, 0x2a, 0xe4, 0xe6, 0xee,
		0x07, 0xce, 0x48, 0x5e, 0xd8, 0x27, 0x23, 0x10, 0xc9, 0x72,
		0x9d, 0x7c, 0xd9, 0x5d, 0xa8, 0xba, 0x8a, 0x1e, 0x22, 0xef,
		0x46, 0x22, 0x6f, 0x79, 0x2e, 0x6f, 0x73, 0xbf, 0xcd, 0xcf,
		0x4f, 0x13, 0xdc, 0x5d, 0x51, 0x0b, 0xd5, 0xa9, 0x76, 0xc5,
		0xe3, 0xc0, 0x90, 0xe9, 0x32, 0x28, 0xfc, 0x9f, 0xf1, 0x88,
		0x1f, 0xa6, 0xb3, 0x45, 0x04, 0xa7, 0x1c, 0xbd, 0x66, 0xe7,
		0xbb, 0xc1, 0x49, 0x42, 0xdb, 0x52, 0x4b, 0xbb, 0xd9, 0x63,
		0x8f, 0x6d, 0x9d, 0x3b, 0xd7, 0xe3, 0x49, 0xbc, 0x7c, 0xd9,
		0xdb, 0x50, 0x67, 0x6f, 0x07, 0x6d, 0x08, 0x2a, 0xea, 0x02,
		0xeb, 0xbe, 0x22, 0xd1, 0x17, 0x1b, 0xfa, 0x39, 0x1c, 0x6c,
		0xda, 0x56, 0x37, 0xd3, 0x8e, 0xf4, 0xe3, 0x22, 0xaa, 0x6c,
		0xc1, 0x75, 0x71, 0x4c, 0x21, 0x36, 0xfb, 0xe0, 0x02, 0x54,
		0x6a, 0x48, 0x31, 0x40, 0x7b, 0x64, 0x7c, 0x76, 0x31, 0xe5,
		0x0f, 0xc1, 0xd8, 0xb4, 0x29, 0xd9, 0x9c, 0x0c, 0x80, 0xce,
		0xa3, 0x7c, 0x2f, 0x0f, 0x4c, 0x11, 0xb5, 0x3d, 0x85, 0x93,
		0xbd, 0x60, 0xd4, 0x5c, 0x3f, 0x41, 0x54, 0xe6, 0x50, 0xb1,
		0xb1, 0xed, 0x85, 0x19, 0x30, 0x81, 0x6a, 0x95, 0xfe, 0x4a,
		0x93, 0x28, 0x12, 0x3d, 0xdc, 0x76, 0xac, 0x59, 0xe6, 0xe3,
		0x56, 0x81, 0x1d, 0x29, 0xf9, 0x65, 0x48, 0xc3, 0x70, 0x50,
		0x2f, 0x2c, 0x4c, 0x69, 0x01, 0xdc, 0x75, 0x3d, 0x14, 0xff,
		0x5f, 0xaf, 0x6a, 0xc7, 0xc1, 0xa3, 0x3b, 0x12, 0xa5, 0x6d,
		0xd7, 0x95, 0xdd, 0x3d, 0x70, 0x67, 0xcb, 0x2f, 0x22, 0x5d,
		0xac, 0x9d, 0xf2, 0x1c, 0x6a, 0x27, 0x45, 0x59, 0x29, 0x5a,
		0x4f, 0x6c, 0x34, 0x43, 0x43, 0x94, 0xd1, 0xe6, 0x1d, 0xde,
		0xda, 0x7c, 0x4f, 0xdf, 0xf6, 0xa1, 0x42, 0xde, 0x88, 0xc7,
		0xcc, 0xbb, 0x99, 0xb0, 0x49, 0xa5, 0x9d, 0x6a, 0xae, 0x3f,
		0xb0, 0x59, 0x2c, 0xfa, 0xb8, 0x66, 0x26, 0xf5, 0xdb, 0xb7,
		0xe8, 0xc5, 0xb3, 0x45, 0xfa, 0xd2, 0xd3, 0xec, 0x1b, 0xcf,
		0x83, 0xc5, 0x96, 0x66, 0x92, 0x27, 0xd4, 0x88, 0xbf, 0x3a,
		0x7e, 0x00, 0x22, 0xb2, 0x8a, 0x80, 0xfe, 0xf9, 0x75, 0x1a,
		0xd1, 0xcc, 0xaf, 0x97, 0x9f, 0x6c, 0xd6, 0x7f, 0x1f, 0x6e,
		0xe1, 0x5f, 0x77, 0x2b, 0x96, 0xd6, 0x21, 0x73, 0xf9, 0x6a,
		0x8e, 0x99, 0x65, 0x19, 0x0f, 0x91, 0xe7, 0x51, 0x57, 0x2c,
		0x59, 0x7b, 0xdc, 0xb2, 0x84, 0x06, 0x12, 0x6d, 0xf9, 0x61,
		0x4c, 0xa7, 0x4f, 0x2c, 0x8a, 0x58, 0xfb, 0xc3, 0x6c, 0x90,
		0x9b, 0xc3, 0x19, 0x72, 0x34, 0x44, 0x9b, 0x2f, 0x53, 0xc3,
		0x6f, 0x4e, 0xf7, 0xff, 0x7f, 0x22, 0x04, 0xd4, 0x5c, 0xd7,
		0x18, 0x07, 0x29, 0xc8, 0x3e, 0x2b, 0xbe, 0xa9, 0xba, 0x17,
		0x3b, 0xcb, 0xa8, 0xb0, 0x63, 0x90, 0x09, 0x22, 0x4d, 0xbf,
		0x54, 0x31, 0x4f, 0x41, 0x8b, 0xaa, 0x8b, 0x8e, 0x19, 0x07,
		0x60, 0x62, 0xdc, 0x4b, 0x99, 0x64, 0x95, 0xea, 0xf6, 0xcc,
		0xfc, 0x97, 0xe5, 0x7f, 0xd4, 0xd9, 0xb9, 0x9b, 0x5a, 0x28,
		0x05, 0x8f, 0x30, 0x7d, 0x5f, 0x11, 0x55, 0x62, 0x64, 0x98,
		0x9c, 0x2a, 0x6b, 0x6e, 0x64, 0xac, 0x3f, 0x1f, 0xbc, 0x45,
		0x98, 0x0e, 0xed, 0x51, 0x92, 0x19, 0x84, 0x11, 0xb8, 0x8f,
		0xe7, 0xb7, 0x7a, 0x9b, 0xc0, 0x41, 0x50, 0x85, 0xc9, 0xc2,
		0x0e, 0xe9, 0x84, 0xb5, 0xcc, 0x72, 0xa6, 0xec, 0xd4, 0xdd,
		0x6f, 0x2f, 0xd0, 0xb6, 0xa6, 0x03, 0xaa, 0xd3, 0x99, 0x1c,
		0xb7, 0x5f, 0x7a, 0xda, 0x9f, 0xe7, 0xff, 0xa5, 0x8e, 0x77,
		0x4c, 0x14, 0x85, 0xd0, 0x78, 0x3c, 0x0f, 0xaf, 0x1c, 0x5c,
		0x09, 0x79, 0x6c, 0xb4, 0x64, 0xa1, 0x86, 0xd0, 0xb2, 0xfa,
		0xeb, 0xfc, 0xda, 0xf3, 0x42, 0x3a, 0x43, 0xb0, 0xd2, 0x24,
		0xfe, 0x04, 0xd1, 0x31, 0x47, 0x77, 0xde, 0x0f, 0xb4, 0x55,
		0x28, 0x83, 0x14, 0x5e, 0xd5, 0x4d, 0x99, 0xce, 0xf1, 0xb9,
		0x48, 0x6e, 0xf8, 0x17, 0xd5, 0x0a, 0xa0, 0xa3, 0x7a, 0xce,
		0x9a, 0x1a, 0x29, 0x83, 0xfc, 0x42, 0x58, 0x73, 0x48, 0x2c,
		0x18, 0xdd, 0x46, 0xb6, 0x0f, 0x5e, 0xbe, 0xa3, 0x3e, 0x2e,
		0xa0, 0xd6, 0xef, 0xe8, 0xf8, 0x88, 0xd8, 0x04, 0x44, 0x53,
		0x09, 0x33, 0x07, 0xfc, 0x70, 0x05, 0xc7, 0x4e, 0x56, 0xea,
		0x27, 0xbe, 0x15, 0x6c, 0x0a, 0xa7, 0x04, 0x5a, 0x35, 0x36,
		0x1c, 0x9e, 0xeb, 0x88, 0xee, 0xdd, 0x63, 0x4c, 0x34, 0xa9,
		0x70, 0x93, 0xc8, 0xea, 0x83, 0x79, 0x6f, 0x0e, 0x25, 0xc5,
		0x0b, 0x32, 0x55, 0xbc, 0x0a, 0x75, 0x90, 0x82, 0xa2, 0x18,
		0x2f, 0x12, 0x26, 0x31, 0x92, 0xeb, 0x47, 0x38, 0x60, 0x62,
		0xe9, 0xce, 0xb0, 0x42, 0x88, 0xe6, 0x84, 0x3c, 0x13, 0x37,
		0x4d, 0x2b, 0x74, 0x01, 0xef, 0x07, 0xb6, 0x9e, 0x99, 0x2a,
		0x84, 0x4b, 0x6f, 0x41, 0x9a, 0x8b, 0x0e, 0x01, 0xbb, 0x51,
		0xf9, 0x73, 0x7b, 0x7a, 0x51, 0x50, 0x68, 0x8a, 0x41, 0xbe,
		0xb6, 0x91, 0xf6, 0x5d, 0xe9, 0x66, 0xef, 0xfd, 0xf3, 0x8e,
		0xe2, 0x24, 0xcb, 0xca, 0x5b, 0xc5, 0x75, 0x87, 0x3f, 0x15,
		0x6e, 0xd3, 0xc0, 0x6f, 0xe2, 0x61, 0xab, 0x97, 0x7d, 0x22,
		0x67, 0xef, 0x3b, 0x84, 0xb3, 0xe7, 0x6c, 0x99, 0x77, 0x2f,
		0xed, 0xa8, 0xd1, 0xf6, 0x60, 0x38, 0x85, 0x1a, 0xef, 0x91,
		0x7d, 0x40, 0xa9, 0x3a, 0x6e, 0x45, 0xe0, 0x99, 0x83, 0x80,
		0x74, 0x02, 0xe3, 0x04, 0xe6, 0x7a, 0x64, 0xc9, 0x08, 0x8a,
		0xa4, 0x9e, 0x64, 0x01, 0xce, 0xa1, 0xb8, 0x3b, 0xf6, 0x22,
		0x0d, 0x8c, 0x7b, 0xf8, 0xfc, 0xd8, 0xf0, 0x92, 0xdc, 0x95,
		0xdd, 0xa8, 0x05, 0xde, 0x59, 0x2f, 0xc2, 0x17, 0x8d, 0xfa,
		0x6f, 0x7b, 0x2d, 0x07, 0x4d, 0x53, 0x3e, 0x80, 0x6c, 0x5d,
		0x62, 0xe4, 0xe1, 0x9f, 0x93, 0x8a, 0x2e, 0x59, 0xde, 0xad,
		0x44, 0xec, 0x47, 0x12, 0x68, 0xfd, 0x6f, 0x87, 0x14, 0xd5,
		0x0c, 0xe1, 0x93, 0xec, 0xc0, 0xa1, 0x93, 0x4d, 0x2d, 0xff,
		0x27, 0x00, 0x18, 0xe2, 0xb3, 0x40, 0xdf, 0x1d, 0x56, 0x05,
		0xb6, 0x42, 0x3c, 0x76, 0xde, 0xd2, 0x0d, 0xa6, 0xaf, 0x8b,
		0x71, 0x93, 0x7d, 0x92, 0x3f, 0x55, 0xf0, 0x99, 0x34, 0x26,
		0xa2, 0xa5, 0xc8, 0x15, 0xe9, 0x91, 0x6f, 0xd7, 0xc1, 0xc7,
		0xe6, 0xc5, 0xa1, 0xba, 0x08, 0x8e, 0x78, 0xac, 0x3b, 0xd2,
		0xd3, 0xc4, 0x46, 0x99, 0x97, 0x7d, 0x71, 0x27, 0xc9, 0x2f,
		0x15, 0x7c, 0x65, 0xc2, 0x2b, 0x40, 0xc8, 0x6c, 0x05, 0x8b,
		0x6f, 0x1e, 0xfc, 0x1f, 0x78, 0xd8, 0x48, 0x9a, 0x2f, 0xb4,
		0x38, 0xb7, 0xfc, 0x92, 0xdb, 0x97, 0x4d, 0x03, 0x4c, 0xf9,
		0xba, 0x69, 0x3f, 0x41, 0xbb, 0x39, 0xba, 0x3e, 0x74, 0xac,
		0x8e, 0x57, 0x8d, 0x45, 0x6b, 0x9d, 0x8a, 0xa1, 0x16, 0xb0,
		0xea, 0x7d, 0x12, 0xe5, 0x08, 0x0d, 0xc2, 0xba, 0x4f, 0x52,
		0xdb, 0xb8, 0xb6, 0x4c, 0x07, 0xb2, 0xdc, 0xbc, 0xc7, 0x77,
		0x01, 0xb0, 0xda, 0xfd, 0x15, 0x2b, 0x5e, 0xce, 0x5d, 0x2c,
		0x6b, 0x55, 0xbb, 0x5b, 0x44, 0xd1, 0xf8, 0x36, 0x9b, 0x81,
		0x19, 0x3a, 0x47, 0x18, 0x93, 0x23, 0x40, 0xaa, 0x14, 0x7f,
		0xe3, 0xb5, 0xd6, 0x1e, 0x70, 0x26, 0x85, 0xb3, 0x51, 0x27,
		0x1c, 0x45, 0xca, 0xb7, 0x47, 0xa9, 0x98, 0xf3, 0xb3, 0xa9,
		0x37, 0x33, 0xbf, 0x10, 0x82, 0x95, 0xce, 0xa9, 0xa4, 0x57,
		0x9c, 0xec, 0xde, 0xfb, 0x2b, 0x2b, 0xd8, 0xc7, 0x6d, 0xe4,
		0x4c, 0x29, 0x7e, 0x6e, 0xfb, 0x26, 0xe6, 0x42, 0x17, 0xdf,
		0xd0, 0x1f, 0xc2, 0xb7, 0x71, 0xd8, 0x44, 0x6b, 0xf6, 0x00,
		0x78, 0x51, 0xa0, 0x57, 0x28, 0x75, 0x00, 0x0e, 0x4f, 0xac,
		0x3d, 0xe6, 0x38, 0x75, 0xa0, 0x1a, 0xcf, 0x42, 0x20, 0xa0,
		0x22, 0x83, 0x80, 0xb3, 0xf6, 0x5f, 0x15, 0x85, 0xb5, 0x8d,
		0xd7, 0xae, 0xf2, 0xb8, 0xb1, 0xc1, 0xb7, 0xd6, 0xe8, 0x9f,
		0x6a, 0x68, 0x31, 0x7d, 0xbe, 0xf3, 0x75, 0xaa, 0x24, 0xc3,
		0x41, 0xa7, 0xfc, 0xaf, 0xd3, 0xd1, 0x71, 0xb0, 0x60, 0xcc,
		0x35, 0xbe, 0x09, 0x49, 0x7f, 0xf6, 0x97, 0x0b, 0xf1, 0x6b,
		0xe5, 0x2d, 0xec, 0x83, 0x53, 0x8a, 0x20, 0xb9, 0x96, 0xbb,
		0x61, 0x1e, 0x47, 0x32, 0x02, 0x2a, 0xe5, 0x1c, 0xa8, 0x9f,
		0xec, 0xbe, 0x2b, 0x46, 0xa6, 0x14, 0x9b, 0x84, 0xed, 0x92,
		0x90, 0xee, 0x08, 0x65, 0x4b, 0x06, 0x99, 0x23, 0x84, 0xba,
		0x97, 0xc1, 0x8c, 0x42, 0xde, 0x2e, 0x95, 0xe4, 0x9d, 0xe0,
		0x6f, 0x8d, 0x15, 0x89, 0xa9, 0x8b, 0x74, 0x2a, 0xf9, 0x4b,
		0xc7, 0x76, 0xd4, 0xf5, 0x53, 0xae, 0x95, 0x44, 0x4f, 0x61,
		0x86, 0x12, 0x9f, 0xf2, 0xe2, 0x37, 0x41, 0x09, 0x86, 0x52,
		0x3f, 0xac, 0xa4, 0x98, 0x4e, 0x42, 0x6c, 0x81, 0xc0, 0x82,
		0x82, 0xc2, 0x4c, 0xca, 0x80, 0x77, 0xd1, 0x08, 0x3b, 0x90,
		0x63, 0x97, 0x8c, 0x80, 0xab, 0x48, 0xba, 0x1b, 0x59, 0x77,
		0x37, 0x78, 0xb0, 0x95, 0xd2, 0x5b, 0xab, 0xd8, 0xf6, 0xf1,
		0xcf, 0x40, 0x1a, 0x6e, 0x40, 0x61, 0x95, 0x39, 0x2a, 0x17,
		0x58, 0x51, 0xda, 0xeb, 0x4c, 0x08, 0xf1, 0x94, 0xed, 0x44,
		0x48, 0xbb, 0xaa, 0xdd, 0x78, 0x76, 0x30, 0x31, 0x86, 0xfd,
		0x91, 0xd7, 0xf8, 0xd9, 0xd4, 0x1a, 0x5a, 0x92, 0x3f, 0x16,
		0x11, 0x48, 0xf8, 0x39, 0x79, 0x63, 0xa2, 0x00, 0x94, 0x13,
		0x70, 0xe8, 0xc1, 0x7e, 0x93, 0xf7, 0x39, 0x58, 0xee, 0x3b,
		0xfa, 0x81, 0xe6, 0xc3, 0x64, 0x31, 0x6b, 0xfd, 0xf8, 0x01,
		0x75, 0x14, 0xac, 0xb5, 0x18, 0x2f, 0xef, 0xd5, 0x42, 0x4d,
		0xa1, 0xa5, 0x46, 0x92, 0x5b, 0x37, 0x3d, 0x3f, 0x99, 0xd2,
		0x23, 0xd8, 0x75, 0x10, 0x81, 0x76, 0x91, 0xdb, 0x3d, 0xf3,
		0x19, 0xbf, 0xfc, 0xfc, 0xb1, 0x6c, 0x3b, 0x6a, 0x67, 0xf0,
		0xb9, 0x0a, 0x6e, 0xc2, 0xc6, 0x12, 0x88, 0x54, 0x78, 0x66,
		0x68, 0xb6, 0xe7, 0x81, 0x39, 0x6f, 0x18, 0x45, 0x34, 0x4c,
		0x99, 0xaa, 0x4d, 0x23, 0xf8, 0x5f, 0xcb, 0xe7, 0xec, 0xbe,
		0xbe, 0xf3, 0x37, 0x93, 0x96, 0x7d, 0x4f, 0x57, 0x95, 0xf5,
		0xea, 0x6a, 0xca, 0xd1, 0x4e, 0x8a, 0xf0, 0x46, 0x34, 0xc5,
		0x79, 0x78, 0xce, 0x79, 0x47, 0x9e, 0xb7, 0x92, 0x51, 0x2d,
		0x8f, 0x5d, 0xab, 0x9a, 0x5e, 0x4a, 0x2d, 0x39, 0x2c, 0x11,
		0x24, 0x22, 0x0f, 0xac, 0x65, 0xbf, 0xd6, 0xda, 0xef, 0x82,
		0x00, 0xfb, 0x40, 0x53, 0x55, 0xf7, 0xe5, 0xa5, 0x6a, 0x84,
		0xc6, 0x36, 0x1a, 0x72, 0xa6, 0x2a, 0xe0, 0xb2, 0xa3, 0xfe,
		0x1f, 0x32, 0x9c, 0xff, 0x8a, 0x3c, 0xcc, 0xd1, 0xf0, 0x8c,
		0x99, 0xbe, 0xd1, 0x82, 0xb6, 0x74, 0x88, 0x82, 0x99, 0x6e,
		0xdb, 0xd0, 0xe2, 0x02, 0x3b, 0xee, 0x44, 0x23, 0x70, 0x14,
		0x1a, 0x37, 0x6b, 0x09, 0xb2, 0x0d, 0x79, 0x4b, 0xf6, 0xa3,
		0x63, 0x9c, 0xb3, 0x29, 0x29, 0xc6, 0xc9, 0x28, 0x0a, 0xce,
		0xf0, 0x96, 0xf9, 0xbb, 0x3b, 0x7a, 0xad, 0x7e, 0x12, 0x17,
		0xb6, 0xc6, 0x5f, 0x16, 0x23, 0xa7, 0xbf, 0xaa, 0x48, 0x60,
		0x60, 0x83, 0xdd, 0xf3, 0x2a, 0xdc, 0x92, 0x11, 0x62, 0x2c,
		0x8d, 0xd8, 0xb6, 0x77, 0x6e, 0x1a, 0x2a, 0xa5, 0x34, 0xda,
		0xad, 0x75, 0x64, 0x88, 0x73, 0x3f, 0x51, 0xfe, 0xca, 0x9f,
		0xe9, 0x65, 0x4a, 0x1e, 0xb6, 0x81, 0x05, 0xb8, 0x61, 0x01,
		0x1e, 0x36, 0x41, 0x6e, 0xd6, 0x70, 0xe3, 0xc9, 0xa4, 0xe7,
		0x1e, 0x6c, 0x55, 0xc6, 0xe8, 0xbf, 0xfe, 0xcf, 0x1e, 0x23,
		0x99, 0xca, 0xf0, 0xa7, 0x57, 0x3e, 0x8f, 0xf5, 0x6f, 0x0e,
		0xe9, 0xbc, 0xef, 0x28, 0x13, 0xe8, 0xf7, 0x2f, 0x5e, 0xc2,
		0xbe, 0xac, 0x79, 0x37, 0xad, 0xbb, 0xe2, 0x01, 0xf4, 0x63,
		0x7f, 0xef, 0x9a, 0xbc, 0x9b, 0xa7, 0x5a, 0x70, 0xfa, 0x02,
		0xc6, 0xa4, 0x4a, 0xbb, 0xab, 0xed, 0x41, 0x6c, 0x5a, 0x38,
		0xbf, 0xdb, 0xb1, 0x6d, 0xd7, 0xe2, 0x3a, 0xfe, 0x23, 0x3f,
		0x6c, 0xb0, 0x18, 0xce, 0xc8, 0xed, 0x82, 0x5d, 0x4d, 0x18,
		0x43, 0x89, 0x57, 0x06, 0xfd, 0xc2, 0x94, 0x0f, 0x95, 0xda,
		0x43, 0xa7, 0xbc, 0xc7, 0x61, 0x39, 0x71, 0x8f, 0x9d, 0x4c,
		0x1c, 0xe5, 0x80, 0x2d, 0x37, 0xd4, 0xd1, 0x4e, 0x9a, 0xa6,
		0x69, 0x44, 0xdf, 0xa3, 0x3d, 0x7b, 0x8c, 0x56, 0x71, 0x0d,
		0xf0, 0x7c, 0xbc, 0xcc, 0xde, 0x2c, 0x00, 0x3c, 0xe6, 0x44,
		0x75, 0xbe, 0x74, 0x23, 0x97, 0xdb, 0xde, 0x19, 0xb8, 0x43,
		0x3b, 0x20, 0x40, 0x3a, 0xa0, 0xde, 0x9c, 0x5e, 0x0d, 0x74,
		0x88, 0x9f, 0xd6, 0x37, 0x3d, 0x0f, 0xce, 0xac, 0xed, 0xb0,
		0x84, 0xf6, 0xe5, 0x23, 0x90, 0x04, 0x79, 0x27, 0x42, 0x01,
		0x9e, 0x40, 0x02, 0x72, 0xe1, 0xee, 0x80, 0x95, 0x69, 0x6f,
		0x44, 0x0c, 0x5c, 0x82, 0x23, 0x2b, 0x43, 0x7f, 0x55, 0xec,
		0xf8, 0xfb, 0x23, 0xed, 0x49, 0x92, 0x1d, 0xcf, 0xef, 0xcf,
		0x02, 0x04, 0xd8, 0x69, 0x7f, 0x3f, 0xf8, 0x8b, 0xee, 0x6f,
		0x30, 0x4d, 0x4b, 0x4e, 0xdb, 0x8a, 0xfc, 0xd3, 0xb5, 0x00,
		0x80, 0x52, 0x99, 0x9c, 0xab, 0x46, 0x94, 0x34, 0xca, 0x0e,
		0x30, 0xae, 0x0e, 0xfb, 0xfa, 0x18, 0x63, 0x8a, 0x26, 0x86,
		0xb9, 0x87, 0x05, 0xdc, 0x62, 0x64, 0x62, 0x3a, 0x53, 0x96,
		0x23, 0xc6, 0x6b, 0xf6, 0xb0, 0x7c, 0xca, 0x0c, 0x30, 0x90,
		0x85, 0x95, 0xb8, 0x7a, 0x1b, 0xa5, 0x3e, 0xb6, 0x08, 0x80,
		0xd5, 0x64, 0xb6, 0x65, 0x8d, 0x6a, 0xf3, 0x5a, 0x70, 0x4a,
		0x11, 0x79, 0x8a, 0xe0, 0xac, 0x29, 0x4e, 0x84, 0x1b, 0xdf,
		0xcb, 0xf8, 0xc4, 0x12, 0x3b, 0x6a, 0xad, 0x42, 0x07, 0x1a,
		0x76, 0x5e, 0xda, 0x5a, 0x23, 0x44, 0x1d, 0x2d, 0xc9, 0xec,
		0x07, 0x80, 0xdb, 0x6b, 0xf9, 0xb3, 0x7d, 0x60, 0xb5, 0xa9,
		0xe4, 0xac, 0x57, 0x1a, 0x62, 0x8b, 0x2e, 0x07, 0x40, 0x5d,
		0x72, 0xa6, 0x55, 0x43, 0xf9, 0xf4, 0xa5, 0xf7, 0xeb, 0x2d,
		0xf8, 0xf6, 0xe1, 0x97, 0x6d, 0x8c, 0x3c, 0xef, 0x9d, 0xf8,
		0xfc, 0xdf, 0xe2, 0xbd, 0xf4, 0x96, 0x69, 0x31, 0xbe, 0x69,
		0xe3, 0x73, 0x12, 0xab, 0x62, 0x8d, 0x8c, 0xbd, 0xf1, 0x25,
		0xd8, 0x9c, 0x13, 0x68, 0x9f, 0x31, 0xb5, 0x23, 0xb6, 0x16,
		0x1f, 0xea, 0x5d, 0x6e, 0x64, 0xdb, 0x02, 0xd4, 0x3f, 0x26,
		0x9a, 0x08, 0x44, 0xa8, 0x05, 0x39, 0x0c, 0xc2, 0x16, 0xe0,
		0x15, 0x71, 0xbf, 0xb4, 0x6e, 0x80, 0x61, 0xf2, 0x9b, 0x4f,
		0xb4, 0x12, 0x33, 0x4d, 0x6a, 0x86, 0xe2, 0xf0, 0xd9, 0x4d,
		0x45, 0x7a, 0x44, 0x4c, 0x3c, 0x6f, 0xa5, 0x31, 0x47, 0xcd,
		0x67, 0x0f, 0x43, 0xa4, 0x2e, 0xd6, 0xe6, 0xdd, 0xc2, 0xf0,
		0xa8, 0x7e, 0x0f, 0x1f, 0x59, 0x81, 0x3a, 0xde, 0xba, 0xa1,
		0x32, 0xe6, 0x98, 0x09, 0x9c, 0xc4, 0x2a, 0xc4, 0x1c, 0xf5,
		0x10, 0x31, 0xfb, 0xcf, 0x96, 0x55, 0x0f, 0xec, 0x5d, 0x63,
		0x4f, 0xcb, 0x74, 0x27, 0x35, 0xf3, 0x3a, 0x6b, 0x99, 0x30,
		0x47, 0xbb, 0xf9, 0x60, 0x08, 0x77, 0x7d, 0xef, 0xd2, 0x8c,
		0xbd, 0xa9, 0x39, 0xda, 0xc1, 0x6a, 0xdb, 0x0b, 0x71, 0x94,
		0xba, 0xe6, 0xc2, 0x09, 0xd6, 0x67, 0x00, 0xe2, 0x7c, 0x1b,
		0x8f, 0x89, 0x09, 0x91, 0xc1, 0xee, 0x51, 0xaf, 0x09, 0x8a,
		0x95, 0xed, 0xd9, 0x86, 0x2a, 0xf2, 0xc6, 0xfd, 0x72, 0x2f,
		0x3d, 0x18, 0x9f, 0x7f, 0x4a, 0x6a, 0x98, 0x9e, 0x6e, 0x48,
		0xe0, 0x69, 0x32, 0x97, 0x80, 0xe9, 0x7a, 0x2b, 0x5a, 0x5c,
		0xdb, 0xcb, 0x58, 0xf3, 0x8b, 0x7f, 0x2c, 0x05, 0x9f, 0xa6,
		0xbf, 0x98, 0xc8, 0x59, 0x20, 0x0e, 0x6c, 0xab, 0x99, 0x76,
		0xc8, 0xa2, 0xe9, 0x78, 0x31, 0xe8, 0xc1, 0x1f, 0xda, 0x9d,
		0x6a, 0xd3, 0x96, 0xbd, 0xbf, 0xda, 0x52, 0x2a, 0x3e, 0x48,
		0x2f, 0xd0, 0x1f, 0xac, 0x85, 0xf9, 0x9c, 0xa7, 0x17, 0xa9,
		0x25, 0x75, 0xb0, 0x54, 0x1c, 0x15, 0xc7, 0x17, 0x29, 0x79,
		0xb9, 0x90, 0x6a, 0x4d, 0x3a, 0x08, 0x26, 0x50, 0x30, 0xa7,
		0x41, 0x56, 0x9f, 0x0b, 0xb2, 0x6a, 0xfa, 0x1b, 0xa9, 0xfc,
		0xab, 0x50, 0x48, 0x85, 0x71, 0x0d, 0x68, 0x55, 0x91, 0x96,
		0xa2, 0x87, 0x51, 0xda, 0xf8, 0x4c, 0xfb, 0x63, 0x6a, 0x5f,
		0xa9, 0xcb, 0xf0, 0x12, 0xa5, 0x07, 0xb8, 0xe0, 0x95, 0xb7,
		0x81, 0x2a, 0x47, 0x1e, 0x34, 0x2c, 0xb5, 0xd0, 0x27, 0x84,
		0xee, 0x33, 0xd1, 0xc4, 0x73, 0xad, 0xc3, 0x05, 0xaa, 0x18,
		0x87, 0xb4, 0x96, 0x0c, 0x5f, 0xe2, 0xd0, 0x30, 0x00, 0xc7,
		0x22, 0x80, 0xe3, 0xd4, 0xf2, 0xd5, 0x55, 0x72, 0x59, 0x61,
		0x3f, 0xea, 0x4a, 0x5d, 0xf7, 0xb2, 0x44, 0x16, 0x96, 0x35,
		0x73, 0x33, 0x77, 0xee, 0xeb, 0x65, 0xef, 0x41, 0xc2, 0xe0,
		0xcc, 0x04, 0x09, 0x90, 0xa0, 0xe3, 0xec, 0x65, 0x8d, 0xe1,
		0xca, 0xa7, 0x75, 0xb4, 0x92, 0x98, 0xdd, 0x88, 0xcd, 0x48,
		0x0c, 0xf9, 0xb0, 0x45, 0xbd, 0x9f, 0x98, 0x8d, 0x5a, 0x50,
		0x47, 0x65, 0x68, 0x6f, 0xc3, 0x08, 0x78, 0x4c, 0xbc, 0xbc,
		0x8f, 0x16, 0x48, 0x69, 0x08, 0xf5, 0x03, 0xc5, 0x25, 0x9c,
		0x90, 0x3f, 0x54, 0x80, 0x7c, 0x9d, 0x90, 0x26, 0x77, 0x56,
		0xca, 0x1c, 0x89, 0xfe, 0xc8, 0xd1, 0x3c, 0x1b, 0xc4, 0x20,
		0x64, 0xc8, 0x61, 0x09, 0x1e, 0x36, 0x21, 0xcd, 0x94, 0xb3,
		0x03, 0xd9, 0xbc, 0x30, 0x37, 0xc6, 0xff, 0x98, 0xe2, 0x7b,
		0xc4, 0x02, 0xcf, 0x48, 0x78, 0x5c, 0xdd, 0xf8, 0x1b, 0x7b,
		0x1e, 0x21, 0xaf, 0x2a, 0x1b, 0xe3, 0x2c, 0x1a, 0x35, 0x33,
		0x8f, 0xff, 0xac, 0x41, 0x3d, 0x41, 0x81, 0x40, 0x13, 0x75,
		0xfc, 0x44, 0x9e, 0x57, 0xe5, 0x7d, 0x4c, 0x3a, 0x61, 0xa4,
		0x17, 0xcf, 0x28, 0x1a, 0x90, 0xc2, 0x56, 0xa9, 0x62, 0x03,
		0xd1, 0xbd, 0x77, 0x54, 0x70, 0x6a, 0x96, 0xce, 0xbd, 0xe0,
		0x86, 0x2f, 0x2c, 0x1b, 0x2c, 0x32, 0x94, 0xd3, 0x81, 0x9b,
		0x45, 0xb9, 0x90, 0x1f, 0xf7, 0xca, 0xb4, 0xe6, 0xcc, 0x29,
		0x12, 0x56, 0x8c, 0x2d, 0xd1, 0x68, 0x63, 0xf1, 0x6f, 0x39,
		0x16, 0xe9, 0x24, 0x55, 0xcd, 0x3d, 0xd7, 0x31, 0xa2, 0x46,
		0xee, 0x9c, 0xa2, 0x73, 0x8c, 0x75, 0x5c, 0xc5, 0x48, 0x12,
		0xd7, 0x18, 0x7e, 0xea, 0x05, 0x23, 0x11, 0x55, 0x1c, 0x07,
		0xdd, 0xa3, 0xca, 0x4b, 0x84, 0x4c, 0xf4, 0xf4, 0xc4, 0xd3,
		0xb1, 0xc2, 0x33, 0xb0, 0x9c, 0x85, 0x86, 0xb8, 0x80, 0xd6,
		0xe3, 0x76, 0xba, 0x38, 0x84, 0xca, 0xa8, 0x1e, 0x30, 0x48,
		0xf3, 0x8b, 0x37, 0xf1, 0x16, 0x78, 0x57, 0xcb, 0xf4, 0x4d,
		0x2c, 0x8f, 0xb8, 0x2c, 0xae, 0x4e, 0x31, 0x4d, 0xee, 0xb9,
		0x50, 0x31, 0xe4, 0x57, 0xd4, 0xaf, 0x3f, 0xf0, 0x8d, 0xdb,
		0xe6, 0x4b, 0x11, 0x68, 0x72, 0x81, 0xb8, 0xe3, 0x36, 0x19,
		0x96, 0xaf, 0x4f, 0x3d, 0x8a, 0x3d, 0x80, 0x5b, 0x68, 0xdb,
		0xb7, 0x3f, 0x97, 0x43, 0xca, 0x7d, 0x3c, 0x26, 0x81, 0x03,
		0xe1, 0xa9, 0x18, 0x79, 0x91, 0xbd, 0x0f, 0x01, 0x2a, 0x0e,
		0xbf, 0x92, 0x83, 0x51, 0x69, 0x08, 0x07, 0x82, 0x3b, 0x02,
		0x8c, 0x8c, 0x37, 0xd9, 0xfb, 0x43, 0xfe, 0x5c, 0x64, 0xe5,
		0x5a, 0x23, 0xe3, 0xec, 0x08, 0xa1, 0x82, 0x6a, 0x0b, 0x9f,
		0x7b, 0x78, 0x06, 0xad, 0xa0, 0xa7, 0x49, 0x6b, 0x67, 0x62,
		0x93, 0x0e, 0x17, 0xa3, 0x35, 0x22, 0x0f, 0x1b, 0x75, 0x30,
		0xff, 0xb1, 0xfc, 0xef, 0x2e, 0x05, 0x08, 0x28, 0x3d, 0x44,
		0x8a, 0x4a, 0x03, 0xe3, 0xde, 0xde, 0x87, 0x1a, 0xf9, 0xd9,
		0x14, 0x82, 0x4d, 0xe4, 0xe1, 0x96, 0xd2, 0x24, 0x22, 0x71,
		0xb3, 0x8d, 0x61, 0xae, 0xbc, 0x25, 0xc9, 0x56, 0xa7, 0xc7,
		0xaa, 0xa6, 0xdb, 0xfb, 0x53, 0x74, 0xa5, 0xe8, 0x24, 0x28,
		0xac, 0x5d, 0x50, 0x1d, 0x1d, 0x52, 0xa5, 0x25, 0xda, 0x6b,
		0xee, 0x4b, 0xba, 0x7d, 0x57, 0x62, 0xe0, 0xaa, 0xd1, 0xa6,
		0xe8, 0x93, 0x69, 0x9e, 0x69, 0x48, 0xa9, 0xf2, 0xc1, 0x43,
		0xfd, 0x80, 0x1a, 0x77, 0x5a, 0xca, 0x31, 0xb1, 0xc6, 0x8c,
		0x2f, 0xb0, 0xb7, 0x04, 0x38, 0x83, 0x76, 0xfc, 0x5f, 0x6b,
		0x62, 0xb2, 0xa4, 0x79, 0xc6, 0x61, 0x82, 0x48, 0xfc, 0x1b,
		0x54, 0x69, 0x13, 0xd5, 0x9b, 0x68, 0x12, 0x5b, 0x4a, 0xcf,
		0xa2, 0xb9, 0x50, 0x9f, 0xc0, 0xcc, 0xe6, 0x18, 0xc8, 0x04,
		0xaf, 0xc0, 0x5b, 0x1d, 0xa5, 0xa7, 0x4c, 0xbe, 0x79, 0xc7,
		0x51, 0xce, 0xbb, 0xfd, 0x2f, 0x65, 0x3e, 0x34, 0x69, 0xed,
		0x8d, 0xc6, 0x09, 0x62, 0x33, 0x12, 0x58, 0xf2, 0x4f, 0x96,
		0x8e, 0xb2, 0x93, 0xf8, 0xbf, 0xf8, 0xeb, 0x83, 0x4c, 0x3b,
		0xc1, 0x26, 0x4c, 0x45, 0x1f, 0x6e, 0x04, 0xe3, 0xcf, 0x13,
		0x3b, 0x34, 0xb4, 0x36, 0x48, 0x4a, 0xd2, 0x3d, 0x8a, 0xd6,
		0x57, 0x98, 0x74, 0x17, 0xba, 0x14, 0x3a, 0x58, 0x99, 0x68,
		0xad, 0x2b, 0x81, 0x6f, 0x57, 0x63, 0x52, 0x19, 0x36, 0xfe,
		0x1d, 0xcb, 0x73, 0xa1, 0x35, 0xb2, 0xf1, 0xf1, 0x26, 0x9e,
		0x2b, 0xf3, 0x1b, 0xcd, 0x10, 0xd7, 0x92, 0x29, 0x65, 0x1e,
		0x02, 0x8d, 0x08, 0x4e, 0x68, 0xd3, 0x70, 0xf7, 0x87, 0xe5,
		0x53, 0x04, 0xa4, 0x43, 0xc5, 0x30, 0x74, 0x4c, 0x5e, 0xe1,
		0xf6, 0xa8, 0xd8, 0x55, 0x95, 0xc8, 0x42, 0xd8, 0xfc, 0x3d,
		0x2e, 0x8d, 0x8f, 0x0a, 0x20, 0x76, 0xd3, 0x7a, 0xfb, 0x07,
		0xba, 0xd8, 0x07, 0xef, 0x29, 0x3a, 0x3d, 0x81, 0x90, 0xaf,
		0x0c, 0x13, 0x96, 0x72, 0xb6, 0x3c, 0x0e, 0x57, 0x96, 0x3e,
		0x51, 0x91, 0xdb, 0xb0, 0x2d, 0xdf, 0x31, 0x39, 0x21, 0x29,
		0xbb, 0x17, 0x0a, 0x23, 0x6b, 0xe4, 0xdc, 0x69, 0x27, 0xc4,
		0x20, 0x98, 0xfd, 0x34, 0xca, 0x7a, 0x66, 0x20, 0x58, 0xd2,
		0x36, 0x7f, 0x2b, 0xa7, 0xd1, 0xde, 0x6f, 0x36, 0xb4, 0xf2,
		0x3b, 0x20, 0x5d, 0x02
	};

	if(target < CHIAKI_TARGET_PS4_10)
		return CHIAKI_ERR_INVALID_DATA;

	const uint8_t *keys_a = chiaki_target_is_ps5(target) ? keys_a_ps5 : keys_a_ps4;
	const uint8_t *keys_b = chiaki_target_is_ps5(target) ? keys_b_ps5 : keys_b_ps4;

	const uint8_t *key = &keys_a[(nonce[0] >> 3) * 0x70];
	for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
	{
		uint8_t v = nonce[i];
		if(chiaki_target_is_ps5(target))
		{
			v -= 0x2d;
			v -= i;
		}
		else
		{
			v += 0x36;
			v += i;
		}
		v ^= key[i];
		ambassador[i] = v;
	}

	key = &keys_b[(nonce[7] >> 3) * 0x70];
	if(chiaki_target_is_ps5(target))
	{
		for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
		{
			uint8_t v = morning[i];
			v += 0x18;
			v += i;
			v ^= nonce[i];
			v ^= key[i];
			bright[i] = v;
		}
	}
	else
	{
		for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
		{
			uint8_t v = (key[i] ^ morning[i]);
			v += 0x21;
			v += i;
			v ^= nonce[i];
			bright[i] = v;
		}
	}
	return CHIAKI_ERR_SUCCESS;
}

CHIAKI_EXPORT void chiaki_rpcrypt_bright_ambassador(ChiakiTarget target, uint8_t *bright, uint8_t *ambassador, const uint8_t *nonce, const uint8_t *morning)
{
	switch(target)
	{
		case CHIAKI_TARGET_PS4_8:
		case CHIAKI_TARGET_PS4_9:
			bright_ambassador_ps4_pre10(bright, ambassador, nonce, morning);
			break;
		default: {
			ChiakiErrorCode err = bright_ambassador(target, bright, ambassador, nonce, morning);
			assert(err == CHIAKI_ERR_SUCCESS);
			break;
		}
	}
}

CHIAKI_EXPORT void chiaki_rpcrypt_aeropause_ps4_pre10(uint8_t *aeropause, const uint8_t *ambassador)
{
	for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
	{
		uint8_t v = ambassador[i];
		v -= i;
		v -= 0x29;
		v ^= echo_b[i];
		aeropause[i] = v;
	}
}

CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_aeropause(ChiakiTarget target, size_t key_1_off, uint8_t *aeropause, const uint8_t *ambassador)
{
	static const uint8_t ps4_keys_1[] = {
		0xc8, 0x48, 0xc2, 0xb4, 0x08, 0xeb, 0x88, 0xf7, 0x5f, 0x4a,
		0x09, 0x2d, 0x59, 0x1f, 0x09, 0xcd, 0x1c, 0x18, 0xf4, 0x7a,
		0x28, 0x4a, 0x96, 0x6d, 0xb3, 0x59, 0x71, 0x53, 0x75, 0x7e,
		0x82, 0x50, 0x57, 0xe4, 0x59, 0xb3, 0xf4, 0x49, 0x69, 0x40,
		0xeb, 0x17, 0xc9, 0x9f, 0x17, 0x97, 0x71, 0xae, 0xc9, 0x60,
		0x7f, 0xf8, 0x2e, 0x08, 0x94, 0xe8, 0x43, 0xea, 0xda, 0x6b,
		0xe5, 0x19, 0x59, 0x33, 0x1f, 0x89, 0xae, 0x47, 0x57, 0x7b,
		0x1c, 0x66, 0xfe, 0xff, 0x95, 0xbf, 0x55, 0x6b, 0xd5, 0x93,
		0x27, 0xea, 0xa6, 0x24, 0x67, 0x39, 0x9f, 0xd3, 0x0c, 0xaa,
		0x26, 0x42, 0xe7, 0x66, 0x4d, 0xd8, 0x18, 0x75, 0xfe, 0x44,
		0x03, 0x46, 0xee, 0x3e, 0xf8, 0x3c, 0xb7, 0x85, 0x97, 0x03,
		0x07, 0x06, 0x92, 0xff, 0x59, 0x17, 0x27, 0x0b, 0x21, 0xf7,
		0x05, 0x7f, 0x69, 0x90, 0x0e, 0x38, 0x91, 0xc6, 0x67, 0x23,
		0x48, 0xba, 0x08, 0x8e, 0x57, 0xdd, 0x91, 0xd0, 0x40, 0x47,
		0x1c, 0x5b, 0xbf, 0xc8, 0x06, 0x3f, 0x96, 0xa0, 0xdc, 0x00,
		0xe5, 0x9a, 0xf5, 0x3b, 0x90, 0x80, 0x66, 0xbb, 0x0f, 0x93,
		0x30, 0x07, 0x2b, 0x56, 0x45, 0xa0, 0x9a, 0xb1, 0xb0, 0x72,
		0x0c, 0xe4, 0xdd, 0x70, 0xdd, 0x7c, 0x5a, 0xbf, 0xd4, 0xe8,
		0x0d, 0xca, 0x37, 0xe5, 0x0e, 0xfd, 0x12, 0xee, 0x79, 0x9a,
		0x5e, 0xa7, 0x1e, 0x31, 0xaf, 0x1f, 0x46, 0x52, 0xca, 0xf3,
		0x42, 0x00, 0x3d, 0xf2, 0x89, 0x7c, 0x1c, 0x77, 0x60, 0xb7,
		0x4a, 0x80, 0x76, 0x47, 0x4b, 0x3f, 0xb6, 0x91, 0x2f, 0x9c,
		0xc2, 0xf1, 0xad, 0x44, 0x29, 0xcb, 0x32, 0x8c, 0x0a, 0x8d,
		0x05, 0x75, 0x46, 0xa1, 0xf8, 0xda, 0x1a, 0xa7, 0x20, 0xde,
		0x32, 0x59, 0xfe, 0x70, 0xb5, 0x87, 0xf3, 0x92, 0xfd, 0xb4,
		0xdf, 0xf4, 0xa6, 0xe3, 0x7d, 0x98, 0x3b, 0xe1, 0xba, 0x18,
		0xdb, 0x61, 0xd1, 0xc2, 0xa6, 0xee, 0x08, 0x25, 0xfa, 0x86,
		0x8a, 0x7b, 0xfe, 0xbc, 0x02, 0xbd, 0x22, 0x5f, 0x25, 0x30,
		0x51, 0x5d, 0x28, 0x36, 0x5a, 0x29, 0x8e, 0x52, 0xeb, 0x49,
		0x14, 0x28, 0x7f, 0x0a, 0xc4, 0x69, 0x25, 0x85, 0x6b, 0xec,
		0x33, 0x33, 0x57, 0xab, 0x50, 0x13, 0xcf, 0xe7, 0x73, 0x78,
		0x23, 0x06, 0x4d, 0x1f, 0xbb, 0x38, 0x11, 0xb1, 0x6e, 0x6c,
		0x6f, 0xcd, 0x4b, 0x0e, 0x42, 0x85, 0x80, 0xbb, 0xa8, 0x39,
		0x68, 0xc9, 0x5b, 0xbb, 0x46, 0x58, 0x86, 0x88, 0x57, 0x88,
		0x5e, 0xea, 0x7c, 0x37, 0xdf, 0xfd, 0x02, 0x39, 0x45, 0x89,
		0x2e, 0xf2, 0xe4, 0xf0, 0x08, 0xc0, 0xb6, 0xeb, 0x9c, 0x6e,
		0x7a, 0x81, 0xa3, 0x26, 0x46, 0xfe, 0xe1, 0x70, 0x3c, 0x3e,
		0x11, 0x7a, 0x32, 0xce, 0x45, 0x02, 0x7d, 0x32, 0xcd, 0x08,
		0x07, 0x06, 0xa3, 0xa8, 0xf7, 0x34, 0xfe, 0xee, 0x06, 0xb0,
		0x14, 0xd6, 0x6b, 0x2d, 0x2e, 0x01, 0xaf, 0x77, 0x11, 0xec,
		0x1f, 0x31, 0x38, 0x17, 0x9c, 0xd0, 0xe0, 0xc5, 0x4d, 0xa4,
		0xd6, 0xad, 0xb9, 0xe6, 0xe1, 0xe3, 0xe2, 0x9e, 0x44, 0x91,
		0x9a, 0x5e, 0x26, 0xca, 0xcc, 0xda, 0x4d, 0xd7, 0x78, 0x6a,
		0x75, 0xa6, 0x19, 0xad, 0xcc, 0x62, 0xc7, 0xb6, 0x0d, 0x14,
		0xb1, 0xbe, 0xeb, 0xcb, 0x10, 0xcf, 0xa9, 0xee, 0xe2, 0x42,
		0x08, 0x35, 0x8a, 0x5c, 0xbc, 0xf1, 0x49, 0xfe, 0x64, 0x78,
		0x03, 0x49, 0x0c, 0x85, 0xf0, 0xe4, 0x77, 0x26, 0xd2, 0x5e,
		0xf5, 0xc1, 0x3b, 0x3d, 0x2d, 0xcc, 0xcf, 0x2a, 0xac, 0xed,
		0x88, 0x52, 0x74, 0x6d, 0xbf, 0xb2, 0xb9, 0xf7, 0x58, 0x51,
		0x1c, 0x50, 0xf6, 0x3d, 0xf2, 0xc4, 0x47, 0x0a, 0x21, 0x30,
		0x47, 0x81, 0x2d, 0xe4, 0x75, 0x0d, 0x8f, 0x2d, 0x22, 0xb0,
		0x63, 0x27
	};

	static const uint8_t ps5_keys_1[512] = {
		0x79, 0x4d, 0x78, 0x30, 0xfe, 0x10, 0x52, 0x4c, 0xa8, 0x90,
		0x5b, 0x9a, 0x7e, 0x5f, 0xd3, 0xe1, 0x13, 0xe0, 0xf1, 0x0f,
		0xa3, 0xe7, 0xbb, 0x45, 0x7f, 0xdc, 0x8e, 0xd5, 0xf1, 0x04,
		0x5c, 0x78, 0x51, 0xef, 0xf8, 0x65, 0x59, 0x03, 0x39, 0x84,
		0x37, 0xae, 0x59, 0xdf, 0x23, 0xb6, 0x60, 0x34, 0xe6, 0x4b,
		0xe2, 0xf5, 0x4c, 0x13, 0xc6, 0xda, 0xf9, 0xfd, 0xb3, 0x65,
		0x84, 0xd6, 0x45, 0xec, 0x2c, 0x00, 0xf2, 0xed, 0xdc, 0xcb,
		0x93, 0x6e, 0x61, 0x46, 0xe5, 0xd6, 0x01, 0x94, 0xee, 0x78,
		0x85, 0x0e, 0x68, 0x5e, 0xb5, 0x5b, 0xcd, 0xd3, 0x63, 0x41,
		0xfc, 0x81, 0x43, 0x1c, 0x6f, 0x7c, 0xba, 0xe8, 0xbd, 0x86,
		0x31, 0xd5, 0x70, 0x7f, 0xb5, 0x4a, 0x90, 0x3e, 0x84, 0xe1,
		0x71, 0xe0, 0x02, 0x99, 0xf4, 0x71, 0xe7, 0x02, 0xed, 0x36,
		0xaf, 0xde, 0x56, 0xc2, 0x90, 0xe0, 0xae, 0xc2, 0xf9, 0xaf,
		0x53, 0xc6, 0xd8, 0x62, 0x16, 0x32, 0x27, 0xfb, 0x6e, 0x9b,
		0x48, 0xc6, 0xea, 0xff, 0x6f, 0x78, 0x02, 0x22, 0x98, 0x2c,
		0x1f, 0xbf, 0xb0, 0x8e, 0xa9, 0x39, 0xbc, 0xdf, 0x17, 0xde,
		0xd7, 0x0e, 0xe1, 0x7a, 0x01, 0x0e, 0xc3, 0x87, 0xfc, 0xaa,
		0xe4, 0x6b, 0x0f, 0x5b, 0x0a, 0xf1, 0x18, 0x19, 0x8a, 0xe5,
		0x2c, 0x36, 0x9b, 0x40, 0x30, 0x99, 0x24, 0x94, 0x48, 0xd7,
		0x47, 0xb2, 0xaf, 0x6b, 0x8c, 0x40, 0x9e, 0x4d, 0x6d, 0x34,
		0x07, 0xc1, 0x26, 0x2f, 0xbb, 0x14, 0xf7, 0xbc, 0x36, 0x52,
		0xbd, 0x84, 0xfe, 0x4a, 0x9a, 0xf4, 0x8a, 0xdb, 0x34, 0x89,
		0xaa, 0xf1, 0x0d, 0x94, 0x0b, 0x92, 0xf4, 0x1c, 0xe4, 0x6c,
		0x79, 0x2d, 0x6e, 0xc0, 0x19, 0x0a, 0xd5, 0x55, 0x94, 0x14,
		0x05, 0x13, 0xc2, 0x62, 0x23, 0xb3, 0xd4, 0x26, 0xc4, 0x44,
		0x56, 0x7a, 0xcd, 0x1c, 0xea, 0xd4, 0x74, 0xb9, 0x36, 0x40,
		0x9f, 0x08, 0xfb, 0x49, 0x62, 0x05, 0x92, 0x98, 0xad, 0x1d,
		0x9f, 0x8a, 0x76, 0x8b, 0xd4, 0x0f, 0x21, 0x40, 0x76, 0xb6,
		0x16, 0x91, 0x45, 0x93, 0x66, 0xcc, 0x12, 0xea, 0x4d, 0xf4,
		0x09, 0xe2, 0xac, 0x33, 0xd0, 0x6f, 0x43, 0x51, 0x07, 0x3e,
		0xd7, 0x95, 0x2c, 0x1e, 0x1f, 0x0c, 0x24, 0xb3, 0x0e, 0x3a,
		0xef, 0x95, 0xf5, 0xeb, 0x77, 0xdd, 0x20, 0xf2, 0x35, 0x98,
		0xf2, 0xae, 0xa9, 0x66, 0xe6, 0x13, 0xef, 0x5d, 0x3a, 0x2d,
		0x66, 0xed, 0xe2, 0x1e, 0xe9, 0x32, 0x4a, 0x40, 0xbf, 0x37,
		0xc6, 0x70, 0x29, 0xd9, 0x8c, 0xa1, 0x61, 0x4a, 0x29, 0x3d,
		0xc7, 0x55, 0x9c, 0x94, 0x9e, 0xc9, 0x11, 0x45, 0x10, 0x28,
		0xa7, 0x27, 0xd1, 0xd3, 0xd0, 0x84, 0x79, 0xc7, 0xa9, 0xb0,
		0xf6, 0xaf, 0x45, 0x8c, 0x3c, 0xd4, 0xdf, 0x3b, 0xf7, 0x0d,
		0xa2, 0x4f, 0x13, 0x97, 0x78, 0x27, 0xf0, 0x48, 0xc0, 0xa5,
		0xab, 0x83, 0x01, 0x05, 0xd0, 0x12, 0xd7, 0x1e, 0x12, 0x3a,
		0x4e, 0x98, 0x77, 0xae, 0xba, 0xb1, 0x4e, 0xb5, 0x3b, 0x59,
		0xca, 0x6d, 0xa5, 0x11, 0x80, 0x91, 0x9c, 0x07, 0x69, 0x59,
		0x5a, 0x53, 0x70, 0x7c, 0x95, 0x97, 0x11, 0x6d, 0x66, 0x8d,
		0xa3, 0xbd, 0xbb, 0x2d, 0xb0, 0xbf, 0x9b, 0x10, 0xcb, 0xc7,
		0x0f, 0x5b, 0x7e, 0x67, 0xe2, 0xb0, 0x4b, 0xba, 0x10, 0x12,
		0xb9, 0xbc, 0x97, 0xfd, 0x48, 0xe4, 0x8a, 0xc1, 0x0f, 0xa1,
		0x30, 0x9d, 0x56, 0x20, 0x24, 0x1a, 0x7d, 0x5b, 0xa0, 0xb4,
		0xbe, 0x9d, 0x38, 0x4f, 0xb4, 0x56, 0xa8, 0x4d, 0x13, 0x7c,
		0x44, 0xe8, 0x84, 0x97, 0xeb, 0x78, 0x2c, 0x52, 0x85, 0xe4,
		0xa2, 0xf6, 0xf3, 0xd9, 0x71, 0x9e, 0xee, 0xb8, 0x11, 0x47,
		0xfb, 0xa9, 0x1b, 0xc7, 0x40, 0xc6, 0xe1, 0x19, 0x6d, 0x50,
		0xa1, 0x2a
	};

	if(target < CHIAKI_TARGET_PS4_10)
		return CHIAKI_ERR_INVALID_DATA;
	const uint8_t *keys_1 = chiaki_target_is_ps5(target) ? ps5_keys_1 : ps4_keys_1;
	uint8_t wurzelbert = chiaki_target_is_ps5(target) ? -0x2d : 0x29;

	for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
	{
		uint8_t k = keys_1[i*0x20 + key_1_off];
		aeropause[i] = (ambassador[i] ^ k) + wurzelbert + (uint8_t)i;
	}

	return CHIAKI_ERR_SUCCESS;
}

CHIAKI_EXPORT void chiaki_rpcrypt_init_auth(ChiakiRPCrypt *rpcrypt, ChiakiTarget target, const uint8_t *nonce, const uint8_t *morning)
{
	rpcrypt->target = target;
	chiaki_rpcrypt_bright_ambassador(target, rpcrypt->bright, rpcrypt->ambassador, nonce, morning);
}

CHIAKI_EXPORT void chiaki_rpcrypt_init_regist_ps4_pre10(ChiakiRPCrypt *rpcrypt, const uint8_t *ambassador, uint32_t pin)
{
	rpcrypt->target = CHIAKI_TARGET_PS4_9; // representative, might not be the actual version
	static const uint8_t regist_aes_key[CHIAKI_RPCRYPT_KEY_SIZE] =
		{ 0x3f, 0x1c, 0xc4, 0xb6, 0xdc, 0xbb, 0x3e, 0xcc, 0x50, 0xba, 0xed, 0xef, 0x97, 0x34, 0xc7, 0xc9 };
	memcpy(rpcrypt->ambassador, ambassador, sizeof(rpcrypt->ambassador));
	memcpy(rpcrypt->bright, regist_aes_key, sizeof(rpcrypt->bright));
	rpcrypt->bright[0] ^= (uint8_t)((pin >> 0x18) & 0xff);
	rpcrypt->bright[1] ^= (uint8_t)((pin >> 0x10) & 0xff);
	rpcrypt->bright[2] ^= (uint8_t)((pin >> 0x08) & 0xff);
	rpcrypt->bright[3] ^= (uint8_t)((pin >> 0x00) & 0xff);
}

CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_init_regist(ChiakiRPCrypt *rpcrypt, ChiakiTarget target, const uint8_t *ambassador, size_t key_0_off, uint32_t pin)
{
	static const uint8_t ps4_keys_0[512] = {
		0xbe, 0xce, 0x5d, 0xf0, 0xc1, 0x7d, 0xb5, 0xd0, 0xcb, 0x30,
		0x13, 0x5d, 0xaa, 0x56, 0x23, 0xfb, 0xc4, 0xbc, 0xf1, 0x8f,
		0x38, 0x57, 0xfb, 0xd4, 0xd4, 0x3f, 0x26, 0x38, 0xb5, 0xce,
		0xed, 0x6a, 0x21, 0xbc, 0x38, 0xd0, 0x1e, 0x68, 0xcc, 0x7b,
		0x45, 0xd1, 0xbe, 0x42, 0x1a, 0x08, 0xaa, 0x16, 0xfd, 0xb0,
		0xc0, 0xf4, 0xda, 0x35, 0xe9, 0x12, 0xfd, 0x21, 0x07, 0x48,
		0x34, 0xc1, 0xfc, 0x9f, 0x8c, 0xb6, 0xcb, 0x5d, 0xb2, 0x9c,
		0x84, 0xe0, 0x1a, 0xfa, 0xa0, 0xc7, 0xeb, 0x3a, 0x93, 0xb3,
		0xb3, 0xf1, 0x15, 0xaf, 0x13, 0xbd, 0x21, 0xab, 0xea, 0x5b,
		0x80, 0x50, 0x6b, 0x31, 0x1d, 0x7c, 0x1d, 0x40, 0xba, 0x3c,
		0x56, 0x0e, 0xe7, 0x94, 0x3a, 0x5b, 0xa1, 0x40, 0x80, 0x74,
		0x0a, 0xad, 0x28, 0xcf, 0x47, 0xdf, 0x42, 0xa6, 0x69, 0xe9,
		0x5e, 0xbb, 0xc0, 0xc0, 0x0e, 0xb2, 0xc5, 0x8a, 0xee, 0x08,
		0x03, 0xd2, 0x84, 0xe5, 0x91, 0x00, 0x1d, 0x46, 0x06, 0x55,
		0x09, 0x9d, 0x39, 0x9f, 0xd8, 0xe7, 0xfd, 0xad, 0x9e, 0x93,
		0x97, 0xc5, 0xea, 0xe7, 0xa3, 0x10, 0xa7, 0xf2, 0xa2, 0x93,
		0x7f, 0x07, 0x04, 0xb4, 0xee, 0xbb, 0xbf, 0x88, 0x23, 0x9c,
		0x6e, 0xa7, 0x62, 0xb1, 0x4b, 0x67, 0x1e, 0xb8, 0x3b, 0x1f,
		0x64, 0x93, 0x5a, 0x99, 0xec, 0xda, 0xfd, 0x0c, 0x6a, 0xb7,
		0xfe, 0xe4, 0x12, 0x76, 0x32, 0x65, 0xb8, 0x41, 0x23, 0xd1,
		0x17, 0x09, 0x9c, 0x24, 0x2d, 0x5c, 0x9d, 0x12, 0x79, 0xde,
		0xa1, 0xce, 0x69, 0xac, 0xa4, 0xbc, 0x39, 0x2f, 0x57, 0x38,
		0x84, 0x61, 0x2d, 0x2a, 0xe8, 0x04, 0xf8, 0xd5, 0x9d, 0x0b,
		0xff, 0x7e, 0x56, 0x0c, 0xec, 0x87, 0x0a, 0x1e, 0xab, 0xdf,
		0x93, 0x81, 0x13, 0xee, 0xcf, 0x32, 0x02, 0x5a, 0xbf, 0xb0,
		0x17, 0xb7, 0xba, 0xb5, 0x7f, 0xf0, 0x01, 0x7b, 0xe1, 0xcb,
		0x39, 0x7e, 0x60, 0x6d, 0xa4, 0x75, 0x6e, 0x29, 0x92, 0x45,
		0xa6, 0x4f, 0x74, 0x00, 0x86, 0x78, 0x73, 0xbe, 0xfd, 0x3e,
		0xe0, 0xd1, 0x0c, 0x6c, 0x0b, 0x49, 0x09, 0x83, 0x6c, 0x85,
		0x8a, 0x1d, 0xcb, 0x16, 0xce, 0x81, 0x7c, 0x49, 0xc9, 0x2c,
		0x63, 0x61, 0xde, 0xe2, 0x3f, 0x98, 0xb2, 0x73, 0xf0, 0x9a,
		0xec, 0x7b, 0x7c, 0xf1, 0xc9, 0xe1, 0x7f, 0xa5, 0x19, 0x8b,
		0x4b, 0xe8, 0x38, 0xa4, 0x34, 0x7d, 0xf4, 0x28, 0xfe, 0x0d,
		0x4d, 0x11, 0x57, 0x0c, 0x95, 0xf1, 0xaf, 0xd7, 0x34, 0x80,
		0xf4, 0xeb, 0x9b, 0x50, 0xe6, 0x6a, 0x5d, 0xea, 0xce, 0x0c,
		0x85, 0x4e, 0xc5, 0x5b, 0x93, 0x44, 0xc4, 0x24, 0x98, 0x80,
		0xfc, 0xf7, 0x72, 0x9c, 0x31, 0x0b, 0xee, 0x89, 0x67, 0xb3,
		0xa2, 0x69, 0x4f, 0xb3, 0x79, 0x5a, 0x14, 0x02, 0x70, 0xed,
		0x50, 0x13, 0x75, 0x00, 0x6a, 0xf3, 0xc6, 0x05, 0x1a, 0x00,
		0x33, 0x34, 0xf5, 0xac, 0x9e, 0x04, 0xdb, 0xc2, 0x00, 0xb0,
		0x1b, 0xc4, 0xf3, 0x97, 0x9d, 0x7f, 0xbe, 0xb8, 0x23, 0x8d,
		0x99, 0xe7, 0xcb, 0x74, 0x37, 0x4c, 0x57, 0xec, 0xd2, 0x69,
		0x49, 0x46, 0x75, 0x74, 0xaf, 0x51, 0x40, 0xa4, 0x11, 0x7b,
		0xb3, 0x2f, 0x51, 0xda, 0xe2, 0xef, 0x33, 0x73, 0x12, 0x18,
		0x25, 0x39, 0x03, 0x09, 0xca, 0x49, 0xdc, 0x8e, 0xf1, 0x94,
		0xd7, 0x80, 0x17, 0x9e, 0x87, 0x46, 0xc1, 0x04, 0x78, 0xd1,
		0xe5, 0x3d, 0x25, 0x88, 0xec, 0x72, 0x3a, 0x28, 0x41, 0x68,
		0x14, 0x6e, 0x10, 0xe4, 0xc9, 0x57, 0x75, 0x90, 0xfe, 0x22,
		0x1a, 0x63, 0x8e, 0xf4, 0xb8, 0x8d, 0x1a, 0x36, 0xfd, 0xb6,
		0xcb, 0x72, 0xc2, 0x97, 0x52, 0x9f, 0x91, 0x72, 0x1b, 0x75,
		0x57, 0x90, 0x3b, 0xfd, 0x5a, 0x93, 0x8c, 0xdb, 0xfc, 0xa3,
		0x03, 0xdf
	};

	static const uint8_t ps5_keys_0[512] = {
		0x24, 0xd8, 0xc2, 0x69, 0x4c, 0x67, 0x78, 0x71, 0xee, 0x31,
		0xbd, 0x2b, 0x83, 0xb2, 0x1d, 0x61, 0xc9, 0xa7, 0x8e, 0xed,
		0x9a, 0xd3, 0x6a, 0x6b, 0x5c, 0xc8, 0x35, 0x79, 0xa7, 0x24,
		0xe2, 0x17, 0x06, 0x60, 0x2e, 0xdf, 0xf4, 0xdb, 0x27, 0x10,
		0x55, 0xd9, 0xea, 0x16, 0x4e, 0x90, 0x0c, 0xbf, 0x40, 0x6f,
		0x54, 0xa5, 0x31, 0x70, 0x2d, 0x5d, 0x1e, 0x27, 0xdf, 0x37,
		0x40, 0xba, 0x9d, 0x5d, 0xff, 0xe1, 0x05, 0x70, 0x80, 0xd4,
		0xb7, 0xc2, 0x96, 0x7f, 0x2f, 0x42, 0xeb, 0x5a, 0x08, 0xde,
		0xc1, 0xb5, 0x52, 0x15, 0xf6, 0xb5, 0xf2, 0xd9, 0x69, 0xa5,
		0xc7, 0xc4, 0x7f, 0x46, 0x64, 0xa4, 0xfd, 0x46, 0x98, 0xa7,
		0xe1, 0x2a, 0x8e, 0x6f, 0xaf, 0x65, 0x42, 0x28, 0xb9, 0xc2,
		0x6f, 0x3e, 0xe3, 0xe4, 0x4e, 0xe4, 0x5b, 0x9d, 0x60, 0x10,
		0xb8, 0x5a, 0xb0, 0x7d, 0x04, 0x0c, 0x4c, 0x24, 0x78, 0xbd,
		0xb8, 0xba, 0xdb, 0x8f, 0xe3, 0xa0, 0x75, 0x6d, 0x28, 0xc2,
		0x33, 0x5b, 0x32, 0x83, 0xdd, 0x51, 0xb0, 0xa5, 0x8d, 0x09,
		0x66, 0xe4, 0x5c, 0xb8, 0x70, 0x0b, 0xe6, 0x82, 0x14, 0xb6,
		0xd2, 0xb0, 0xc2, 0xe0, 0x55, 0xf3, 0x84, 0xad, 0x9d, 0x3a,
		0xf8, 0x77, 0xf5, 0x9d, 0x9a, 0xa9, 0x7d, 0xf1, 0x45, 0x1b,
		0x9b, 0x55, 0x25, 0xd8, 0xc1, 0xff, 0x03, 0xa5, 0x48, 0x0b,
		0x1b, 0x19, 0x0c, 0xbd, 0xe0, 0xcd, 0x48, 0xf3, 0x2c, 0x99,
		0x19, 0xd6, 0xb8, 0xbb, 0xd6, 0x35, 0x43, 0x6f, 0x71, 0xe3,
		0xef, 0x3e, 0x97, 0xb8, 0xe9, 0x40, 0xa8, 0x47, 0xe0, 0xe0,
		0x01, 0x16, 0x9d, 0xa7, 0xe5, 0x94, 0x4b, 0x1d, 0xd2, 0x80,
		0xa2, 0x7f, 0xf2, 0x98, 0x10, 0x38, 0x0d, 0xb8, 0x56, 0xc3,
		0x7a, 0x4b, 0x4c, 0x85, 0xec, 0x2f, 0x23, 0x89, 0xaf, 0xd5,
		0xba, 0x9a, 0xad, 0xb0, 0x61, 0x9c, 0x51, 0xb4, 0x6d, 0x02,
		0x49, 0x26, 0xa4, 0x34, 0x84, 0x20, 0x35, 0x30, 0x23, 0x0a,
		0x47, 0x14, 0x32, 0x1a, 0x96, 0x0e, 0xe8, 0x0f, 0x96, 0x96,
		0xd4, 0xba, 0x68, 0x3a, 0x67, 0x15, 0x74, 0xe0, 0xd6, 0x60,
		0x4c, 0x68, 0x50, 0x73, 0x14, 0x2f, 0x11, 0x59, 0xac, 0xc8,
		0x32, 0xd1, 0xdb, 0x4c, 0x8a, 0x94, 0x75, 0x33, 0x61, 0xd1,
		0xd4, 0xfd, 0xaa, 0x6a, 0x61, 0x68, 0xd8, 0xae, 0x31, 0x4f,
		0xb8, 0x07, 0x7b, 0x27, 0x0f, 0xf9, 0x0b, 0xb0, 0xc2, 0x64,
		0xb3, 0x72, 0xea, 0x8b, 0x87, 0x40, 0x09, 0xb4, 0x82, 0xb4,
		0xad, 0x76, 0xf9, 0x36, 0x05, 0x60, 0x89, 0xc8, 0x20, 0xeb,
		0xa5, 0xf1, 0x51, 0x0b, 0x27, 0xa7, 0xf0, 0x76, 0x84, 0x96,
		0xeb, 0xb1, 0x2e, 0xc2, 0x85, 0x28, 0xbc, 0x48, 0x34, 0xd4,
		0x01, 0x8d, 0x5b, 0x25, 0x54, 0xe0, 0xc4, 0x4f, 0xa0, 0xfa,
		0x99, 0x8d, 0x6d, 0x7a, 0x64, 0xb1, 0xa9, 0x5d, 0xa4, 0xf9,
		0xf5, 0x22, 0xeb, 0x9a, 0xf4, 0xa8, 0x7a, 0x78, 0x4b, 0x7f,
		0xe2, 0x8b, 0x04, 0x50, 0x43, 0x7d, 0x26, 0x2d, 0x19, 0x98,
		0x38, 0x6a, 0x4f, 0x2d, 0x30, 0x15, 0x2e, 0x4f, 0xcd, 0xb9,
		0xce, 0x9e, 0x8d, 0x12, 0xc9, 0xfe, 0x33, 0x8b, 0x84, 0xce,
		0x5b, 0x40, 0xe3, 0x7f, 0x72, 0x6d, 0x6c, 0x8a, 0x6a, 0x9e,
		0x54, 0xf1, 0xe3, 0x64, 0x5d, 0x6e, 0x7f, 0xac, 0x1a, 0xe7,
		0xf7, 0xfa, 0x00, 0x22, 0xed, 0x2b, 0x23, 0xfa, 0x58, 0xc5,
		0xeb, 0x44, 0x92, 0x5d, 0xcc, 0xaa, 0x82, 0x9f, 0x23, 0xfb,
		0xa6, 0xc9, 0x65, 0x2a, 0xe0, 0x79, 0x12, 0x65, 0x2c, 0x34,
		0xc5, 0x23, 0x16, 0xc9, 0xcc, 0x05, 0x30, 0xf3, 0x96, 0x0b,
		0x90, 0x67, 0x1a, 0xa7, 0x69, 0x4c, 0x3e, 0x43, 0x24, 0x9d,
		0x4e, 0x68, 0xbd, 0x8b, 0x75, 0x6e, 0x9d, 0x07, 0x6f, 0x1a,
		0x6a, 0xba
	};

	if(target < CHIAKI_TARGET_PS4_10)
		return CHIAKI_ERR_INVALID_DATA;
	const uint8_t *keys_0 = chiaki_target_is_ps5(target) ? ps5_keys_0 : ps4_keys_0;

	if(key_0_off >= 0x20)
		return CHIAKI_ERR_INVALID_DATA;

	rpcrypt->target = target;
	memcpy(rpcrypt->ambassador, ambassador, sizeof(rpcrypt->ambassador));

	for(size_t i=0; i<CHIAKI_RPCRYPT_KEY_SIZE; i++)
		rpcrypt->bright[i] = keys_0[i*0x20 + key_0_off];
	rpcrypt->bright[0xc] ^= (uint8_t)((pin >> 0x18) & 0xff);
	rpcrypt->bright[0xd] ^= (uint8_t)((pin >> 0x10) & 0xff);
	rpcrypt->bright[0xe] ^= (uint8_t)((pin >> 0x08) & 0xff);
	rpcrypt->bright[0xf] ^= (uint8_t)((pin >> 0x00) & 0xff);

	return CHIAKI_ERR_SUCCESS;
}

#define HMAC_KEY_SIZE 0x10
static const uint8_t hmac_key_ps5[HMAC_KEY_SIZE] = { 0x46, 0x46, 0x87, 0xb3, 0x49, 0xca, 0x8c, 0xe8, 0x59, 0xc5, 0x27, 0x0f, 0x5d, 0x7a, 0x69, 0xd6 };
static const uint8_t hmac_key_ps4[HMAC_KEY_SIZE] = { 0x20, 0xd6, 0x6f, 0x59, 0x04, 0xea, 0x7c, 0x14, 0xe5, 0x57, 0xff, 0xc5, 0x2e, 0x48, 0x8a, 0xc8 };
static const uint8_t hmac_key_ps4_pre10[HMAC_KEY_SIZE] = { 0xac, 0x07, 0x88, 0x83, 0xc8, 0x3a, 0x1f, 0xe8, 0x11, 0x46, 0x3a, 0xf3, 0x9e, 0xe3, 0xe3, 0x77 };

static const uint8_t *rpcrypt_hmac_key(ChiakiRPCrypt *rpcrypt)
{
	switch(rpcrypt->target)
	{
		case CHIAKI_TARGET_PS5_1:
			return hmac_key_ps5;
		case CHIAKI_TARGET_PS4_8:
		case CHIAKI_TARGET_PS4_9:
			return hmac_key_ps4_pre10;
		default:
			return hmac_key_ps4;
	}
}

#ifdef CHIAKI_LIB_ENABLE_MBEDTLS
CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_generate_iv(ChiakiRPCrypt *rpcrypt, uint8_t *iv, uint64_t counter)
{
	const uint8_t *hmac_key = rpcrypt_hmac_key(rpcrypt);

	uint8_t buf[CHIAKI_RPCRYPT_KEY_SIZE + 8];
	memcpy(buf, rpcrypt->ambassador, CHIAKI_RPCRYPT_KEY_SIZE);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 0] = (uint8_t)((counter >> 0x38) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 1] = (uint8_t)((counter >> 0x30) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 2] = (uint8_t)((counter >> 0x28) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 3] = (uint8_t)((counter >> 0x20) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 4] = (uint8_t)((counter >> 0x18) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 5] = (uint8_t)((counter >> 0x10) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 6] = (uint8_t)((counter >> 0x08) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 7] = (uint8_t)((counter >> 0x00) & 0xff);

	uint8_t hmac[CHIAKI_RPCRYPT_KEY_SIZE];
	unsigned int hmac_len = 0;


	mbedtls_md_context_t ctx;
	mbedtls_md_type_t type = MBEDTLS_MD_SHA256;

	mbedtls_md_init(&ctx);

#define GOTO_ERROR(err) do { \
	if((err) !=0){ \
		goto error;} \
	} while(0)
	// https://tls.mbed.org/module-level-design-hashing
	GOTO_ERROR(mbedtls_md_setup(&ctx, mbedtls_md_info_from_type(type) , 1));
	GOTO_ERROR(mbedtls_md_hmac_starts(&ctx, hmac_key, HMAC_KEY_SIZE));
	GOTO_ERROR(mbedtls_md_hmac_update(&ctx, (const unsigned char *) buf, sizeof(buf)));
	GOTO_ERROR(mbedtls_md_hmac_finish(&ctx, hmac));
#undef GOTO_ERROR
	memcpy(iv, hmac, CHIAKI_RPCRYPT_KEY_SIZE);
	mbedtls_md_free(&ctx);
	return CHIAKI_ERR_SUCCESS;
error:
	mbedtls_md_free(&ctx);
	return CHIAKI_ERR_UNKNOWN;
}


static ChiakiErrorCode chiaki_rpcrypt_crypt(ChiakiRPCrypt *rpcrypt, uint64_t counter, const uint8_t *in, uint8_t *out, size_t sz, bool encrypt)
{

#define GOTO_ERROR(err) do { \
	if((err) !=0){ \
		goto error;} \
	} while(0)

	// https://github.com/ARMmbed/mbedtls/blob/development/programs/aes/aescrypt2.c
	// build aes context
	mbedtls_aes_context ctx;
	mbedtls_aes_init(&ctx);

	// initialization vector
	uint8_t iv[CHIAKI_RPCRYPT_KEY_SIZE];
	ChiakiErrorCode err = chiaki_rpcrypt_generate_iv(rpcrypt, iv, counter);
	if(err != CHIAKI_ERR_SUCCESS)
		return err;

	GOTO_ERROR(mbedtls_aes_setkey_enc(&ctx, rpcrypt->bright, 128));
	size_t iv_off = 0;
	if(encrypt)
	{
		GOTO_ERROR(mbedtls_aes_crypt_cfb128(&ctx, MBEDTLS_AES_ENCRYPT, sz, &iv_off, iv, in, out));
	}
	else
	{
		// the aes_crypt_cfb128 does not seems to use the setkey_dec
		// GOTO_ERROR(mbedtls_aes_setkey_dec(&ctx, rpcrypt->bright, 128));
		GOTO_ERROR(mbedtls_aes_crypt_cfb128(&ctx, MBEDTLS_AES_DECRYPT, sz, &iv_off, iv, in, out));
	}

#undef GOTO_ERROR
	mbedtls_aes_free(&ctx);

	return CHIAKI_ERR_SUCCESS;

error:
	mbedtls_aes_free(&ctx);
	return CHIAKI_ERR_UNKNOWN;
}

#else
CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_generate_iv(ChiakiRPCrypt *rpcrypt, uint8_t *iv, uint64_t counter)
{
	const uint8_t *hmac_key = rpcrypt_hmac_key(rpcrypt);

	uint8_t buf[CHIAKI_RPCRYPT_KEY_SIZE + 8];
	memcpy(buf, rpcrypt->ambassador, CHIAKI_RPCRYPT_KEY_SIZE);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 0] = (uint8_t)((counter >> 0x38) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 1] = (uint8_t)((counter >> 0x30) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 2] = (uint8_t)((counter >> 0x28) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 3] = (uint8_t)((counter >> 0x20) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 4] = (uint8_t)((counter >> 0x18) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 5] = (uint8_t)((counter >> 0x10) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 6] = (uint8_t)((counter >> 0x08) & 0xff);
	buf[CHIAKI_RPCRYPT_KEY_SIZE + 7] = (uint8_t)((counter >> 0x00) & 0xff);

	uint8_t hmac[32];
	unsigned int hmac_len = 0;
	if(!HMAC(EVP_sha256(), hmac_key, CHIAKI_RPCRYPT_KEY_SIZE, buf, sizeof(buf), hmac, &hmac_len))
		return CHIAKI_ERR_UNKNOWN;

	if(hmac_len < CHIAKI_RPCRYPT_KEY_SIZE)
		return CHIAKI_ERR_UNKNOWN;

	memcpy(iv, hmac, CHIAKI_RPCRYPT_KEY_SIZE);
	return CHIAKI_ERR_SUCCESS;
}

static ChiakiErrorCode chiaki_rpcrypt_crypt(ChiakiRPCrypt *rpcrypt, uint64_t counter, const uint8_t *in, uint8_t *out, size_t sz, bool encrypt)
{
	EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
	if(!ctx)
		return CHIAKI_ERR_UNKNOWN;

	uint8_t iv[CHIAKI_RPCRYPT_KEY_SIZE];
	ChiakiErrorCode err = chiaki_rpcrypt_generate_iv(rpcrypt, iv, counter);
	if(err != CHIAKI_ERR_SUCCESS)
		return err;

#define FAIL(err) do { EVP_CIPHER_CTX_free(ctx); return (err); } while(0);

	if(encrypt)
	{
		if(!EVP_EncryptInit_ex(ctx, EVP_aes_128_cfb128(), NULL, rpcrypt->bright, iv))
			FAIL(CHIAKI_ERR_UNKNOWN);
	}
	else
	{
		if(!EVP_DecryptInit_ex(ctx, EVP_aes_128_cfb128(), NULL, rpcrypt->bright, iv))
			FAIL(CHIAKI_ERR_UNKNOWN);
	}

	if(!EVP_CIPHER_CTX_set_padding(ctx, 0))
		FAIL(CHIAKI_ERR_UNKNOWN);

	int outl;
	if(encrypt)
	{
		if(!EVP_EncryptUpdate(ctx, out, &outl, in, (int)sz))
			FAIL(CHIAKI_ERR_UNKNOWN);
	}
	else
	{
		if(!EVP_DecryptUpdate(ctx, out, &outl, in, (int)sz))
			FAIL(CHIAKI_ERR_UNKNOWN);
	}

	if(outl != (int)sz)
		FAIL(CHIAKI_ERR_UNKNOWN);

#undef FAIL
	EVP_CIPHER_CTX_free(ctx);
	return CHIAKI_ERR_SUCCESS;
}
#endif

CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_encrypt(ChiakiRPCrypt *rpcrypt, uint64_t counter, const uint8_t *in, uint8_t *out, size_t sz)
{
	return chiaki_rpcrypt_crypt(rpcrypt, counter, in, out, sz, true);
}

CHIAKI_EXPORT ChiakiErrorCode chiaki_rpcrypt_decrypt(ChiakiRPCrypt *rpcrypt, uint64_t counter, const uint8_t *in, uint8_t *out, size_t sz)
{
	return chiaki_rpcrypt_crypt(rpcrypt, counter, in, out, sz, false);
}
