//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleView/RealspacePanel.cpp
//! @brief     Implements class RealspacePanel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleView/RealspacePanel.h"
#include "GUI/View/SampleView/RealspaceWidget.h"
#include "GUI/View/Widget/StyledToolbar.h"
#include <QVBoxLayout>

RealspacePanel::RealspacePanel(QWidget* parent)
    : QWidget(parent)
    , m_widget(new RealspaceWidget)
{
    setWindowTitle("Real Space");

    auto* toolbar = new StyledToolbar(this);

    const auto createAction = [&](const QString& text, const QString& tooltip) -> QAction* {
        auto* action = new QAction(text, this);
        action->setToolTip(tooltip);
        if (!toolbar->actions().empty())
            toolbar->addSeparator();
        toolbar->addAction(action);
        return action;
    };

    auto* action = createAction("Save Picture", "Save 3D real space view as .png file");
    connect(action, &QAction::triggered, m_widget, QOverload<>::of(&RealspaceWidget::savePicture));

    action = createAction("Default View", "Reset view and zoom level to default");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::defaultView);

    action = createAction("Side View", "View sample from the side at current zoom level");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::sideView);

    action = createAction("Top View", "View sample from the top at current zoom level");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::topView);

    action = createAction("Enlarge", "Increase layer size");
    connect(action, &QAction::triggered, [this]() { m_widget->changeLayerSize(1.25); });

    action = createAction("Reduce", "Decrease layer size");
    connect(action, &QAction::triggered, [this]() { m_widget->changeLayerSize(0.8); });

    auto* mainLayout = new QVBoxLayout(this);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(toolbar);
    mainLayout->addWidget(m_widget);
}

QSize RealspacePanel::sizeHint() const
{
    return QSize(300, 300);
}

RealspaceWidget* RealspacePanel::widget()
{
    return m_widget;
}
