var count = (function toFraction(BigNumber) {
    var start = +new Date(),
        log,
        error,
        u,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function assertException(func, message) {
        var actual;
        total++;
        try {
            func();
        } catch (e) {
            actual = e;
        }
        if (actual && actual.name == 'BigNumber Error') {
            passed++;
            //log('\n Expected and actual: ' + actual);
        } else {
            error('\n Test number: ' + total + ' failed');
            error('\n Expected: ' + message + ' to raise a BigNumber Error.');
            error(' Actual:   ' + (actual || 'no exception'));
            //process.exit();
        }
    }

    function T(expected, value, maxDenominator){
        assert(expected, new BigNumber(value).toFraction(maxDenominator).toString());
    }

    log('\n Testing toFraction...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    // Tests generated using Python's fraction module.

    T('1,10', '0.1', u);
    T('-1,10', '-0.1', u);
    T('1,100', '0.01', u);
    T('-1,100', '-0.01', u);
    T('1,1000', '0.001', u);
    T('-1,1000', '-0.001', u);
    T('54301793,100000', '543.017930', u);
    T('-484693350148251574449,1000000000000', '-484693350.1482515744490', u);
    T('-200111074540568980159529,25000000000000', '-8004442981.622759206381160', u);
    T('-5965700213104829,1000000000000000', '-5.965700213104829', u);
    T('-5124110209739113904928179,100000000000000', '-51241102097.39113904928179', u);
    T('-1655017182533520595664837541,200000000000000', '-8275085912667.602978324187705', u);
    T('-5,8', '-0.625', u);
    T('-42394617,200000000', '-0.211973085', u);
    T('-147453708173963291829,2500000000000000000', '-58.9814832695853167316', u);
    T('-2733,100', '-27.33', u);
    T('0,1', '0.0', u);
    T('-2599897551866392365731491641,10000000000', '-259989755186639236.5731491641', u);
    T('-9326331230765073371,5000000000000000', '-1865.2662461530146742', u);
    T('-2483061,10000', '-248.3061', u);
    T('-92864854181,10000000', '-9286.4854181', u);
    T('-758516850228215193,100000000000000000', '-7.5851685022821519300', u);
    T('0,1', '0.0', u);
    T('-19570408085559589,20000000000', '-978520.404277979450', u);
    T('-211,100', '-2.11', u);
    T('-4789334439841,10000000000', '-478.9334439841', u);
    T('2978320663077,10000000000000', '0.2978320663077', u);
    T('-1221,25', '-48.84', u);
    T('12456254053601331,5000000000000', '2491.25081072026620', u);
    T('31969983,100000', '319.69983', u);
    T('31852901149,400000000000', '0.0796322528725', u);
    T('-456674099059,10000000000', '-45.6674099059', u);
    T('-361907787,5000000', '-72.3815574', u);
    T('-6521,100', '-65.21', u);
    T('-11438080592941644302261353,10000000000000', '-1143808059294.1644302261353', u);
    T('670431292887199,100000000000000', '6.70431292887199', u);
    T('0,1', '0.0', u);
    T('-12179876267417657784053129,390625000000', '-31180483244589.20392717601024', u);
    T('-883842861294459426792636233,1250000000000000', '-707074289035.5675414341089864', u);
    T('93,100', '0.93', u);
    T('-4013,250', '-16.052', u);
    T('-1813352184996169886155932687,10000000000000', '-181335218499616.9886155932687', u);
    T('-79491022129556305898477,25000000000000', '-3179640885.18225223593908', u);
    T('-6955532202842357103065812079,100000000000', '-69555322028423571.03065812079', u);
    T('1,2', '0.5', u);
    T('117679941425840963432274357541,6250000000000000', '18828790628134.55414916389720656', u);
    T('-44419548531115044568503,5000000000000000000', '-8883.9097062230089137006', u);
    T('-48,25', '-1.92', u);
    T('3026554851926703631,10000000000000000000', '0.3026554851926703631', u);
    T('-27597130550621,10000000', '-2759713.0550621', u);
    T('-80047992603251,50000000000000', '-1.60095985206502', u);
    T('-77860395508386749,1000000000000', '-77860.395508386749', u);
    T('48601655379903990436811,10000000000000000', '4860165.5379903990436811', u);
    T('310385178311182627346467559,100000000000000', '3103851783111.82627346467559', u);
    T('-1866981502342301,5000000000000', '-373.3963004684602', u);
    T('-263843686400376847,1000000000', '-263843686.4003768470', u);
    T('2124759225302861,2500000000000000', '0.84990369012114440', u);
    T('-664881551,125000', '-5319.052408', u);
    T('-15340092011,500000000', '-30.6801840220', u);
    T('-579129,1000', '-579.129', u);
    T('31618159823455571971,6250000000000', '5058905.57175289151536', u);
    T('-84886224665539,2500000000000', '-33.9544898662156', u);
    T('-912648758578220220986367,1250000000000000000', '-730119.0068625761767890936', u);
    T('-62977599666149191581,10000000000', '-6297759966.6149191581', u);
    T('-5497168411400642219272521,100000000000000000', '-54971684.11400642219272521', u);
    T('-5616409194059,1000000000', '-5616.409194059', u);
    T('-283871646617001,1000000000', '-283871.646617001', u);
    T('-485222200141938641,100000000000000000', '-4.85222200141938641', u);
    T('-13782530592809417884043,2500000000000000000', '-5513.0122371237671536172', u);
    T('-386323443,1250000000', '-0.3090587544', u);
    T('-38961,1000', '-38.961', u);
    T('21551091008117488357,2500000000000000000', '8.62043640324699534280', u);
    T('173,100', '1.730', u);
    T('5333399,6250000', '0.85334384', u);
    T('-229251439697497,62500000000', '-3668.0230351599520', u);
    T('-212582095021077222386880441,100000000000000000000', '-2125820.95021077222386880441', u);
    T('-4509791479,500000', '-9019.582958', u);
    T('-7,10', '-0.7', u);
    T('171494582705307,50000000000', '3429.89165410614', u);
    T('0,1', '0.0', u);
    T('-116465991083,5000000', '-23293.19821660', u);
    T('-123982957,1250000', '-99.1863656', u);
    T('-11006669641979567,2000000000000', '-5503.3348209897835', u);
    T('-2313,1000', '-2.313', u);
    T('15409800999774967,10000000000000000', '1.5409800999774967', u);
    T('-6614709,25000000', '-0.26458836', u);
    T('0,1', '0.0', u);
    T('-200386023,2500000', '-80.1544092', u);
    T('-364278093789203,1000000000000', '-364.278093789203', u);
    T('68468339471261366060229,1000000000000000000', '68468.339471261366060229', u);
    T('-470519242528519,500000000', '-941038.485057038', u);
    T('11,2', '5.5', u);
    T('-9976509289,10000000', '-997.6509289', u);
    T('0,1', '0.0', u);
    T('-6916459729727082955260053,1000000000000000000', '-6916459.729727082955260053', u);
    T('-140541,16000', '-8.7838125', u);
    T('-49,5', '-9.8', u);
    T('-47,50', '-0.94', u);
    T('272783,100000', '2.72783', u);
    T('4116487650822625882405778003,50000000000000000000', '82329753.01645251764811556006', u);
    T('2919425337211,250000000000', '11.677701348844', u);
    T('-4159828773,1250000', '-3327.8630184', u);
    T('-79717565915722094623,10000000000000', '-7971756.5915722094623', u);
    T('-317320236283,500000000', '-634.640472566', u);
    T('-14147732160552271137,2500000000000000000', '-5.6590928642209084548', u);
    T('-12059857,250000', '-48.239428', u);
    T('681420956722259461,12500000000', '54513676.53778075688', u);
    T('17774376511238461243,200000000000000', '88871.882556192306215', u);
    T('4678001741059099871336323918049287,50000000000000000000', '93560034821181.99742672647836098574', u);
    T('0,1', '0.0', u);
    T('-7,10', '-0.7', u);
    T('-23351644276078989,25000000000000', '-934.06577104315956', u);
    T('-1765063,200000', '-8.825315', u);
    T('16389704981,5000000000', '3.2779409962', u);
    T('18121833053827129891,2000000000000', '9060916.526913564945500', u);
    T('-2523281598830311,10000000000', '-252328.1598830311', u);
    T('994977441308813,250000000', '3979909.765235252', u);
    T('-50961713844809094829,5000000000000000', '-10192.3427689618189658', u);
    T('-123997423829,20000000', '-6199.87119145', u);
    T('-549246278950790194833129957,625000000000000', '-878794046321.2643117330079312', u);
    T('-21647408712889002435857,200000000000000', '-108237043.564445012179285', u);
    T('-2820570956057812319823086593,1000000000000000', '-2820570956057.812319823086593', u);
    T('-110806033317345864067,20000000000000000', '-5540.30166586729320335', u);
    T('-8353311,10000000', '-0.8353311', u);
    T('-1357114760839431,200000000', '-6785573.804197155', u);
    T('209320962852697355183332757976049,625000000000000000', '334913540564315.76829333241276167840', u);
    T('0,1', '0.0', u);
    T('-87920815397,10000000', '-8792.0815397', u);
    T('42482569836133,1000000000000', '42.482569836133', u);
    T('0,1', '0.0', u);
    T('-29518137461,1000000000', '-29.518137461', u);
    T('5192895093,100000', '51928.95093', u);
    T('-40587909849427164111,1000000000000000', '-40587.909849427164111', u);
    T('824746069,10000000', '82.4746069', u);
    T('-245537748603430439,400000000000', '-613844.3715085760975', u);
    T('-717771547151675126007258009,1250000000000000', '-574217237721.3401008058064072', u);
    T('-532430787549785291,12500000000', '-42594463.00398282328', u);
    T('-44797367524497867,5000000000000000', '-8.9594735048995734', u);
    T('-468580303722092912184256627,2000000000000000000', '-234290151.8610464560921283135', u);
    T('5657719391,1000000000', '5.657719391', u);
    T('14090298821989,50000000', '281805.97643978', u);
    T('-80762928268461244253,10000000000000000000', '-8.0762928268461244253', u);
    T('3,5', '0.6', u);
    T('-92113139197545733,10000000000000', '-9211.3139197545733', u);
    T('-1556299,10000000', '-0.1556299', u);
    T('2418948368754047020942979,250000000000000', '9675793475.0161880837719160', u);
    T('19652697,62500', '314.443152', u);
    T('-87839127276245656603537,125000000000000', '-702713018.209965252828296', u);
    T('-14005169230761299788063,20000000000000', '-700258461.53806498940315', u);
    T('-279,40', '-6.975', u);
    T('0,1', '0.0', u);
    T('-1806754525468682323,100000000000000', '-18067.54525468682323', u);
    T('247972458841,10000000000', '24.7972458841', u);
    T('451607951414070493,500000000000000', '903.215902828140986', u);
    T('281482401773084158055523,50000000000000000000', '5629.64803546168316111046', u);
    T('33,10', '3.3', u);
    T('-290099,62500', '-4.641584', u);
    T('-899598441,1000000', '-899.598441', u);
    T('193325773941340501463237,100000000000000', '1933257739.41340501463237', u);
    T('27357881161156398470035203,500000000000000000', '54715762.322312796940070406', u);
    T('7215187086319853416331,100000000000000000', '72151.87086319853416331', u);
    T('-823577,125000', '-6.588616', u);
    T('-127409327138611,312500000000000', '-0.4077098468435552', u);
    T('-142974608134492239968107843,1000000000', '-142974608134492239.9681078430', u);
    T('-4537660598987301975574620667,10000000000000', '-453766059898730.1975574620667', u);
    T('-18374189280291,5000000', '-3674837.8560582', u);
    T('-4597156458983,250000000', '-18388.625835932', u);
    T('-71670733247098307,1250000000000000', '-57.3365865976786456', u);
    T('1987,2500', '0.7948', u);
    T('0,1', '0.0', u);
    T('-6097315591260183077181430951,100000000000000000', '-60973155912.60183077181430951', u);
    T('30252232444541,100000000000', '302.52232444541', u);
    T('-980689,125000', '-7.845512', u);
    T('-535844087,1000000', '-535.844087', u);
    T('-193035003,20000', '-9651.75015', u);
    T('-1007194349453595010508023,6250000000000000000', '-161151.09591257520168128368', u);
    T('-3658094692361441,200000000', '-18290473.461807205', u);
    T('-719,100', '-7.19', u);
    T('-958804210658146103495976847,20000000000000', '-47940210532907.30517479884235', u);
    T('-89094216351,1000000', '-89094.216351', u);
    T('-3582499868182265571,100000000000000000', '-35.82499868182265571', u);
    T('2067,10000', '0.2067', u);
    T('-9433,100', '-94.33', u);
    T('-217068626544319,10000000000000', '-21.7068626544319', u);
    T('37585828244413865266713,4000000000000', '9396457061.10346631667825', u);
    T('-91498021630276586782019,25000000000000', '-3659920865.21106347128076', u);
    T('-3836906826908540628227081,2000000000000', '-1918453413454.2703141135405', u);
    T('-4353266163985119533,200000000000000', '-21766.330819925597665', u);
    T('-9383241486201,25000000', '-375329.65944804', u);
    T('0,1', '0.0', u);
    T('-341601768966781,5000000000', '-68320.3537933562', u);
    T('-1567555398070069,2500000000', '-627022.1592280276', u);
    T('-33306019510179,100000000', '-333060.19510179', u);
    T('-4863716727376251454133,1000000000000', '-4863716727.376251454133', u);
    T('-574277977,100000000', '-5.74277977', u);
    T('5350626629797780458100053761,1000000000000000000', '5350626629.797780458100053761', u);
    T('-7636341311139689,1000000000', '-7636341.311139689', u);
    T('-235940193945989693,100000000000', '-2359401.93945989693', u);
    T('-6648926524329262739,1000000000000000', '-6648.926524329262739', u);
    T('53408696765404895507753,100000000000000000000', '534.08696765404895507753', u);
    T('-608550193099791,1000000000000000', '-0.608550193099791', u);
    T('23166918429420757228938433,2500000000000000', '9266767371.76830289157537320', u);
    T('8273,20', '413.650', u);
    T('227757,5000', '45.5514', u);
    T('348417664650569,100000000000', '3484.17664650569', u);
    T('53142735643470026332735643192361,10000000000000000000', '5314273564347.0026332735643192361', u);
    T('-89788064954218810252057051,100000000000000', '-897880649542.18810252057051', u);
    T('-1306089141,250000000', '-5.224356564', u);
    T('-45086702607022861843,1250000000000', '-36069362.0856182894744000', u);
    T('-377,50', '-7.540', u);
    T('-258447712404123545579762063,20000000000', '-12922385620206177.27898810315', u);
    T('-628719551197,10000000', '-62871.9551197', u);
    T('60612440731,312500', '193959.8103392', u);
    T('-1597151323402131,50000000000', '-31943.026468042620', u);
    T('-115594971935575014446009257,4000000000000000', '-28898742983.89375361150231425', u);
    T('1,10', '0.1', u);
    T('-18621,250', '-74.484', u);
    T('-637961611566238281399,500000000000000000', '-1275.923223132476562798', u);
    T('-584061819097005054409,100000000000000000', '-5840.618190970050544090', u);
    T('-18813090881,200000000', '-94.065454405', u);
    T('303,250', '1.212', u);
    T('448557308974238577920704115565781,10000000000000000000', '44855730897423.85779207041155657810', u);
    T('-1193270826256326041220780273,500000000', '-2386541652512652082.441560546', u);
    T('-896023,2000000', '-0.4480115', u);
    T('-41206366129433,10000000000', '-4120.6366129433', u);
    T('-17988697610527989991481961,500000000000000', '-35977395221.055979982963922', u);
    T('-1817173764903,10000000', '-181717.37649030', u);
    T('-3,10', '-0.3', u);
    T('-92549,1000', '-92.549', u);
    T('-279096500654837006467612407,50000000000000000000', '-5581930.01309674012935224814', u);
    T('-259968688776686799,500000000', '-519937377.553373598', u);
    T('-5,2', '-2.5', u);
    T('-3917,50', '-78.34', u);
    T('0,1', '0.0', u);
    T('-75607642540312935927,10000000000000000', '-7560.76425403129359270', u);
    T('-2176542221453118621443981521,500000000000000000', '-4353084442.906237242887963042', u);
    T('-5648864378490501415429342121,100000000000000000', '-56488643784.90501415429342121', u);
    T('-76539862671139,1000000000', '-76539.862671139', u);
    T('141796873549,500000', '283593.747098', u);
    T('-948224946050543083414093,1250000000000', '-758579956840.4344667312744', u);
    T('-2205228733,500000', '-4410.457466', u);
    T('25694008107,1000000', '25694.008107', u);
    T('-196113,2000', '-98.0565', u);
    T('-66885450169789234267297,1000000000000', '-66885450169.789234267297', u);
    T('-48529788189,500000', '-97059.576378', u);
    T('-324221069,10000000', '-32.4221069', u);
    T('-116146483434824230068673,50000000000000000000', '-2322.92966869648460137346', u);
    T('-4051770395293,62500000', '-64828.326324688', u);
    T('-83031469,100000000', '-0.830314690', u);
    T('-28756998587475224546963,50000000000000000', '-575139.97174950449093926', u);
    T('-1799450063972413799,10000000000', '-179945006.3972413799', u);
    T('9167099921651200679035713391,1000000000000000', '9167099921651.20067903571339100', u);
    T('1451,25', '58.04', u);
    T('8253,200', '41.265', u);
    T('12869,2500', '5.1476', u);
    T('13856945667,1000000000', '13.856945667', u);
    T('-43346967,100000000', '-0.43346967', u);
    T('-3588675849064697,5000000000', '-717735.1698129394', u);
    T('-940992729300541871141570909,2000000000000000000', '-470496364.6502709355707854545', u);
    T('-956832531182962661827,2500000000000', '-382733012.4731850647308', u);
    T('-152399,500', '-304.798', u);
    T('-178527,10000', '-17.8527', u);
    T('-342133,1000', '-342.133', u);
    T('57078299482481,1000000', '57078299.48248100', u);
    T('-79,10', '-7.9', u);
    T('4747223909107687084386231,5000000000000000', '949444781.8215374168772462', u);
    T('1731900848696547452986210971,5000000000000000', '346380169739.3094905972421942', u);
    T('122108665581,500000000000', '0.244217331162', u);
    T('-16865419490082861973881,200000000000', '-84327097450.414309869405', u);
    T('213918705805395433,2500000000000', '85567.4823221581732', u);
    T('-3281991687080891663567529581,1000000000000000', '-3281991687080.891663567529581', u);
    T('-9341191082726085332133,500000000000000000', '-18682.382165452170664266', u);
    T('-101666590217047,10000000000000', '-10.1666590217047', u);
    T('-78831482291567,1000000000000', '-78.8314822915670', u);
    T('0,1', '0.0', u);
    T('188098837362734027,50000000000000', '3761.97674725468054', u);
    T('7446591289029764406983,1000000000000', '7446591289.029764406983', u);
    T('0,1', '0.0', u);
    T('-237495928119,50000000000', '-4.74991856238', u);
    T('-5470021580414601267,62500000000', '-87520345.286633620272', u);
    T('20893480589331570804768497,100000000000000', '208934805893.3157080476849700', u);
    T('-427402721,10000', '-42740.27210', u);
    T('-418854065753,5000000', '-83770.81315060', u);
    T('41378721951369,5000000000', '8275.7443902738', u);
    T('11188608560931794787022677022582217,2000000000000000000', '5594304280465897.3935113385112911085', u);
    T('-46338373,50000', '-926.76746', u);
    T('-4074951117564368871187,100000000000000000', '-40749.51117564368871187', u);
    T('-1707089291648182063827,100000000000', '-17070892916.48182063827', u);
    T('-295010270384580559,100000000000000000', '-2.95010270384580559', u);
    T('475514217,12500000', '38.04113736', u);
    T('-456027,5000', '-91.2054', u);
    T('-7485101,5000000', '-1.4970202', u);
    T('-15596055181,100000000', '-155.96055181', u);
    T('-5451370075707939,12500000000000000', '-0.436109606056635120', u);
    T('0,1', '0.0', u);
    T('-6218012505973,2500000', '-2487205.0023892', u);
    T('92703,1000', '92.703', u);
    T('-58672783,62500000', '-0.938764528', u);
    T('26951106753311882431,1250000000000', '21560885.4026495059448', u);
    T('0,1', '0.0', u);
    T('72561,25000', '2.90244', u);
    T('-370800597281,500000000', '-741.601194562', u);
    T('161529914311,2000000000', '80.7649571555', u);
    T('0,1', '0.0', u);
    T('-7921733,1000000', '-7.921733', u);
    T('-108370284181334449019037137,1562500000000', '-69356981876054.04737218376768', u);
    T('-988987,1250', '-791.1896', u);
    T('-22885050541036899,2500000000000000', '-9.1540202164147596', u);
    T('-12248601426509,100000000', '-122486.01426509', u);
    T('-53006782802519,7812500000', '-6784.868198722432', u);
    T('-4098212714968971,1000000000000', '-4098.212714968971', u);
    T('-916,25', '-36.64', u);
    T('-1849979811,10000000000', '-0.18499798110', u);
    T('-1163263320065980213721068337,250000000000000000', '-4653053280.263920854884273348', u);
    T('-63,100', '-0.63', u);
    T('-74849063976673,10000000000000', '-7.4849063976673', u);
    T('-2627,50', '-52.54', u);
    T('-76247965138404535127,50000000000', '-1524959302.76809070254', u);
    T('-3226091329574799,5000000000000000', '-0.6452182659149598', u);
    T('-4774481559,625000000', '-7.6391704944', u);
    T('-3932199215959,10000000', '-393219.9215959', u);
    T('-6463072565956442868764219,10000000000000', '-646307256595.6442868764219', u);
    T('-1852637393024464507520764889,5000000000', '-370527478604892901.5041529778', u);
    T('3118876937753549,20000000', '155943846.887677450', u);
    T('-8469141306546569061,10000000000', '-846914130.6546569061', u);
    T('9573949,2500000', '3.8295796', u);
    T('54039659558876732100021006177771,10000000000000000000', '5403965955887.6732100021006177771', u);
    T('886406857838383,100000000', '8864068.57838383', u);
    T('-64180091,100000000', '-0.64180091', u);
    T('-6864846571598899651,20000000000000', '-343242.32857994498255', u);
    T('-2493931443,1000000', '-2493.931443', u);
    T('31828328526786083786647,500000000000000', '63656657.053572167573294', u);
    T('-3445415293,1000000', '-3445.415293', u);
    T('-5024242230618045468013,10000000000000000000', '-502.4242230618045468013', u);
    T('-9,5', '-1.8', u);
    T('8099,25000', '0.32396', u);
    T('-2055329312700326377528694739,500000000000', '-4110658625400652.755057389478', u);
    T('-30505199589,100000000', '-305.05199589', u);
    T('-1490759877124225470407,1000000000000', '-1490759877.124225470407', u);
    T('-4618629729980580667366682107,10000000000000', '-461862972998058.0667366682107', u);
    T('-1130319192716875137,625000000000', '-1808510.7083470002192', u);
    T('-1691412373887639934138807167,5000000000000000', '-338282474777.5279868277614334', u);
    T('-298703448990358147,2000000000000000000', '-0.1493517244951790735', u);
    T('-33,10', '-3.3', u);
    T('33370280255133778539,1000000000000000', '33370.280255133778539', u);
    T('-4702252998317805383728740913,100000000000', '-47022529983178053.83728740913', u);
    T('-6351,1000', '-6.351', u);
    T('-8284856781150816282422064531,100000000000000', '-82848567811508.16282422064531', u);
    T('-62846620813,12500000', '-5027.729665040', u);
    T('92502823,50000', '1850.05646', u);
    T('171431071233669112837,50000000000', '3428621424.673382256740', u);
    T('-79113158708464277609,1000000000000000', '-79113.158708464277609', u);
    T('-3588793696497311131,500000000000', '-7177587.3929946222620', u);
    T('624073,1000', '624.0730', u);
    T('-2911659018203,50000000', '-58233.18036406', u);
    T('-69,100', '-0.69', u);
    T('375641,10000', '37.5641', u);
    T('2578379695534414471,2000000000000000', '1289.1898477672072355', u);
    T('57798423261,20000000000', '2.88992116305', u);
    T('-96869430091,250000000000', '-0.387477720364', u);
    T('-54859400821,10000000', '-5485.9400821', u);
    T('-38775303,50000000', '-0.77550606', u);
    T('-460471243144930261,1250000000000000', '-368.3769945159442088', u);
    T('-3335175516487387,10000000000000000', '-0.33351755164873870', u);
    T('-174667238638380908738985811,20000000000000', '-8733361931919.04543694929055', u);
    T('828129,1000', '828.129', u);
    T('-117136909799345874651365657,2500000000000000000', '-46854763.9197383498605462628', u);
    T('1779595760369,2500000', '711838.3041476', u);
    T('-142341250379,250000000', '-569.365001516', u);
    T('986200911,2500000000', '0.3944803644', u);
    T('-182333559482602851520637,2000000000000', '-91166779741.3014257603185', u);
    T('-60885611373,10000000000', '-6.0885611373', u);
    T('-741206875556328299282294609,10000000000000000000', '-74120687.55563282992822946090', u);
    T('1156674883,12500', '92533.99064', u);
    T('-3627419,10000', '-362.7419', u);
    T('-733771061870739,1250000000000000', '-0.5870168494965912', u);
    T('-61116071,1000000000', '-0.061116071', u);
    T('354993212041625507168357968471,5000000000000000000', '70998642408.3251014336715936942', u);
    T('-34655833652810542670257,4000000000000000000', '-8663.95841320263566756425', u);
    T('7,10', '0.7', u);
    T('-47990699,100000', '-479.90699', u);
    T('-3488630351881,5000000', '-697726.0703762', u);
    T('123501469355491081,125000000', '988011754.843928648', u);
    T('-3,5', '-0.60', u);
    T('-605101,20000', '-30.25505', u);
    T('-13853,5000', '-2.7706', u);
    T('-251,125', '-2.008', u);
    T('-332431900771,10000000', '-33243.190077100', u);
    T('-83457932143284973896761,125000000000000000', '-667663.457146279791174088', u);
    T('-1599464683,1000000', '-1599.464683', u);
    T('107247619602970114431,125000000000', '857980956.8237609154480', u);
    T('106276234212253,200000000000', '531.381171061265', u);
    T('-78321709403185426102854771,100000000000000', '-783217094031.85426102854771', u);
    T('-1382454139,1000000', '-1382.454139', u);
    T('-22,5', '-4.4', u);
    T('-1551647648312410077781031687,100000000000000000000', '-15516476.48312410077781031687', u);
    T('-7,10', '-0.7', u);
    T('7610856965538431,1250000000', '6088685.5724307448', u);
    T('-8644394501214748574126749,1000000000000000', '-8644394501.214748574126749', u);
    T('-935485113175581,25000000000000', '-37.419404527023240', u);
    T('-1319,10000', '-0.1319', u);
    T('-47315292288529,5000000000', '-9463.05845770580', u);
    T('-111004268629921001873757,500000000000000', '-222008537.259842003747514', u);
    T('-13,25', '-0.52', u);
    T('197308716353,200000000000', '0.986543581765', u);
    T('-21514078956751937905331,2000000000000000000', '-10757.0394783759689526655', u);
    T('-56583509,100000', '-565.83509', u);
    T('-310660495303969,5000000', '-62132099.06079380', u);
    T('-4588911263109638518958719353,500000000000000', '-9177822526219.277037917438706', u);
    T('638787883838178844837013,10000000000000', '63878788383.81788448370130', u);
    T('-90023153,125000', '-720.1852240', u);
    T('-24719448065802851827137557,10000000000000000000', '-2471944.8065802851827137557', u);
    T('-1692163664077,5000000000', '-338.43273281540', u);
    T('-3,10', '-0.3', u);
    T('-37969027,10000000', '-3.7969027', u);
    T('0,1', '0.0', u);
    T('-729239769116052893303137,1000000000000', '-729239769116.052893303137', u);
    T('-248567011649,5000000', '-49713.4023298', u);
    T('0,1', '0.0', u);
    T('33976492623521690540454280357369,10000000000000000000', '3397649262352.16905404542803573690', u);
    T('-324129431756502447158554419,500000000', '-648258863513004894.3171088380', u);
    T('-185535293743,250000', '-742141.174972', u);
    T('-9511387,10000', '-951.1387', u);
    T('-48277609,10000', '-4827.76090', u);
    T('-5305507073516086201,1000000000', '-5305507073.5160862010', u);
    T('4220388699670847,5000000000', '844077.73993416940', u);
    T('-65603972997,10000000000', '-6.5603972997', u);
    T('26,5', '5.2', u);
    T('-499,1000', '-0.499', u);
    T('-997797821500381,12500000000', '-79823.82572003048', u);
    T('-35613001,50000000', '-0.71226002', u);
    T('-34818109982526308537,100000000000000', '-348181.09982526308537', u);
    T('-7249385948502190907,5000000000000000', '-1449.87718970043818140', u);
    T('-4,5', '-0.8', u);
    T('27901580717,1000000', '27901.580717', u);
    T('-37138444464274424296356197,100000000000000000', '-371384444.64274424296356197', u);
    T('-24952519,10000000', '-2.4952519', u);
    T('-1183871,10000', '-118.3871', u);
    T('-323836725046470278148447,100000000000000000', '-3238367.25046470278148447', u);
    T('-458296522853,1000000', '-458296.522853', u);
    T('-9896488459499919018798992969,10000000000000000', '-989648845949.9919018798992969', u);
    T('-800498844471573993218284031,100000000000000000000', '-8004988.44471573993218284031', u);
    T('19,100', '0.19', u);
    T('-8819541616157661,1250000000000000', '-7.0556332929261288', u);
    T('2464295177881259,100000000', '24642951.77881259', u);
    T('-240160051143,25000000', '-9606.40204572', u);
    T('-27,50', '-0.54', u);
    T('-670912890009,500000000000', '-1.341825780018', u);
    T('-41735377076387,100000000000', '-417.35377076387', u);
    T('-18351814413232593,25000000000', '-734072.57652930372', u);
    T('-1,10', '-0.1', u);
    T('-120533530679991899967,50000000000000', '-2410670.61359983799934', u);
    T('-341521322123,1000000', '-341521.322123', u);
    T('4469363,5000000', '0.8938726', u);
    T('-2300185559,500000', '-4600.371118', u);
    T('-9,5', '-1.8', u);
    T('-19061913987,312500000', '-60.9981247584', u);
    T('216663475511333,1000000000', '216663.475511333', u);
    T('-1659477,100000', '-16.59477', u);
    T('-989884121025000977,200000000000', '-4949420.60512500488500', u);
    T('-1913,100', '-19.13', u);
    T('-425227820481,100000000000', '-4.25227820481', u);
    T('-8376041,1000000', '-8.376041', u);
    T('-2990304956561539891971,50000000000', '-59806099131.23079783942', u);
    T('0,1', '0.0', u);
    T('-46410885173793408759388939,500000000000000', '-92821770347.586817518777878', u);
    T('-9499781269249,100000000', '-94997.81269249', u);
    T('-1597534101,25000', '-63901.36404', u);
    T('811,100', '8.11', u);
    T('-4789,400000', '-0.0119725', u);
    T('4757,50', '95.14', u);
    T('-733,100', '-7.330', u);
    T('-2546590947,5000000000', '-0.5093181894', u);
    T('-79318790998372451,10000000000', '-7931879.0998372451', u);
    T('-97970835166736806859507401,12500000000', '-7837666813338944.548760592080', u);
    T('-729119841,1000000', '-729.119841', u);
    T('-4324730954076108671,5000000000', '-864946190.8152217342', u);
    T('-17759,20000', '-0.88795', u);
    T('223387,500000', '0.446774', u);
    T('-3495739634217069,1000000000000000', '-3.495739634217069', u);
    T('2080645759193349226042432238271,25000000000000000', '83225830367733.96904169728953084', u);
    T('-567002479,125000000', '-4.536019832', u);
    T('-3271614601090267536275654641,500000000000', '-6543229202180535.072551309282', u);
    T('-554616993923712942818718997,20000000000000000', '-27730849696.18564714093594985', u);
    T('-112053,250', '-448.212', u);
    T('-607248757361461263304116301,2500000000', '-242899502944584505.3216465204', u);
    T('-18331,50', '-366.620', u);
    T('-24317134245313276003253,250000000000000', '-97268536.981253104013012', u);
    T('-1061159932891264014512925703,20000000000000', '-53057996644563.20072564628515', u);
    T('36731261,12500000', '2.93850088', u);
    T('318736713583604277266673749,5000000000000000000', '63747342.7167208554533347498', u);
    T('-74227605737234405747389,100000000000', '-742276057372.344057473890', u);
    T('0,1', '0.0', u);
    T('-102451481711,250000000000', '-0.4098059268440', u);
    T('3368162492642619086982967909376363443,10000000000000000000', '336816249264261908.6982967909376363443', u);
    T('-17663710859989775919,1000000000000000000', '-17.663710859989775919', u);
    T('-49264796837,10000000000', '-4.9264796837', u);
    T('1381043641897743,500000000000', '2762.087283795486', u);
    T('-9849,10000', '-0.9849', u);
    T('-77,500', '-0.154', u);
    T('-264191,6250', '-42.27056', u);
    T('-8027691,100000', '-80.27691', u);
    T('-18434760024216076621,25000000000', '-737390400.96864306484', u);
    T('-535677556431210667261,1000000000000000', '-535677.556431210667261', u);
    T('-8080033681,10000000', '-808.0033681', u);
    T('-2793157211,250000', '-11172.628844', u);

    T('5,1', '5.1582612935891', '3');
    T('14645,1797', '8.14969395596340', '4682');
    T('350921,81990', '4.28004634702', '82418');
    T('923754598113149102531799372705758450549,100000000', '9237545981131491025317993727057.58450549');
    T('23665467152910525354658972818814752082,49447', '478602688796297558085606261629921.9787246948249979233505266225354557384', '57466');
    T('12401124714135748113774134798397,2', '6200562357067874056887067399198.66971421701', '2');
    T('5747456325233732815460960864983211610837351385715799969377,100000000000000000000000000000', '57474563252337328154609608649.83211610837351385715799969377');
    T('4744,61', '77.7705294446793765233206385561288346447', '86');
    T('230406626515463254737498487452583023685806305331639,100000000000000000000000000', '2304066265154632547374984.87452583023685806305331639');
    T('8452816572418429518944119647,5000', '1690563314483685903788823.9294');
    T('48239499471878294436129777581,10000000000000000', '4823949947187.82944361297775810');
    T('45172634468362586403517152343110493066053,74300', '607976237797612199239800166125309462.530995962577562', '89241');
    T('245629009658126699398365025147112883705717,1250000000000000000000000000000', '196503207726.5013595186920201176903069645736');
    T('19575206081236584421562241755643406299,500000000000000000000000000000000000', '39.150412162473168843124483511286812598');
    T('124335953351113119989680198250484096210027176457861877,20000000000000000000000', '6216797667555655999484009912524.20481050135882289309385');
    T('62359286968916651021817694022163661098254437249041482587,100000000000000000000000000', '623592869689166510218176940221.63661098254437249041482587');
    T('1406253392003122107350210973,50', '28125067840062442147004219.46', '90');
    T('736337826,1', '736337826.0', '1');
    T('2294354321,3019', '759971.62007285863174441704674479534091075', '5946');
    T('342555042697121035185957633789368546657896318423873,50000000000000000000000000', '6851100853942420703719152.67578737093315792636847746');
    T('3511582655801640591,52', '67530435688493088.2885316751682999639624370547426320', '73');
    T('697806103903081,1', '697806103903081.3297023089715266281593302923590', '1');
    T('504239849856718524776121635107896338763895,26083', '19332126283660565302155489595057943440.70448184108567544983900489517214731', '26482');
    T('3513961,10', '351396.0988885676988110229927700324193', '11');
    T('1514998954038665195379885,15263', '99259578984384799540.056673', '55079');
    T('1190687310674838338273481559,50000', '23813746213496766765469.63118');
    T('31747884008318370382077439489,40', '793697100207959259551935987.225181146039', '45');
    T('39451289109544785156061546711681,59993', '657598204949657212609163514.271348320970259397527780637349273932', '61325');
    T('3859539118592118171061,65', '59377524901417202631.7076617756627530296590420377861', '248');
    T('2529860848590369124121129138,36761', '68819152052184900414056.449443703377131320899467509', '59208');
    T('17331073,2', '8665536.3781112033855216846406980971689', '2');
    T('11088761069693879,25000', '443550442787.75516');
    T('37298162459148947522489,474', '78688106453900733169.80801275320205353359142143262532', '517');
    T('12411782390212011049587005051032466579057,100000000000000000000000000000000000000', '124.11782390212011049587005051032466579057');
    T('288377889236302421024476469650301879883847,50000000000000000000000', '5767557784726048420.48952939300603759767694');
    T('64269805556832147,1', '64269805556832146.8921574533060466484388410550211', '2');
    T('1019428223041680221,2', '509714111520840110.44623631764611004359625363775303233110', '2');
    T('15446409508173177876456900755482517,250000', '61785638032692711505827603021.9300680');
    T('794012664285496747924956753856029102507539,10000000', '79401266428549674792495675385602910.2507539');
    T('1254937597799095507639361,5', '250987519559819101527872.2');
    T('6200289913593672418541428454415821,29', '213803100468747324777290636359166.2447131068603257501', '29');
    T('28363420172665408049802993720278,1', '28363420172665408049802993720278.0', '53164');
    T('614972069955669535218947919309413188717428438,76097', '8081423314397013485668921499000133891183.99461213951158548435230604703', '85075');
    T('456397843623966327008858405616124567381,62500000000000000000', '7302365497983461232.141734489857993078096');
    T('18578778016377890885386246496020651,1000000000000000000000000', '18578778016.377890885386246496020651');
    T('22738060524436955,1', '22738060524436954.73553568', '1');
    T('631381816,2355', '268102.68195329664239686517327023024865590172', '20904');
    T('146503259847606731525818341,125000000000000000000000000', '1.172026078780853852206546728');
    T('14904502285493,500000', '29809004.570986');
    T('46687620560206903024816575649873835,3349', '13940764574561631240614086488466.35861452906998315', '6745');
    T('91662449024242718476521823,100', '916624490242427184765218.23');
    T('47716161226447671719905908024708011827667,1000000000000000000000000000000000000', '47716.161226447671719905908024708011827667');
    T('720670502490670418796980571060493068249181,5000', '144134100498134083759396114212098613649.8362');
    T('37873375501866534444667697433,2000000000000000000000000', '18936.6877509332672223338487165');
    T('9837105973,284029', '34634.1605012164224956', '586619');
    T('275962770820946621106879920859,1250000000000000000000000000000', '0.2207702166567572968855039366872');
    T('706513249010161459257,1', '706513249010161459256.864532197276', '1');
    T('595980268265874363487011727978412247,781189', '762914311729779046411318807584.86390233349524542', '1162874');
    T('94990393655983362634,1', '94990393655983362633.93540186786386734642706043938', '2');
    T('478995160185850544676215707045658234,1', '478995160185850544676215707045658234.2072935316895051189693', '1');
    T('900791612975363289,100000', '9007916129753.632890');
    T('3640161366680621782,5508865835', '660782359.87401239405933', '31221561655');
    T('153304,2409', '63.63802393447707212496311847412489674893', '2919');
    T('48894805199,100000', '488948.05199');
    T('764607208068,1', '764607208068.0', '9883952909');
    T('5247038043357340126406507,10000000000000000000', '524703.8043357340126406507');
    T('40752843984968450495935900024466301,100000', '407528439849684504959359000244.66301');
    T('1508393360078152986207294205073063553774739,10000000000000000000000000000000', '150839336007.8152986207294205073063553774739');
    T('15049415517,2500000', '6019.7662068', '25809809256');
    T('4679780279632085707608834877135150316333,500000000000000000000', '9359560559264171415.217669754270300632666');
    T('175108726296810783549,699340', '250391406607388.0852646781250551', '1790242');
    T('44702883860465549,10', '4470288386046554.9');
    T('7808104088649916822858776184466,1', '7808104088649916822858776184466.47052438758636053342', '1');
    T('281411919337,465805657858', '0.6041401914933113740581694868115147', '980851387140');
    T('695754534752369598767,100000', '6957545347523695.98767');
    T('14331404543988591592392002000575758391172,21068409', '680231931323745974002688195419775.569724890', '37960275');
    T('3320644402459621722827,625', '5313031043935394756.5232', '6613572437131');
    T('627120338836675613788452534272072859,1000000000000000000000000000000', '627120.338836675613788452534272072859');
    T('21062775182251,383134912', '54974.8261579748702201285', '452907837');
    T('18956932146202765099219634795464927,10000000', '1895693214620276509921963479.5464927');
    T('2600986773742994608412878821259213,10', '260098677374299460841287882125921.3');
    T('566726653824685741382851080165717726527122873,100000000000000000000000', '5667266538246857413828.51080165717726527122873');
    T('497588039427229066346312712229773035708653408397,10000000000000000000000', '49758803942722906634631271.2229773035708653408397');
    T('41666367526388418811711752302612627697101,2000', '20833183763194209405855876151306313848.5505');
    T('15363941484953032312420486,10851', '1415900975481801890371.43913003328099204929551975491363327', '49992');
    T('2409694437794006212469,622301', '3872232951247075.309968970', '5615242');
    T('39667592557377399116572645358849947597,100000000000', '396675925573773991165726453.58849947597');
    T('6631914348494357725061167,1000000000000000000000000', '6.631914348494357725061167');
    T('190389016168952767733346161974389684092039842699,50000000000', '3807780323379055354666923239487793681.84079685398');
    T('869903981761555555786475820853808311250191616816131,1000000000000000000', '869903981761555555786475820853808.311250191616816131');
    T('371292976710505868722177667462589039897488370705001117485935571,100000000000000000000000000000000', '3712929767105058687221776674625.89039897488370705001117485935571');
    T('46257442183069435630977238493,1000000000000000000', '46257442183.069435630977238493');
    T('440475560214540234869187803670278339051473119,1000000000000000000000000', '440475560214540234869.187803670278339051473119');
    T('1617847693417,5000', '323569538.6834');
    T('182074175724591638134600468349034330834991283,50000000000000000000000000000', '3641483514491832.762692009366980686616699825660');
    T('7589333034185993241367598309953157841,1000000', '7589333034185993241367598309953.157841');
    T('439885064966692635331338940060375403,74133607290968', '5933679488173046564651.89839254336877899980063174325', '77141789811679');
    T('1179158450722067213186968484863836861457157,100000000000000000000000000000000', '11791584507.22067213186968484863836861457157');
    T('12935263022381831236481986920256744104451045320357,50000000000000', '258705260447636624729639738405134882.08902090640714000');
    T('2395282895205309,25123683202843', '95.33963933020212599179070033', '38426420751904');
    T('17850307,215', '83024.683723', '963');
    T('32544916036253,50', '650898320725.06', '5793653961');
    T('60773532817161262237,66672547', '911522591407.24385761053946236159069413709389', '338084891');
    T('914770176848677528292843944387964798627453476861720317700876791,10000000000000000000000000000000000000000', '91477017684867752829284.3944387964798627453476861720317700876791');
    T('4587186380417513404538237108175442069452907,50000000000000000000000', '91743727608350268090.76474216350884138905814');
    T('8531233654030763235701612513100153212942822881,100000000000000000000000000', '85312336540307632357.01612513100153212942822881');
    T('107773450530064909662187359086117316399733950067153,500000000000000000000000000000000000000', '215546901060.129819324374718172234632799467900134306');
    T('769619176587614353053,1250000000000000', '615695.3412700914824424');
    T('12281666497073480840695662,211', '58206950223097065595714.0379184486359', '683');
    T('457196542212636540687669731893020947887,6883500', '66419196951062183582141313560401.0965187767850813634641', '9593745');
    T('4268227996137613294377196928631180903821983,40735441', '104779226426875145266678147135590870.4614731678', '830069295');
    T('575458315285885743858391,24', '23977429803578572660766.29105216999286300600008393459353040482', '52');
    T('8436494536336345240276183670676635057499,38398604', '219708365864976373627441864049970.0212408503184125498085096963579394842', '68059426');
    T('437912761,10', '43791276.1');
    T('18901384502804686897816143307188176228147,2500000000000000000000000000', '7560553801121.8747591264573228752704912588');
    T('25477273836466129096074485,6558', '3884915193117738501993.66956453300', '6559');
    T('7590779151577004043224428413071780976089,2000000000000000000000000000000000000', '3795.3895757885020216122142065358904880445');
    T('6056953730608206494408538120559,10', '605695373060820649440853812055.9', '5171809704');
    T('31849179082257267370211,88449', '360085236489471530.14970208743', '92072');
    T('178285814692139084253431718197878271715779878432662563,10000000000000000000000000000000000000000', '17828581469213.9084253431718197878271715779878432662563');
    T('1449138221474956614898,2029', '714213021919643477.03203547420265509578', '3820');
    T('183,821', '0.222897429282815755169043594043902608174', '974');
    T('433492537953390549272160099280439360005070313,5000000000000000000000000', '86698507590678109854.4320198560878720010140626');
    T('3862824701391243701393001605164434946489851,148867231', '25948119511884006906751705519157771.8436225296620181042509021019411', '552982978');
    T('2589933131557013289797231668,47', '55104960245893899782919822.723478083911298417430269312', '80');
    T('10890199899142835887,2137', '5096022414198800.13430078158019717882', '2446');
    T('3293945324962080603125,382', '8622893520843142940.11780071049874178362583458818054294', '421');
    T('436594379,1000000', '436.594379');
    T('168301463949156607668854677190355669536114773797741179468731574716495111,25000000000000000000000000000000000', '6732058557966264306754187087614226781.44459095190964717874926298865980444');
    T('24609496918349161226843361015480093061391925879787537781518362891719192377,5000000000000000000000000000000000000000', '4921899383669832245368672203096018.6122783851759575075563036725783438384754');
    T('101528584171956890122633497199277785333799099,100000000000', '1015285841719568901226334971992777.85333799099');
    T('21735,239', '90.941418', '302');
    T('118224122484187225351526044435841806374099,2693763154', '43888091018185790119960206581636.911933980295284713291', '2834067906');
    T('1,1', '0.775084028626235358553642', '1');
    T('19134122533504389120891,25000000', '765364901340175.56483564');
    T('342810849655223057065770739830569,10000000000000000000', '34281084965522.30570657707398305690');
    T('6725614005586035111245829205493452074108331,1000000000000000', '6725614005586035111245829205.493452074108331');
    T('1376371700067530025202571904028,1409', '976842938302008534565345567.088715501208', '2249');
    T('76801072988978594413391998963,10000000000000', '7680107298897859.44133919989630');
    T('12704762005671512260146338889,20', '635238100283575613007316944.45');
    T('2492573677322062578671959051439,108734', '22923590388673851588941444.731537513573028882768807925690476231785', '231654');
    T('31123084257678338385173981570213479239607677464746977165210081,12500000000000000000000', '2489846740614267070813918525617078339168.61419717975817321680648');
    T('3786296291638531133538327635741,100', '37862962916385311335383276357.41');
    T('586924625297711213018263802128176897964674915689,1000000000000000000000000000000000', '586924625297711.2130182638021281768979646749156890');
    T('9645358,1', '9645357.912453778903438267262181232435591944111', '4');
    T('5316031065626771333799,10', '531603106562677133379.9', '19');
    T('67278211304547861090569521693426588943,79', '851622927905669127728728122701602391.68360228044574465611893', '159');
    T('7021540690227758957176483359885192690352067153,81703854', '85938916543982845132084997604705314027.8066583370718350934', '97264150');
    T('60669129292299639447664369918689547710597272939435803927977438559,25000000000000000000000000000000000000', '2426765171691985577906574796.74758190842389091757743215711909754236');
    T('6288763899326986566293639459006397625,120589', '52150394308991587676269306976642.957691', '210871');
    T('615,1', '615.0', '56760255');
    T('17799,19', '936.7892332671682401846243', '97');
    T('1303385080467541077973,1000000', '1303385080467541.077973');
    T('21448859436681680760040806307936789975345193025273495023,500000000000000000000000000000', '42897718873363361520081612.615873579950690386050546990046');
    T('1291898345053434859844591463516855305911784995979,12500000000000000000', '103351867604274788787567317081.34842447294279967832');
    T('60507,1', '60506.9', '5');
    T('616317355277164831416386217627324701901,1000', '616317355277164831416386217627324701.901');
    T('2904921181742034,1', '2904921181742033.970138508789748538132238870774712', '1');
    T('3440569493092390376625126875391799,50000000000000000000', '68811389861847.80753250253750783598');
    T('15571280591820679565505108997031,5', '3114256118364135913101021799406.195951', '8');
    T('392321264085832586910672967559,611830431', '641225483741642440322.9347831490911899434269', '695282372');
    T('5577690959101259996527703333121313876972011192208077,625000000000000000000000000000000', '8924305534562015994.4443253329941022031552179075329232');
    T('16539986799609190872098179009446893,331524797', '49890647545164444734124002.82518502831629815787962712124518347', '544553707');
    T('8981681843839458445912808176057,20203', '444571689543110352220601305.55150225273', '22042');
    T('99325333489133639887283,11', '9029575771739421807934.81761', '13');
    T('6522,1', '6522.0');
    T('7187454479696854214600838,2125352003', '3381771334607886.2281528797655830003743', '3307131440');
    T('28117786498578019978223647606249,72838872', '386027209462799203950105.756803166858487810', '81837562');
    T('478254578956578786562047524831361309532201229,10000000000000000000000', '47825457895657878656204.752483136130953220122900');
    T('77058140608147483002424280179365952862962922218441255276651,1000000000000000000000000000000000', '77058140608147483002424280.179365952862962922218441255276651');
    T('803865379172081337096308068958767613527926677,327563', '2454078693784344804194332293203956532111.1562569643063947120941944927', '375003');
    T('2360034347428712012050231582155883,250000000000000000000000', '9440137389.714848048200926328623532');
    T('203318509580360624351812461951622123031368062580663,250000000000000000000', '813274038321442497407249847806.488492125472250322652');
    T('221965264758349084399,6470046354', '34306595751.2225088455680019382813053906996309752644', '7317562156');
    T('5099265696964671,12500000', '407941255.75717368');
    T('321755130,1', '321755130.0', '1');
    T('681406688936672531143794851254005123313841616787245936785722503,7812500000000000000000000', '87220056183894083986405740960512655784.171726948767479908572480384');
    T('153955479454984280174500893,4927686668', '31242952287278887.5492886539595216', '9598118777');
    T('23227,100000', '0.23227');
    T('43326473463554790328447035534276163421779557,1000000000000000000000000000000000000', '43326473.463554790328447035534276163421779557');
    T('41089006913283492613083959033772531785,2', '20544503456641746306541979516886265892.530283312918885713244232199421', '5');
    T('139407109288420981406975879448852798787333071448546598243307,1000000000000000000000000000000', '139407109288420981406975879448.852798787333071448546598243307');
    T('9781691985049606719084834150405960550336089,122674', '79737287322901403060834685022139659180.7236170663710128362032', '57088498');
    T('14022980999863417630135058657283616,2321', '6041784144706341072871632338338.481688949941106013243041338', '8154');
    T('5574653114539338322522057935820934,1', '5574653114539338322522057935820933.952578779342397522948776164237003865', '2');
    T('698597423090735413329970853695704980420,1', '698597423090735413329970853695704980420.0');
    T('41770388588086435767794101997,1', '41770388588086435767794101996.87800912334551920117198', '2');
    T('3969460977515807861628298607881363600942403072746116154451251370411,500000000000000000000000000000000000', '7938921955031615723256597215762.727201884806145492232308902502740822');
    T('818626449980780390596498633228169970328017564,951613', '860251436225419777363800865717649895837.927354922641714029', '2934871');
    T('2864942859671509919660119,10000000', '286494285967150991.96601190');
    T('145096137530603,3125', '46430764009.79295997627678788068274740859435028566', '3415');
    T('3175128188,8421', '377048.82888017296266613549723116278', '9892');
    T('353211267011180153249,10000000', '35321126701118.0153249');
    T('897447070318612896975557401374492140028592343673,10000000000', '89744707031861289697555740137449214002.8592343673');
    T('17536546982566421053425616271159751448175509509,5000000000000000', '3507309396513284210685123254231.9502896351019018');
    T('1498867800352844887470319343495958,17', '88168694138402640439430549617409.293', '22');
    T('14305398689362,15467073', '924893.7203155374', '22113082');
    T('436102839806788235096903,4', '109025709951697058774225.7362272869251737724', '9');
    T('308087699066,7807', '39463007.437684133317', '10839');
    T('2628342736773923291423096554670735302124820185693541213399,50000000000000000000', '52566854735478465828461931093414706042.49640371387082426798');
    T('64300989158077788880534179354276244873359,2195', '29294300299807648692726277610148630921.8036447', '3984');
    T('1593668215027228291921575913818,1', '1593668215027228291921575913817.9011053', '2');
    T('9464555700726947268792864782,1', '9464555700726947268792864781.94741904335358', '2');
    T('86501031784980810599118109920903,367173188', '235586460591400292003669.151138843504008795198121680193', '457966587');
    T('9745513313209888804669228133888956813799941097,10000000000000000', '974551331320988880466922813388.8956813799941097');
    T('31556786915280245846977588242899282449,10000000000000000', '3155678691528024584697.7588242899282449');
    T('472322148083391454920144185678364119,100000000000', '4723221480833914549201441.85678364119');
    T('518007801929481005201363369,625000000', '828812483087169608.3221813904');
    T('299535536454040965333171155703500161446096189553447,500000000000000000000000000000', '599071072908081930666.342311407000322892192379106894');
    T('5330869452533820313561195628056700122127092795967358371,250000000000000000000000000000000000', '21323477810135281254.244782512226800488508371183869433484');
    T('47795736745489295393396455460987004248591698840612598072778499,100000000000000000000000000000000000000', '477957367454892953933964.55460987004248591698840612598072778499');
    T('206470771234730604991708607179250628073439743340221,1000000000000', '206470771234730604991708607179250628073.439743340221');
    T('907,1', '907.239004086069', '1');
    T('19077064083517899785847151076434612550555316546848569573481919,500000000000000000000000000000000000000', '38154128167035799571694.302152869225101110633093697139146963838');
    T('88133212648357648517,100', '881332126483576485.17', '87089410');
    T('92145848597976720763683816246415023,100000', '921458485979767207636838162464.15023', '521344865');
    T('68539971794,6880585', '9961.358197595117284', '7875099');
    T('935168426589766229927456282413,1000000000000000000000', '935168426.589766229927456282413');
    T('97,1', '97.0');
    T('12846464199528950327,5000000000', '2569292839.9057900654');
    T('2196110114554495743126928176989085775661,247228124', '8882929979901864818287939510429.9936995072615605828834737982875357', '360433698');
    T('388166039,1', '388166039.0');
    T('341539978,1', '341539978.0', '195382934');
    T('781034514540009,100000000', '7810345.145400090');
    T('20877130490298631791495412013,5000000', '4175426098059726358299.0824026', '2413029349');
    T('620026697301098193277327524609895859849707,1000000000000', '620026697301098193277327524609.895859849707');
    T('899179602300830741337813,100000', '8991796023008307413.37813');
    T('122432635,1316', '93033.9171732715553792824649171231580', '8140');
    T('119126340085680938295485816698905,46045039', '2587169924770417466591478.32840156786489', '73781454');
    T('40751167179588164798660481,1000000', '40751167179588164798.660481');
    T('68346625258654521574,1', '68346625258654521574.0');
    T('750578575419957257,100000', '7505785754199.57257');
    T('1082551,2000000', '0.5412755');
    T('638082,1', '638082.0');
    T('142488629984756867662276702019849226999,23', '6195157825424211637490291392167357695.608', '50');
    T('738115949031,1', '738115949031.49784834425903489305746005970674', '1');
    T('762953733493512712065532091292988047,1000', '762953733493512712065532091292988.047');
    T('178822248219150690173,50', '3576444964383013803.46');
    T('1984025728,43281', '45840.570411959105408010583324095855', '43320');
    T('20761534807588043,10000', '2076153480758.8043');
    T('924518404356149235469334759956600273757163609,20000000000000', '46225920217807461773466737997830.01368785818045');
    T('129411352783937937234426479,4112', '31471632486366229872185.427772374363428597742550717671565', '6699');
    T('3411360513380150634213140127292339282533595063273539901300312481,400000000000000000000000', '8528401283450376585532850318230848206333.9876581838497532507812025');
    T('76,1', '75.994673795016', '1');
    T('494578703756754347,5770935', '85701659047.754713404327025764320339913', '7644022');
    T('6890929,100', '68909.29');
    T('1166392938478006378987508440139351590646756345929819,100000000000000', '11663929384780063789875084401393515906.46756345929819');
    T('313075076294541347576328789584368750721345183624469,100000000000000000000000000000000000000', '3130750762945.41347576328789584368750721345183624469');
    T('2662023247705491854869433606657653587,2000000000000000000000', '1331011623852745.9274347168033288267935');
    T('806080010264194,4727', '170526763330.694732378488764174454796737', '7478');
    T('65545519,78105263', '0.83919465196602580801751582896749717', '79437231');
    T('6248476043461963589910078747895049611946214109708513753,1000000000000000000000000000000000', '6248476043461963589910.078747895049611946214109708513753');
    T('54531559010653599,3030958223', '17991524.461422310735689741', '7731481918');
    T('429099569700734470147551061212192591773651008518,670838555', '639646553559722681931349430582731774821.2176154931941859', '704304148');
    T('844049999410175821526,1', '844049999410175821525.66', '1');
    T('60701048676262579463126309740310059202124189359,100000000000000000000000000000000000', '607010486762.62579463126309740310059202124189359');
    T('5279410855406353186734225099558276,77', '68563777342939651775769157137120.467524470763581', '95');
    T('842239448004271461702793995095761523551628413356036645106333,250000000000000000000', '3368957792017085846811175980383046094206.513653424146580425332');
    T('1182610690103,312500000000', '3.7843542083296');
    T('68524947508627870390245103423700023635275808281519987514918614853613,100000000000000000000000000000000000', '685249475086278703902451034237000.23635275808281519987514918614853613');
    T('98142576359019755098761,10000000000000', '9814257635.9019755098761');
    T('241800962281991517205884817297371,5000000000000000000000000', '48360192.4563983034411769634594742');
    T('79836993918410993594506591,1000', '79836993918410993594506.591');
    T('1973810204163989,4534064777', '435329.0703248347084653925795494', '5332316364');
    T('401580056542974086016353741391922659742781536879698656843919,50000000000000000000000000000', '8031601130859481720327074827838.45319485563073759397313687838');
    T('590663,750977', '0.7865260853527516321730', '780179');
    T('280079,1000', '280.079000');
    T('8287753156729,1000000', '8287753.156729');
    T('0,1', '0.123545182349443609735032', '1');
    T('57057377258668955092844388184203953190814195154676204418983523542264563023,100000000000000000000000000000000000', '570573772586689550928443881842039531908.14195154676204418983523542264563023');
    T('147604349147007341987769096030122333191422929,6471015', '22810076803562863320169880000297068263.853959386587721161799958796244241051', '8303141');
    T('6105498122564848444576189774,80267', '76064860061604998873462.192108836736339050132605', '141567');
    T('670639675445600472707497548714250172221625881188854267533121,100000000000000000000000000000000', '6706396754456004727074975487.14250172221625881188854267533121');
    T('478196505300332351297,1000000000000000', '478196.505300332351297');
    T('5375051998674067089293,50000000', '107501039973481.34178586');
    T('66584033541591402334601184478033648516971762649454147,1000000000000000000000000', '66584033541591402334601184478.033648516971762649454147');
    T('6689748165117193396726141393,1250000000000000000000000000', '5.3517985320937547173809131144');
    T('460631023076,645385145', '713.7304393270471076595464454721208755', '940592764');
    T('5276069320082913776121941721,56', '94215523572909174573606102.16045907376298740945874743718869112', '78');
    T('27476132817352,13', '2113548678257.84612197961789955018136', '856');
    T('202602298856965754329,41868', '4839072772928388.13244005', '42848');
    T('4464608676391498186364792490985185921,5000000000', '892921735278299637272958498.1970371842');
    T('5031980754741001166,1', '5031980754741001166.0');
    T('8023557799983,1294504', '6198171.500422555666059971291979899', '4419145');
    T('168103197472408099843,2', '84051598736204049921.4325681106454432215021', '4');
    T('22889257701907256349217661636051,500000', '45778515403814512698435323.272102');
    T('892403,100000', '8.92403');
    T('1914024621531154563928585,31', '61742729726811437546083.38708655997995899644906913601366359', '352');
    T('1895934054286044231883939226027166734,19', '99786002857160222730733643475114038.631477380741702788316870213097972253591', '65');
    T('488907941212895608156694986135107572664818936553,809421353', '604021551199299295180189033306992053945.5041242938398241170607073', '811259873');
    T('9558758078456554336102679663,10000000000000000', '955875807845.6554336102679663');
    T('2281310565857,50362', '45298251.972856518761138181026483534977097', '82487');
    T('2890669262797093295990028537,799', '3617858902124021647046343.6007512810038540029262532383', '994');
    T('7463509030548608549549784845908192540652431,1000000000000000000000000000000000000', '7463509.030548608549549784845908192540652431');
    T('2506595022495145502116175,2852121886', '878852700790623.04917082880938279787754319212190899', '5003324771');
    T('12700439554099,50', '254008791081.98', '925168108');
    T('1628787692156684658130881,10000000000000000000000000', '0.1628787692156684658130881');
    T('44847136057058496707418799843,50000000000000000', '896942721141.16993414837599686');
    T('21592764541,25000000', '863.71058164');
    T('47801148603152505431027216366549,100000000000000000000000', '478011486.03152505431027216366549');
    T('11423000733021287947376,5', '2284600146604257589475.18575178', '5');
    T('2732203040520419846369019510610,1', '2732203040520419846369019510609.688387071616656538889144285', '1');
    T('568956,96605', '5.889508824590992627401001', '840534');
    T('71266649783967665510339357353436865700959295147,1000000000000000000000000', '71266649783967665510339.357353436865700959295147');
    T('47566615560121136721952891543681047713870191381209033,500000000000000000000000000000', '95133231120242273443905.783087362095427740382762418066');
    T('46719848692300081220895684593,79867', '584970622313346954573173.959119536234', '447174');
    T('6512959508348312854935682614687474683779,45270', '143869218209593833773706264958857404.1037994256595241153200438542903873871', '89045');
    T('5439094235028724828058147496706,1', '5439094235028724828058147496706.08739593', '5');
    T('6914775066832852797746427420897025001305226852,9731571', '710550749394198819260161326562486673662.990985936391792772356968236196619', '9877188');
    T('9132456017491382812010830613993084921418431583790758069448731138967157,10000000000000000000000000000000000000000', '913245601749138281201083061399.3084921418431583790758069448731138967157');
    T('167000488634822153427980339,200', '835002443174110767139901.6950', '209');
    T('141817570803201010469485291685322472531441481,200000', '709087854016005052347426458426612362657.2074050');
    T('588070510408392162256205164465497,78125', '7527302533227419676879426105.1583616', '5624320');
    T('19513055373531609310745811785217,200', '97565276867658046553729058926.085', '662');
    T('138474825950618076159440491355711,250000000000000000000', '553899303802.472304637761965422844');
    T('415063278182543292471143901597284671215994230954413,5000000000000000000000000000000000000', '83012655636508.6584942287803194569342431988461908826');
    T('36464063274630813862074158403502015663924,4479', '8141117051714850158980611387252068690.315695463672930753117541745392976362780', '9949');
    T('77574321944149904065550232031471931458854752374018266928805594322125477,1000000000000000000000000000000000000', '77574321944149904065550232031471931.458854752374018266928805594322125477');
    T('860630077,2', '430315038.403217142', '2');
    T('15558187771473513998987651232806625351751827119199,10000000000', '1555818777147351399898765123280662535175.1827119199');
    T('3231091733918849478252448666476580370445734299,10000000000000000000', '323109173391884947825244866.6476580370445734299');
    T('11956064398957062636192043300176336503346154370545469406163,25000000000000000000', '478242575958282505447681732007053460133.84617482181877624652');
    T('13021206583561217782,1393', '9347599844623989.793252229114225593843', '1438');
    T('1205029771231704090691706271,10', '120502977123170409069170627.1');
    T('39427566041566908772215108783864070289307704169373367199,100000000000000000000000000000000000', '394275660415669087722.151087838640702893077041693733671990');
    T('20653796995853981466925025099001736339,1', '20653796995853981466925025099001736339.01178318512428457087673690149249703', '2');
    T('49185188690848465217210092151,1000000000000000000000000', '49185.188690848465217210092151');
    T('97698380055293281205698719426020,264416813', '369486262794088215584455.74119380979', '553766419');
    T('104240787201124446779050832403,11', '9476435200102222434459166582.0896', '42');
    T('23908923428339619,2000000', '11954461714.1698095');
    T('6443200626654394417539232110121866815091625569790751249399973169,10000000000000000000000000', '644320062665439441753923211012186681509.1625569790751249399973169');
    T('522824108366566300342606573753440692777606807910783,10000000000000', '52282410836656630034260657375344069277.7606807910783');
    T('9331495570471,10000', '933149557.0471');
    T('2288353143307282848483,625000000000000000', '3661.3650292916525575728');
    T('2540011169177460230281,658036', '3859988160491918.7252384368027830049697409', '870300');
    T('201,500', '0.402');
    T('10226950029331489136943365681209055845834606737970478522990929,10000000000000000000000000000000', '1022695002933148913694336568120.9055845834606737970478522990929');
    T('17113571783735766467,6250000000000', '2738171.48539772263472');
    T('607162793632634650613626558,109', '5570300859014996794620427.13759324128530102744', '279');
    T('23435545976038973604189644358770194,65959', '355304749557133576982514052043.999969678005', '66113');
    T('90547763901776187620780219732687,11', '8231614900161471601889110884789.730553504', '14');
    T('538495939889,1000000000', '538.495939889');
    T('892775,1771', '504.1078487171945391982883708896239694983', '2610');
    T('5143213613655381342618850465608,5333', '964412828362156636530817638.4039', '5396');
    T('39901895138105505,1', '39901895138105504.7809060487341642826622382411986', '1');
    T('1381529243092626455345386156196319406364914937,2000000000000000000000000000000000000', '690764621.5463132276726930780981597031824574685');
    T('9499945404778753956432849082641761055489266837114251,100000000000000000000000000000000000000', '94999454047787.53956432849082641761055489266837114251');
    T('83156369636439388647124221367308574466072835786176325890310492691,1000000000000000000000000000000000000000', '83156369636439388647124221.3673085744660728357861763258903104926910');
    T('476709938523129116704811850643225303462101,4991', '95513912747571451954480434911485735015.448006383504104143813484355284523646562', '6199');
    T('157890355707984,275', '574146748029.0327252791231876', '499');
    T('8019262547628511834980873806916156016716,947', '8468070272047002993644006131907239721.9809925284381006401', '5117');
    T('622115359848494409270460619,10000', '62211535984849440927046.0619');
    T('832436043530692186307700916007638657221,9338', '89145003590778773431966257871882486.3162347563616628086057549945343341', '9703');
    T('1197345815215160429787134483429457424748454,3797', '315339956601306407634220301140231083684.08058994774186577044230365470', '5784');
    T('423735152100855061347214859430967270991323541,10000000000000000000000000000000', '42373515210085.5061347214859430967270991323541');
    T('24083749929262503747426239253781575649811815722386688065407,50000000000000000000000000000000000000', '481674998585250074948.52478507563151299623631444773376130814');
    T('5303019589668854255151682937590376324897,10000000000000000000000000000000000', '530301.9589668854255151682937590376324897');
    T('85737211216718643688831414562125117156828541666351591,1000000000000000000000000000000', '85737211216718643688831.414562125117156828541666351591');
    T('528032868848420465249059811366153020793348383745543,100000000000000000000000', '5280328688484204652490598113.66153020793348383745543');
    T('1866752363992468250754020414,35', '53335781828356235735829154.68522878762736138731952941267', '47');
    T('35716735,553', '64587.224236228224605', '625');
    T('217921488209893967858247239220161357646029,31250000', '6973487622716606971463911655045163.444672928');
    T('74900978196398249235645033181449,56659483', '1321949552494120873564007.5145311332967863901290698034953', '95809931');
    T('4181810551795274201109494238031,6850', '610483292232886744687517407.011824823463664815517814', '9863');
    T('173659759502325661822,36351627', '4777221099411.194492670163016336584109466723444167542', '40565652');
    T('975617306643846126337774473247881968707228143,100000', '9756173066438461263377744732478819687072.281430');
    T('107162744108339267900049017055818023,400000000000000000000000000000', '267906.86027084816975012254263954505750');
    T('172824702408627980142888156188664041173157111,5000000000', '34564940481725596028577631237732808.23463142220');
    T('4661,1000', '4.661', '725484239');
    T('85630686880,2563', '33410334.3269603915296445267071067965485682', '2749');
    T('3650614799062559649462274868183699,1000', '3650614799062559649462274868183.6990');
    T('228678,30631', '7.46557409177', '61028');
    T('351824463570227805968918689708954570800044191,500000000000000000000000000000000000000', '703648.927140455611937837379417909141600088382');
    T('488087329866628899580567813575317229184103113,100000000000000000000000000000000000000', '4880873.29866628899580567813575317229184103113');
    T('15364099762357780872271701893453303984731614629318015521954234513,25000000000000000000000000', '614563990494311234890868075738132159389.26458517272062087816938052');
    T('4191173857019885158075164088272787562741,2000000000000000000000000000000000', '2095586.9285099425790375820441363937813705');
    T('891299363389677085414626945889,4000000000000000000', '222824840847.41927135365673647225');
    T('27963428850565537315601806701616541673936819779,10000000000000000000000000000000000000', '2796342885.0565537315601806701616541673936819779');
    T('571401796386043738811955172893452900347780492106439462254983,1000000000000000000000000000000', '571401796386043738811955172893.4529003477804921064394622549830');
    T('366690412973,10000000', '36669.0412973');
    T('12875015217651127,146130874', '88106057.70859296304489358653746605217', '448798288');
    T('24696724779197162265975356937723632,5', '4939344955839432453195071387544726.4');
    T('2985624149040533370866,254972185', '11709607261829.4947382829229', '310816918');
    T('205145083837395729,3524939951', '58198178.3205122545930144839521', '3928265791');
    T('792952739262136340728303572788453059228501933,2000000000000000000000000000000000000', '396476369.6310681703641517863942265296142509665');
    T('1320227697516244557850056741,15625000000000000000', '84494572.6410396517024036314240');
    T('37219214107473809786834936594940667709,10000000000000000000000000', '3721921410747.38097868349365949406677090');
    T('5575556813194831758332268154,56297415', '99037527978768328143.170839265', '60858604');
    T('249706733132241788592404281174823647611951769482541265522444180644991,50000000000000000000000000000000', '4994134662644835771848085623496472952.23903538965082531044888361289982');
    T('295382366467493833160483758,359', '822792107151793407132266.73533894441115275877029700384684359997', '368');
    T('295805371652277510092410438965536011,673881', '438957874835879791969814312861.671439022616798329375629138173', '2521756');
    T('5844753078362685375500413573063352892766873,10000', '584475307836268537550041357306335289276.6873');
    T('638178878607660002139073283695192160794,6631617', '96232770771843428554313870010163.759576887507235421822943361938', '8897582');
    T('22432938154584,39169', '572721748.1831039851864432568917', '41092');
    T('106706350515532352182646089407,29', '3679529328121805247677451358.861708158318085', '61');
    T('1006727,2905', '346.54974185099960', '2920');
    T('506555946665035572178858654593077910567905546515259539016213,62500000000000000000000000000000000000', '8104895146640569154861.738473489246569086488744244152624259408');
    T('5667347843,1', '5667347843.0');
    T('494398222717274926488483225499160,1', '494398222717274926488483225499159.98', '2');
    T('837794972781085082959914787434100934327883671639127,1000000000000000000000000000000000', '837794972781085082.959914787434100934327883671639127');
    T('2557367721562609913879206622896959,50000000000000000000000', '51147354431.25219827758413245793918');
    T('87843192004882899,10', '8784319200488289.9');
    T('19,25', '0.760207454590321431534164054874637479559', '39');
    T('147128541247,43', '3421593982.48867', '44');
    T('368946937,500', '737893.8740');
    T('190832298553383334575954,647488127', '294727100923942.246397290308923302', '2820909477');
    T('2073843380993671359645467586457708477,10', '207384338099367135964546758645770847.7', '90210557');
    T('142023011523698529831928453508889478348921,10000000000000000000000', '14202301152369852983.1928453508889478348921');
    T('131523705761031644569208707101725314627,20000000000000', '6576185288051582228460435.355086265731350');
    T('107614457850081791201223888580314663474567,100000000000000000000000000', '1076144578500817.91201223888580314663474567');
    T('7424203392370085459633617400060995558887,1', '7424203392370085459633617400060995558886.92627', '2');
    T('1150595474009947183249,500000000000', '2301190948.019894366498');
    T('107132561823987752087271387138926902232412924373,100000000000000000000', '1071325618239877520872713871.38926902232412924373');
    T('66592083512795708829,97656250000000', '681902.93517102805840896');
    T('2383461165589395538962556182433727539343280817092285028907,500000000000000000000000000', '4766922331178791077925112364867.455078686561634184570057814');
    T('4752189583313890783247219222481531,10000000000000000000000000000', '475218.9583313890783247219222481531');
    T('742246581737,1', '742246581736.60', '1');
    T('217582569729319048394786382926204527678365,278', '782671114134241181276210010525915567188.36330596838736296846323412', '353');
    T('8,5', '1.6');
    T('77516567,1', '77516567.0', '2529');
    T('187771783050301517419414783195509184163970763,5000000000000000000000000000000000000000', '37554.3566100603034838829566391018368327941526');
    T('75775984617422245988100638579,50000000000000000000', '1515519692.34844491976201277158');
    T('27025750270934149627304743094,27', '1000953713738301838048323818.295980629760086538462', '55');

    T('45331,5000', '9.0662');
    T('1,2', '0.26678469618283923562166896359509235073451784352063763', '2');
    T('2533,681', '3.719523969530026850479571156532323136048917552796680071910875141219321593975575609179027', '697');
    T('6,1', '5.95670176075546656061641857369584534119416380474162703019059921749378', '4');
    T('1611,263', '6.1254796583797468286994317214330688018450504512208169605280714777631573', '348');
    T('4575219,841429', '5.437439165988931940772741700237239451817319934504577438827546421794517317551497622237902', '905300');
    T('544,87', '6.25275513583617822405786324552333236913767738896504459666064910547852535280664715894565', '90');
    T('9398,2267', '4.14556674388769106671553518476960309922345613865422241156', '3348');
    T('3,1', '3.1912389255634557180255978654316265495460980977830957259345681346972112849133326200582984', '2');
    T('90833,188310', '0.482358876323596164805577916312592462921530658660338105311161525274025', '829071');
    T('79,8', '9.87749219827141149', '9');
    T('0,1', '0.0', '484');
    T('126,13', '9.695091451492871', '20');
    T('2344,401', '5.8453758518246010129661980734406695366765558069122949118835193553349723', '492');
    T('1288,2105', '0.611876458127545613370', '6810');
    T('15,8', '1.91843965975015244765459863481866326921442203181383237784729', '8');
    T('0,1', '0.409838221056570951533841', '1');
    T('1783,21429', '0.0832050036602382863605408078022770284912449098792064253887456750250300671442273197567215669397533539', '37533');
    T('45,8', '5.6372753722674653472810042478599375756026972396886320541634716311399615308115964737620290729', '9');
    T('3731,503', '7.41749596112475033273240678217035352204496655512439751763538750951822', '879');
    T('68,261', '0.2605320118176922', '418');
    T('1,1', '0.873516707890829872596691063080472230824856618229083662104844156119886415612785045', '2');
    T('298,5781', '0.051548', '5872');
    T('1,2', '0.493490546390034810538790', '2');
    T('194103,39860', '4.8696186653773687510967391401128197073966271363341369329403328795737499798898813154928081356102', '67047');
    T('8,1', '8.24255476352493964464539856548405279481376105856089797340403340770555106189715421415058338308', '1');
    T('3504,6943', '0.504680921122223324202661672653248024256463340120222953245027413565995719', '7749');
    T('1,2', '0.491444664060778917796646216056433250881566732430439598453611', '8');
    T('5,1', '5.174287049199459942513839348622840616091614563710554596545393655504425427310533622274177087224827743', '2');
    T('48,17', '2.8244956913919378299574044583308527', '38');
    T('13,2', '6.5123694703010445805148101381793203139145534810051830232704726181238111389305273667', '10');
    T('24566962284749476680485,474039', '51824770292633046.39593999650204923557888735082449292', '494735');
    T('3368493619886411272795065960341766914829350143,646', '5214386408492896707113105201767441044627477.00108781345538244652891489457688743808616951764106898054577612254593670957661783381212694042072880', '646');
    T('2586853155389978187,4', '646713288847494546.759357782181814033868900465999081587869086726952721777767314153318', '7');
    T('7790857,1', '7790857.0819506998029689765', '6');
    T('20316021176247877123927349918982151279842758053,328679', '61811132370026308720445632118213062835906.030056681445118198111988670764482507178630313908793827175849924091992', '379056');
    T('1859,400', '4.6475', '506');
    T('3049623920905,54', '56474517053.79685033296077', '54');
    T('955619588917617544364192167390384622044661,1', '955619588917617544364192167390384622044660.58683083203739003688389119086331771452663428', '1');
    T('23729431076324650550683769,2', '11864715538162325275341884.628306443723097500514117908401946557366479217553160881729', '2');
    T('602177639398434193,12', '50181469949869516.0836841857315775159797349427875', '57');
    T('140119378421058440906693944186090478181701459,23487', '5965826986037316000625620308514943508396.196151063587153500718548824147376690362082681', '32245');
    T('7923555977836074006224698706360493883275129309417457,87511', '90543542844169007395923926207682392879468058980.213424597970479050473902729928963099224064941742', '92677');
    T('54774151891479696845053139444,1', '54774151891479696845053139444.32158102581962967272783934830487680739269', '1');
    T('19487924786960507453065915479,493', '39529259202759650006218895.4949277990322529722833768257687422896', '761');
    T('140444,271', '518.243545484623519007809525661667018545612793583718813418801027794258962294070595146', '447');
    T('448303887586360873569623,41995', '10675172939310891143.460483390962390167622207695401198601781636467015533654852801916441417159108770772582258763991743677', '93301');
    T('5308934360728177543766425862582,19', '279417597933061975987706624346.42105975966853513862473278487385291152772263985624372', '565');
    T('671757917955898917816836249302738,1', '671757917955898917816836249302737.83748346114250347811329609441825552892855517748573774083', '1');
    T('6923438,7041', '983.3032239648939942067850769366628571834182477091616462640', '8574');
    T('8681,1', '8680.554499202744390742254279390116166894742281104651898357648636131885616626172960139623', '1');
    T('8296090754,1', '8296090753.77808799719854507879129386685898155948713328498119893342154782002735', '1');
    T('449550842508002469586005256134656634,263', '1709318792806093040250970555645082.25858528116754388519315876056635084792675358257245998246599064750318560', '313');
    T('2384258830294857351266823482459306002060,89', '26789425059492779227717117780441640472.584265827825756685558321547464547903622405705748281747905025875364873', '244');
    T('3099059767,369', '8398535.95393019688934', '716');
    T('1118567,8', '139820.8675991643206836269', '11');
    T('54501975717322825161701857159922629,617907', '88204172662427881803737224468.9291899913754653160726311160386', '687952');
    T('292065600267078038277,360587', '809972628705632.866068382943176993810432', '473708');
    T('60865545773557068449101563,352490', '172673113488487810857.33372010556289892591424474886122627772863302106575690', '362845');
    T('43910364284205515720513,57', '770357268143956416149.350857167011834016095423796125853', '82');
    T('2768,1', '2768.0056248196618573', '9');
    T('24707,61104', '0.40434341444406601572079746049739999661378', '138159');
    T('62398,1', '62398.049758341992', '3');
    T('106402023318778195146268840340215196296057694978,1711', '62187038760244415631951397042790880360057098.175336027856651274924796443646769049038890419014107906', '5296');
    T('0,1', '0.0', '50251');
    T('1,1', '0.977479149688', '1');
    T('286761116711598046630263181694725313446508186,9', '31862346301288671847807020188302812605167576.2167633900367016409879668147081261089855665558671456', '11');
    T('493854667771423,1', '493854667771423.254959180508897738136194526024260', '1');
    T('59770,50369', '1.186642577786729', '657935');
    T('50670898227060364508301169119,2', '25335449113530182254150584559.576505095152042462067029896885530025096832963272421507985849118924294', '2');
    T('4,7', '0.539885108909598446106149715395519', '7');
    T('7436629,41', '181381.195088', '90');
    T('31167367322965637719365091694651,71', '438977004548811798864297066121.844953873132264903410589240583520033403555493637587973009095', '84');
    T('2598472288366433446623870668432105,96', '27067419670483681735665319462834.427090009020254549190095557820154764669675729935', '626');
    T('1,1', '0.922067456510097025954699039976943', '5');
    T('294229548281191410,241', '1220869494942702.94605133401949538', '427');
    T('241625399655500291,253225', '954192515176.227825056771711519082815948456167', '682290');
    T('1195655939518580983871186713470668235614408053376357058667333,423677', '2822093102808462540735481778502652340378184450362793020.78548752891313966613182857481605202157821315686948713904923560', '424333');
    T('2618811517,1', '2618811516.931208050937507288604652795688621819503341112255782020992459900662114311111088545100179', '5');
    T('4131144162729913,1055', '3915776457563.8985781325507805186370493303', '1489');
    T('582466848848638,23255', '25046951143.7814663542320116521870304194984491696195', '27057');
    T('6455766824570,83', '77780323187.590347796893974704799', '265');
    T('53455832886999353767497945728788540143157488885248223281,59', '906031065881344979110134673369297290561991337038105479.336253413994200448040788970880967169296448933722265614219121230464551', '60');
    T('34684817227394711572160619503375408,71983', '481847342114036808304191538326.7633746855360502891294804519887966739426931804349338553', '73666');
    T('170771191765582963,51940', '3287855059021.6203889101733530', '91639');
    T('59489,802', '74.175817224433680910672', '860');
    T('79898665639094718393482676981,4', '19974666409773679598370669245.22684378060267947984678026612709838', '8');
    T('7517924493744156919811183,286', '26286449278825723495843.297196347604166787095605578145803015483232937733237769524617002712850157698634151602400', '475');
    T('27132562808975269,342', '79334978973611.897660511616985808318420102330091', '4677');
    T('60272291702537718907780263,10573', '5700585614540595754069.8253097510065828573367376877437092127528369415566888', '94840');
    T('1,2', '0.51', '2');
    T('216897821528130673257,53', '4092411726945861759.566036761275728746134069907357892874', '94');
    T('3163857791132342485290267882552370855322887,545326', '5801773234968335427414551814056859301.26729149169554685226093744763268316765389373680859617450067621955512471523', '888336');
    T('31893079,51', '625354.490182800115515495968616197567291321495007057703788292149', '218');
    T('45189870731246404,15181', '2976738734684.566497593258376965224195020258733405076577662', '20203');
    T('239861,472', '508.180083770484715', '880');
    T('33499101710863017,2', '16749550855431508.69914410240779698148517248391235817', '2');
    T('2141207850036582856465048433,329419', '6499952492225957994120.097605177654579789782944540260769802900107062218301666704054646611', '330997');
    T('22193392902987826122559599877876897,58', '382644705223928036595855170308222.362212252887839179829725912772009160714747747554090412224', '62');
    T('90543357024529043405978879144258275090780578649,2', '45271678512264521702989439572129137545390289324.6726991725126164245656717069363903985212602605238565780325239301101476316356964391514481', '2');
    T('37600782476781126884139203431600472982966839780023429919897,1', '37600782476781126884139203431600472982966839780023429919897.24287512927437642710013783433910340431924976940125246127078', '1');
    T('64961559075,8', '8120194884.3724681021129254915288817', '8');
    T('620318090519481009,8560', '72467066649472.0804906532075275', '9472');
    T('8163482,849', '9615.4087173', '869');
    T('7377,17', '433.9414759956934328436361859748327855391332306426', '49');
    T('611,3', '203.6725705400', '8');
    T('89171,1000', '89.171', '55217');
    T('51270493819699846152993496,65', '788776827995382248507592.24720791353857126515444468431677377534114643955968022584267028953553278893503', '65');
    T('952437397670186727964934896572895455572,2039', '467110052805388292282949924753749610.383521318013377289176993384359404280274806527340713677500761934497938212', '6194');
    T('115107039410294795197,2', '57553519705147397598.52569349659929993375319668989846544486260551837544', '2');
    T('22,1', '22.365923', '1');
    T('1990972334583731,2697', '738217402515.2877271240567969345615', '8197');
    T('85750361597402603631603273118666,21', '4083350552257266839600155862793.619499612682355814294752644386270178753962225708928418572214290985761742439416042', '70');
    T('8635686977961,268079', '32213216.9172557343179410005696124075', '549886');
    T('66531984494429,15', '4435465632961.93350703161015018', '61');
    T('0,1', '0.18167201504679', '2');
    T('10529,27', '389.9630', '38');
    T('2934260971563580194491472182109802377517454,5583', '525570655841586995251920505482679988808.42808527488170902078803003616929756884458514431718406108322520', '8718');
    T('5108200665966863073353288070176030051511632153746,7109', '718554039382031660339469414851038127938054881.66352511082081319890207684459946872434550056671', '8537');
    T('1,3', '0.3465753923796479102166177077330555737271920233165265624927052833', '5');
    T('16875360180472543035698945274090527182525755485603,531840', '31730144743668289402261855584556496657877849.51414523164777408433594873822056099197640611496223398710564', '965422');
    T('114634251053546388025309,12', '9552854254462199002109.0865339064408202873873764199', '18');
    T('183409270415172374488271,315', '582251652111658331708.79682520032387306155850071007553282614176099535489744326559', '2814');
    T('2,1', '1.94', '2');
    T('182903874405130637649185200486,34217', '5345409428212018518548826.6208609757260577676285922746405052706839876854656056193106274576096695371939222', '61801');
    T('5095786559079370391313919886663340905109333948827,1', '5095786559079370391313919886663340905109333948827.456737121375509845692166705486257168106577499425369656222842111805457906', '1');
    T('781826702775423639656188,34805', '22463057111777722731.10725470482914830289184811512575254583329325', '310121');
    T('17,21', '0.81', '24');
    T('642449847968682862604540,1', '642449847968682862604539.50719286706858188848922666523', '1');
    T('2232932,29507', '75.6746534717435458614870249', '35207');
    T('188417858747526768690461499725073001733,3250', '57974725768469774981680461453868615.917846152487377706532523813340718234379', '80249');
    T('2512235463,11', '228385042.090496856693737836687110835528901075269402072', '104');
    T('844375191952747415306,142731', '5915850039253893.09474465953436249', '562376');
    T('5398821357017249826283471756217066750564780611585,71', '76039737422778166567372841636860095078377191712.464788995235824488719493358582451477426725287196562573274150399', '20788');
    T('2678627417245,1', '2678627417244.5539595554458498441512623087660882055', '1');
    T('5937979,886', '6702.0079', '913');
    T('93,5', '18.60', '58090');
    T('726330087886628639542796583938445152543562444469390135628,1', '726330087886628639542796583938445152543562444469390135627.99091884025141082950358349062777831341655621667599142195853205636838136485595', '15');
    T('80273,1', '80273.01388365199', '14');
    T('65,4221', '0.0153991811534756810765664526560', '9509');
    T('537861620698135920565351,1', '537861620698135920565351.273301952640253473856107229450', '1');
    T('21589045207819,26', '830347892608.4230723287683895554447840899', '972');
    T('577595426226192934612626685,1', '577595426226192934612626685.0286363631751288945392409001532685640518541279892202642453992056486281846257604', '1');
    T('1675,4', '418.7484751712328113764323953', '26');
    T('19556084872373,345', '56684303977.892786055289803608591940101556713', '352');
    T('19,7', '2.71902394', '9');
    T('5529474308079205564524774009772451131,1', '5529474308079205564524774009772451130.51761287042846642080574332180546078965538812745557608038', '1');
    T('209346348003047912995736,11471', '18250052131727653473.6061372099726072866405051', '11515');
    T('776986810,1', '776986810.1594219281755754257102', '1');
    T('244164980163078814691433439911704751189317,21291', '11467990238273393203298738429933058625.20863275581273538021661214252911409518043401674563445995768940072042555829137104208000', '113505');
    T('1335963025004702580265523343526,35', '38170372142991502293300666957.8857742784242579029128174713642', '173');
    T('29021817758099544970572197889,34', '853582875238221910899182290.852626803328881395374231956097046035', '36');
    T('25607,5205', '4.919692611', '9786');
    T('101904493912965638284134197405,2', '50952246956482819142067098702.5713155960069352658952883488314532333725811763112', '2');
    T('137744409497385418682848482707483773,4443', '31002567971502457502329165587999.94890844640229953492819622422667457964936363221', '4647');
    T('3153045199217207368076364837,1', '3153045199217207368076364836.85821697552566256162690119145512', '1');
    T('1,1', '0.98820163097883359337765041974973089352398769777740690629811505014740114696', '9');
    T('6347903215,1', '6347903214.98333714357446671154140415601041950', '1');
    T('7865439544,1', '7865439543.64842980606939915292766451085112', '1');
    T('3237690453781443948931054435307536847022633061,63987', '50599191301068091158064207343796971994665.0579180143954472511934222589835395025939549529049341055670604818599814635787218355355744623771', '68323');
    T('16547,500', '33.094', '35944');
    T('198,25', '7.92', '879');
    T('48136900151439,5', '9627380030287.78227662306223327607211090', '8');
    T('810188023876443978252916,240291', '3371695252325072425.73760981476647528684572725155145126012755201643744911640005908457166013810637', '594953');
    T('29122019,133', '218962.54888005640775', '570');
    T('42600929421,11', '3872811765.545957776549217228', '37');
    T('1835567,2179', '842.389628282959984577403026165282107385446131435327480129911', '9427');
    T('10715045877924720610637328562303816539523,609238', '17587619087983219383290813380491395.0533666645856507178226823958979342511581839203713355424821734358638', '614310');
    T('5255724839155,1', '5255724839155.35294116304683192133028279893695924503895137408737492483521789962049', '1');
    T('1,1', '0.8978468872480801693946442849195311956942575461556562092198261658', '2');
    T('262004018028579753610099210362465283,291450', '898967294659734958346540436995.935093498025522742208259700262919468727383538368456777271618508283980760681', '449006');
    T('7719603971110112789459092487129510355,11', '701782179191828435405372044284500941.367257069803941074325532535698775501470540336095', '11');
    T('191185328300455108074421,29218', '6543409141640601960.244404134198278885577292621401052797106803852078359839960114', '36394');
    T('10465166381522661,74', '141421167317873.7972970342414003516709434322011311188209619349951980125136084535342139027758', '240');
    T('7902040135126365394659608,1', '7902040135126365394659608.077726021750747291286722149059246102412021295790775165293274649572871360', '2');
    T('116492401903663116224172913052825042248229707047214,155', '751563883249439459510792987437580917730514239014.28387332355496359942769428955017512246414862817961697422654477855510889819244572169952641417683018', '190');
    T('1161,2', '580.6213', '2');
    T('4999366848067334,51', '98026800942496.74502522271320672697996', '83');
    T('155052967013059677537669297640306143,21064', '7361040970995996844743130347526.877278769440031446596394275560016869303377072002659486154804941895737040887923', '42555');
    T('4915589395,87363', '56266.26140356895501876786714596805105223390662', '96564');
    T('1996814563251707684865255053446343344056628802563603747544300835419,35111', '56871480825146184525227280722461432145385457621930555881185407.29170344881337650276681031440473191594935108291821236856625185470036', '36542');
    T('350611450,530451', '660.96859087832849141227', '828317');
    T('595025607,730', '815103.5712333824167995605685', '994');
    T('91,1', '90.9474373813', '7');
    T('520640397828785313386029881258886527,57796', '9008242747400950124334380947797.19231434781804453269843126509668', '58523');
    T('1,1', '0.7', '1');
    T('2432192303,2', '1216096151.53824349736', '2');
    T('182088255527725530185924136198820811105,5994', '30378421008963218249236592625762564.415248549875433131587553202118349363105537060676313739107816391730307', '6684');
    T('2852721959,44162', '64596.75646483388', '93026');
    T('22240697755,6364', '3494767.08909490781227389975332975981042527092609989513703885638542988746801587017415743745', '31237');
    T('638,89', '7.16857182142575386243468931653698455419954180851', '119');
    T('6365209086547966173670265664612892931202087175837077590,1', '6365209086547966173670265664612892931202087175837077590.307044947141883547127069233591193089391933585385866946904448320602686297092942', '1');
    T('447415936707462873637430969941367936988468633406659,17686', '25297746053797516320108049866638467544298803200.6479136045407721264362934913054716728108054995868628807078498679271321502402341540158940227697648', '41076');
    T('556766636479654782236989106,6405', '86926875328595594416391.74176424353622475555994', '9157');
    T('16,5', '3.1934183459202528284347244885465199377401', '11');
    T('69366814480303229226256659588662,1', '69366814480303229226256659588661.85385468147818494724215139782478923263860687041151715994378816519729624732503782164002', '3');
    T('206872032674988736528870510540636795271597443,1590', '130108196650936312282308497195369053629935.4987418677695164526182963222692423472284907274366595734056060718094172585913', '1956');
    T('3296663,352883', '9.342085053684726', '920944');
    T('1901467916531,38371', '49554817.87107450941333', '87559');
    T('35236497328086505353582711620562,1', '35236497328086505353582711620561.9401992755513250318288838292367937207', '1');
    T('2156393081917549337270143486668378869841014072477,9077', '237566716086542837641306983217844978499615960.3918695610681834957819094833282531753966844370947863377761477800766935462547138240866128882311', '40639');
    T('80227083104542559289996173436021300960849044229167256111161292357,10', '8022708310454255928999617343602130096084904422916725611116129235.70048092770865523338951907230464003175896000457756548821368253282172627341860630323893179952355', '40');
    T('14646654367694222221940854473824917,2', '7323327183847111110970427236912458.2637426177732172114494620983511178446999858023257859333622402678207150389963999824040809470078503', '2');
    T('2,3', '0.663', '7');
    T('27963875636216786940514230252872638062266663119606641289638039128214075044862928694740648165,713284', '39204406149888104794884268051537169012997155578432491531617194733393816551139418092569.927497322247096619019101878129041174149753190001476011543505686258781332861014187754125', '898421');
    T('11522420634707,298017', '38663635.41243284778334005538792', '304106');
    T('157263777436938477441584837,17', '9250810437466969261269696.2904128954351819175969727761318768772961877204484176247921068491788133398523250', '23');
    T('2276529,37', '61527.810269929938376521343109850668138767761631527', '52');
    T('4677043,136', '34390.022052748042318976850337684966011768593413342120365', '222');
    T('49253974531233847037400608287727,1', '49253974531233847037400608287727.060540048192515048222477969304021589766722976717090718681939735386', '2');
    T('2160607509685172123,79', '27349462147913571.177199397582134523253856176684228053461559176285030516574874756952', '105');
    T('78052129404159796516380569,250594', '311468468535399077856.53514848720455080879540146109259080789155707071914631548609', '287111');
    T('40103944258853846056256735013082291791174170619985429502015036,737', '54415121111063563169954864332540423054510407896859470151987.83718927418708473127959038816066084224533120453216547984773003205060234', '775');
    T('9213413,115215', '79.967131015930047430988598240344339327429', '146382');
    T('1313141867085165037876810634168452559775493,28', '46897923824470179924171808363159019991981.892596651603443078583217313699611602451988188888733555496625642547664131296842', '37');
    T('8533218224569738665099332127984434063538646794116239739804343876879184515887519,24546', '347641905995670930705586740323654936182622292598233510136248019102060804851.605923571866483092045258463240562511296361654119265411211336042019198176927208580711456953', '79843');
    T('214162254366433222852,219', '977909837289649419.41552439968900800332241887585088624500224888', '461');
    T('1524141790335346850,3649', '417687528181788.6681282293544866931614625951230414', '6410');
    T('8305836543976713538118348277180340222586336935040977205971736934,1289', '6443628040323284358509191836447121972526250531451495117123147.349883517025494372135199719073787587581983279739058143878476420396334557651148941312712594381002', '2421');
    T('12625344856297573143960857366368117442436018000143718905329627,17413', '725052825836878949288511879995871902741401137089744380941.2293688631111047258667617297890350092651444230980392383355854871853724988293848', '29957');
    T('49799008287031995307745299546,201', '247756260134487538844503977.8407984495287628804649309589792969473029', '886');
    T('44230,1', '44229.8640940746684332711646269', '2');
    T('695341392387473187615708533688053494612321911735732263004072039989,12', '57945116032289432301309044474004457884360159311311021917006003332.419123618616690370149051021044082250337110267237814259673029440937922237595010738707535760151768', '12');
    T('1048610192448716663976442300874466291435985376302590107183025006833039111667021956,3711', '282568092818301445426149905921440660586360920588140691776616816715990059732423.05470231283500539106621844843522303682651413708748707464182560598910098632280380429225', '4185');
    T('564179730,41', '13760481.2195468173', '99');
    T('1535358,3677', '417.55724768958258887221999802356746577772806585929098618110903690609575310', '4909');
    T('9432220623393,1', '9432220623393.0361844246328616666243628939101171241757', '2');
    T('2564823,38662', '66.33963581834778293420676990546', '54629');
    T('1,10', '0.1', '5209');
    T('45861865321787548161276827663395729,7180', '6387446423647290830261396610500.797910864788713778286585490522140448230159951171', '8297');
    T('19345,2', '9672.47045', '2');
    T('1621235027393185648213757671997956512719559180709550368213,46478', '34881772610550919751576179525753184575918911758456697.10858900988357153434887154833422609206799928958299650146498864771982234211', '152688');
    T('1,10', '0.10', '571');
    T('14037083559893857042646225061800058376951690953612422931418795682,15209', '922945858366352622963128743625488748566749355882202835914182.1080938918800941797499465697485803927015797682325657123828892678', '74299');
    T('214990,3', '71663.30911497342580428530022113794390987325723075', '5');
    T('307488726044879510002660981395,529', '581264132409980170137355352.35349658712969215704186070305948528667158789544493835023232021586095779408143518', '1469');
    T('7986140723,12950', '616690.4033204624183575117582891638', '45213');
    T('186442582640609698457372644742287508248773821,608', '306648984606265951410152376220867612251272.73190810532616003961033029139820173162304625273178016', '636');
    T('402342388451303346208638031,162144', '2481389311052541852974.134294207606755500782850967202793230491650542666651668050739706230056896340969275037488', '320928');
    T('78241853523003572280986839112347001594408862797249859718,8531', '9171475034931845303128219330951471292276270401740693.906693228378227059405060756165776641079149233773669394894276125419046799506956656383', '12127');
    T('400,2969', '0.1347255413647743121776508902102719647076843003868088', '3453');
    T('3923238860902840839626291104694492525371269916682029,446965', '8777507994815792824105446969437187532292841534.97931381652037574964251891911468943081965790985704', '758290');
    T('35646747251,457', '78001635.1225357380236558357960166239621978669699089151772761258494596715203598593743346261565262019417704', '706');
    T('2683468334240429143171526,38457', '69778410542695195755.558831942122856930461609481624295211857604995876954854137975872813613', '199109');
    T('857821143787449574697977573812019342762480425225461812882950230507,2', '428910571893724787348988786906009671381240212612730906441475115253.25776495579890393989612727541338942514948359370113611786812146845315324593620', '2');
    T('201510238869457784262139645580084027788,1', '201510238869457784262139645580084027788.433065800897233736582903597431923327846329', '1');
    T('38608781052273175033830581491417961344934552,7', '5515540150324739290547225927345423049276364.575944943254543085974802343286568634911541631917053639543590794169', '10');
    T('292623487,68388', '4278.87183424', '74336');
    T('793792872031,9', '88199208003.45664265953995372', '9');
    T('10201,1338', '7.6240657658623830734267630', '54376');
    T('22992066679053461303,552597', '41607295513825.556966469235264236204961015671182279735261042008326101516531', '606489');
    T('314113031689,42767', '7344752.5355764958912381877823', '945136');
    T('22786552661743835954280087582255152144518703,34', '670192725345406939831767281831033886603491.26455918123167096367780477819074132459587344526801847361091447403001267800', '64');
    T('17573479348385910412048883205603677,38771', '453263504897627360966931036228.20347682580886838529942674548009547499946688868816182510007585234388', '46381');
    T('75473594247,8350', '9038753.802035928500', '73489');
    T('261517467644150958487,5053', '51754891677053425.3882842133773049001841242098677038847975074900593966523958', '7047');
    T('7134449311548606923639346547,1', '7134449311548606923639346547.112804404073069945060714830706555901856818370048110134601278840469', '1');
    T('37806932836871856297021727733724124238343972270363,43', '879230996206322239465621575202886610194045866752.62806530892044737038187480030241392596263055431188445162729153784215811022', '47');
    T('7333274067095990846,1', '7333274067095990846.351138936980520182083044922', '1');
    T('1216547636340968,255', '4770775044474.3843159801874529573386574217', '767');
    T('504431408656563019198096470328071071180957480250246834023199409,5242', '96228807450698782754310658208330994120747325496040983216939.986455552080186728491640321801261184902364920421299986856683243744', '5930');
    T('11078314997991539069143123025050347457581,2', '5539157498995769534571561512525173728790.5320269718590661495828504888068010731730753957949', '8');
    T('1564271697783336061228018441408289997,3440', '454730144704458157333726291107061.0456395345856333688836200385200217', '4460');
    T('186593194550751772022221789989,2117', '88140384766533666519707978.265942337331545158189132168003333366607627322769794', '3564');
    T('7628876005219,132254', '57683518.1183102212280', '310461');
    T('85478687995924280983954436,2811', '30408640340065557091410.329420030100682263063726019261554456880639675662867805458887878168290', '4191');
    T('6470726792910130860277335858754095217788330802569343716026302897754173285980528060,691', '9364293477438684312991802979383639967855760929912219560095952095157993177974714.99276540727387009268720823298457991691888717815907421435881615376920831023127122321923253385357011', '939');
    T('4523198057239453672050446673969751,1', '4523198057239453672050446673969750.982888715114504843151109238914542962690753455459880331059163673987075278756', '15');
    T('73447073503191914407442232,2281', '32199506139058270235616.93643140080846067688709303', '6162');
    T('15,2', '7.68', '2');
    T('78438145023100581358518585507076222143917169875433,211', '371744763142656783689661542687565033857427345381.19905314983923110968122150777959792708922490953099', '571');
    T('1,2', '0.5', '7499');
    T('1150752865721854191422680744753413845124279744986,33211', '34649750556196868249154820533962056099613975.6401794585917699904699257964671446961247221389248149631404474350964027899932797090', '288970');
    T('16076493965732650145955199661686357106768139698004095413613703910,17831', '901603609765725430203308825174491453466891352027597746262896.2991419437676397998410373969724327013846829822983652684993660181094261958331254042736453820712', '526347');
    T('59952990463187638,6003', '9987171491452.21355988365672831891871115833105277946211529145', '8517');
    T('437481101068975804785076879586290013500861083645889947003529075931870456655071,479', '913321714131473496419784717299144078289897878175135588733881160609332894895.7640894704400779282366881441709170514130408442820253311598876929674239971303006388', '571');
    T('942821117939,1', '942821117939.0565371890037804928059719232795905869562990', '2');
    T('2935001420859380000157762707121042737,1', '2935001420859380000157762707121042736.786463962708882551445456874895215017511777450740305104656019625159421350155320318', '2');
    T('1202935830992033229631564824298161983619110510518805373783051328257731091663742,273', '4406358355282173002313424264828432174428976229006613090780407795815864804629.091573522116851474330839252731895239804995755007457036768565079804281386796112572872744', '988');
    T('214575955168277704587734045462020765792071421909315615775398203182854444483,74', '2899675069841590602536946560297577916109073269044805618586462205173708709.230122092385266218497768440824927539562064322829675284011146374975633484236446356139989037833789440', '77');
    T('50045992670555904787210785904349993086628482049506558835722589483755503086379483349,2470', '20261535494152188172959832349939268456124891518018849731061777118929353476266997.3072874276195536175288188310939039502675866973828697179032965565983562956493102277329078', '7679');
    T('108,19', '5.682899045', '20');
    T('95920780265488699585348723900916407585247076296,1', '95920780265488699585348723900916407585247076295.9722011782231104515584864013490244939631472605433519900458718986879', '11');
    T('553600997,74', '7481094.55435674169522', '77');
    T('144433407482676852197679175735237771398114384568707349,171', '844639809840215509927948396112501587123475933150335.374342960334521507675621856380306927386460319339374458276206878079117', '173');
    T('6868766813211978712457243,8542', '804116929666586128828.991219864750604264260222291459642492956899214044143458845929414919104910105083368993', '8858');
    T('1208445098905305130465909241337086457990767085995,831', '1454205895192906294182802937830428950650742582.42478544512467032570845257550424907273378267749042187915529412467328062395677178', '847');
    T('71541469,14', '5110104.926816824', '14');
    T('85305914129449392893425526497554757292362463826540038888586101763289670816704873,1716', '49712071171007804716448442014892049704173929968846176508500059302616358284793.0495345021247989668017828545182900215671406666272584768815635473370187911747761623186498587', '1955');
    T('34910355329147790705808594414928780898400374709520341716387506640744,260471', '134027800903547000264169886148280541397700222710091878621372462.349912274303723867421927305257380152124131927422266182255099973219', '500555');
    T('2443149449559899,68934', '35441863950.44388835629755412356920533972949735', '72105');
    T('10235833030110696450984055472910862445487735415,46713', '219121722649170390490528449744415097413733.5520090766481292920641152585608933353122716914574714147115271201821924', '81445');
    T('2412282,65', '37112.030602327213004965512285789841303110099812346432158134090838533603521058186494', '68');
    T('197727336539381948963943498780602681817294393124640440907866490545352,205', '964523592875033897385090237954159423498997039632392394672519466074.8877983219803431837713331765228187870876208300323215077046996150117195232993', '319');
    T('15,2', '7.5', '68659');
    T('29693788745631893429990333506515902428051472103452,706235', '42045195643988040000835888205081739687287478.110617570638404585755831163301839410670137680785270602419771', '744617');
    T('73297180366617098,1', '73297180366617098.3144078057048487400', '1');
    T('1183788966238,17', '69634645072.82347802778257521139329875246046832067292347716607881216167245956951211505074357948', '580');
    T('337857977447058978741478048573512513964055992298298429352553483782737183,5899', '57273771392957955372347524762419480244796743905458286040439648039114.626716392720723625021095618796943542634113675561622196537826531821852407006351', '373605');
    T('59745671209268865703423430945,2', '29872835604634432851711715472.453145834917249137580363606422723099328289141421318324960362064961', '6');
    T('859671161301,148', '5808588927.709477965543', '158');
    T('327826595930709915652253567185,117', '2801936717356495005574816813.547008571112938049893857162154471', '2682');
    T('208334,583', '357.34820', '985');
    T('417920309434403655587245399148447,5', '83584061886880731117449079829689.3970122523691827620394154188956126026896151043485425601', '11');
    T('332161,36', '9226.69454628762590761188160545639599510876405724295839599630525', '95');
    T('492939727089,697975', '706242.669277552920397338754', '908453');
    T('1060742245237932,1', '1060742245237932.1437804740224071805053424640287668142484799633802423091065073873258946063378900279156089046', '1');
    T('43273220994074199614838155038679231161,1576', '27457627534311040364745022232664486.777284267145378185527688171290545838090904705362680992', '21199');
    T('37269,200', '186.345', '5642');
    T('10102884,718535', '14.060392326052129', '967773');
    T('1219084532009380132581392727265570983021563997,14', '87077466572098580898670909090397927358683142.64373446162321728317853561861789635745516311592864', '32');
    T('171,307', '0.557', '460');
    T('356651164544964166146468975252,5', '71330232908992833229293795050.392421805278026288006060731316131740551778431198546677842557461594293258596467893254118405', '5');
    T('2206833385925938227905409311,36', '61300927386831617441816925.3056000492614958215838075939699', '95');
    T('2717956424176196011003730333667350,1', '2717956424176196011003730333667349.7306970440606935095598372413709729268', '1');
    T('5785393455764129633036790208876423850860836737707710175673874728149225089964815672236,7767', '744868476344036260208161479190990582060105154848424124587855636429667193248978456.577314278497503536146945605079240467181174559062273212358677378627319841643644350778240', '76564');
    T('1161,5', '232.1980', '44');
    T('24726923024624909624212763782,25299', '977387368062963343381665.8279773904190662818049691495617687914', '309921');
    T('90987255907509062168543931363687667620909919141734393421873391,368455', '246942654890038300928319418555013957256408297191609269576.6739249026294236002555892691179015779033299594359428884187399481034374064629205214720091827', '444997');
    T('6107945601448519394102630769235213774292217827152610948371675819819794650452964,1', '6107945601448519394102630769235213774292217827152610948371675819819794650452963.911972351804484891326252430492737744797210771982506527128972981079255477625007979629605229', '2');
    T('60582631603433937,8795', '6888303763892.4317225732362317829730210699185957', '13023');
    T('924269334814001,207', '4465069250309.1835743819175830981731288956258299854260859884448829672511937457501', '293');
    T('6639539158116313497,206', '32230772612215114.0631244318936864176600857945579315355413334651806017', '294');
    T('5041241320711878795159,22', '229147332759630854325.410203402656522969600', '34');
    T('583,1', '583.3709375101662808436673731501019705754894240048302559234903760275308517229172', '1');
    T('78633259182155920440744871,101874', '771867789447316493322.58349529813193835020381', '205541');
    T('82102,271', '302.9594088297030760145', '393');
    T('14074604229494197,668215', '21062987555.64331390345832', '882097');
    T('38059055655073430300661325288520231229,4615', '8246815959929237334921197245616518.14279523235188926660945402784286683812836715240105189956436644363941258265156731324437', '9220');
    T('486761604513222981,17381', '28005385450389.6772912944793568949303', '31501');
    T('175551139999215326573815807,242', '725417933881055068486842.1776848404920362490446391323254203047651712', '654');
    T('83,1', '83.4546729624154107891414754194986886501010566249251149259469106168', '1');
    T('414475,1', '414474.886867910455898589', '2');
    T('10888695234363590494551796761105710263060721374178557358201,12', '907391269530299207879316396758809188588393447848213113183.416301978225238330934261320589911002579118705166590028299156884288694543667054408155068376964606', '79');
    T('286464034178,3', '95488011392.68165187332682534905110794125997507098491840188910335502196279864989624086867102546849244', '4');
    T('668447227940,43947', '15210303.955673880085109578181067218136526836139659839', '95990');
    T('1967,6688', '0.2941088588343008701843', '9768');
    T('75109477871,485', '154864902.826805976232816926890145049168468569260391930749888426437423962', '607');
    T('213183802966400860090647,263', '810584802153615437607.022810961591907292757102947541669', '598');
    T('43056714838957186,32235', '1335713194942.05633628090090583329650340', '42219');
    T('268353852074,5', '53670770414.80032170547125902473828206966175034334621779372380824973952480291058107718671403832311691055877', '170');
    T('2186335470375799,701', '3118880842190.868759139435558849448310039566842312194008045', '2332');
    T('57411860,1', '57411859.9022270896768046667038700171202296361618483869857700616072411266642815342383915741', '2');
    T('267977324713872263310162785716701711718,1', '267977324713872263310162785716701711717.723363678989761781788396613638357558534985574238', '1');
    T('1,1', '0.93975044', '2');
    T('5220050569,33611', '155307.8030704234564212140039371642554614853869565892270988882219659485221018', '40474');
    T('929196388431713792942501463987504635208031773362379819037,10', '92919638843171379294250146398750463520803177336237981903.6998915282781857375325494762319240056440144680509967422813830936344214429604941', '161');
    T('135189175314691708540514427,34', '3976152215137991427662189.029089859359115789727429472083152955', '44');
    T('178496793807305,3', '59498931269101.67341651289764334798299', '10');
    T('28953864542244521672518662959329064963,1302', '22237991199880585001934456958010034.533794189847734064794558196958783605867923015359980997857492', '7709');
    T('63453473492536727313152722,1', '63453473492536727313152722.117501523352379698983442205150925604583299973094118972659206526799924556265155676677565655996784819', '1');
    T('2354616796181372,3', '784872265393790.7899857377275372165161613720867003677895', '3');
    T('751530355586715733980909778114787014730961080,9', '83503372842968414886767753123865223858995675.55044820224824632303844521455280789663171560785577862318186544211959819', '9');
    T('22356002,299325', '74.68805478994431561057', '475984');
    T('1004346799,3847', '261072.7317390373573439501681537210331007400524043782883748343985532566718630458', '6781');
    T('153494881923896158957371,262882', '583892704422121556.2776112476168651952528665744272702680374729523657500274346100536469900767949261833977217955', '303274');
    T('914429817632267,109905', '8320183955.5276557027198219649678748529779134912036046830590971420068279283602748367072877', '158163');
    T('9787288468125405218949937571075629551994420536796208677387264,5', '1957457693625081043789987514215125910398884107359241735477452.8020344719128402729740667465646566172893416633248347553225784565', '9');
    T('972383851579928284342475475240744,1', '972383851579928284342475475240743.93833057061207062377573942707899328841206929761487155707619', '1');
    T('227161,15754', '14.4192586', '42794');
    T('24,43', '0.55814928', '50');
    T('86815,883', '98.318233883469928701242143703889452452189640274700389087359375180495206739', '1295');
    T('158102075545494626716142158943501696850901702,27', '5855632427610912100597857738648210994477840.8148706240869250708802459205761650371721955732529677375775213714657834564033767693895', '53');
    T('4573874572374703,123057', '37168747591.560845786911529343095322680071068130368028', '713297');
    T('94137508309,1', '94137508309.12944203063142389', '2');
    T('128263524729405205764013306079835664513219670002964257195470632782718792,96061', '1335229955230584792621493697544640015336293292834389161006762711014.03058473261439886528254630252502628596554777041358913090264220653599909490115329', '343998');
    T('62696361050395475983517,313445', '200023484344607430.278093445431579320407235', '369284');
    T('11477001729,2', '5738500864.606603032325764061', '2');
    T('6522584,7473', '872.8200187578967040279', '9244');
    T('52287796137932885846742481073302755432,37439', '1396613054246451183171091136870716.510376879576631687165374051269033365927122943423452', '44514');
    T('11042,541', '20.41035109316777251868202719496081724', '7581');
    T('3607458421641129939414052,74321', '48538884321270299638.245610258206316802894', '274601');
    T('5695785474134027899942879,1422', '4005475016971890224994.99226471327230623639080467989', '2042');
    T('8878654273655,9012', '985203536.801486935043294', '9665');
    T('65,16', '4.06247646638986069242804', '411');
    T('29908,3', '9969.299545', '6');
    T('3234129973440988418535787398987835234217977885694939183658,1', '3234129973440988418535787398987835234217977885694939183658.0555777579527219596665429746310422846678601273328325482789110754280233239741', '1');
    T('38763,524', '73.97519', '728');
    T('7,1', '6.56', '1');
    T('287773507776094924831763292813488774703439511,5539', '51954054482053606216241793250313914913060.030871977269534757984962396972028182875803654912773216689075089542071374319', '6975');
    T('135902709964850252402968509809,145', '937260068723105188985989722.82062844541535607143554252535789346214610335738887862350472', '173');
    T('12272393029782069191611244866640477536622317952693272516,1915', '6408560328867921248883156588324009157505126868247139.695038408447774384790132755871611355838250229705452357706632277610603203384972', '2031');
    T('2225,192', '11.5885417757514787571566159059881098551576095026723816', '9297');
    T('53371779627153865605630667958238412899341574421446540565563726321002579114955251781202163,5836', '9145267242486954353260909519917479934774087460837309898143201905586459752391235740439.02724468894046585153819806105344749411841071667196589490983824705161267037058650756988034162768018', '73957');
    T('2446436726419359505665677336229199860520232227,534', '4581342184305916677276549318781273147041633.3838917648266110125209840196281133496676870967557119215942174589504706059079700', '557');
    T('492176,8321', '59.14866', '8502');
    T('9,1', '9.01348435261992079442197040', '9');
    T('328435344544083927300588344986968685085,1', '328435344544083927300588344986968685084.696665368175202653355800835276350404235962273399764329774312731281859536060585323082333783861', '1');
    T('159247659795083,438555', '363119015.391645289643', '560287');
    T('6,1', '5.70', '1');
    T('72150594429471845730,319963', '225496680645799.18843741307358458329580279139701535416802994965032378655564757028', '549282');
    T('8,1', '7.95841526551959133597120', '2');
    T('426677,11', '38788.81813330', '24');
    T('1751990915929005698023297071945011392910508303508376261581982019497491952385038529603989010476,18689', '93744497615121499171881698964364674028065081251451456021294987398870563025578603970463321.2304564170567292002043552709059704140869790828468039869462307264644682137220664645749338191124', '21564');
    T('247438600592095556152258502462038301212815837584618493182,487', '508087475548450833988210477334780905981141350276424010.64065691758941436115799603577450664221966964524772848474790743487607130855363168196131673164233', '4945');
    T('1830602979532381493889680621382105980461277,751704', '2435271036913973444187713011214661596.1352833030042512463024221180036627596838788009350350', '861635');
    T('3920365916916176786055106017,46', '85225346019916886653371869.93476900513189071699081282383598839961655611874932469229156603202305772110380915189992118499498858', '520');
    T('472649930038,123', '3842682358.03251262860394055442674', '147');
    T('1322110218557217605016482876416321878749760304812247,155948', '8477891467394372515303068179241297604007491630.6220470926214830053246154267443752408560398235344548438873690519655363049727184587649', '186638');
    T('23547504929176,55', '428136453257.745545520225605', '65');
    T('17121761490301749308279011219417223490311,232710', '73575529587476899610154317474183419.235576468560490943788124174995507987278744551812', '543753');
    T('16210026155636340189350564224854481593345828818610444963909203142193849898,239', '67824377220235733009834996756713312106049492964897259263218423189095606.26777934017026855940357410680016571583242046733124370322513732722378950028598', '692');
    T('190721792400863327458476761464009950717029,30', '6357393080028777581949225382133665023900.9666170639928048559949862724415791705119899615018742118160129182', '79');
    T('61,200', '0.305', '244093');
    T('1716239054042743503675603092,2003', '856834275607959812119622.1128307556027562949206592525403122428', '7036');
    T('43,1', '42.926', '2');
    T('27358642114,8563', '3194983.31355833866', '9623');
    T('26693152565727887582,12127', '2201134045165983.9681701980199628544994221693300315787693850757330347383069440481917705451090', '48076');
    T('6108091753110008516780,1', '6108091753110008516779.73479487754085449867983587832803285557110100750079855828473226', '1');
    T('2,3', '0.718905305925227067205880082792710', '3');
    T('3473047534252762578370964997,58594', '59273091686055954165460.02998600522350350905681210811307517203796586454683', '61501');
    T('214822703242583,56', '3836119700760.4108921624057841119514683803433710971136', '64');
    T('26939714,3', '8979904.64498692634635572320542201896798645625719108607536315732247799297935360779199341966787552114794600', '5');
    T('2888884587551576320,29', '99616709915571597.240972811959289242975553142', '34');
    T('947939103218542770831,1', '947939103218542770831.0521624243633591645983271158385778064925689940216692520225367', '1');
    T('9933088375957272302206358007975451939487,2', '4966544187978636151103179003987725969743.72438467636183210768718000533720409321549827235614', '2');
    T('2384620985464,48617', '49049118.322068409482196547648333', '50827');
    T('1481114193489213622796229711,159566', '9282141518175636556636.31168920697604560950832652295984016490111948393975418960516033185', '434728');
    T('83229715325,52', '1600571448.5577172712032806916796048084329678112100319702869128418611147806748', '97');
    T('259,9', '28.786385', '9');
    T('18421000231117295985083192807209645332,25', '736840009244691839403327712288385813.280233799772105420526583646770509765903251570390', '65');
    T('699729069573093,1', '699729069573092.9415209617669715638690428130440588', '2');
    T('1269145537981521474122771320340177027273014491011926684206752265265583667980863639117069,46', '27590120390902640741799376529134283201587271543737536613190266636208340608279644328631.9347579670840456037995587812140760559559266417811221041968023988665383416259781405442396223206087', '88');
    T('7035913928382432889095670790,1', '7035913928382432889095670789.9451603279584465650713190452458435039596863298954339307519647608553073011741266856074291808762823441', '1');
    T('5380967249116311236814136453793462678035221359070439177808760415,14197', '379021430521681428246399693864440563360936913366939436346323.900471930297520912068842457020834544905293996118477045554870916446118758909841288498400493527105', '91913');
    T('5340404069347070913584555456280093652528470496019424627166576,7', '762914867049581559083507922325727664646924356574203518166653.71250635103031314100742496440500618527676618848777566910526532273', '12');
    T('198596768231311216444471847625266,1', '198596768231311216444471847625265.83726507122160159383594314037463792212479812445545510304752094547024953', '1');
    T('334,53', '6.301655789150', '56');
    T('9482145824073019,308', '30786187740496.81493576082236077814823570323411336136297997538465529623724716057733663574274358863', '518');
    T('49564672028692779312869265853690250389636933591279,1', '49564672028692779312869265853690250389636933591279.05690047648396367999804886767002033615206952728024860778041603073475913559839990', '2');
    T('180043559436755584546341153685052792171032546504943274982733,271928', '662100112664953901570787685288211556629080295169836408.83885808006294519499163145424494176495048682043552813068297869819085051', '323702');
    T('135402176306147,40', '3385054407653.6752566904201890867915343261767717538895', '41');
    T('487749,5656', '86.23567892818914544812557542982186431015804303', '27597');
    T('689353268,161', '4281697.316772446980582095053374575467875610055', '661');
    T('2341601,3', '780533.72419690', '3');
    T('133253,2', '66626.369877360981293176690373014842055920029', '2');
    T('23614,1', '23614.044401474469046155210490395', '1');
    T('281757167099338050985151947541415731483,42890', '6569297437615715807534435708589781.568733970826316638832067132912947392456994401031191534637582563566923232', '64820');
    T('7,30', '0.2334', '77');
    T('188159408735932593,55153', '3411589736477.3011984841415382', '62884');
    T('49893755664003300784200999639,5755', '8669636084101355479444135.471589937556905646874989257298383710889254914096760398878896706072509550992971893614682', '5837');
    T('180209,22', '8191.3248002765501146181', '22');
    T('70415704682823093030143170,135531', '519554232484251522014.4702687945814801414708882557', '370767');
    T('309703964089501625654278146528831017909611280781,531191', '583036919092194004895184870468119787250934.7499882339908947319021034086598173362092896910180810566025185828', '568168');
    T('6311026676431831888549757346533448414306654210393,1', '6311026676431831888549757346533448414306654210392.54151415748259382280340911118846529021380341028941701654077953763414807744375826120008106076828300', '1');
    T('677239979042028152601158,7949', '85198135494027947238.79204931221625179624149473557938172126847707', '9883');
    T('1829735554262281,259', '7064616039622.706568544570626683229950243512536015093535101968887493794302873092483253629579230290', '331');
    T('1,2', '0.317925082', '2');
    T('65422,9439', '6.9310308267754', '17120');
    T('46961497053061382777609130118847271892089775387846767743314411513423583155027,5', '9392299410612276555521826023769454378417955077569353548662882302684716631005.400654014794318218848509928000496866762016629616754055629753036755426927256260779921696553', '67');
    T('500194857457135627698,665', '752172717980655079.2451135360411912947058454904697242572660767', '687');
    T('47471355958725273634,1', '47471355958725273634.16168212923451461593072522210510640420155857994234198580802495272039820633735570319533813', '1');
    T('36696342857496890219,1', '36696342857496890218.90572359364080164761627035720048698', '3');
    T('4695131986657142516305076,510781', '9192064674796326637.649160794940805998501189986', '747475');
    T('349103010796287846089941527434273199124602123798955453463589836132521373235273,71', '4916943814032223184365373625834833790487353856323316245966054030035512299088.352127776626217594251476581025885809792618906667335429116230898910825304240224', '299');
    T('1358220866465871480816463658707268378971,16', '84888804154116967551028978669204273685.687886576615307696608953854116834104494900462161862843643799426800222', '91');
    T('27194039930406868209865840701464121302724394008919831130512956946096037284286,47', '578596594263975919358847674499236623462221149125953853840701211619064623069.9146866857902214919864171822857086850016715830174570427578437592194338070650774544060278', '58');
    T('2751902407120,23', '119647930744.3479655249971735656329573846182167', '43');
    T('4469030866140915638532462562432305966504,1', '4469030866140915638532462562432305966504.192178572338167275494797670860545892956073433782578010264482847', '2');
    T('4704489769335751301445685485486877217294,103', '45674657954716032052870732868804633177.6116627025867182700549754236229291028015465033707982024', '443');
    T('1475216297524085707694311943977667,151', '9769644354464143759564979761441.503308131238849125841196958844356073790374671626586526675819233898553472', '734');
    T('4907983733090076808453942390,57', '86104977773510119446560392.807235262788300753240131745761023198496136680383768897782992', '59');
    T('585074587,61012', '9589.5002130729144884804491184088195014563319127651629128080417190511729', '67046');
    T('7,1', '6.9', '2');
    T('504,601', '0.8386', '776');
    T('47984,267191', '0.17958688728350164035953663206762777096380577875511144865128', '335603');
    T('10388140798372,2093', '4963277973.421882452088446982138192961421705231905173423353291761058587', '13798');
    T('20168990044874146939541,7160', '2816898050960076388.204050292237501110723265705833631209539445571975161631843126782', '8609');
    T('46896058387748469269291854711564296983,110836', '423112151176048118565194113027935.8419917716416221340193798283264901', '155719');
    T('10955549766043717068313994547,128018', '85578198113106883940648.9286428471045878019343568927449610831989116', '378555');
    T('647078048579323459391277902984368763031859387250426744686475450876664538395892114032,1', '647078048579323459391277902984368763031859387250426744686475450876664538395892114032.3902894368682962019659983250422182774256220836412025664532446636812180965587379981258', '1');
    T('18638867424143570538275250698558378620490371,4', '4659716856035892634568812674639594655122592.7613196710958960564528035771871961517785715362059753295820571', '4');

    T('-766693107209001147673676223300000000,1', '-7.666931072090011476736762233E+35', u);
    T('-449394937314220211511925695600000000000000000000000000000000000000000000000000000000000000,1', '-4.493949373142202115119256956E+89', u);
    T('579061232699508282365391682698798449087639,5000000000000000000000000000000000000000000', '0.1158122465399016564730783365397596898175278', u);
    T('-5514336245609206034070613442000000,1', '-5.514336245609206034070613442E+33', u);
    T('-1626302750634169116141012001000000000000000000000000000000000000000000,1', '-1.626302750634169116141012001E+69', u);
    T('-1795503669658780056331087359,20000000', '-89775183482939002816.55436795', u);
    T('-478536837039153,200000000', '-2392684.185195765', u);
    T('-19898168880712128313811543240000000000,1', '-1.989816888071212831381154324E+37', u);
    T('-363762350297028160717513248300,1', '-3.637623502970281607175132483E+29', u);
    T('-8468355353839925110875467363,10000000000000000000000', '-846835.5353839925110875467363', u);
    T('-45775371,10000', '-4577.5371', u);
    T('113017973534801596808086521,1250000000000000000000000', '90.4143788278412774464692168', u);
    T('-60641938108001991714010900430,1', '-6.064193810800199171401090043E+28', u);
    T('-6268106205338927353362914493000,1', '-6.268106205338927353362914493E+30', u);
    T('-62639169752501205441274039740000000000000000000000000000000000000000000000000,1', '-6.263916975250120544127403974E+76', u);
    T('538570638075066218029616939267858613539095601141862844311,10000000000000000000000000000000', '53857063807506621802961693.9267858613539095601141862844311', u);
    T('-458667133551,1000000000', '-458.667133551', u);
    T('-1856541268082778545036889427,2500000000000000', '-742616507233.1114180147557708', u);
    T('401060653077651395227747,12500000000000000000', '32084.85224621211161821976', u);
    T('-4144323558304702676396445443000000000000000000000000,1', '-4.144323558304702676396445443E+51', u);
    T('-5542449392944384114719343577000,1', '-5.542449392944384114719343577E+30', u);
    T('33218113354409142794187676923560201901246573,125000000000000000000000', '265744906835273142353.501415388481615209972584', u);
    T('-313400727241063772788345397,10000000000000', '-31340072724106.37727883453970', u);
    T('-2001790975721803137,2500000000', '-800716390.2887212548', u);
    T('-2000194173750641732491991877,5000000', '-400038834750128346498.3983754', u);
    T('-4665510204000006565975869513,50000000000000000000', '-93310204.08000013131951739026', u);
    T('2153528479648597306374181609165668498526259590105002419225534999299,39062500000000000000000000000000000000000000000000000', '55130329079004.0910431790491946411135622722455066880619321736959820544', u);
    T('-30666677738567071,5000000000000', '-6133.3355477134142', u);
    T('-4703545825363662923055913431,50000000', '-94070916507273258461.11826862', u);
    T('-7623267021806768883120286367,100000000000000000000', '-76232670.21806768883120286367', u);
    T('-47613686004517,1000000000', '-47613.686004517', u);
    T('-4426382462876620839114533,10000000000000000000000000', '-0.4426382462876620839114533', u);
    T('-3208666589586606174447191,100000000000000', '-32086665895.86606174447191', u);
    T('-7729062348415188688065314918000000000000000000000000000000000,1', '-7.729062348415188688065314918E+60', u);
    T('-2160223698870209381629363153,25000000', '-86408947954808375265.17452612', u);
    T('-2004385076106459382854411831000000000000000000000,1', '-2.004385076106459382854411831E+48', u);
    T('-8002741891355462634094119996000000000000,1', '-8.002741891355462634094119996E+39', u);
    T('-4368891335671811,100000000', '-43688913.35671811', u);
    T('-400258929245951237,2000000000000000000', '-0.2001294646229756185', u);
    T('42762676503070237126609626887821888428451673288895043501,100000000000000000000000000000', '427626765030702371266096268.87821888428451673288895043501', u);
    T('-58304052117107641964012899950000000000000000000000000,1', '-5.830405211710764196401289995E+52', u);
    T('15248669391693474071160842935287056367141428182921376132980388727,5000000000000000000000000000000000000', '3049733878338694814232168587.0574112734282856365842752265960777454', u);
    T('-8078047970318036600976065350000000000000000000000000000000,1', '-8.078047970318036600976065350E+57', u);
    T('7,1', '7.0', u);
    T('-6226798387179777853909992923000000,1', '-6.226798387179777853909992923E+33', u);
    T('-7448516509843407495470415780000000000000000000000000000000,1', '-7.448516509843407495470415780E+57', u);
    T('-78852027500722155037504331190000000000000000000000000000000,1', '-7.885202750072215503750433119E+58', u);
    T('-3342348042119829837180052129,500000000000', '-6684696084239659.674360104258', u);
    T('15672687176259210859051407089930945124489111085110817031431,40000000000000000000000000000', '391817179406480271476285177248.273628112227777127770425785775', u);
    T('39866154729780717195080337049315868519050979213540479786966580509961183,1000000000000000000000000000000000000', '39866154729780717195080337049315868.519050979213540479786966580509961183', u);
    T('-993671543846886085686711917800000,1', '-9.936715438468860856867119178E+32', u);
    T('532028339563122259746625760616140111884633179604572642613837,500000000000000000000000000000000000000', '1064056679126244519493.2515212322802237692663592091452852276740', u);
    T('-697966803816803579573431799,25000000000000000000', '-27918672.15267214318293727196', u);
    T('-48787459362042945333251298680000000000000000,1', '-4.878745936204294533325129868E+43', u);
    T('-353168187829651949299,4000000000000000000', '-88.29204695741298732475', u);
    T('-1403470827451558036088375821,20000000000000000000000000', '-70.17354137257790180441879105', u);
    T('-68304813987155896442052862570,1', '-6.830481398715589644205286257E+28', u);
    T('-28707593804742896781267400000000000,1', '-2.870759380474289678126740000E+34', u);
    T('-1791426377870978262043979561,1000000000000000000000000', '-1791.426377870978262043979561', u);
    T('179838406674297924775572988142470977688025003111919,25000000000000000000000000000000000', '7193536266971916.99102291952569883910752100012447676', u);
    T('-2423075550414669291292735759,5000', '-484615110082933858258547.1518', u);
    T('-776725396418999866033600668900000000000000000000000000000000000000000000000000000000000000,1', '-7.767253964189998660336006689E+89', u);
    T('-369087919583602506700335119,50000', '-7381758391672050134006.702380', u);
    T('-65769870411112553029542951,200000000000000', '-328849352055.5627651477147550', u);
    T('-1198038835922050392798019,200000000000000000000', '-5990.194179610251963990095', u);
    T('-40097191351465303957388254440000,1', '-4.009719135146530395738825444E+31', u);
    T('-3236635346912723615834792627,5000000000000000', '-647327069382.5447231669585254', u);
    T('-2365771,250000', '-9.463084', u);
    T('-160497076958746128902700099,3125000000000', '-51359064626798.76124886403168', u);
    T('157938109732784046764321035938547478521587349076387,20000000000000000000000000000000000000000000', '7896905.48663920233821605179692737392607936745381935', u);
    T('3516380051744022949626313427473,5000000000000000000000', '703276010.3488045899252626854946', u);
    T('-86489135514398778941814310290000,1', '-8.648913551439877894181431029E+31', u);
    T('-301393313166931162873106807500000000000000000000,1', '-3.013933131669311628731068075E+47', u);
    T('-518869594507859770521243546000000,1', '-5.188695945078597705212435460E+32', u);
    T('-1950909743929332443100687271,50', '-39018194878586648862013745.42', u);
    T('-8773131005595091487081140354,1', '-8773131005595091487081140354', u);
    T('-4355596294471508983960024769,100', '-43555962944715089839600247.69', u);
    T('-2880570808953415045636806327,50000000000000000000000', '-57611.41617906830091273612654', u);
    T('-9320214110050561347793210341000000000000000000,1', '-9.320214110050561347793210341E+45', u);
    T('-2028302171,1000000000', '-2.028302171', u);
    T('-232946686791834523955513553,5000000000000000000000', '-46589.3373583669047911027106', u);
    T('32153248599816465463513302064276865531758911323110724293,2500000000000000000000000000000000000000000', '12861299439926.5861854053208257107462127035645292442897172', u);
    T('-8744838688036249566782650227,1000000000000000', '-8744838688036.249566782650227', u);
    T('-623804321303869879672219319900000000000000000,1', '-6.238043213038698796722193199E+44', u);
    T('-5226891267180881946811097711000000000,1', '-5.226891267180881946811097711E+36', u);
    T('6104731674327122118125159140803482511,50000000000000000000', '122094633486542442.36250318281606965022', u);
    T('-3631918259327646858473441459,100000000000000000', '-36319182593.27646858473441459', u);
    T('-1848392047279159333633594727,500000000000000', '-3696784094558.318667267189454', u);
    T('-264510547222964010201402067,2500000000', '-105804218889185604.0805608268', u);
    T('23965604410399646104889267,100000000000000', '239656044103.99646104889267', u);
    T('-7462093867393361276933908373000000,1', '-7.462093867393361276933908373E+33', u);
    T('24323093395343447761780819,25000000000000000', '972923735.81373791047123276', u);
    T('820159670097111265573137140169274052235176656378637635665429413451,10000000000000000000000000000000000000000000000000000', '82015967009711.1265573137140169274052235176656378637635665429413451', u);
    T('-13218678465057612022150417,10000000000000', '-1321867846505.761202215041700', u);
    T('-6699947761211673976391555178000000000000000000000000000,1', '-6.699947761211673976391555178E+54', u);
    T('-6087775300361116653641781722,1', '-6087775300361116653641781722', u);
    T('-1174267411076461674383873197,12500000', '-93941392886116933950.70985576', u);
    T('-9622612849795128416976470129,10000000000000000', '-962261284979.5128416976470129', u);
    T('-430267205006530807363136683,200000000', '-2151336025032654036.815683415', u);
    T('-1842468826711949238742823267,2000', '-921234413355974619371411.6335', u);
    T('-896792855550853822680165595000000000000000000,1', '-8.967928555508538226801655950E+44', u);
    T('-7249373999522375642630010999,10000000000', '-724937399952237564.2630010999', u);
    T('-867584543901195289988590669,5000', '-173516908780239057997718.1338', u);
    T('-7172260193178638618675644585000000000000,1', '-7.172260193178638618675644585E+39', u);
    T('-589873963727757525194937079400,1', '-5.898739637277575251949370794E+29', u);
    T('-5890981801970379206238029411,10000000000', '-589098180197037920.6238029411', u);
    T('-452864647493799097082782472200000000000000000000,1', '-4.528646474937990970827824722E+47', u);
    T('-600618187610370877245711717700000000,1', '-6.006181876103708772457117177E+35', u);
    T('-667,100', '-6.67', u);
    T('-535739501101785482356483511,20000000000000000', '-26786975055.08927411782417555', u);
    T('-2189002367733133,500000000000000', '-4.378004735466266', u);
    T('-4135862533810979636019397675000,1', '-4.135862533810979636019397675E+30', u);
    T('-14984699501047789,1000000000000', '-14984.699501047789', u);
    T('-2644213426631746862661931503,10000', '-264421342663174686266193.1503', u);
    T('-2092299259157497414090361017,5', '-418459851831499482818072203.4', u);
    T('-77,100', '-0.77', u);
    T('-54276151887,1000000000', '-54.276151887', u);
    T('-7316738873979211718601112500000000000000000000000000000000,1', '-7.316738873979211718601112500E+57', u);
    T('-389683579126928886377631531,2500000000000000', '-155873431650.7715545510526124', u);
    T('46789726161807116524384304297700033991397937633588752307,10000000000000000000000000000000000000', '4678972616180711652.4384304297700033991397937633588752307', u);
    T('-9196858034762943045297291137,10000', '-919685803476294304529729.1137', u);
    T('-12757819,25000', '-510.31276', u);
    T('-2751580109664699528322719473,50', '-55031602193293990566454389.46', u);
    T('-1397067073903866232334961397,200000000000', '-6985335369519331.161674806985', u);
    T('-4812745393643600076996562763000000000000000000000000000000000000000000000000000000,1', '-4.812745393643600076996562763E+81', u);
    T('-13515907799732096505704651000000,1', '-1.351590779973209650570465100E+31', u);
    T('5425915792163208468016304413313938329266256242679179675690564113030627,1000000000000000000000000000000000000000000', '5425915792163208468016304413.313938329266256242679179675690564113030627', u);
    T('-10827300142961420614583582960000000000000000000000000000000000000000000,1', '-1.082730014296142061458358296E+70', u);
    T('488478745829,50000000', '9769.57491658', u);
    T('-2319703638146144907642415179,250000000000000000000000000', '-9.278814552584579630569660716', u);
    T('8199609,1000000', '8.199609', u);
    T('-1132520281920903148966859641,50000000000', '-22650405638418062.97933719282', u);
    T('-9435372448691264711,10000000000', '-943537244.8691264711', u);
    T('-5430432705270926972283794348000000000000000,1', '-5.430432705270926972283794348E+42', u);
    T('2547106921,10000000', '254.7106921', u);
    T('-2013361060932518973304030049,250000000000', '-8053444243730075.893216120196', u);
    T('-299071373209294751127694543,4000', '-74767843302323687781923.63575', u);
    T('161261,5000', '32.2522', u);
    T('-2149015945426180798371334522,1', '-2149015945426180798371334522', u);
    T('-1219052557726696173105236523,1250000000000000', '-975242046181.3569384841892184', u);
    T('-6635702255618430700913831019,1000000', '-6635702255618430700913.831019', u);
    T('-4229092410987972953057006873,100', '-42290924109879729530570068.73', u);
    T('800876757958087347097942687332875742929708200305858327085977609,100000000000000000000000000000000000', '8008767579580873470979426873.32875742929708200305858327085977609', u);
    T('-2665484396670548074136945193,50', '-53309687933410961482738903.86', u);
    T('-3156306908006626264325676067,100000000000000000', '-31563069080.06626264325676067', u);
    T('-3,5', '-0.6', u);
    T('-8680573119149141897627467752000000000000000000000000,1', '-8.680573119149141897627467752E+51', u);
    T('-82860324718555460591215170260000000000000000,1', '-8.286032471855546059121517026E+43', u);
    T('-4531829930266176401806361441000000000000000,1', '-4.531829930266176401806361441E+42', u);
    T('-5737601975600871638965999673,1000000000000', '-5737601975600871.638965999673', u);
    T('-37760607898500795218600331790000000,1', '-3.776060789850079521860033179E+34', u);
    T('-27369129037741,50000000000', '-547.38258075482', u);
    T('2477983988516065743406448097218451318402450481,10000000000000000000000000', '247798398851606574340.6448097218451318402450481', u);
    T('-705385142506364347385362693,1', '-705385142506364347385362693.0', u);
    T('-4969675047511558769703779774000000000000000000000000000000000000000,1', '-4.969675047511558769703779774E+66', u);
    T('-85382944582940147310728016560000000000000000000,1', '-8.538294458294014731072801656E+46', u);
    T('-1782322041330569224195797189000000000000000,1', '-1.782322041330569224195797189E+42', u);
    T('-290994034468272029485971361100000,1', '-2.909940344682720294859713611E+32', u);
    T('-988966648479804202907677518800000000000000000000000000000000000000000000000000000,1', '-9.889666484798042029076775188E+80', u);
    T('-4566915368931602815051783606,5', '-913383073786320563010356721.2', u);
    T('270653156797430209033263481,625000000000000000000000', '433.0450508758883344532215696', u);
    T('-48999796485280572833341497940000000000000,1', '-4.899979648528057283334149794E+40', u);
    T('-72525550030522147167279582110000,1', '-7.252555003052214716727958211E+31', u);
    T('-2432675193330427851698809509,100000000000000000000000', '-24326.75193330427851698809509', u);
    T('-403266406094322401449840190200000000000000000000000000000000000000000000000000000000000000000,1', '-4.032664060943224014498401902E+92', u);
    T('-7328622424224698690222269711,10000000000', '-732862242422469869.0222269711', u);
    T('-47237342255413479773733697,25000000000000000000', '-1889493.690216539190949347880', u);
    T('78835631855459381,10000000000000', '7883.5631855459381', u);
    T('-14,5', '-2.8', u);
    T('-9866329513673835238514096493,100', '-98663295136738352385140964.93', u);
    T('-13089959190012611177126518580000000000000000000,1', '-1.308995919001261117712651858E+46', u);
    T('-4967358948172051919497159739,500', '-9934717896344103838994319.478', u);
    T('-6833646028032836687349866821,100000', '-68336460280328366873498.66821', u);
    T('649637135613968795105908969145470711,250000000000000000000000000000', '2598548.5424558751804236358765818828440', u);
    T('-61363709817705515400922815730000000000000000000000000000000000000000000000000,1', '-6.136370981770551540092281573E+76', u);
    T('-100998083624843426300911688400000000000000000000000000,1', '-1.009980836248434263009116884E+53', u);
    T('-3590001814306412202538592043,1000000000000000000000000000', '-3.590001814306412202538592043', u);
    T('-873319590748158544964033381600000000000000000000000000000000000,1', '-8.733195907481585449640333816E+62', u);
    T('1753,50', '35.06', u);
    T('-52231905436704456949438551980000000000000000000000000000,1', '-5.223190543670445694943855198E+55', u);
    T('-4348525293879767179803795097,5000000000', '-869705058775953435.9607590194', u);
    T('-1628925349793270623943546973,1000000000000000000000000000', '-1.628925349793270623943546973', u);
    T('-78494014309,5000000', '-15698.8028618', u);
    T('-9245876360737547772734914910000000000000000000000000000,1', '-9.245876360737547772734914910E+54', u);
    T('-807754225571147376128169812100000000,1', '-8.077542255711473761281698121E+35', u);
    T('-6150194754154627890562649755,1', '-6150194754154627890562649755', u);
    T('-2471953954534824409680038307,25000', '-98878158181392976387201.53228', u);
    T('-970474827218905069338978816700,1', '-9.704748272189050693389788167E+29', u);
    T('-24039,10000', '-2.4039', u);
    T('-19253676936551703,1000000000000000', '-19.253676936551703', u);
    T('-3673854325985073740352036342000000000000000000000,1', '-3.673854325985073740352036342E+48', u);
    T('-160934801,25000', '-6437.39204', u);
    T('-32833225394240539484189003,2500000000000', '-13133290157696.21579367560120', u);
    T('-879576295241677088569196885100,1', '-8.795762952416770885691968851E+29', u);
    T('-34723705196925586591,2500000000000000000', '-13.8894820787702346364', u);
    T('58699728651,2500000', '23479.8914604', u);
    T('25989415788377713434433189687,1000000000000000000000000000', '25.989415788377713434433189687', u);
    T('-1528267418236634262738364747,1000000', '-1528267418236634262738.364747', u);
    T('-6671686359029212455119744847,100000000000000', '-66716863590292.12455119744847', u);
    T('144580847646431960012829985973482546613221338871,200000000000000000000000000000000000000000', '722904.238232159800064149929867412733066106694355', u);
    T('-33981629178302642259388602190000000000000000000000000000000000,1', '-3.398162917830264225938860219E+61', u);
    T('-64240817973350524445496942010000000000000,1', '-6.424081797335052444549694201E+40', u);
    T('-990485401974653806265390505600000,1', '-9.904854019746538062653905056E+32', u);
    T('-6028091377718570445490225453,1000000000', '-6028091377718570445.490225453', u);
    T('-231645924266279,200000000', '-1158229.621331395', u);
    T('-356048892709504881374103191700000,1', '-3.560488927095048813741031917E+32', u);
    T('-96293143562099332540946672560000000000000,1', '-9.629314356209933254094667256E+40', u);
    T('-637588725404145616579243109,200000000', '-3187943627020728082.896215545', u);
    T('-5533803323992652056012670321,10000000000000000000000000', '-553.3803323992652056012670321', u);
    T('82096165597399152387073083,10000000000000', '8209616559739.9152387073083', u);
    T('-84046624737814576667252743050000000000,1', '-8.404662473781457666725274305E+37', u);
    T('-760070190521772257789348512700000000000000000000,1', '-7.600701905217722577893485127E+47', u);
    T('33848173936588017661795890663826550420632434137913501,500000000000000000000000000000', '67696347873176035323591.781327653100841264868275827002', u);
    T('-67816336411542138315185300640000000000000000,1', '-6.781633641154213831518530064E+43', u);
    T('-1204357960160460298667925175000000,1', '-1.204357960160460298667925175E+33', u);
    T('-7970186008547782569239312038000000000000000000000,1', '-7.970186008547782569239312038E+48', u);
    T('-3710232124874852163479109505000000000000000000000000000000000000000000000000000000,1', '-3.710232124874852163479109505E+81', u);
    T('-4889555265921636364438094567,5000000000000000', '-977911053184.3272728876189134', u);
    T('3117,500', '6.234', u);
    T('-699740571071362462935070191800,1', '-6.997405710713624629350701918E+29', u);
    T('-70248014967,250000000', '-280.992059868', u);
    T('-857100231280874948737999933,200000000000000', '-4285501156404.374743689999665', u);
    T('467297342335727,10000000000', '46729.7342335727', u);
    T('-27212862586658598118090814040000000000000000000000000000000000000000000,1', '-2.721286258665859811809081404E+70', u);
    T('-3777590780331387653794428909,100000', '-37775907803313876537944.28909', u);
    T('-207858091537398998838496056000000000000000000000,1', '-2.078580915373989988384960560E+47', u);
    T('-6020540505342374731212772184000000000000000000000000000000,1', '-6.020540505342374731212772184E+57', u);
    T('-38563435471069083574670403660000000000000000000000000000000000000000,1', '-3.856343547106908357467040366E+67', u);
    T('12925583620733632059,250000000000', '51702334.482934528236', u);
    T('-64127811,100000000', '-0.64127811', u);

    /*
    T('30396,38195', '0.79581097', '74409');
    T('2125057574,3289', '646110.5424140842277760913716223149175043516912866283522989430448583852673477181054', '4314');
    T('18251,16', '1140.688086121119279582816178812726653910', '37');
    T('61348765027081784476232889631422618917241085602257,2', '30674382513540892238116444815711309458620542801128.559859020834676491532682063017825303001913079937413732204504474950978211332225018371420', '2');
    T('238354711696434965227174225040046402291933791409568617910,25981', '9174193129457486826033417691391647830796882006449660.0558100148468970808679278895365813792370050009058167954469', '61710');
    T('7355967799241738309628763754835483150582416687842544374137,8', '919495974905217288703595469354435393822802085980318046767.1338999658963927824509929378965804777221331548975659545056488845295998946309871474144010438', '9');
    T('1125354937387110207464744146511518,247481', '4547237716782743755943866989.8356560705677241346146759866583370949', '863202');
    T('22627259137083636694372928354252733658397903083652793951,13169', '1718221515459308732202363759909843849828984971042052.8476725643837742907709281227796422726432771727035925851544494828706899020', '266434');
    T('8704956917621873401344326514156520003279612669326092644945779992398046340851791,127', '68542967855290341742868712709893858293540257238788131062565196790535797959462.9212696722706718117183841431994103691974707194398390091180486522725873390891717', '171');
    T('58400904696,8737', '6684320.09797413893379904985072385250774250364224704531308514779353223', '14426');
    T('4765998700799,481932', '9889359.28886025414277', '551989');
    T('84446319074008288318841993796319916,1', '84446319074008288318841993796319916.4290214679797552828327089099139280188', '1');
    T('208579935771852505416338,29', '7192411578339741566080.621098672624218880520368888346887640903578755437931806539999680994582938', '38');
    T('1222506164843897615007151221018141468,521', '2346460968990206554716221153585684.199616425445388739418159378389564040736054', '706');
    T('63993391879749765751,9', '7110376875527751750.114919085901505100980521292031066797810839927388501365930465', '11');
    T('36707708314,1', '36707708314.345688574511854258055215898538289601221976670300216981023491', '1');
    T('1006862373895650813218257996159822111779222738385202718573234201031,335', '3005559325061644218561964167641260035161858920552843936039505077.70447752609278095294701366298104650604976387611568135143174113323780682', '7063');
    T('12180512204,290477', '41932.7940043445804168810482554121918927813650259', '895566');
    T('972851014381591,18170', '53541607836.080957621939096312239677792878740188648258030024875', '31555');
    T('383576424003665,59', '6501295322096.01707059843465672720153308', '88');
    T('456059695649,731697', '623290.372447884850674344473049816049238992052574671', '770999');
    T('1,2', '0.5716202415873544262986953173753852278285193332218646822626651', '2');
    T('13802410789886736953383903846755867001633124596437299605748175709981308048573814,2523', '5470634478750193005701111314607953627282253109963257869896225013864965536493.782798186996737913044082377221550214034629772236938011947896897722075720890727218714', '2580');
    T('1720546690653497270570006270472130851,1', '1720546690653497270570006270472130851.14006845543964920673232647491053082791301585407937041734157349507526701334664232654756777642845206', '1');
    T('275492732803212658371830121443811803,6273', '43917221872024973437243762385431.5005579559793624838126175850466341579997661657132448932942532299', '11453');
    T('88013352536654826423613115136850148349568967,93', '946380134802740069071108764912367186554505.02139414761736578247940104106009403877307019264568966335689948628094', '100');
    T('0,1', '0.16188799510329961', '2');
    T('31257777609885493076027897164441747647216000811363789460,63', '496155200156912588508379320070503930908190489069266499.364683462978985476107209780379101750930177884309470340386147403736530', '73');
    T('1,4', '0.247', '5');
    T('139940391692584759336872579028566139849626387938102,9', '15548932410287195481874731003174015538847376437566.916769682876782912934506396938577003712568789804114047347942880', '9');
    T('2695380025444014713279737860623,4', '673845006361003678319934465155.7153220774931558972103241774927225665361206328689645904684305800', '6');
    T('212541,1', '212541.15765900911813807562731698106746376864402272303473059993656512959159467385761060217667309', '2');
    T('24404077187137405965287142997346562978162191083070727134647177951283348456202109976721557879,358982', '67981339418515151080798321356910828337248639438943253797257739806684871264303251908790.85268620710740817674022927818400975732259671102671740536686565452412022811045315287709605606', '524757');
    T('4639813710478702096,63125', '73501999373920.033203960389538229718685812256959445512319514858711642307834404391479542025114608466188853802', '975593');
    T('6313823485926856046033387,79', '79921816277555139823207.43038835415731205809303875306226403799816963841913838777893750', '309');
    T('156447551951667418528348,10581', '14785705694326379220.1444097907069652381615089410990740501255021747836566834256552197426931481109', '28853');
    T('5939913817183935713420272651319175,1', '5939913817183935713420272651319175.15560997078079301798352616505127598762535051904836053714139307939449564', '2');
    T('2612459,3290', '794.06048625538280973740786635286809420954121949534447211174', '3834');
    T('44771360338621,79', '566726080235.708661152424012356777601', '84');
    T('19411423,337', '57600.661728879757757929764351499408564549978771807173536885876073886506463832955', '390');
    T('1847,2', '923.31354', '2');
    T('2652776,545', '4867.4789', '848');
    T('2469,20', '12.345e1', new BigNumber('123'));

    T('-2991910323532143367251432461,50000000', '-59838206470642867345.02864922', u);
    T('-972745200863972039012044288500000000000000000000000000000,1', '-9.727452008639720390120442885E+56', u);
    T('-759,100', '-7.59', u);
    T('-27827258151452494934863201,100', '-278272581514524949348632.0100', u);
    T('-284626800809,50000000', '-5692.53601618', u);
    T('-74421663208247269188483431,80000000000000000', '-930270790.1030908648560428875', u);
    T('-1082016448333999416859837516000,1', '-1.082016448333999416859837516E+30', u);
    T('-21,25', '-0.84', u);
    T('-67246554173689300336047211,8000000', '-8405819271711162542.005901375', u);
    T('-2196812392790282979771359770000000000000,1', '-2.196812392790282979771359770E+39', u);
    T('-93367638440645772883402889310000,1', '-9.336763844064577288340288931E+31', u);
    T('-35989199713272470893896397640000000000,1', '-3.598919971327247089389639764E+37', u);
    T('-21641213684333939876179569040,1', '-2.164121368433393987617956904E+28', u);
    T('-91309127692139566014327039220000000000000000000000000000000000000,1', '-9.130912769213956601432703922E+64', u);
    T('-1991444477322661912100056959,50000000000', '-39828889546453238.24200113918', u);
    T('-553515865616749148505592781,10000000000000000000000000', '-55.35158656167491485055927810', u);
    T('-2736653830100392487693621031,500000', '-5473307660200784975387.242062', u);
    T('-1434910778309587281399577109,5000000000', '-286982155661917456.2799154218', u);
    T('4088837688828123100698969622178669612073,5000000000000000000000000', '817767537765624.6201397939244357339224146', u);
    T('-43637769341168890990138963370000,1', '-4.363776934116889099013896337E+31', u);
    T('-972943330653707493150695821,2000000000', '-486471665326853746.5753479105', u);
    T('-1782595711624723281402457991,1', '-1782595711624723281402457991', u);
    T('-1587179980733649965762666577,25000000000000000', '-63487199229.34599863050666308', u);
    T('-1869019007722241613978741151000000000000000000000000000000000000000000,1', '-1.869019007722241613978741151E+69', u);
    T('-9534407406159040849329043501,1000000000000000000000000000', '-9.534407406159040849329043501', u);
    T('884870737759390069055513857366081428271482267261449,1000000000000000000000000000000000', '884870737759390069.055513857366081428271482267261449', u);
    T('-139160500164081906232225785000000000000000000000000000000000000000000000000000000000000000,1', '-1.391605001640819062322257850E+89', u);
    T('-9077283717131739693270257533,1000000000', '-9077283717131739693.270257533', u);
    T('-3546952154463187872567980253000000000,1', '-3.546952154463187872567980253E+36', u);
    T('-2998349233065239662673433121,50000000000000000', '-59966984661.30479325346866242', u);
    T('-49202917142412402713948777780,1', '-4.920291714241240271394877778E+28', u);
    T('181577327136618327510814084031821584885623912603171067071894881,2500000000000000000000000000000000', '72630930854647331004325633612.7286339542495650412684268287579524', u);
    T('-5299277021247306829333111683000000000000,1', '-5.299277021247306829333111683E+39', u);
    T('-859302244477761340074208133600000000000000000000000000000000000,1', '-8.593022444777613400742081336E+62', u);
    T('-6501829741620523766984105281,1', '-6501829741620523766984105281', u);
    T('-2677415014831874236017310386000000000000000000,1', '-2.677415014831874236017310386E+45', u);
    T('-4734450690396580389539539453000000000000000000000000000000000000000000000000000000000,1', '-4.734450690396580389539539453E+84', u);
    T('573,1000', '0.573', u);
    T('170553245440322276130662336624985791859,390625000000000000000', '436616308327225026.89449558175996362715904', u);
    T('-198470325592748654691786431,100000000000000', '-1984703255927.486546917864310', u);
    T('-3234474304795759429822944851,100', '-32344743047957594298229448.51', u);
    T('-76846093771110548259822724440,1', '-7.684609377111054825982272444E+28', u);
    T('-131857323335983571776045123,156250000', '-843886869350294859.3666887872', u);
    T('-4663443251439907783224263307,5000000000000000000', '-932688650.2879815566448526614', u);
    T('63123252215861327828078184797456511902545223118271893387,100000000000000000000000000000000', '631232522158613278280781.847974565119025452231182718933870', u);
    T('-618918292260085378971410121400000000000000000000,1', '-6.189182922600853789714101214E+47', u);
    T('-999,1000', '-0.999', u);
    T('-5117190266823165989390291807,10000000000000000', '-511719026682.3165989390291807', u);
    T('-7326556356501682719117834871,100000000000000000000', '-73265563.56501682719117834871', u);
    T('-6104631144035504571891715845000,1', '-6.104631144035504571891715845E+30', u);
    T('-553580276933846729294149065500000,1', '-5.535802769338467292941490655E+32', u);
    T('-2029603585167950725930322651,2500', '-811841434067180290372129.0604', u);
    T('-3840678439599650124602555357,500000000000', '-7681356879199300.249205110714', u);
    T('-1811414587273530335470170277,10000000000000000000000000000', '-0.1811414587273530335470170277', u);
    T('-3184069685558819408138516175000000000000000000000000000000000000000,1', '-3.184069685558819408138516175E+66', u);
    T('-149520028664451586711286719,250000000', '-598080114657806346.8451468760', u);
    T('-39829,100000', '-0.398290', u);
    T('-76787660914545531460526626510000000000000000000000000000000000000,1', '-7.678766091454553146052662651E+64', u);
    T('-2015496795339941642277997653,5000000000', '-403099359067988328.4555995306', u);
    T('-1318336013779584857221166649,20000000', '-65916800688979242861.05833245', u);
    T('-724089949757367,100000000000', '-7240.89949757367', u);
    T('-15952866815190919594236189370000000000000000000000000,1', '-1.595286681519091959423618937E+52', u);
    T('-1342078101586766284616393,100000000000000000000', '-13420.78101586766284616393', u);
    T('-6812055517286162590000532611,10000000000000000000000', '-681205.5517286162590000532611', u);
    T('-567847685641009361520749225200000000000000000000000000000000000000000000000,1', '-5.678476856410093615207492252E+74', u);
    T('-1580251794203221398959714431000000000000000000000,1', '-1.580251794203221398959714431E+48', u);
    T('-6574603652290596029643069989,10000000000000000000', '-657460365.2290596029643069989', u);
    T('-7311541940635831348361450069,100000000000000000000000000', '-73.11541940635831348361450069', u);
    T('-14723103323,10000000000', '-1.4723103323', u);
    T('3285075251527965969147,12500000000000000000', '262.80602012223727753176', u);
    T('-820320298626362566428108475000000000000000000000000000000000000000000000000000000000000000000,1', '-8.203202986263625664281084750E+92', u);
    T('252919151006836181107958661434555503428356126246362605748877,500000000000000000000000000000000', '505838302013672362215917322.869111006856712252492725211497754', u);
    T('-1179378935267,5000000000000', '-0.2358757870534', u);
    T('-1383313883094030650884742084,5', '-276662776618806130176948416.8', u);
    T('-3721825945463317461862875941,5000000000000000', '-744365189092.6634923725751882', u);
    T('-772585031312025977266466033,20000', '-38629251565601298863323.30165', u);
    T('-1022325206901044018628878242000000000000000000,1', '-1.022325206901044018628878242E+45', u);
    T('-9331233103994018282451022848000000000,1', '-9.331233103994018282451022848E+36', u);
    T('-76778786132507991606924452380,1', '-7.677878613250799160692445238E+28', u);
    T('-1075535026526469098011554507000000,1', '-1.075535026526469098011554507E+33', u);
    T('-6734357,2000', '-3367.1785', u);
    T('292826785068446808328398273711158229435617047159402888673901,5000000000000000000000000000000000000000000000000', '58565357013.6893616656796547422316458871234094318805777347802', u);
    T('-5742288086224317836910104718000000000000000000000000000,1', '-5.742288086224317836910104718E+54', u);
    T('8423,25000', '0.33692', u);
    T('48097700873717796851032061929605079278183,20000000000000000000000000000000000', '2404885.043685889842551603096480253963909150', u);
    T('6023634370460703429713558622417776109096525391939,1000000000000000000000000000000000000000000000000', '6.023634370460703429713558622417776109096525391939', u);
    T('106758008007385450139,100000000000000000', '1067.58008007385450139', u);
    T('-3849580463489122904112780999000000000000000000000000000000000000,1', '-3.849580463489122904112780999E+63', u);
    T('-88275520431316339743958679610000000,1', '-8.827552043131633974395867961E+34', u);
    T('-472042305926729955808146449,1000', '-472042305926729955808146.4490', u);
    T('-6245867883108091,1000000000', '-6245867.883108091', u);
    T('-183356236298435289867506669,2000000000000000', '-91678118149.21764493375333450', u);
    T('-74562808732179833652840115180000000000000000000000000000000000000000000000000000000000,1', '-7.456280873217983365284011518E+85', u);
    T('-5104625575258744026028197501,1000000000000000000000', '-5104625.575258744026028197501', u);
    T('-344922559321915510291490087500000000000000000000000000000000000000000000000000000000,1', '-3.449225593219155102914900875E+83', u);
    T('-8563598459050660434249519489,100000000000000000', '-85635984590.50660434249519489', u);
    T('17500814267583629590589899612077,250000000000000000', '70003257070334.518362359598448308', u);
    T('45401405910540076895910506821437320717666680221,6250000000000000000000000000000000000000', '7264224.94568641230334568109142997131482666883536', u);
    T('-84509536465048048632507,1250000000000000', '-67607629.1720384389060056', u);
    T('-1751326137777370498636433863,25000', '-70053045511094819945457.35452', u);
    T('-867111630064241604638707847400000000000000000000,1', '-8.671116300642416046387078474E+47', u);
    T('-21318256746158533732802277550,1', '-2.131825674615853373280227755E+28', u);
    T('-67579403706761435205295261900000000000000000000000000000000000000,1', '-6.757940370676143520529526190E+64', u);
    T('-1417367504555887528141746601,2500000000000', '-566947001822355.0112566986404', u);
    T('-25974847707227382380557472,625', '-41559756331563811808891.95520', u);
    T('-981439911205743478966598319,100', '-9814399112057434789665983.190', u);
    T('-610789624400704048725520972100000000000000000000,1', '-6.107896244007040487255209721E+47', u);
    T('-1893232078819503172511076559,50000000000000', '-37864641576390.06345022153118', u);
    T('-27975984885306222968321285120000000000000000000000,1', '-2.797598488530622296832128512E+49', u);
    T('-2612911727298316430264079,62500000000000000000', '-41806.587636773062884225264', u);
    T('50680230444767491,1000000000', '50680230.444767491', u);
    T('-373274225495445311979038119300000000000000000000000000,1', '-3.732742254954453119790381193E+53', u);
    T('-8327620769338584623732488387,100000000000000000000', '-83276207.69338584623732488387', u);
    T('-4868628906895320112426939731,100000000', '-48686289068953201124.26939731', u);
    T('-1193567448524866936588184537,20000000000000000000000', '-59678.37242624334682940922685', u);
    T('-86830341,10000', '-8683.0341', u);
    T('-6646694931396004582423920897,100000000000000', '-66466949313960.04582423920897', u);
    T('-5348172887765880895603201513,100000000000000', '-53481728877658.80895603201513', u);
    T('-8919495454891628197011164031,10000000000000000', '-891949545489.1628197011164031', u);
    T('1559360912667,1000000000000', '1.559360912667', u);
    T('-7711382430758007094636047839000,1', '-7.711382430758007094636047839E+30', u);
    T('-6741391809448013052641540557,1000000', '-6741391809448013052641.540557', u);
    T('-301,500', '-0.602', u);
    T('-86974760786594923670130517580000000000000000000000000000,1', '-8.697476078659492367013051758E+55', u);
    T('-3,5', '-0.6', u);
    T('-83297684778142804104535372320000000000000000000000000000000000000000000000000000000000000,1', '-8.329768477814280410453537232E+88', u);
    T('-163673076457158327,1250000000', '-130938461.1657266616', u);
    T('-38956087033937387393159766230000000000,1', '-3.895608703393738739315976623E+37', u);
    T('-1534311654615354489262951441,200000000000000000', '-7671558273.076772446314757205', u);
    T('-8060394441780008075571792928000000000000000000,1', '-8.060394441780008075571792928E+45', u);
    T('-280124687988949117330427391,2500000000000000000000', '-112049.8751955796469321709564', u);
    T('-30114151670681354095135826370000000000000000000000000000000000000000,1', '-3.011415167068135409513582637E+67', u);
    T('-812,125', '-6.4960', u);
    T('-1370772417711,2000000000000', '-0.6853862088555', u);
    T('-9001273695034747863562885281,100000000000000000000', '-90012736.95034747863562885281', u);
    T('-94895895660508854551459723340000000000000000000000000000000000000000000000000000,1', '-9.489589566050885455145972334E+79', u);
    T('33927,1000', '33.927', u);
    T('-221228229074405139018236415000000000000000000000000000000000000000000000000,1', '-2.212282290744051390182364150E+74', u);
    T('-72357217092268759,1000000000000', '-72357.217092268759', u);
    T('3,5', '0.6', u);
    T('-8999705506879032909995262466000,1', '-8.999705506879032909995262466E+30', u);
    T('-34801868215741663768869349060,1', '-3.480186821574166376886934906E+28', u);
    T('-638899147521386374389705727,250000000000', '-2555596590085545.497558822908', u);
    T('-576341894025935781,10000000000000', '-57634.1894025935781', u);
    T('-4499426107906193213910836749,1000000000000000000000000000', '-4.499426107906193213910836749', u);
    T('-1187772132418630901316882007,12500000', '-95021770593490472105.35056056', u);
    T('-328174941647878676105511874800000000000000,1', '-3.281749416478786761055118748E+41', u);
    T('-3472215593316048786969586545000000000000000000000000000000,1', '-3.472215593316048786969586545E+57', u);
    T('-819049308491114964745499902800000000000000000,1', '-8.190493084911149647454999028E+44', u);
    T('-6323108380621192986595369,200000000000000000000', '-31615.541903105964932976845', u);
    T('-111264401026058917823810189,100000000000000000000000', '-1112.644010260589178238101890', u);
    T('-375536277598104783062275796800000000000,1', '-3.755362775981047830622757968E+38', u);
    T('-101660024857,1000000', '-101660.024857', u);
    T('-2004125726691987151280986843,25000000000000000000000000', '-80.16502906767948605123947372', u);
    T('-7642170031922907778643381,100000000000000000000', '-76421.70031922907778643381', u);
    T('-3602317098842106506606624929,1000', '-3602317098842106506606624.929', u);
    T('-1299346412819078024158481,20000000000000000000', '-64967.32064095390120792405', u);
    T('-1239554900011232365384901487,125000000000000000', '-9916439200.089858923079211896', u);
    T('-4626908109136214769997540471,50000000000000000000000', '-92538.16218272429539995080942', u);
    T('393786126653440528814656629984919695629424483,20000000000000000000000000000000000000', '19689306.332672026440732831499245984781471224150', u);
    T('79,400', '0.1975', u);
    T('-70926474960683674023948242490000000000000000000000000000000000000,1', '-7.092647496068367402394824249E+64', u);
    T('-717448177428239400724351357200000000000000,1', '-7.174481774282394007243513572E+41', u);
    T('93965062239421527657876691002319056463449204972859618248841032392337,5000000000000000000000000000000000000000000000', '18793012447884305531575.3382004638112926898409945719236497682064784674', u);
    T('-742455691111299272393671478500000000,1', '-7.424556911112992723936714785E+35', u);
    T('4327353232599424133161482714110773218878025952012887889286000677,10000000000000000000000000000000000000', '432735323259942413316148271.4110773218878025952012887889286000677', u);
    T('-6406522453382260785242123219,100000', '-64065224533822607852421.23219', u);
    T('-8755758445067880087438735551000000000000000000000000000,1', '-8.755758445067880087438735551E+54', u);
    T('-3641369340200941843410397,100000000000000000000000', '-36.41369340200941843410397', u);
    T('-784260557922398380334392781600000,1', '-7.842605579223983803343927816E+32', u);
    T('-662071018037510735810792187000000000000000000000000000000,1', '-6.620710180375107358107921870E+56', u);
    T('2,1', '2.0', u);
    T('-6471005670399478287255928531,10000000000000000000000', '-647100.5670399478287255928531', u);
    T('-8213251496116568092258251293,10000000000000000000000000000', '-0.8213251496116568092258251293', u);
    T('-2433095402981113505171684571,500000000000000', '-4866190805962.227010343369142', u);
    T('-7073268678570322756074595191,100000000000000000000', '-70732686.78570322756074595191', u);
    T('-5922619520639844865780847779000000000000000,1', '-5.922619520639844865780847779E+42', u);
    T('765719072830121,100000000000000', '7.65719072830121', u);
    T('-3110127455285391882932859,10000000000000000000000000', '-0.3110127455285391882932859', u);
    T('-626969950393688092623498764800000000000000000000000,1', '-6.269699503936880926234987648E+50', u);
    T('-45963737194150293757063245860000000000000000000000000000000000000000000000000,1', '-4.596373719415029375706324586E+76', u);
    T('778051713899,100000000', '7780.51713899', u);
    T('-4459718171731138401213154589,50000000000000', '-89194363434622.76802426309178', u);
    T('-1071777034967899856015183850000000000000000000000,1', '-1.071777034967899856015183850E+48', u);
    T('-17597792014189525803930088410000,1', '-1.759779201418952580393008841E+31', u);
    T('-73583607563995328344161936390,1', '-7.358360756399532834416193639E+28', u);
    T('78478899969144273,1000000000', '78478899.969144273', u);
    T('-355286094740091330430709,1250000000000000000', '-284228.8757920730643445672', u);
    T('-41,50', '-0.82', u);
    T('-9907765598680476853217019311,100000000000000000', '-99077655986.80476853217019311', u);
    T('-2404392782688999157305680689,100000000000000', '-24043927826889.99157305680689', u);
    T('-2650994750421622104977287881,1000000000000000000', '-2650994750.421622104977287881', u);
    T('39983073582040014866844804541291,100000000000000000000000000', '399830.73582040014866844804541291', u);
    T('-8807898557101190300230101932000000000000000000000000000000000000000000,1', '-8.807898557101190300230101932E+69', u);
    T('-695393182113783444341458924,5', '-139078636422756688868291784.8', u);
    T('-1706659473239464584561002977,100000', '-17066594732394645845610.02977', u);
    T('-165933152156143937454202761,2000000000000000000000', '-82966.57607807196872710138050', u);
    T('1538291813397103572148216063922845985226928827759,250000000000000000000000000000000000', '6153167253588.414288592864255691383940907715311036', u);
    T('-13,250', '-0.052', u);
    T('-2294062859474990814031336269,5000000000000000000', '-458812571.8949981628062672538', u);
    T('149418055166367757421653263494051015159636829951,5000000000000000000000000000', '29883611033273551484.3306526988102030319273659902', u);
    T('-4629861263429484977299994121000000000000000000000000000000000,1', '-4.629861263429484977299994121E+60', u);
    T('-76174501687600226360492968090000,1', '-7.617450168760022636049296809E+31', u);
    T('-6350678364890072941143667763,1000000000000000000', '-6350678364.890072941143667763', u);

    T('7325643,10000000', '0.7325643');
    T('8129753468709152191211051855224273832281,100000000000000000000000', '81297534687091521.91211051855224273832281');
    T('26846080372036394639591396,6672793', '4023214922452471497.25630571786057289667012386513324609', '72150416');
    T('151169465630612379481675652813336817561189321,200000000000000000000000000000000000000', '755847.328153061897408378264066684087805946605');
    T('48650152878444478499756261104359,1000000', '48650152878444478499756261.104359');
    T('310291026486294288245548,3136069', '98942665638509321.1423434879780971949357099446690852', '4194429');
    T('281148949800335203180214,28247', '9953232194581201656.11264913171363754779711915', '40003');
    T('8070991737,10', '807099173.7');
    T('2105837647496797835243555428994974365746550495259,2500000000000000000000000', '842335058998719134097422.1715979897462986201981036');
    T('7050549406715449007260719040610995681,100000000000000000000', '70505494067154490.07260719040610995681');
    T('4712375599056588823223,5', '942475119811317764644.6', '23351170');
    T('1812890500714000446600040219589,6454218482', '280884588237897901176.13812435378291552542223517750505532', '8240308776');
    T('1485521639981158765714509531187624755829128849,100000000000000000000000', '14855216399811587657145.09531187624755829128849');
    T('179195624045698698709602380291841534992963,397', '451374367873296470301265441541162556657.33752912', '654');
    T('60786881043136,81', '750455321520.1975303545477874468301224681956212', '6955');
    T('54350058946186575767248586,111', '489640170686365547452689.9639497075875189553277274425649764720088', '136');
    T('207448313353211937938706154109749,2500000000000000000000000', '82979325.34128477517548246164389960');
    T('18637328030895697373,20000000000', '931866401.54478486865');
    T('77127230808472816894238711,10000000000000000', '7712723080.8472816894238711');
    T('29126173556928305215214880165709856062,7', '4160881936704043602173554309387122294.5709328', '7');
    T('43333053556734019842,614489', '70518843391393.531604308619475952242596538499476346', '675048');
    T('53377793855957,559250504', '95445.231562915140439462747444833608100863907', '1363996915');
    T('347448038601,1158528605', '299.9045833667611513140004251012084', '3206002475');
    T('4723559025392992686163722415183139,100000000000000000000', '47235590253929.92686163722415183139');
    T('178091379773085237300450763077988415217238,1811184275', '98328691470714782625004163686209.3533896422549273776361277', '1811232885');
    T('805514724831305204420874185699445105715750883,100000000000000', '8055147248313052044208741856994.45105715750883');
    T('65819,6849', '9.610016056348', '8529');
    T('319168767950612554460346534967,625000000000000', '510670028720980.0871365544559472');
    T('5710296907,88986', '64170.7336772078034895641687140263', '92775');
    T('556588363946502498333308725135986051213680185539426551525214239244852201,500000000000000000000000000000000000000', '1113176727893004996666617450271972.102427360371078853103050428478489704402');
    T('20826873980146981695459788919411033,31250000', '666459967364703414254713245.421153056');
    T('2499361049528080561202511540068978284974386343138634241,1000000000000000000000000000000000000', '2499361049528080561.202511540068978284974386343138634241');
    T('418298807255269058527019236340986131039447727889611856636829077,50000000000000000000000000000000000', '8365976145105381170540384726.81972262078895455779223713273658154');
    T('6913324275323537689724784638569909,100000000000000', '69133242753235376897.24784638569909');
    T('13962383460375585294405129665013837633,2000', '6981191730187792647202564832506918.8165');
    T('373341224221778946139808778111541154755014783291201,500000000000000000000000', '746682448443557892279617556.223082309510029566582402');
    T('2605595932904747914340060901727110356316231797,100000000000000000000000000000000000000', '26055959.32904747914340060901727110356316231797');
    T('5404218768000268681499,58562496', '92281223259341.07524206277', '59176667');
    T('837600757847923350654714829279011772308108410631278941774977080884126219,1000000000000000000000000000000000000', '837600757847923350654714829279011772.308108410631278941774977080884126219');
    T('39732046116239133,500000000000000', '79.464092232478266');
    T('3508566582308615186460661,50', '70171331646172303729213.22', '689');
    T('55037110921169089073232054697568652125721273823943129,1000000000000000', '55037110921169089073232054697568652125.7212738239431290');
    T('566293124809467513,2', '283146562404733756.42441', '2');
    T('43320564587705861,41', '1056599136285508.80459063125943036359093410684247', '43');
    T('959454884256,1', '959454884256.0872639565266385531414960733727818427', '1');
    T('91278743653117380466990456156,1', '91278743653117380466990456155.68579170481078286369193', '1');
    T('10122,25', '404.88', '8317919834');
    T('12366919721,2285', '5412218.69628', '3581');
    T('6717391976887166914502653755927330777815,1', '6717391976887166914502653755927330777814.993', '7');
    T('13432031244267392897695975,26625836', '504473596407166065.98553281106366012670736', '962581771');
    T('198826963117874099525548139368203,2', '99413481558937049762774069684101.4789260648567440', '2');
    T('1348726,601', '2244.1364219063828677', '603');
    T('976694301668980861868663901012070434704655861333,1000000000000000000000000000000000000', '976694301668.980861868663901012070434704655861333');
    T('297178549846227440886808646486219,32786695', '9063998364160444988029706.7602031555788099950276275841962', '66781952');
    T('22339526413138422840472802051271868021571230079710624149228880060499,100000000000000000000000000000000000000', '223395264131384228404728020512.71868021571230079710624149228880060499');
    T('9135248819601092928115011665372155588037116079546743946559,1000000000000000000000000', '9135248819601092928115011665372155.588037116079546743946559');
    T('5286330697037762023749,10', '528633069703776202374.9');
    T('107227092928577441190850028,357', '300356002601057258237675.148458846976702', '1739');
    T('9567629701592580680115114528171473874982843,10000000000000000000000000', '956762970159258068.0115114528171473874982843');
    T('16087152277929474327956249483660452531,7812500000000000000000', '2059155491574972.713978399933908537923968');
    T('89875669677258766402915335437973697536104973046538128991,1000000000000000000000000000000000', '89875669677258766402915.3354379736975361049730465381289910');
    T('923938919,1', '923938919.0', '9691412392');
    T('48428060142308830168757150336059283056510851633772347,1000000000000000000000000000000000000', '48428060142308830.168757150336059283056510851633772347');
    T('93051527972639804714713550878952900355,496672', '187350057930867463264918398619114.62767178339', '4005220');
    T('3497280820822833818,1', '3497280820822833818.0', '7323674');
    T('923369267213275370382645033,200', '4616846336066376851913225.165');
    T('83964374249481,1000000000', '83964.374249481');
    T('194123595638199705708781601234299234156024384,573715909', '338361883630805339422410197159617608.156695518792036842720875013789726', '847457002');
    T('5637607,236111', '23.87693500091710301251170853372825835205', '293373');
    T('96319079573841658175275890975870,1', '96319079573841658175275890975869.869419481278952979', '2');
    T('35413177676239984779155768701462876116,5', '7082635535247996955831153740292575223.20');
    T('17525144964454,27791', '630605050.716203087332881753643518191177425472803', '459536');
    T('99069927604575118603830348579,20000000000', '4953496380228755930.19151742895');
    T('4625959927338367711360075575124826216422187184789056538633051077924197,500000000000000000000000000000000', '9251919854676735422720151150249652432.8443743695781130772661021558483940');
    T('16687954096803,461', '36199466587.4251659487445747440075096128888226597', '715');
    T('48983711168012156360736451565279496359797,62500000000000000000000000000000', '783739378.688194501771783225044471941756752');
    T('951604336538762886268589543789943349215119,100000000000', '9516043365387628862685895437899.43349215119');
    T('3170484800431125268583,9881458', '320851922907644.324206306397298875', '10955847');
    T('2209917901057085830155650409703230456398682421,500000000000', '4419835802114171660311300819406460.912797364842');
    T('257559146244597067867426507016,341085825', '755115362078142848262.38823327237360274380050695878', '370161150');
    T('52302828211946727812235521578754597,100000000000000000000', '523028282119467.27812235521578754597');
    T('1548611009582576944501513330034309618,3581483467', '432393734007589359758871484.98019567096887564653', '4247950750');
    T('4160735782556259525559,100000000000000', '41607357.82556259525559');
    T('7810032276912931435247371001442767211661887,10000000000', '781003227691293143524737100144276.7211661887');
    T('2699214678525489726487560119191725243,6772462978', '398557317669177478741406878.339929884663593948389215779911509998918', '8700679367');
    T('234005756036522310409974828052077,2792665138', '83792987871115935565481.6170273247418609771373578473125', '3496871917');
    T('888930141417205,472', '1883326570799.16313674359847883331', '494');
    T('93731,249276', '0.3760129334673260070153210504', '257553');
    T('1884739237,5388890', '349.745353310236416512548553007633265749648', '7464665');
    T('43774993705091173299491664598042468178382578189423014141380153967959,1250000000000000000000000000000', '35019994964072938639593331678433974542.70606255153841131310412317436720');
    T('4990606690039884835950094377597511034291,50000000000000000000000000', '99812133800797.69671900188755195022068582');
    T('880848777307,10', '88084877730.70', '10550849');
    T('117189165353141,500000000000000', '0.234378330706282');
    T('3696766657880426699915213404988822821636621907807,500000000000000000', '7393533315760853399830426809977.6456432732438156140');
    T('136878149428912649758117905461610548699832354530413221,1000000000000000', '136878149428912649758117905461610548699.832354530413221');
    T('247112873960438003987375929845431961097478236423759,50000000000000000000000000000', '4942257479208760079747.51859690863922194956472847518');
    T('32090761171034587720265676015278308236143,372372', '86179307711198983060664271253688000.8060299915140599804368969', '803032');
    T('3614723886086134485880643254470683475022113,2000000', '1807361943043067242940321627235341737.5110565');
    T('865224903023822325262781,300218', '2881988764910239643.4017314085126964034449762676', '532601');
    T('611185005509628,9139', '66876573532.07440639019695013008058748', '4159302');
    T('1518595063109297520311975384,5', '303719012621859504062395076.8');
    T('49679411118006488713997,857', '57968974466752028837.802807564217277428', '895');
    T('88166434473746490289165987929145035778940246272,282622141', '311958695669729889595472238422909108805.2448313736325421133448741444545', '385416267');
    T('18629257493748797005,3', '6209752497916265668.3441723459', '10');
    T('3105722390606004795903600587554457,8015259', '387476236339462616978889963.2007471', '8184974');
    T('186363420925320576328934,21515', '8662022817816433945.104996514626241666816484709452865', '37212');
    T('2487576402639632794634694150901775257367,500000000000000000000000', '4975152805279265.589269388301803550514734');
    T('124421130067284418607320404,18230353', '6824943547022069106.79789930562507372336112189652537764238', '26966531');
    T('7137130635754072199223,3556', '2007067107917343138.139201473525306155357', '3856');
    T('112942288681912873203,25000', '4517691547276514.92812', '505601099');
    T('1431788138645392811436351585868528725106020877729648867279053,1562500000000000000000000000000', '916344408733051399319265014955.85838406785336174697527505859392');
    T('3234595666430613200570074711,125000000000000000000000000', '25.876765331444905604560597688');
    T('3771991535846550543543187642939268809849,113246271', '33307865261599214543171912856532.545869426464382214773370432169717071927', '1862546449');
    T('159328743982472447,10889954', '14630800459.07195264553', '62191463');
    T('19156099060539,200000000000', '95.780495302695');
    T('2459812065130292014604235393568931103851693517,25000000000000000000000000000000', '98392482605211.68058416941574275724415406774068');
    T('14081093976643671519,2', '7040546988321835759.5');
    T('1398718931796139131570704959571446329,250000', '5594875727184556526282819838285.785316');
    T('226057240496116726752712879,73578', '3072348263014987180308.147530511915', '99443');
    T('794112996051185221451708830987876881997,100000', '7941129960511852214517088309878768.81997', '106276919');
    T('8216044054806771996810629067,22837', '359768973805962779559952.229583565335247922821497', '25456');
    T('23807055304320404163339151165491853782535135991908506357626995217,500000000000000000000000000000000000', '47614110608640808326678302330.983707565070271983817012715253990434');
    T('3531469840127598977918199412295084056732447544259170233,500000000000000000000000', '7062939680255197955836398824590.168113464895088518340466');
    T('7074869431756164165861127913754839263,188036', '37625079409028931512375970100166.13447956773725419869171', '551328');
    T('116727432190485304215216347036665233873946214267,200000000000000000000', '583637160952426521076081735.1833261693697310713350');
    T('322222287817074340496094,4519846351', '71290540163115.0139280683703046524993660939348992959', '4821696804');
    T('3220195518339602136340818267702090777236019441295,6573210637', '489896900642954724826783041016686466674.0816389412472740753393', '6620677169');
    T('4979158772221132031066953552941723189,1000000', '4979158772221132031066953552941.723189');
    T('8737425880160915911998771,1', '8737425880160915911998771.12725422920', '1');
    T('323045767198973747297480019029873,446', '724317863674829029814977621143.2130109419902482321169979691406', '573');
    T('640289857462018471811912,23805611', '26896594145893523.6659925258797173744700212383', '26827286');
    T('4449327148350034505583562687233,50000000', '88986542967000690111671.25374466');
    T('25753316144627889753,100', '257533161446278897.53', '6528003920');
    T('3986906378997449549838631393557,100000000000000000', '39869063789974.49549838631393557');
    T('6265774677875908010545250165153745,700772', '8941245765920881557118792082.380210681934346581057360805031205825124', '928104');
    T('230083713653,500000', '460167.427306');
    T('19377991,21577587', '0.89806107606008', '49239725');
    T('38710794736631001569771,1', '38710794736631001569771.012260896821016342565', '28');
    T('10037882653,349139894', '28.750317066316116827', '647871852');
    T('3333245366687,4471008', '745524.3575245224343350924369527004181', '6167413');
    T('26769493351047347344386695666057,50000000000000', '535389867020946946.88773391332114');
    T('1621363436291186251405436340171,5232692', '309852641105416915691853.512526821758252648366658861942981640', '7299049');
    T('4550392744783143007699048628173,20000000000000000000000000000', '227.51963723915715038495243140865');
    T('797141000219512098778764678119786879,867317228', '919088165765700780912845741.5119936704405', '962488946');
    T('14904132191968276587297737169053358300443,200000000000', '74520660959841382936488685845.266791502215');
    T('91470701131037383295747,1000000', '91470701131037383.295747');
    T('27753892633682075,3', '9251297544560691.601', '4');
    T('1788334345204020429620861096201,50000000000000000000', '35766686904.08040859241722192402');
    T('4612688956079856295802362391039,100000000000000000000000000000', '46.12688956079856295802362391039');
    T('161347204015,23', '7015095826.738876208769833554', '47');
    T('20600465972959782785256431550761005019024,339', '60768336203421188157098618143837772917.474927302472', '502');
    T('1062891113749632541,702810', '1512344892289.00064170970827727498908573320667399924', '2943629');
    T('286820046488752611155983,5611503', '51112874124588833.1799845781067795486147680004500944874455', '9222645');
    T('55268920261272895079460750602209,6', '9211486710212149179910125100368.0869540216384846614405', '6');
    T('147110482766595420803446985119783296,1', '147110482766595420803446985119783295.962905368965408966', '8');
    T('8584260220485230830750630108611597500009840276389875279,100000000000000000', '85842602204852308307506301086115975000.09840276389875279');
    T('45997967047497611745453,100', '459979670474976117454.53');
    T('25322562031741252483400559,1000000000000000000000000', '25.322562031741252483400559');
    T('7588887,2056', '3691.09289882762466516742', '8887');
    T('823890318784964048554080494714815498182966138912381,100000000000000000000000', '8238903187849640485540804947.14815498182966138912381');
    T('9304855287990530136771405094519,1', '9304855287990530136771405094518.82504', '2');
    T('47011776728314188266315200417504146431,1000000000000000000000000000000000000', '47.011776728314188266315200417504146431');
    T('2466085134442399070795709748333417449542855462250787166784205121205109,500000000000000000000000000000000000', '4932170268884798141591419496666834.899085710924501574333568410242410218');
    T('3442044952553264476,7', '491720707507609210.91163844234598216', '7');
    T('80868288238431,874004803', '92526.1371114353018034864163941998778367260', '931966805');
    T('6985103397660981071,7353', '949966462350194.624098996515677191782270599392137132201', '7793');
    T('468653409753396391457767631,5000000000000000000000000', '93.7306819506792782915535262');
    T('60,1', '59.9083995049887984453397720', '2');
    T('1585744058865312687896395129349,250000', '6342976235461250751585580.517396');
    T('20479199716624053465947561943251653,500000', '40958399433248106931895123886.503306', '4535856588');
    T('3479780591056475652402107167313793111293,5', '695956118211295130480421433462758622258.6');
    T('467879214344174541429051133613371644281991908668101132848827627,50000000000000000000000000000000000000', '9357584286883490828581022.67226743288563983817336202265697655254');
    T('3780398268842275439375533,500000000000000', '7560796537.684550878751066');
    T('599108489,11', '54464408.05188181950959', '11');
    T('215021045713727250783388053394109267617,4166009', '51613197598403472192063928185010.9463558528077204341344511373832016128195', '4179908');
    T('159241433857,5000000', '31848.2867714');
    T('160936670389208853641372968978236488980668850585131,500000000000000', '321873340778417707282745937956472977.961337701170262');
    T('2370282706660426694332478,23799509', '99593765008363268.9368708404866667088069121270797650', '24298070');
    T('597763381558982555051400082606522098999937129,100000000000', '5977633815589825550514000826065220.98999937129');
    T('4431461,1000', '4431.461');
    T('8250115774778471280985677882646125194730659155862382409,100000000000000000000000000', '82501157747784712809856778826.46125194730659155862382409');
    T('12205438,45', '271231.955313', '47');
    T('19202320927991171,2', '9601160463995585.5');
    T('73746355649075260,3', '24582118549691753.331713657589540437', '9');
    T('47309,1769001', '0.026743342711584657', '4015485');
    T('1926643365791669364845561,8927875671', '215800873218910.819759058112', '8935017174');
    T('8952538174202125514384339,585', '15303484058465171819460.40854848482726775573160880288331876238', '946');
    T('171044475643258691052591881190402011,2500000000000000000000000000000000', '68.4177902573034764210367524761608044');
    T('1417176227509505,17149674', '82635753.164142070572304998841', '20372906');
    T('3380729603876017655487973251773669209031,27845', '121412447616305177069059912076626655.0199676781941', '40016');
    T('890269765213404348432913996063275824595152162733,100000000000000000000000000000000000', '8902697652134.04348432913996063275824595152162733');
    T('664601293959812682633636004618957911435231667680903297856881,20000000000000000000000', '33230064697990634131681800230947895571.76158338404516489284405');
    T('778481676645324115540463089347,20000000000', '38924083832266205777.02315446735');
    T('75705502543433,7812500000000', '9.6903043255594240');
    T('81546124479956359876565726669992644826500740941,100000000', '815461244799563598765657266699926448265.00740941');
    T('24701,4', '6175.250206214726482', '51');
    T('5311817589061598103376746873687003094342952495326331677278701943,12500000000000000000000000000000000', '424945407124927848270139749894.96024754743619962610653418229615544');
    T('587612851301963849841025647818575969,100000000000000000000000000000000000', '5.876128513019638498410256478185759690');
    T('717873356,808272029', '0.88815810796788070', '942302647');
    T('8684149473480042958819491590345594042855,92', '94392929059565684334994473808104283074.5108197400739669029736', '109');
    T('557213,6836', '81.51155646233081505773189164', '13939');
    T('6358852540742501026007669889675383610909273172987401669,100000000000000000000', '63588525407425010260076698896753836.10909273172987401669');
    T('9613,1', '9612.9773812129992689657249691195343', '10');
    T('286113583488621159688740779858763238199941269901,1000000000000000000000000000000000000', '286113583488.621159688740779858763238199941269901');
    T('462642039300426299179091176857436950199,413144', '1119808200773643812276327810297225.54411778943845784669037258980', '1448446');
    T('627912942633907772006040,1', '627912942633907772006040.0');
    T('8887831015869015364122580184912527,100000000000000000000000', '88878310158.69015364122580184912527');
    T('25269814867297462150639281025731,5000000000000000000000000000', '5053.9629734594924301278562051462');
    T('99529184605,1', '99529184605.0', '650785');
    T('1658758680465486368314728898678949517,1981', '837334013359659953717682432447728.17617366544553', '11197');
    T('7,1', '7.056759118252905562', '2');
    T('2194117028293242686416582588432860977,5000000000000000000', '438823405658648537.2833165176865721954');
    T('47876011607044241,59126929', '809715850.573674154461835825714251915075523523', '4052259112');
    T('10497951457730554025955004980575188440216805387873,200000000000000000', '52489757288652770129775024902875.94220108402693936500');
    T('185573471761473313640214816617129543,200000000000000000000000000000000000', '0.927867358807366568201074083085647715');
    T('13136066791,2000', '6568033.3955', '857278646');
    T('786449314441,10000000', '78644.93144410');
    T('258069327759691253346816792459688368610750867,50000000000000000000000000000', '5161386555193825.06693633584919376737221501734');
    T('2400603934155970960994087661933,100', '24006039341559709609940876619.33', '52973663');
    T('53787312868755909573338050263817591013,1000000000000000000', '53787312868755909573.338050263817591013');
    T('65846043893884622068872344771627703174979234269,1000000000000000000000000000000', '65846043893884622.068872344771627703174979234269');
    T('413208107310727259011301805,1', '413208107310727259011301804.818566766164839933798622982', '1');
    T('380767438123951525794980657112504797825,680920177', '559195410835874710458140319012.02539607663880400492014675272', '689501446');
    T('3834864268417157885795059233263411,50000000000000000000', '76697285368343.15771590118466526822');
    T('291722167388405104196504,669403279', '435794350789854.8913389831184259847904358843278386', '904770649');
    T('32620215359,500', '65240430.718');
    T('1796562823633852909577,2', '898281411816926454788.285872', '2');
    T('427388755061110023176602282445173618523399130350681998042193349558569077343,5000000000000000000000000000000000000000', '85477751012222004635320456489034723.7046798260701363996084386699117138154686');
    T('6120788215815611148178349402985667902090005391,2500000000000000000000000', '2448315286326244459271.3397611942671608360021564');
    T('5152513511677523,168', '30669723283794.7797667708156960783', '283');
    T('310212151,2', '155106075.4712372', '4');
    T('4699524488896952877010,3', '1566508162965650959003.35067266814212548734090212433985', '5');
    T('70596591,100000', '705.96591');
    T('69579827425857277909,7355044059', '9460151002.1841295279996092270403052', '7426098425');
    T('96855367597936633192863,100000000', '968553675979366.331928630');
    T('8688367572560606063,100000', '86883675725606.06063');
    T('28879513,100000', '288.79513');
    T('18800652833943208658107123506275565438787,40', '470016320848580216452678087656889135969.675');
    T('1146931396095000518282937400043083943334700,4602527', '249196016904409364308549933556736102.435618520', '8225878');
    T('2019756551427915738961933054973199346876419,5000000000', '403951310285583147792386610994639.8693752838', '741757710767379773640708586558');
    T('159260774150855645667623300326629,10000000000000000000000000', '15926077.4150855645667623300326629');
    T('63365344813375864020105079652680969461686025291,10000000000000000', '6336534481337586402010507965268.09694616860252910', '688235748647978162333833362258291330722279331');
    T('1461532669,5000000000', '0.2923065338');
    T('390058219943639629654109687634774504333495620881,1000000000000000000000000000000000', '390058219943639.629654109687634774504333495620881');
    T('299225805773161690264821787540667,5000000000000000000', '59845161154632.3380529643575081334', '2584812000841991553143215804559645');
    T('81603234217578447206418101677952309588084613059984751910997827257167,156250000000000000000000000000000000000', '522260698992502062121075850738.8947813637415235839024122303860944458688');
    T('2565677724908492708768279520615817013536305135641817470099879036,374196298701793888137550171191', '6856502145557413849706463648425482.701677508664756140796680003271200', '869790929515477340097538635993');
    T('9612780561,1000', '9612780.561');
    T('936512323689541548094571523023523188200531,1000000000', '936512323689541548094571523023523.188200531');
    T('152234006509072541435610651574042873,25000000000000000000000000000', '6089360.26036290165742442606296171492', '11857604157972831619815396931384638829511784605');
    T('64208688601822739753865109352559333654371,2631857012', '24396723799606914113715957967308.95259380109514855361003', '576672018138');
    T('13680813615256177,5000000', '2736162723.0512354', '5926568749173865070557');
    T('12399570290333756330011977724487093294866653,500000000000000000000', '24799140580667512660023.955448974186589733306', '67333132803983266789655187746269204');
    T('4506349014426198904317777807334981419,500000000000000000000000', '9012698028852.397808635555614669962838');
    T('725431855797035508085239096174824869288283239,100000000', '7254318557970355080852390961748248692.882832390');
    T('355915386841717685353688119260713,100000000000', '3559153868417176853536.88119260713');
    T('2914867717398323136217,100000000', '29148677173983.23136217');
    T('1326209433713162649634239109961037850584868175033,10000000000000000000000000000000000000000', '132620943.3713162649634239109961037850584868175033');
    T('4778160350176337979760052281234320029,10000000000000000000000000000000', '477816.0350176337979760052281234320029');
    T('86665339051641131146005186162442032968947419505854245963,125000000000000000000000000', '693322712413129049168041489299.536263751579356046833967704');
    T('430351447085339973547588264756339950843789831043,500000000000000000000', '860702894170679947095176529.512679901687579662086');
    T('55525130114430055347694274663,763982638008', '72678523505724782.1378481069695424350', '850916842740');
    T('220792158460126762184413874181,10000000000', '22079215846012676218.4413874181');
    T('94783059874905327086792198452757929148500148142136001437,250000000000000000', '379132239499621308347168793811031716594.000592568544005748');
    T('6488407113121033385211896005019711681817,100000000000000000000', '64884071131210333852.11896005019711681817');
    T('47333842438492360793095254962373810441359677680855935665786394332060804133207,500000000000000000000000000000000000000', '94667684876984721586190509924747620882.719355361711871331572788664121608266414');
    T('2367138419900331631616148353601561797867243729737608115661,100000000000000000000000000', '23671384199003316316161483536015.61797867243729737608115661');
    T('88626039212907148865042680324936804431893182827,10000000000000000000000', '8862603921290714886504268.0324936804431893182827');
    T('134036367111046300205853082693215909752923821078333787,2000000000000000000000000', '67018183555523150102926541346.6079548764619105391668935');
    T('1290426619008005556475919972597725061539536069,25000000000000000000000000000000000000', '51617064.76032022225903679890390900246158144276', '964910734608101784280721264706455020684');
    T('51515831610611387382745267,6250000000000', '8242533057697.82198123924272');
    T('7952216990048092855359353056111362078447011828178913246364701327,10000000000000000000000000000000000', '795221699004809285535935305611.1362078447011828178913246364701327');
    T('538015045243626212422430662564753243027945021916339128227,100000000000000000', '5380150452436262124224306625647532430279.4502191633912822700');
    T('43577529369368712265446116906547230280422309612453,500000000000', '87155058738737424530892233813094460560.844619224906', '796419722408512060993');
    T('1161347707886748476637875642925084188220298315962357581,891732452430846866981741483', '1302349942206247176394632274.42078366656260573729837386635563', '979302956680189066606017794');
    T('1528624826136387009578521,25000000000000', '61144993045.45548038314084', '8024287939412517568729347651281075');
    T('3355746698476190015402680252180735699888853,50000000000000000000', '67114933969523800308053.60504361471399777706', '8915896738291537715389083695879542442064029');
    T('980289651663460623618247173,100000000000000000000000000', '9.80289651663460623618247173', '644505847543429115712324396');
    T('51055624915620324003432130818561734442539,10000000', '5105562491562032400343213081856173.4442539', '2809024769655863489136966521');
    T('41875396716460469922985854768609021869387681761,125000000000000000000000', '335003173731683759383886.838148872174955101454088', '216625103073443887003029267945');
    T('105780502837966983,13631', '7760289255224.633775993192903826796476494397758857808', '13861');
    T('1938155266888975965598512083832127,251657941372', '7701546219135603482029.870015176732906490144727404299117584', '547888565206');
    T('99954595750029,10', '9995459575002.9');
    T('4249542002102445996878781,10000000000000000000000000', '0.4249542002102445996878781');
    T('485671,250000', '1.942684');
    T('13356987302624015208420862387472998,746681507193', '17888466734413902028886.0311842409055156116317714302570867', '849696537110');
    T('2668448503233496883227988430416595281601,1000000000', '2668448503233496883227988430416.595281601');
    T('3640373983168722911281713865270438495806072129379784031313632242881734911,50000000000000000000000000000000000', '72807479663374458225634277305408769916.12144258759568062627264485763469822');
    T('1818728887,200', '9093644.435', '476185163436332077972781497090111043581420');
    T('802345758447757541858014527779737950265044017,1000000000000000000', '802345758447757541858014527.779737950265044017', '8387249919619107142865224089883458341020176321');
    T('595208804134258162637737,100000000000000000000000', '5.95208804134258162637737', '900644264372865329335187168932351592714013');
    T('13988441958679191876839735942604595286414864789,100000000000000000000', '139884419586791918768397359.42604595286414864789', '5655831590440584955875');
    T('2085083269139810791270306641073124298284478872355022690995133,50000000000000000000000', '41701665382796215825406132821462485965.68957744710045381990266');
    T('63143145632769116262051260423942651269,1000000000000000', '63143145632769116262051.260423942651269');
    T('1407997631298057561573924769518255515279313425669860717,10000000000000000000000000', '140799763129805756157392476951.8255515279313425669860717', '327937261477572180619558003617237646356733892');
    T('270833075343099303551865182416448009,1000000000000000000000000000000000', '270.833075343099303551865182416448009');
    T('9108779,100000', '91.08779');
    T('1290586935123071,1000000', '1290586935.123071');
    T('469132037299556835463,1000000000', '469132037299.556835463');
    T('455380269621149826,1', '455380269621149826.0', '29816950548028');
    T('59036688354560778646766626225702807845034056189868056799,100000000000000000000000000000000', '590366883545607786467666.26225702807845034056189868056799');
    T('134877717700773438669846465447856699885143,400000000000000000000000000000', '337194294251.9335966746161636196417497128575');
    T('16122989466646638395384532997642495,812372', '19846805978845453062617289859.377840447480619999019072107458825800245', '1022628');
    T('41676689473,419541', '99338.776122', '477053');
    T('948783386589836295531621640054416694045428069792652292815184843252620407087997,10000000000000000000000000000000000000000', '94878338658983629553162164005441669404.5428069792652292815184843252620407087997');
    T('183987075285353368451987553220870945911,100', '1839870752853533684519875532208709459.11', '4771405818636333119008616393205342343236');
    T('8284467081338201094106418401321,1000000000000000000000000000000', '8.284467081338201094106418401321');
    T('126835799338565188592427149928234243606839,20000', '6341789966928259429621357496411712180.34195', '487302037789409112102552330066366980508855813174');
    T('202777274382581430728017680787579982546489517,229892480925189939516479', '882052660298045344051.7919993163515770841250643782458114275328', '746044076719153861455150');
    T('3738015590431,5', '747603118086.2');
    T('301106896154595422209552298959107137373547203141065553859057389421821,250000000000000000000000000000', '1204427584618381688838209195836428549494.188812564262215436229557687284');
    T('341334346359,543174817', '628.406059478637427329224571381873725166216', '876128137');
    T('465138295089535296259774745663038553386930899198195570591,20000000000000000000000000000000000', '23256914754476764812988.73728315192766934654495990977852955');
    T('82769483306049,10000000000', '8276.9483306049', '33839355334');
    T('556953498133884269,68', '8190492619615945.13225248677717', '92');
    T('45318755494241836794770073,1000000000000000', '45318755494.241836794770073', '1253935394801806614019883297829417163126525');
    T('1858914525035831826768925339743,200000000000', '9294572625179159133.844626698715', '31391560509860145405596');
    T('425343351349790045976513159948568763820438862683367,100000000000000000000000000000', '4253433513497900459765.13159948568763820438862683367', '7721074673031296384304307266121470119427');
    T('71354211162897369216945850343222823881026041367233,100000000000000000000000000000', '713542111628973692169.458503432228238810260413672330', '144939713832937271061627357504826456742');
    T('144714939860168363453308091038982,4284977', '33772629318703079025466902.398538428561', '5618239');
    T('2317035080099534328156555098955006964997947989671,5000000000000000000000000000000000000000', '463407016.0199068656313110197910013929995895979342');
    T('12220710712628394678137945125840492835,295726726654365551281703', '41324336325243.639147174633347897967505321910146183513', '461406359261841272137912');
    T('4164103,50000000', '0.08328206');
    T('7502614012109051682244916192266750188051123,100000000000000000000', '75026140121090516822449.16192266750188051123');
    T('323880445913,500000', '647760.891826');
    T('58136053747539046641739287759965480752,70045887', '829970983842906388518419186553.60136637287497', '86239491');
    T('50255437,100', '502554.37');
    T('450233554293111245876449378888745591234681102067569,5000000000000000000000000', '90046710858622249175289875.7777491182469362204135138', '3810754204488023014210763363');
    T('200222822546125679699429097695077,100000000000000000000000', '2002228225.46125679699429097695077');
    T('3783671589181030205501527752342356874517572828977,500000000000000000000000000000000000000', '7567343178.362060411003055504684713749035145657954');
    T('40048820082402526973,30', '1334960669413417565.766541760088934259233455362', '55');
    T('19682241,5000000', '3.9364482', '2938089409064663302584899743202354591655563930');
    T('3847570170988597157763381309021862223325329,1000000000', '3847570170988597157763381309021862.223325329');
    T('302871942581513778978519519,50000000000000000000000000', '6.05743885163027557957039038', '7739846393033927255652731261656491478657146');
    T('326080663059742589574506673375771249877,100000000000000000', '3260806630597425895745.06673375771249877', '550868828960159419120535470417267167');
    T('96832187539664765931343834203105130885992737,2500000000000000000000', '38732875015865906372537.5336812420523543970948', '3389413421383652579250716211200348');
    T('271852185141591302785258831339,100000000', '2718521851415913027852.58831339', '326060519878426625637075641056508361');
    T('48700692997319228087245485443026803016745947787367383,5000000000000000000000000000000000', '9740138599463845617.4490970886053606033491895574734766');
    T('40758214623530792633726096531289862600019220056526601,50000000000000000', '815164292470615852674521930625797252.00038440113053202', '5221210587873615917315733');
    T('1241221035324719632397392464340037510556481149089904847941523,10000000000000000000000', '124122103532471963239739246434003751055.6481149089904847941523');
    T('48037863508976458691221362323501021215551,4882812500000000', '9838154446638378739962135.0038530091449448448');
    T('193622000346959273707446061164100672839811,2143395561', '90334235952520606021468783458071.49539011805390223029', '2265033200');
    T('67783685294777359438349711011,20000000000000000000000000', '3389.18426473886797191748555055');
    T('12617473737636720451218269740609659622541624,159627839', '79043065524659019227957284697750369.36095854808884558276454809642992365632', '174329681');
    T('9684474818082811551223783413257727824992067,1562500000000', '6198063883572999392783221384484.94580799492288', '49209400621412262419');
    T('17505591753702885777016987703565333702388539,2500000000000000000000000000000000000000', '7002.2367014811543108067950814261334809554156');
    T('947913360646751474137651443537325162671,95614802879262668687699', '9913876639412481.794682748986875680160639360860844', '97032668484031432706648');
    T('87432797,2000', '43716.3985');
    T('52266359279671923770562530491,1000000000000000000000000000', '52.266359279671923770562530491');
    T('934519820781022674420129570582560225765920330071,12500000000000000000', '74761585662481813953610365646.60481806127362640568', '8642213980970552163513');
    T('50083810531494711281387558,56570303851847', '885337484887.12586165119274913582149678253727190', '78326898482642');
    T('3972190938628200176537,10000000000', '397219093862.8200176537');
    T('18121256332358987866268770477428593607665037,200000000000000000000000000000', '90606281661794.939331343852387142968038325185', '921193442621072417175235574539201255016');
    T('14987898155158472518478544881232078802611860873609524963,2500000000000000000000000', '5995159262063389007391417952492.8315210447443494438099852');
    T('3239212309823789165605527377995418545725605705,14170665490304070723367973333508266', '228585757813.57202362946598425863064253567493626119', '86979626336746925002222279115462537');
    T('122185041583802144366142,5', '24437008316760428873228.4', '8000951389854267175050');
    T('657574695679001861295665451464661555661,10000', '65757469567900186129566545146466155.5661', '2507718086460574385741706208476049260047621469');
    T('6360428604897138048594396059747553,10000000000000', '636042860489713804859.4396059747553', '244160781758980');
    T('21351223921251662111677047871929368,23', '928314083532680961377262950953450.7838973425314927418923934606452990487', '30');
    T('289499461511853909335456535686028099405454280671300094239293194461506601099,5000000000000000000000000000000000000000', '57899892302370781867091307137205619.8810908561342600188478586388923013202198');
    T('976859569751666674770433143725255060903,2000000000000000000000000000000000', '488429.7848758333373852165718626275304515', '192870444813125961651197355653979526967');
    T('8711911,24', '362996.29144884884', '91');
    T('47961961919420699035752271439053656196891,1000000000000000', '47961961919420699035752271.439053656196891', '1432836513794575009230872452783621');
    T('36002186243558704970796134460620318473988856831,2500000000000000000', '14400874497423481988318453784.2481273895955427324');
    T('347658515811,125000000', '2781.268126488', '18168322069934165333666594401837841045656351551');
    T('4744824007215869949704323431169019043851337919936522927179,10000000000000000000', '474482400721586994970432343116901904385.1337919936522927179');
    T('18647457854156074231876861461127052818609195520980488636185434563,50000000000000000000000000000000000', '372949157083121484637537229222.54105637218391041960977272370869126');
    T('88636422031310834818570712943870555589,10000000000', '8863642203131083481857071294.3870555589');
    T('8907393333063962550030633954046846122487647,500000', '17814786666127925100061267908093692244.975294', '738024193644869313081788449951849');
    T('5347810730899001839326028336703,1000000', '5347810730899001839326028.336703');
    T('83499427757,10000000', '8349.9427757', '537893050789928685742496508297');
    T('91030793284677869802,1', '91030793284677869802.0', '54365179780137798417812339693');
    T('139576870289544319,15393617308715296', '9.0671911280086886680514259430164', '34314606681915854');
    T('526728365945557273665550256298589878104412618423937,1250000000000000000000', '421382692756445818932440205038.8719024835300947391496', '914852587643922476861639338196683814291555');
    T('818477009119923314771842558536624613582668934883,20000000000000000000000000', '40923850455996165738592.12792683123067913344674415');
    T('164078868342751098016,64915', '2527595599518618.1624586', '69194');
    T('293978517263308243537157587247782681150732925,5389019', '54551397436770633678811966936428073671.80054941353888320018', '5423880');
    T('16896047083967,200', '84480235419.835');
    T('104972899545933771066622411344760258240458360899093,10000000000000', '10497289954593377106662241134476025824.0458360899093', '7205746832408993529175409871720727602599');
    T('721550565141436165,78491572038293', '9192.7138978617930051035079340183063425', '114711958041391');
    T('642808470971740957339375857676583667488830045491443,2500000000000000', '257123388388696382935750343070633466.9955320181965772');
    T('25547609239638147055212848154480190930531,1000000000000000', '25547609239638147055212848.154480190930531', '956611854763149744692116271165');
    T('63725524120161766602391195314216892192697,100000000000000000000000000', '637255241201617.66602391195314216892192697');
    T('5135812052028819027098843325324594885149,100000000000000000000000000', '51358120520288.19027098843325324594885149');
    T('53823225503436626111601,100000000000000000', '538232.255034366261116010');
    T('84772164557665415263075633416115167894928760949624252554563147439736263703,10000000000000000000000000000000000', '8477216455766541526307563341611516789492.8760949624252554563147439736263703');
    T('6769664484487816999120299051126228678705642768668574797742,193743613880784562681728605', '34941355479481073234753500993749.9459542322571041925382270549294049381061', '211314918688969418804736477');
    T('924366516845377679336424,1162633928094070467838273', '0.7950623962614875520469822998', '1975805181618643839348076');
    T('77572007179,100000000', '775.72007179', '11829149976320');
    T('334089623626745537032388435915351225103753617,50000000000', '6681792472534910740647768718307024.50207507234');
    T('13498748436005386978369839564419223032439008199537017,100000000000000000', '134987484360053869783698395644192230.32439008199537017');
    T('90490381500830768647702025423,125000000000', '723923052006646149.181616203384', '961797459451554264950219276775915317946969734');
    T('5122161348451940740499291,10', '512216134845194074049929.1', '394574323186626013065489993123031');
    T('647855253,1', '647855253.0');
    T('6739958677034739744526166360174781560887228448458551401840878517716909,1000000000000000000000000000000000000000', '6739958677034739744526166360174.781560887228448458551401840878517716909');
    T('29091323665776070746692915891291406784815,515462320066', '56437342814992192098354407772.812600292121', '529393438698');
    T('496334264543794411305432918384584691,50000000000000000000000000', '9926685290.87588822610865836769169382', '29049581566777794866190221032687140418');
    T('8753315153638170826059223897506472318301157473753,1000000000000', '8753315153638170826059223897506472318.301157473753', '85364248144953441121602575793592689255772062');
    T('403542021675968571796035012464419951,4287', '94131565588049585210178449373552.589456498753402006170187050920091', '7569');
    T('594350241966914532560460828210265427161371137,10000000000', '59435024196691453256046082821026542.7161371137', '37574267571308361093683177199409609584373');
    T('15073005193012506,4955', '3041978848236.630877878990635453002381471', '5133');
    T('13262456470877201216542736093570139,200000000000000000000000000000000', '66.31228235438600608271368046785069500', '774102365994926801596084747934965835393944101916');
    T('294594117792262322334048360348006989133,100000', '2945941177922623223340483603480069.89133', '5284650921380');
    T('2036856646580353988754209669686648294707,1000000000000000000', '2036856646580353988754.209669686648294707', '29099065293427480893828927261338');
    T('180007401117801177,24077', '7476321847314.9136935645446918134', '26301');
    T('754812580262694694896295635929668953,1000000000000000', '754812580262694694896.295635929668953');
    T('46737787953832186777355613171050728218666711,500000000', '93475575907664373554711226342101456.437333422', '691648681170749354');
    T('190706002804671951924923171009850382212424001290162408147668760121797397,100000000000000000000000000000000000', '1907060028046719519249231710098503822.12424001290162408147668760121797397');
    T('876656568077577748604467778917147906790921181,5000000000000000', '175331313615515549720893555783.4295813581842362');
    T('6537572759474510978019763055947,2500000000000000000', '2615029103789.8043912079052223788');
    T('4084064084496559348260554164312115397804071,10000', '408406408449655934826055416431211539780.4071');
    T('7624072166846972160236154899045401668245306574381,10000000000000000000', '762407216684697216023615489904.5401668245306574381');
    T('177905487516777613587301880159113906479,195312500', '910876096085901381566985626414.66320117248', '6176184286');
    T('61831932067541525703246230050884,1', '61831932067541525703246230050884.0', '38989042121550969073469994280949');
    T('2726,501', '5.4411176792808', '776');
    T('303592203692208117214,635873', '477441570395673.5342025844991732249', '649536');
    T('550810457267856795577333963103167983,1000000000000000000000000000000', '550810.457267856795577333963103167983', '613145053953638371365146050566423');
    T('9612212037,125000000', '76.897696296');
    T('19854238142631550950262260304626082745471,37901467699', '523838240257787937603272504463.70854524809625103906310953', '40952250789');
    T('4544923928273434665236266014745010426628274,1096333144070264351987790901', '4145568299978486.279365051939148738458218639823299', '8537712898310208075262736912');
    T('290281008735066620365246906258552421932457445333,10000000000', '29028100873506662036524690625855242193.2457445333');
    T('1241037210471406591442469574953248591923111373602168667041493,2500000000000000000000000000', '496414884188562636576987829981299.4367692445494408674668165972');
    T('1861288631961340435812093,5000000000000000', '372257726.3922680871624186');
    T('4606419796132285145444663293643444818654488971283369022438656603,1000000000000000000000000000000000000000', '4606419796132285145444663.293643444818654488971283369022438656603');
    T('1669553057719,10', '166955305771.9', '8904226427748599439');
    T('12986349851069138804910894156891141,10000000', '1298634985106913880491089415.6891141', '899784234279572148861700827');
    T('67976496070837962006325629822530505039572859443166349265322123,1000000000000000000000000000000000000', '67976496070837962006325629.822530505039572859443166349265322123');
    T('14888274078142413593939188117,30', '496275802604747119797972937.2341911747299802284464386794810513', '39');
    T('1864264547760926364924249007253274324985177177877,100000000000000000000', '18642645477609263649242490072.53274324985177177877');
    T('641495287700451270339673,1782139719136440677413', '359.957909479339969376522560097960864', '3056779573484499009088');
    T('35386938150365066827946983062788302882742764047401931324672517379,500000000000000000000000000', '70773876300730133655893966125576605765.485528094803862649345034758');
    T('4777468662711829819443534803439171840234159,178583972921', '26751945230971166111811483364592.179982673703975838988713621959299431584', '437519908206');
    T('3483541015477817879719467546289485218183998679,2000000000000', '1741770507738908939859733773144742.60909199933950', '83159072643084822474226334121133568627');
    T('111029882872498,4311275609', '25753.3716101836903', '5409449457');
    T('5720723061390066,1169615', '4891116359.990309631802', '4121879');
    T('231690198058125307182393583380655131,100000000', '2316901980581253071823935833.80655131');
    T('785496570070925429293,20', '39274828503546271464.652502190252240978711699', '22');
    T('32844860194853812869198888570842117997979071035777472888657362673,5000000000000000000000000', '6568972038970762573839777714168423599595.8142071554945777314725346', '905302824861571794974789992');
    T('16498218387700361806672647974301337909169,6250000000000000000000000', '2639714942032057.88906762367588821406546704');
    T('242421647746889969,6067935407799844', '39.951257133567439718340905', '190386947376923275');
    T('84558420761363869643064360953349,10000000000', '8455842076136386964306.4360953349', '6083651091899923443690');
    T('3598344761346319138456455141169710488001767817863349,500000000000', '7196689522692638276912910282339420976003.535635726698', '6036430781596647112651702874946851436073');
    T('824416533850205237891728394872441,1', '824416533850205237891728394872441.0', '1270229614801297875');
    T('335030447307372600715767872105686598725232815951617004014831,50000000000000000000000000000000000', '6700608946147452014315357.44211373197450465631903234008029662');
    T('5315911067632204213000620089949667373095344325293668193104565,2501970004127297621628283', '2124690167693048092976376585227236996.721538833089666153400428951073', '171422422880513480794848980');
    T('82399859561,1000000', '82399.859561', '8007741850781315');
    T('9892854397114191944592435550624273099,500000000', '19785708794228383889184871101.248546198');
    T('482407510494718039597752127429518561787157,100000000000000000000000000000000000000', '4824.07510494718039597752127429518561787157');
    T('999219844563549589130339540247689,5000000000000000000000000', '199843968.9127099178260679080495378');
    T('32624665975434667774707373884930901627548640830986443495723665279,50000000000000000000000000000', '652493319508693355494147477698618032.55097281661972886991447330558');
    T('73322735201975172815448646322842399037093611,10000000000000000000000000000000000000', '7332273.520197517281544864632284239903709361100');
    T('5584344580393,2000000', '2792172.2901965', '67978301406759380237407609039081408522371172962');
    T('170698813505081259998189073431460004483981154768324955353,100000000000000000000000000', '1706988135050812599981890734314.60004483981154768324955353', '8303823206850919910010652520558244731932');
    T('703,10000', '0.0703');
    T('21213217171037545905806,1377', '15405386471341718159.6267247682855942563690178641195095', '2077');
    T('8296927175846011,250000000000', '33187.708703384044', '211191546934584837648');
    T('409552981550509729869340257571088152877592694961334449207,50000000000000000000000000000', '8191059631010194597386805151.42176305755185389922668898414');
    T('155027340589770073651612776,1737956327', '89200941462880657.099280939526163363777810267831118712467', '4986922301');
    T('167120807435010524556182779027297683324205827407507983577,436666647901481563398', '382719422786590845258091621853314922.96446004593674086717010', '702557026070523204786');
    T('441178720520318407454911830068962225120047224526619130534738920629112747,100000000000000000000000000000000', '4411787205203184074549118300689622251200.47224526619130534738920629112747');
    T('1869381106689425601212408179,250000', '7477524426757702404849.632716', '4624982');
    T('3317200487984067615668861775794559427256935200,6585479539550110485794411', '503714341235457458568.866804056313036956801153194', '6681245709808538985864323');
    T('3,4', '0.76011435975445459973959', '6');
    T('354884114822406966217620182813482001432151618849817803161826753,50000000000000000000000000000000000', '7097682296448139324352403656.26964002864303237699635606323653506');
    T('25087,50', '501.74', '85274653480326161797');
    T('650965999980720974038905679127117623356653858122009681,100000000000000000000000000000', '6509659999807209740389056.79127117623356653858122009681', '183925089257431510839976366871018952574324990');
    T('1751490441040021871522373727331,2000000000000000000000000', '875745.2205200109357611868636655', '75633648534698941212843963259058');
    T('26023193025631228939744774315993248032983335603365475283635011873374371768389,40000000000000000000000000000000000000', '650579825640780723493619357899831200824.583390084136882090875296834359294209725');
    T('321404575157883362782850293105646894530775907836460116367,3125000000000000000000000000000', '102849464050522676090512093.79380700624984829050766723723744');
    T('676431378140908951740507719346273515877844188802588789,125000000000000', '5411451025127271613924061754770188127022.753510420710312');
    T('3527866022768742962936331430909250221,1000000000000', '3527866022768742962936331.4309092502210');
    T('768670768316183532755914505999,1000000000000000000000', '768670768.3161835327559145059990');
    T('2363639937483203621,39891', '59252461394379.775413', '43820');
    T('1935991763481,5000000', '387198.3526962');
    T('84869849518884530417484124084533612955218764283953737717972,10107032631283871345', '8397108490200226398800083828224629264362.359197032737297920889324', '13064222726791451522');
    T('204886710340038538694354698793188156503362390502152775076648235731,5000000000000000000000000000000', '40977342068007707738870939758637631.3006724781004305550153296471462');
    T('1835971874232282120848076240479727,50000000000000000000000000', '36719437.48464564241696152480959454');
    T('18757216144126861202054952112882561,31250000000000000000000000000000', '600.230916612059558465758467612241952');
    T('1513207436951935998024078252323,21502756', '70372720452761311062827.400000399948732291662947549700417847500', '79362321');
    T('492152380966786320942977,1000000000000000000000000', '0.492152380966786320942977');
    T('322784976894366795182672726861732071108792529212189953293,500000000000000000000000000000000000000', '645569953788733590.365345453723464142217585058424379906586');
    T('6599835778515480731397541233194527935698039249154668674834348842207421,100000000000000000000000000000000000000', '65998357785154807313975412331945.27935698039249154668674834348842207421');
    T('2763733882920963372134959827,6250', '442197421267354139541593.57232');
    T('4493052127468834914347681261358393,1000000000000000000000', '4493052127468.834914347681261358393', '1917366645721044702863943');
    T('4165243072706279649913855214769337205676562643099363,50000000000000000', '83304861454125592998277104295386744.11353125286198726', '3369568246882905246798');
    T('40214993092834747603004055,592133223718', '67915447878984.246803347100480455192908487360291259358', '593784905929');
    T('104927737922243,5000000', '20985547.5844486', '74521293546');
    T('447244182767234345092627470873244541,1000000000000000000000', '447244182767234.345092627470873244541');
    T('3729,5000', '0.74580');
    T('1984341587249242120813988003214936078446236041737736626984039187,25000000000000000000000000000000000000', '79373663489969684832559520.12859744313784944166950946507936156748');
    T('109334915848204042748090967078741604879,500000000000000000', '218669831696408085496.181934157483209758');
    T('799850280431550021052452766407947249271905583022189,100000000000000000000000', '7998502804315500210524527664.07947249271905583022189', '2265487815011911855578344877035753');
    T('15005645256328535134146956704896053721,50000000', '300112905126570702682939134097.92107442', '959098942294647858855310128844058600617');
    T('284513580039738865694794020417628789,51896427813614457', '5482334565715520355.9164123103552051805461339353164', '608885290702033051');
    T('5611435160230836724036677212897508778352647894406598273,2500000000000000000000000', '2244574064092334689614670885159.0035113410591577626393092', '22586430234193957631479883247974');
    T('2257504637009351837615338885305884515151,24014872062235743869129', '94004441546010134.51153698433238353406039907857785325251', '337497026107271498961543');
    T('11961030675420279947461068675448705351,125000000', '95688245403362239579688549403.589642808');
    T('55042147389211604287,250000000000000000', '220.168589556846417148');
    T('8096641935000626379617912817593887056966796040651323,10000000000000000000000000000', '809664193500062637961791.2817593887056966796040651323');
    T('80403698814496577881,100000000', '804036988144.96577881');
    T('56530236925584442258099719084525496045476821,100000', '565302369255844422580997190845254960454.768210', '2085478950022');
    T('176510083304138788597014159,100000000', '1765100833041387885.970141590');
    T('194695119504918441713815809,100000000', '1946951195049184417.13815809');
    T('10419880141383547192692795964599686190836397501179348604845271327,10000000000000000000000000000', '1041988014138354719269279596459968619.0836397501179348604845271327');
    T('39904704373914275932765408749964211,10000', '3990470437391427593276540874996.4211');
    T('377154718786378953829563586363,500000000000000000', '754309437572.757907659127172726', '71111432644498937002116249698');
    T('4887809210859915472422967794086028102845277,100000000', '48878092108599154724229677940860281.02845277', '3695479757465231553301692211346874715665');
    T('2793736636935987847283419755276112581519780097583042817577,2000000000000000000000000000000000000', '1396868318467993923641.7098776380562907598900487915214087885');
    T('659750104347511676142213993755263489310617528836766246359807,1000000000000000000000', '659750104347511676142213993755263489310.617528836766246359807', '627804966532681570527289793642365197');
    T('465885672960721647139447,50000000', '9317713459214432.94278894');
    T('24074397538666586878367593,25000', '962975901546663475134.70372');
    T('5245705435294637571119386528576989827690767746698047845736131163241291317,100000000000000000000000000000000000', '52457054352946375711193865285769898276.90767746698047845736131163241291317');
    T('5068049704,72321', '70077.1519199125397539945071314731245643424', '83843');
    T('874564740796402183581390256952998902,9589359007', '91201585023356732021180261.663447689293503994887169401', '9860123179');
    T('181553923535320179726588305557,250000000000000000000000000', '726.215694141280718906353222228');
    T('6556143591220725641,1000000000000000000', '6.556143591220725641');
    T('2352354979582086906566789472880974692741621,500000000', '4704709959164173813133578945761949.385483242', '784087081050622276');
    T('1488083617458385838438378349131388911543068115734032699085102951249497989,5000000000000000000000000000000000', '297616723491677167687675669826277782308.6136231468065398170205902498995978');
    T('264579815467469,1000000', '264579815.467469');
    T('1538352201070219502279514358536457,20000000000000', '76917610053510975113.97571792682285', '21354906117089650290111715638');
    T('15722329841355833,100000', '157223298413.55833');
    T('4714260957941010403796629429189999850855093929869,50000000000000000000000000000', '94285219158820208075.93258858379999701710187859738', '30803072331334882951224000809043660');
    T('618425801728323030200807,10000000000000', '61842580172.8323030200807', '105985138218996171314403646219921');
    T('17448382674796873946013108235649557747459,50000000000000000000000', '348967653495937478.92026216471299115494918');
    T('45529397794072630857138473073,100000000000000000000000', '455293.977940726308571384730730');
    T('141465087,250000', '565.860348');
    T('17246099925359016508879876148711553304736078191,250000000', '68984399701436066035519504594846213218.944312764');
    T('2101258579700648429883582010633539413193707979287,50000000000000000000000000000', '42025171594012968597.67164021267078826387415958574');
    T('19873369192189503070833114340215357,100000000000000000000000000000000000', '0.19873369192189503070833114340215357');
    T('466815777276437858935139719637380455775922,237005', '1969645270253529921036010715543471470.120554418689640661226062605044220', '326529');
    T('269959,670534', '0.402603', '711489');
    T('34387358377391124546143226397815411955083664951,100000000000000000000000000000000', '343873583773911.24546143226397815411955083664951');
    T('37847045428231017995951348454896363,10000000', '3784704542823101799595134845.4896363', '720339608750233689017491098687232050524085');
    T('1735960832955491713580107966534128965678953034168488818485380597,25000000000000000000000000000000', '69438433318219668543204318661365.158627158121366739552739415223880');
    T('224192437577613620930013362114213113494689048262177491313573197479539,5000000000000000000000000000000000000', '44838487515522724186002672422842.6226989378096524354982627146394959078');
    T('12484340282845391902600,13508059694802606664561', '0.9242141776771164633050356530775932610539', '37252541001935408317724');
    T('115214077261487,10000000000000', '11.5214077261487');
    T('2958050889912784841,3643159', '811946689648.4026200887745906887963452131998', '5561403');
    T('827694940730584623592642676724234945522897,100000000000000000000', '8276949407305846235926.42676724234945522897');
    T('2546353200289710117989769402906610426930099,50000000000000000000000', '50927064005794202359.79538805813220853860198');
    T('6948521267340375164028079892940653337857575858806325681799,10000000000000000000000', '694852126734037516402807989294065333.7857575858806325681799');
    T('51227168953999260924,54854698290641', '933870.216231584893753966795960727850255527896', '81456633392018');
    T('3268599004367296653150629333,250000000000000000000000000', '13.074396017469186612602517332', '63486303090385390571705929886419628708915825065754');
    T('2942166843387188808794605445,4769444899383008874', '616878254.273950668243175338972426161', '5339655230978177566');
    T('239413121903268737233555547588419461449,5000000000000000000000000000', '47882624380.6537474467111095176838922898');
    T('63412373159131395132401586380914861773878046941905927937911236161547,1000000000000000000000000000000000', '63412373159131395132401586380914861.773878046941905927937911236161547');
    T('471747654685695831508453690945802340660145053437,800000000000000000000', '589684568357119789385567113.68225292582518131679625');
    T('891764513109420861505160319263,1', '891764513109420861505160319263.0', '918689638838436368865522030712717248752750323');
    T('3191887595,4', '797971898.75');
    T('121856597397345862613024522353232674108567324439,5000000000000000000000000000000000000000', '24371319.4794691725226049044706465348217134648878', '607506204583168148325583885842871519015786327');
    T('339040389167328399663612233561450019484519844493,50000000000000000000000000', '6780807783346567993272.24467122900038969039688986');
    T('38243,100000', '0.38243', '29337860446');
    T('8250342179921020599,1000', '8250342179921020.599');
    T('103452041851274523230988200830011,2500000', '41380816740509809292395280.3320044', '183359155743182533907906050197722073594351162073');
    T('252848083743402827,100000000000000', '2528.48083743402827', '503743383711879521120013355828612195');
    T('98069574418320559223875778928515122947287741711497297,1000000000000000000000000', '98069574418320559223875778928.5151229472877417114972970');
    T('63210643808681380191879692959261598285204021,10000000000000000000000000000000', '6321064380868.1380191879692959261598285204021');
    T('18241982295796176045680696249,250000000000000', '72967929183184.704182722784996');
    T('940517435169296832108127451,2', '470258717584648416054063725.5');
    T('55842359310373546663302494925523543342262143787999046184363835703,10000000000000000000000000', '5584235931037354666330249492552354334226.2143787999046184363835703');
    T('298991506390580492859083808694073988657502789821,10000000000000000000000000000', '29899150639058049285.9083808694073988657502789821');
    T('6930195055948617,10000000000000', '693.0195055948617', '5757907609830118853131884806500');
    T('3614496879689787374430705036518,101841455572128247685', '35491410245.30874016786935666286319487941134579', '195190430043812451482');
    T('153016231039501975524480588156505131693512848094041350606332993,34188657170984476841125800', '4475643201610302041685982836005246034.654308295479849660834145713924093080', '46378851978993763287521043');
    T('10030484490,1', '10030484490.0', '6598579459658967203976644647349896921');
    T('1355896469090076356140549902302865199527128695725221,2500000000000000000000', '542358587636030542456219960921.1460798108514782900884');
    T('38099363522125304202166422476325079802687,500000000000000000', '76198727044250608404332.8449526501596053740', '1090889184708266087570263218992');
    T('45445894379730855842101669571055419344067331,50000000000000000000000000000000', '908917887594.61711684203339142110838688134662');
    T('48387017662752417884217,5000000000000000000000', '9.6774035325504835768434');
    T('143145652784899539,50', '2862913055697990.78');
    T('89829636046825131956593468,1', '89829636046825131956593467.96758068078087', '1');
    T('2396938114331990504523927741558933050260578381,4400452737740717', '544702615204686384863075977771.218589328031376415328281017228242917', '4414423939286874');
    T('3647034409547,125000000000', '29.176275276376', '315898686604743');
    T('5657225048612832243372055670023410281,100000000000000000000000000000000000', '56.57225048612832243372055670023410281');
    T('2048923042589181698150868951283041,500000000', '4097846085178363396301737.902566082');
    T('22265,3654', '6.09332237707570340312178966387447', '7353');
    T('248385231997710097553022283051971775382681228677873,1000000000000000000', '248385231997710097553022283051971.7753826812286778730', '807422358051844497008514434773778070855');
    T('263427757666534908045161273191351480,3', '87809252555511636015053757730450493.35240699180867869447737776738648', '4');
    T('280765588944227665193509257500853588850373,4000000000000000000000000', '70191397236056916.29837731437521339721259325', '66471574477550907802775072350732179636393120');
    T('36902302597391526140987033687,5000', '7380460519478305228197406.7374');
    T('467727844444957446621571,530243516999456689', '882100.0718532815202598878323222458558', '2374163188160596325');
    T('7485818045096143742403867459891607342912851163218831484413,1000000000000000000000000000', '7485818045096143742403867459891.607342912851163218831484413');
    T('6617976493555127256572099691726271,878490192697848883004', '7533352732409.31694657454587818166126423', '1597982807060692539038');
    T('329306133931896629549389727842553925953320055555,37084658199', '8879848161598439046985423392402720037.627926029870431056848013', '90166908885');
    T('4454671579284887267233133029196547900999664983,100000000000000000000000', '44546715792848872672331.33029196547900999664983', '312066601531489140643249426');
    T('2118844361164150744985300682261162411140053906424859259689007681661137,500000000000000000000000000000000000', '4237688722328301489970601364522324.822280107812849718519378015363322274');
    T('18513646046835945,109', '169849963732439.86237889670792909799300509767365', '190');
    T('80056584734477457616078211168206873374318213792897,50000000000000000000', '1601131694689549152321564223364.13746748636427585794', '7664647609675678867596695298');
    T('297303657379426370759382568,33333', '8919198913371924842029.8973389736300643077988817', '64991');
    T('103432169127096395317863672765426134926680268209844331529227,5000000000000000000000000000000000000', '20686433825419279063572.7345530852269853360536419688663058454');
    T('71251826759954175648215320529575384132956249082986677350937829,1000000000000000000000000000000', '71251826759954175648215320529575.3841329562490829866773509378290');
    T('641201328368559843571245009607921346857269477,1000000000000000000000000', '641201328368559843571.245009607921346857269477');
    T('4882048845842468103103485015911,10000000000000000000000', '488204884.5842468103103485015911');
    T('224791477,1', '224791477.0', '726177065211931040302050912338320716');
    T('476719932128485235601740674,631706300099407547', '754654389.3158369544946287762236', '936155391544004023');
    T('56347229254417904350177815,1', '56347229254417904350177815.0', '10148255980165242466385019381283504');
    T('3247176969,50000', '64943.53938', '5737853151156754534369320427420819473538787');
    T('148680609085343851516935023,25000000000000', '5947224363413.75406067740092', '3299703385992240915628547545324147430534');
    T('1,10', '0.10', '60676600');
    T('33888285831283524427,5', '6777657166256704885.4', '8419970403584865379780209059074');
    T('708030949802968823378286961,10000000000000000000', '70803094.9802968823378286961');
    T('544304014187971143672362107687657,1000000000000000', '544304014187971143.6723621076876570');
    T('205971513796048941402022334203083,50000000000000000000', '4119430275920.97882804044668406166');
    T('77292325648780699681,25000000', '3091693025951.22798724', '39764580440463487');
    T('2309917737614290424010537409984194531,26393218442', '87519365729890563984994483.37889805167096566858300647613', '41528749227');
    T('1350005508097927361466501541112951711,1', '1350005508097927361466501541112951711.0');
    T('479515933263403410783578189085467270740481972776756061209,1000000000000000000', '479515933263403410783578189085467270740.481972776756061209');
    T('5468820184309309381471350304120512393962862818693211,100000000000000', '54688201843093093814713503041205123939.62862818693211');
    T('129723834937825709698545211076130124,26403603915209861863899', '4913110928129.700171713296448609990870295676076015185', '55134610153115101837033');
    T('232780859367591812749509510144676014553783,400000000000000000000000000', '581952148418979.5318737737753616900363844575', '8362349325310445745344747979264146251506102');
    T('26526026292549267547430447894201,31250000000000', '848832841361576561.517774332614432');
    T('19890543181073850618089831198136526047676310776189255009170995711093373,2500000000000000000000000000000', '7956217272429540247235932479254610419070.5243104757020036683982844373492');
    T('30493213502290966437312425842604482987,1000000000000000000000000', '30493213502290.966437312425842604482987', '724744520119961789856512230962935140768926990');
    T('275344194003530259835120378143,50000', '5506883880070605196702407.56286');
    T('280641866494800916387325846054898697443,500000000', '561283732989601832774651692109.79739488600');
    T('816489328,1', '816489328.211266', '1');
    T('186929316741636687784817442181,50000000000000000', '3738586334832.73375569634884362', '847366608343837425110540459931506231838');
    T('112563301820705703136709010962115946,426227402670606553', '264092128088010144.981654762325530158196731698433', '529348397551579750');
    T('435191728671575057019159870521595631,1000000', '435191728671575057019159870521.595631');
    T('297734036862960068140512298506553,33525048786443771', '8880942687348218.29539231419286944799431547446739362', '66580157811010440');
    T('644491386590497383264428696888480290002513,100000000', '6444913865904973832644286968884802.90002513');
    T('692870436323024990207,2000000', '346435218161512.4951035');
    T('16731325404663519753103973595257782597,5000000000', '3346265080932703950620794719.0515565194');
    T('7678382551,10', '767838255.1', '563996631510634040120890403050803455723');
    T('301101154009241217644497759979222592173429957444321800273,5000000000000000000000000', '60220230801848243528899551995844.5184346859914888643600546');
    T('3287369581548004161063138286884405777483908660733707180700133,500000000000000000000000000000000', '6574739163096008322126276573.768811554967817321467414361400266');
    T('7072616405346678239433829996523626553492305980858871567461,100000000000000000000000', '70726164053466782394338299965236265.53492305980858871567461', '4156731482608469668334695763987365235929934687');
    T('4206726322500320286718351,500000000', '8413452645000640.573436702');
    T('8042280713465517964325469588870746568338611,100000', '80422807134655179643254695888707465683.38611');
    T('26905465406799944428340969750110533073398836737,10000000000000000000000', '2690546540679994442834096.9750110533073398836737', '8335668443865329061594075961775870790796125');
    T('781397951264870786241984162,53412969887938978591', '14629367.2286759680365043324657384094905107442051', '79799519873530341684');
    T('11272468788367824600929293072031224512269133,500000000000', '22544937576735649201858586144062.449024538266');
    T('35062259331242087386880791,3774248', '9289866307471604247.225087222673259247488484429220179469', '5015595');
    T('40684596467754525130528154364962148818,4282358620351909', '9500511301038867618025.61680375343530528695710569699373987', '4516893654064672');
    T('11718831363175661697070030246145843821,3243', '3613577355280808417227884750584595.689485', '4424');
    T('723075924168109306363,129', '5605239722233405475.682148', '246');
    T('1425315029,100000000', '14.25315029');
    T('23081969548598774686163,261431419053327', '88290725.0864537257072290999928', '439109518021570');
    T('74432727113,5', '14886545422.6');
    T('5043881924600298710923,5283123823581648', '954715.82590711316746962783430632190', '8464867636721848');
    T('1199912974622736146774960640527,101153498187505481', '11862298349766.31444753914108283160', '864586429503386891');
    T('2671282759101377354629,5000000000000', '534256551.8202754709258');
    T('4428522968683635809406603927474774548389467299,1000000000000000', '4428522968683635809406603927474.774548389467299');
    T('29178029301,5000', '5835605.8602');
    T('606209140346077832367770959,20000000000000000', '30310457017.30389161838854795');
    T('950780223241033310711461629624186798117525059147,10000000000000000000', '95078022324103331071146162962.4186798117525059147');
    T('12431119464175333997,5000000000000000000', '2.4862238928350667994');
    T('234188333,2500', '93675.3332');
    T('8882443557897602881136097481314340616279449789015613,10000000000000000000000000000', '888244355789760288113609.7481314340616279449789015613');
    T('620853587690846221319415358259599862766390740728717,20000000000000000', '31042679384542311065970767912979993.13831953703643585');
    T('18330643901258352830379667572558501125731138590723,2500000000000000', '7332257560503341132151867029023400.4502924554362892');
    T('1737914673731044895006236355997286362604696463,20000000000000000000000000', '86895733686552244750.31181779986431813023482315');
    T('32321383496895512945858052467433508644049961463611439,500000000000000000000000000000000000', '64642766993791025.891716104934867017288099922927222878');
    T('30443644280497735598449,100000000000000000000000', '0.304436442804977355984490');
    T('33441569099242498502093191685727884000199,1000000000000000000000000000000000000000', '33.441569099242498502093191685727884000199', '226046621508705333950998644181135613143302286173');
    T('398604346574198750962183419866817869,1', '398604346574198750962183419866817869.04266056', '7');
    T('512097566031,1000000000000', '0.512097566031');
    T('398681694350124818470993924383729,500000000000000000000000000000', '797.363388700249636941987848767458');
    T('3812252051639510486930285769517024400609099822879069290116480023961048395781,50000000000000000000000000000000000000', '76245041032790209738605715390340488012.18199645758138580232960047922096791562');
    T('17640229131583372766764324994200509903251266518839,20000000000000000000000000000000', '882011456579168638.33821624971002549516256332594195');
    T('560154998022191785354816414733070384411658872186803,1000000000000000000000000', '560154998022191785354816414.733070384411658872186803');
    T('71890032720748761728115659399,11502294600531586', '6250060115607.39517761094265713855993887187878900', '56396172263945605');
    T('239652696866514152551465514613,1000000000', '239652696866514152551.465514613');
    T('624116519630700792,1167196367', '534714241.130516466043798094841834', '1620799323');
    T('354713305113764443146383722,1', '354713305113764443146383722.4', '1');
    T('291430659582700164960933388735,37370964395748226', '7798317873109.4467055700449997519564282043354166', '60816081804303776');
    T('488,1', '488.292453', '1');
    T('5375378978249541387849408245604751505152302194421,25000000000000000', '215015159129981655513976329824190.06020609208777684');
    T('13453827695164848671202758912258191501381686420571586492041,200000000000000000000000000000000000000', '67269138475824243356.013794561290957506908432102857932460205');
    T('89089683441815473,1000000000000', '89089.683441815473');
    T('9209451109,5000', '1841890.2218', '134792391785549571059323696404');
    T('5753101370471889620242487730093198832286126348947,12500000000000000000', '460248109637751169619399018407.45590658289010791576');
    T('97943035611348452670762369622371455167581899341662817,1000000000000000000000000000000', '97943035611348452670762.369622371455167581899341662817');
    T('156481263905560941381243892108,159892048517', '978668203684460155.744443223268507096551689891232818887', '322004247450');
    T('2555714776189801984122292922264279394187,5000000000000000000000000000000000000000', '0.5111429552379603968244585844528558788374');
    T('54279806605191090207,1', '54279806605191090206.69', '1');
    T('31837316259061,16072996650298', '1.980795302316617044769286885040', '49459229769021');
    T('4271818608155766034681095050304504858956311601618369719,500000000000000', '8543637216311532069362190100609009717912.623203236739438');
    T('176478464295519818379214397692871852257325839802281,1000000000000000', '176478464295519818379214397692871852.257325839802281', '90779039488828051777510007885720349007138577');
    T('348438519284667659789950964099607679,250000000000000000000000000000000', '1393.754077138670639159803856398430716');
    T('65822230159387850562143906302474539651870314716129059,2000000000000000000', '32911115079693925281071953151237269.8259351573580645295');
    T('45672374821762348481787087840824627394334023792813,100000000000000000000000000', '456723748217623484817870.87840824627394334023792813', '23761423870035685067842276095584074976971583');
    T('464877382392713428729345554492047701,50000000000000000000', '9297547647854268.57458691108984095402', '79229417025138906364159421018995634243');
    T('84227563396748648999111809845309,5000000000000000000000000000000', '16.8455126793497297998223619690618');
    T('4557298469196086700221846516327,100000000000000000000000000000', '45.57298469196086700221846516327');
    T('80439239502527,1000000', '80439239.502527', '533338416082600676071203071632265123199117929');
    T('935236327881877011999858293276481942539,100000000000000000000000000000', '9352363278.81877011999858293276481942539');
    T('42714429794630694344783675903666677070293499362190798,2299869685120688616083', '18572543510172490391080421152535.67751350101760083140434', '2523895099168949296133');
    T('947258061205512702708184333018496368818813104995057,1000000000000000000000000000000000000000', '947258061205.512702708184333018496368818813104995057');
    T('0,1', '0.00');
    T('1357605379576214126002356605597838987734159077021017189,10000000000000000000000000000000000000000', '135760537957621.4126002356605597838987734159077021017189');
    T('689179106336557793377586103585498426183,1000000000000000000000000000000000000', '689.179106336557793377586103585498426183');
    T('3185931020723678246265927634227924895843827321051,5000000000', '637186204144735649253185526845584979168.7654642102', '8348820773767166375670963549814381');
    T('181704427890961813309386683087009072816236283562619436859899,20000000000000000000000000000000000000', '9085221394548090665469.33415435045364081181417813097184299495');
    T('206086803,50000', '4121.73606', '164519844857384467378024');
    T('2446815034234629509921183262469736744807867506113159757,50000000000000000000000', '48936300684692590198423665249394.73489615735012226319514');
    T('11921542076626216037383,2467379936129', '4831660459.77076950587336328803734586299', '4032388674437');
    T('4821454596269229904070747116,5', '964290919253845980814149423.2', '63058918491');
    T('8059615507653467966326957937996724771202206977059344880847,100000000000000000000000000', '80596155076534679663269579379967.24771202206977059344880847');
    T('72323768597570082020606327010228744367968599,10000000000000000', '7232376859757008202060632701.0228744367968599');
    T('28149894654702826785477357889173350109389,332175002107', '84744169417165234660977522265.35323457903133812593052374237901963', '660733805280');
    T('211462938819720593425336680167945416724957,25000000000000000000000000000000000000', '8458.51755278882373701346720671781666899828');
    T('8669768238324809349466041,100000000000000', '86697682383.24809349466041', '65200093702980892');
    T('9229478586505609450209233741970260794135057090509829,200000000000000000000000000000', '46147392932528047251046.168709851303970675285452549145');
    T('788251666453791388583584838872536454639199125815589911,20000000000000000000', '39412583322689569429179241943626822.73195995629077949555');
    T('20943607945514035985620131263207,5000', '4188721589102807197124026252.6414', '4835242474956309626165311527265229');
    T('1772029297651420947345744911718400232486579,1000000000000000000000000000000', '1772029297651.420947345744911718400232486579', '887760316568762719383237942823678');
    T('757397481749167231261,12500000000000000', '60591.79853993337850088', '66725488326793109720958265319522090494147096792756');
    T('254427806433,20000', '12721390.32165');
    T('665415819,1000000000', '0.665415819');
    T('746095752604743115329558773090119,10000000', '74609575260474311532955877.3090119');
    T('200405698845550169272337116046884055,37753768', '5308230395587274077446709850.17665137424163872012965483562443432875', '52358187');
    T('876243876010368000888263,6250000000000000000000000', '0.14019902016165888014212208');
    T('314444059696144669541009651,100000000', '3144440596961446695.41009651');
    T('102685572797675093,500', '205371145595350.186');
    T('53720823257553465809425585560295424700501,10000000000000000000000000000', '5372082325755.3465809425585560295424700501');
    T('144228,1', '144228.0', '893072502');
    T('2249884045617817731,500000000000000', '4499.768091235635462');
    T('6149980092464732650245703095614058473554397900,7551277', '814429147873231593841108344405066649462.653522046668397742310767776', '99624556');
    T('1005318427,100', '10053184.27', '56258703');
    T('675360018584114704817648366594913164055503162783551547601259390568622759863323,5000000000000', '135072003716822940963529673318982632811100632556710309520251878113.7245519726646', '763113868956966253237445913550376861628125707761719940487681719401160610436');
    T('487813882528400826023428640609378101429959398115157937005578981194998958082862673744095665022036524299189099,10000000000000000000000', '48781388252840082602342864060937810142995939811515793700557898119499895808286267374409.5665022036524299189099', '319053905671710373413564483718075497688524361485928741160092292043716624896461985792922106492711599194534899554643542426');
    T('4363269813583297680660811455498012143883949072442011552361660186260454870353040104340521418931006735842679876713781061,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '436326981358329768.0660811455498012143883949072442011552361660186260454870353040104340521418931006735842679876713781061');
    T('341893039028161520368729363564608981945974443020224230053218402863771456413927271762947534978276261913698192670665537,1000000000000000000000000000000000000000000000', '341893039028161520368729363564608981945974443020224230053218402863771456.413927271762947534978276261913698192670665537');
    T('91901269276951481268589508998910356181948890482769886151337833001536520925704410737790986004918759962586403387644931014593138475668109,200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '459506346384757406342947544994551780909.744452413849430756689165007682604628522053688954930024593799812932016938224655072965692378340545');
    T('160085901137799630792793617081212539406350610856533352536667873727,50000000000000000000000000000000000000000000000000000', '3201718022755.99261585587234162425078812701221713066705073335747454', '82302986588243438196352421107435286399958881274722031838421651877101115');
    T('19730270740072250726350191888457903937782226838056969873977606036442991786321705617019361264080014612697781414129464904485361465812168904497,500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '39460541480144501452700383776915807875564453.676113939747955212072885983572643411234038722528160029225395562828258929808970722931624337808994');
    T('554184362944725960497789037,625000000000000000000000000', '0.8866949807115615367964624592', '3519583398754511869875010683565244740317978393251156588940794481183604900896512');
    T('638231282580449825792515353133591102373788595106470546093619096974527294935424922680855148938101691766274567269680519810515529956690038641911538125173665109699,10000000000000000000000000000000000000000000000000000000000000000000000000000', '63823128258044982579251535313359110237378859510647054609361909697452729493542492268.0855148938101691766274567269680519810515529956690038641911538125173665109699', '4108116384125526836507938785056766891019126123715803937416172409988923081826654552');
    T('47734373761311372617250467020779084445512709682911994169631,625000', '76374998018098196187600747233246535112820335492659190.6714096', '612857840076182176420280571134792');
    T('96501773711552295237818469319540275439821230156574621025922681890456142035559160552388101621981467063,1000000', '96501773711552295237818469319540275439821230156574621025922681890456142035559160552388101621981.467063');
    T('74403513794353613906027668774054715460457535829576006466757980055646532561807,100', '744035137943536139060276687740547154604575358295760064667579800556465325618.07', '9062598597523005153489301824610095730026103190392060661534');
    T('4552464673010589780215760669790574819454182498449417341631679,50000000000000000000000000000', '91049293460211795604315213395811.49638908364996898834683263358');
    T('553156651153146542558050182057356237357767613058590716666708160610420683997653491,10000000000000000000000000000000000000000000000', '55315665115314654255805018205735623.7357767613058590716666708160610420683997653491');
    T('10094472168930635850294295702643512345309187651848439883217715065443928296148508896180724198805240643680251877332506709166622938770815681788987679097027,500000000000000000000000000000000000000000000000000000000000000000000000', '20188944337861271700588591405287024690618375303696879766435430130887856592297017.792361448397610481287360503754665013418333245877541631363577975358194054');
    T('1390638862389813656941579907173114981020090536680671582853648156827291166367509597465818901222618393320613440799359074159,10000000000000000000000000000000000000', '139063886238981365694157990717311498102009053668067158285364815682729116636750959746.5818901222618393320613440799359074159');
    T('764577472361146278858344672554647493530542460178759013102132345200903874086918631004687608105278297,1000000000000', '764577472361146278858344672554647493530542460178759013102132345200903874086918631004687.608105278297');
    T('127046433646264975767457966930607201956534938099951672926121899531366533718900568517910151332761,10000000000000000000000000000000000000000000000000000000000', '12704643364626497576745796693060720195.6534938099951672926121899531366533718900568517910151332761');
    T('759706682454180966323586844671034018022403078651480091043022480725,1', '759706682454180966323586844671034018022403078651480091043022480725.0', '8491246168928786773792854508283380644357724943352621981745640028894860319462721890015437671665726136602020425847570035');
    T('1232133064589580204554541761202884749167806439036647915064417866743368822300712460904153,50000000000000000000000000000000000000000000000000000000', '24642661291791604091090835224057.69498335612878073295830128835733486737644601424921808306');
    T('860753657944772685510679876822980161245604845668538717795313417907394077573430859608260608605457173684835477710478072161454145765516186246987,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '860753657944772685510679876822980161245604845668538717.795313417907394077573430859608260608605457173684835477710478072161454145765516186246987');
    T('16005705331614697482755365912151704073591971169105093634086192436675140166715310612622644294152431935263181364819,250000000000000000000000000000000000000000000000000000000000000000', '64022821326458789931021463648606816294367884676.4203745363447697467005606668612424504905771766097277410527254592760');
    T('69113805838501162731578029205573692448157213720275881273819982233592866266473,100000000000000000000000000000', '691138058385011627315780292055736924481572137202.75881273819982233592866266473', '58975855894427885852484710799418772114376247');
    T('4729199753900213112131177820583529490731573946050705866217107256295944829467960577257186,510681', '9260575102461640656556985320745297927143508268470348155144027790922209421278568376.8481419907926541887178704058086735', '7975458');
    T('23498489129913177044438966039588756148779460123795350021472351713417447381957515629514242536444852177161448159204317722351313614332682938279261185510112182173899064135103,2500000000000000000000000000000000000000000000000000000000000000000000', '9399395651965270817775586415835502459511784049518140008588940685366978952783006251805697014577940870.8645792636817270889405254457330731753117044742040448728695596256540412', '2738063341239025028739321372348930298237495900370115886451023412898232579270');
    T('581214945232966023983197807568271384582625395798108238988909577356391,100000000000000', '5812149452329660239831978075682713845826253957981082389.88909577356391');
    T('755774297159354724303490783235728345982079,10', '75577429715935472430349078323572834598207.9', '976346638366081178717993549430125376917323536668391354073736050931714508949029849095898475324947663457561691565869');
    T('72952666104960722610055921852408716115501920420152774589434925111108095463,1000000000000000000000000000000000', '72952666104960722610055921852408716115501.920420152774589434925111108095463');
    T('84253414454971232257480071766397766368674340440243975710448609630373480623069,50000000000000000000000000000000000000000000000000000000', '1685068289099424645149.60143532795532737348680880487951420897219260746961246138');
    T('36916365097825321454559738506509326979820327574234098190319159712574451468922233909969299557,384885741807809995033', '95915127758251045669622249994875027216126584339530370592512741225931996.041944546489526354777674857433920571240428120906031224034258694793804737624806', '996972807953924088414');
    T('46202911095785852843473956927532538280968392442342938578610785707745261965259051602639282297839,500000000000000000000000000000000000000000000000000000000000000', '92405822191571705686947913855065.076561936784884685877157221571415490523930518103205278564595678');
    T('1772839597150920835362349139129708947,250', '7091358388603683341449396556518835.788', '85650512041659658831614');
    T('342934812509126807028655148458814382887953695727579025045242138980820292534408239977896632855835901410506357,500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '685869625018.253614057310296917628765775907391455158050090484277961640585068816479955793265711671802821012714');
    T('5467569331238493371328759073003072869784239507210662,37362334854454456899796270284749', '146339069882476370230.94111136842283676758455168072986640234', '96171449439665788570414679267610');
    T('9014126789799805335062120695538270131901418610794009238567600799878815000839260540538480396467469718753863,1000000000000000000000000000000000000000000000000000000000000000000000', '9014126789799805335062120695538270131.901418610794009238567600799878815000839260540538480396467469718753863');
    T('1250041788678023084610616992418436156068571496356335523903697743016099193022025463752931903884161856935693369085205315349974809,2000000000000000000000000000000000', '625020894339011542305308496209218078034285748178167761951848871508049596511012731876465951942.0809284678466845426026576749874045');
    T('25239061830805058260347549564363328809074990929185276918644989105557079183929665109,500000000000000000000000000000000000000000000000000000000000000000000000', '50478123661.610116520695099128726657618149981858370553837289978211114158367859330218');
    T('23046884534040883773,25000000000', '921875381.361635350920');
    T('17516790639377003056238183959898186484431649497446483054036345045419619227552721302646069807169936889243905650537200431291875228387189659891591,2000000000000000000000000000000000000000000000000000000000000', '8758395319688501528119091979949093242215824748723241527018172522709809613776360651.3230349035849684446219528252686002156459376141935948299457955');
    T('830206819686442727061104242686969514890459473419575742090385225994337814887819198514017,1000000000000000000000000000000000000000000000000000', '830206819686442727061104242686969514.890459473419575742090385225994337814887819198514017');
    T('413893751926246540100720057983571518390906128684895499155815473199,1000000000000000000000000000000000000000000000000000', '413893751926246.540100720057983571518390906128684895499155815473199');
    T('206500127024184753713560420566631778621,500000000', '413000254048369507427120841133.263557242', '5224151114621627696369936014682664216594925452865329899315399663005407747319945317435179613');
    T('16217662953881464354149036371967353081296233553367,125000', '129741303631051714833192290975738824650369868.426936');
    T('158877139692735544114713014099098201170707224624720234416065901772598705859728066434630087577,10000000000000000000000000000000000000000000000000000000000', '15887713969273554411471301409909820.1170707224624720234416065901772598705859728066434630087577');
    T('1363481494416319010330583135309100415177784161418502772461,5', '272696298883263802066116627061820083035556832283700554492.2', '504783341926451844554');
    T('558130898291826876915105441493420754206140253419478295461642459649585459789613029845418176825679124538566594627314537483,14998270055051685963765623', '37213018317658468525204154347997553646488930517742862611997889963271621447764872970409325420092.841681521850527813475859312769151840560733279569364227935675286869135262631', '77352003382953136988715543');
    T('10451566350931936938155976509981211336966461645889115229513600263067775576558118998602357161325,346893908613991685517980584558', '30128999360902532302715650110413576813750396896824593726469912912.366199216341942668442860487226805935762749663328322107767450167435735148060307', '351381928759382906855046900016');
    T('1468542461374793122289224886453463993847932190034840253613540817203177345513896917033925255992523630246969352648225086758781193593730856733866872342988998661495531701242165439378823,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1468542461374793122289224886453463993847932190034840253613540817203177345513896917033925255992.5236302469693526482250867587811935937308567338668723429889986614955317012421654393788230');
    T('209271529724729052617837048480332267987578229392112165277475305343450561725459538265834951255626969900314942469226110140479983692763731004233,40000000000000000000000000000000000000000000000', '5231788243118226315445926212008306699689455734802804131936882633586264043136488456645873781390.674247507873561730652753511999592319093275105825');
    T('7101983556936994042497298646151570859081364497353591813399109991310416906158547491525491881998229621477347427929,100000000000000000000000000000000000000000000000000000000000000000000', '71019835569369940424972986461515708590813644.97353591813399109991310416906158547491525491881998229621477347427929', '56744786973831246136639355921544534171994029324959759143702819842099548907292114036606647');
    T('700608601940127639463635186370519157157680192656598879705304237171077875720684209649489000891686101,1000000000000000000000000000', '700608601940127639463635186370519157157680192656598879705304237171077875.720684209649489000891686101');
    T('8602847585675548348838808002159088443209,100000000000000000000000000000000', '86028475.85675548348838808002159088443209', '72582545116505333796075480243333118971380354201482192123094188476893348588726112639735928329133275172401442');
    T('148408052390556415124137719435490471,25000000000000000', '5936322095622256604.96550877741961884');
    T('416641482734579229938341545166425850147492151317072668023493195705213636353836873961255897570763844236,25717644180352342203393925081331448253691044279201083210974082080156217889361', '16200608415481665314529154.120823068329960470035846076484357364396252835780302742135244075134790170921737797562758009637507765', '67934439808176948911255062541382522278265771795756778046815888409619479526534');
    T('25995084389459522138350250382942032604375065037616720903582722822063333331140405124796661617605799564677801133030110,310603780302310842094866507130236732405959190501476949', '83692105627814610657066372727390901983001818916459785375020003.3353553456353151128930868769216270423551208386173443046109736307958268123', '3982369319326564077663582191574752696545616233919787956');
    T('3588237426497060004880870257545324138618112195268225592118612244849912586535502781039993176234955972397277,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3588237426497.060004880870257545324138618112195268225592118612244849912586535502781039993176234955972397277');
    T('3313686586024934130977871482209887492640993452578557026807334010909893148780133822458503199230307206620292985108899540304687909083,1000000000000000000000000000000000000000000000000000000000000', '3313686586024934130977871482209887492640993452578557026807334010909893.148780133822458503199230307206620292985108899540304687909083');
    T('273781703818906790628718083951503889627525115303814013952356530599951005452333198523390628974188239287896869929063813635816988840128233022694039881132781253023590270236688227083953341213,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '27378170381890679062871808395150388962752511530381401395235653059995100545233319852339062.89741882392878968699290638136358169888401282330226940398811327812530235902702366882270839533412130');
    T('19909985415765292964414127154401583912053874960439188258560175396645802964673383822417554033845591984786339,50000000000000000000000000000000000000000000000000000000000000000000000', '398199708315305859288282543088031678.24107749920878376517120350793291605929346767644835108067691183969572678');
    T('85820359411270844647913420510043686839128620275465076863336894062316958345993146300360121970304754792703695315584328722640236913591546766980436151189616776567197756454666891371391,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '8582035941127084464791342051004368683912862027546507686333689406231695834599314630036012.1970304754792703695315584328722640236913591546766980436151189616776567197756454666891371391');
    T('35000532278070509562103057820976975041119512140519835960943403473094631,3662495203502602060139335', '9556471840454013866302210174557670158611986330.928972391824443058488437395568693146286054952317466368062069862106460410265233312282378988449302', '7332300780944167684576224');
    T('39376042039638531344672309024889855506726524477396754533061336162078676158805401717447421956134946657,125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '315008336.3171082507573784721991188440538121958191740362644906892966294092704432137395793756490795732560');
    T('6597859015029510292915275130297833881131020029929841154557256370393361962811461577212908969927685870101608101607420224161,100000000000000000000000000000000000000000000000000000000000000000000000000000000', '65978590150295102929152751302978338811310.200299298411545572563703933619628114615772129089699276858701016081016074202241610');
    T('77941951791579411765197932683235264335432882206721156588976877215382711588365187167594098027719350761193,100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '779419517915794.11765197932683235264335432882206721156588976877215382711588365187167594098027719350761193');
    T('7469527965820984707499768784473976650257429936748963414116722131553049029423204476861,10000000000', '746952796582098470749976878447397665025742993674896341411672213155304902942.3204476861');
    T('793689283759962354945365347,1000000000000000000', '793689283.759962354945365347');
    T('197236579651099174237,25000000000000', '7889463.18604396696948');
    T('13248974903544268426929651334228835034267588350879238246365359559734640920519013385480529227732605812835471479125497458201213631616739806389677655600244904631323,100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '132489749035442684269296513342288350342675883508792382463653595.59734640920519013385480529227732605812835471479125497458201213631616739806389677655600244904631323');
    T('6385188721690120379276122832047303271584663620275352024639252661807196391,2500000000000000000000000000000000000', '2554075488676048151710449132818921308.6338654481101408098557010647228785564', '69360752312105986272643762566971174625835381312809147456026971');
    T('25862063700127446496519773714020332299021654523004379222786870074851196534089686069439903327778111173914467314241700595801,100000000000000000000000000000000000000000', '258620637001274464965197737140203322990216545230043792227868700748511965340896860.69439903327778111173914467314241700595801');
    T('337561464417705983649060127652172317890452693275809295120807312918687,1000000000000000000000000000', '337561464417705983649060127652172317890452.6932758092951208073129186870', '8489487941822174030068781645476719602281025013758808050237847942377537968055889085564660960488663905553823');
    T('12655042328012036252407420053218831648099,781250000000000000000', '16198454179855406403.08149766812010450956672', '64236488636162905463925496010047829405545831499743306133923178415645494');
    T('5922194307177579374811484837474903626993993529770932798657322377651387,10000', '592219430717757937481148483747490362699399352977093279865732237765.1387', '3152877460250162477684029172894929736548191904765877685303248');
    T('77272660141369492538273738266098041603520972222166296362371597167705245085548532311984761778478186732722925744042787931177199124350594600293303988626922427396336198562206707,10000000000000000000000000000000000000000000000000000000000000000000000000000', '7727266014136949253827373826609804160352097222216629636237159716770524508554853231198476177847818.6732722925744042787931177199124350594600293303988626922427396336198562206707');
    T('192725006484481955518141547878555444458245002321169158256767255032910672226968644775201483730442494510679299467564928048217,1000000000000000000000000000000000000000000000000000000000000000000000000000000000', '192725006484481955518141547878555444458245.002321169158256767255032910672226968644775201483730442494510679299467564928048217', '439652030092743058167706039163812494429147760995240096210156202016745715173125314627');
    T('4119,1', '4119.0');
    T('356296672182939168877855246845022740124409939589391646979258814297585852593514729523,775462837988393944311111588325395377444992149494144758081106318', '459463245340290213739.77440565240861533938923822785461681533785064251235446563843782326420183106592724997999481060381275', '908934942330192208051413123327017049131882835053534224912068502');
    T('768098015293590626357287707160911,1000000000000000000000000', '768098015.293590626357287707160911', '4970166725632570631012333623050617477446500886979586460886675511529151792998381196736807216434719029198488026413277');
    T('17308584498708223882873554139102525513289773342431835066642028579392867094694512432111520248839813,1250000000', '13846867598966579106298843311282020410631818673945468053313622863514293675755609945689216.1990718504');
    T('730329569763986956693151189671893374471654148260912717254713032732776230933990274047334434047257585286285228208365006513375988013830063457878469670357195374629549829394279454843199,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '73032956976398695669315118967189337447165414826091271725471303273277623093399027404.7334434047257585286285228208365006513375988013830063457878469670357195374629549829394279454843199', '6972155039468853521357852529419312954662546952741049465777419800482878657794987116597422340793948647312389878781628');
    T('406720109833232022397234646712546646136794280590981538261378858703302037675499399765989883329,100000000000000000000000000000', '4067201098332320223972346467125466461367942805909815382613788587.03302037675499399765989883329');
    T('306423719310620474116542871532316659191577143943784441511526985433945123437819539085214867633133,1000000', '306423719310620474116542871532316659191577143943784441511526985433945123437819539085214867.633133', '66262964973805318142587348908691096');
    T('1354706619236923062203233937522312950517745299920180807301978313151489452805373608391524296715118535473,250000000000000000000000000000000000000000000000000000000000000000000000000000000', '5418826476947692248812.935750089251802070981199680723229207913252605957811221494433566097186860474141892');
    T('1974457002526436695523552349738485726387619749364549221535073398987258972810290980762750581643854743176718698008795411899,500000000000000000000000000000000000000000000000000000000', '3948914005052873391047104699476971452775239498729098443070146797.974517945620581961525501163287709486353437396017590823798');
    T('156783373050685706995418777471852885862528591164845287226607990619037741515473450079378844849264334400620506174471,100000000000000000', '1567833730506857069954187774718528858625285911648452872266079906190377415154734500793788448492643.34400620506174471', '1475852993423867930895375531269870744877795635101316802275045394546252736459');
    T('817062512854985934298222856248923943142561938985062588141950589526461,100000000000', '8170625128549859342982228562489239431425619389850625881419.50589526461');
    T('6290321680681892372665821719454390401908672910524568767187416242561415491887619927859962686607232816600821305765944442019799,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6290321680681892372665821719454.390401908672910524568767187416242561415491887619927859962686607232816600821305765944442019799', '9105579897058381459068780523330349175438124909135209330622101030511881627579118208200106153986510746652194');
    T('291019633702648422081986065259017934111952044975205808908304848900279460934291632580590482482887983287848959317,5000000000000000000000000000000000000', '58203926740529684416397213051803586822390408995041161781660969780055892186.8583265161180964965775966575697918634');
    T('9761851721,10000000', '976.1851721');
    T('46347552512954346442636579916791964575641708792980525139317489275282600482060794741187192532852855496225264908653845892622887557367998590985412813,2500000000000000000000000000000000000000000000000000000000000000000000000000000000', '18539021005181738577054631966716785830256683517192210055726995710.1130401928243178964748770131411421984901059634615383570491550229471994363941651252', '20131139962522780309461689017191472470532273184373694231458313824124123308217129198334611710977665987264');
    T('4583104655599717564687861567606979451521059259267795570154284167355331986003319384,66188846884496355722931850901', '69242853914610712768120180522495339911077166565137157.256079758842112249952337457360998041147', '73473311912842459042771817157');
    T('238700990196510062211759023156089849525744243141210514794850464466693722026273452038656315746884677478,30092123405681931755343047697485765975508737462761615936547801', '7932341197013669060995249486649815002914.388775664475545030214338121665357005207009166397688180912492427405071534798797537035114944', '65853245521247303460683253084956246647481048311248038291232869');
    T('42298914901,5000000', '8459.7829802', '500230246408163621933114312035010656400108045812177724426229');
    T('8074196380821158892995075889038998746407004456023946886130297182819587,100000000000000', '80741963808211588929950758890389987464070044560239468861.30297182819587', '32771612136576235765298661712341320992877938190809773521921351747798365048023003');
    T('8237639161158330611197745589676406384164235975022473710629638053,1', '8237639161158330611197745589676406384164235975022473710629638053.0', '2445104217464463695693500282135171214096744322818309314494766605989146939852441404618950981337350157904852939939373515');
    T('67543757294923929626940280513013641414086200749124590622532346827999900040177969865717432810934479,1916727', '35239111931393427247041587306389298744206243637787014333565680886219007735675435190153544459.348920842665681332000268271160278288144867678', '1995694');
    T('658743889770265303742933377935250192966496461555942991475714429268442962813046457181311534066346113237896378659793723,20000000000000000000000', '32937194488513265187146668896762509648324823077797149573785721463422148140652322859065576703317.30566189481893298968615');
    T('2805501179441768134530084886604077457872799801301882910075637983,100', '28055011794417681345300848866040774578727998013018829100756379.83', '9443876125195970145695293537833903272964339319463666915497252065195075053919409140815614968385107901341125179527914879');
    T('4517797327291151860616480864207613008757043659519224972937765633329645192613384083939856763691292386455457794604201829021,100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '45177973272911518606164808.64207613008757043659519224972937765633329645192613384083939856763691292386455457794604201829021');
    T('7397503339464272269855273915959903960816642518570157919,100000000', '73975033394642722698552739159599039608166425185.70157919');
    T('7222002295276272486455390693063960627773827847231792065164227238340366235889305566200968538315044537097,1000000000000000000000000000000000000000', '7222002295276272486455390693063960627773827847231792065164227238.340366235889305566200968538315044537097');
    T('367219394873481267603265750320061360395709498756776364710800723463,406', '904481268161283910352871306207047685703717977233439321947785033.1600984661747', '6541');
    T('1444394660367662282713282645691209012033688835789995095825382219,23057197513854003298629056495', '62643981754495200591799923072694095.830589947154613883171592329886789430042859888542274546466937863915', '68126015383576305118873089291');
    T('66486456721902783143382958103393588590534783949309048501514976388834547,100000000000000000000000000000000000000000000000000', '664864567219027831433.82958103393588590534783949309048501514976388834547');
    T('79393077119427399,1000000000000', '79393.077119427399', '457680453236362518491204537541456509655751496155359270049043835757460913773646217639420768944492');
    T('79965035093,100000', '799650.35093');
    T('4981989182126281993008748299751823310785501417823357606800265327971466995164429423070200992890363497791927,6250000000000000000000000', '797118269140205118881399727960291729725680226851737217088042452475434719226308707.69123215886245815964670832', '56677887877573113002659257334697952352644607188634909253465284091874992869314792073934863690164809324007357194');
    T('2815316100629806442280650620776290692376915362259448998756135028144140939975030651743612709055129972829840588253,312500000000000000000000000000000000000000000000000000000', '9009011522015380615298081986484130215606129159230236796.0196320900612510079200980855795606689764159130554898824096', '259630086511147416886758802731895884065033396642518234216979248');
    T('46695586464432642825171060403079900323010491862383753361945816046325281445639764147,10000000000000000000000000000000000000000000000', '4669558646443264282517106040307990032.3010491862383753361945816046325281445639764147', '42779209808392670932751246137054085319776080319163738333884406583882155006114876471576920');
    T('141870841510827737902203370743784262520488460033697822757850614639775801850436993371020540186361144664063,10000000000000000000000000000000000000000000000000000000000000000000000000000000', '14187084151082773790220337.0743784262520488460033697822757850614639775801850436993371020540186361144664063');
    T('6957664010752546415724323132519018928019295298179423339867424767111181183144993110875961376863,1250000000000000000000000000000000000000000000000000', '5566131208602037132579458506015215142415436.2385435386718939398136889449465159944887007691014904');
    T('421618093371672938615524326204048137268986623787424222206605428544644224123,125000000000000000000000000000000000000000000000000000', '3372944746973383508924.194609632385098151892990299393777652843428357153792984');
    T('79740490393758467762763825028765531324197343705389133253196874446069,1000000000000000000000', '79740490393758467762763825028765531324197343705.389133253196874446069', '3734356963544490606916998560924054030945103');
    T('35316130946513480000918112491378283158238075089400011285127542852378740666170346667104579433857777012085159078825132200346010522593453834556380919575663964381,2500000000000000000000000000000000000000000000000000000000000000000000000000', '14126452378605392000367244996551313263295230035760004514051017140951496266468138666.8418317735431108048340636315300528801384042090373815338225523678302655857524');
    T('11479496353311542550681317423708387702866260566865784453684955378292320572395800527156920358638182034688285765378247985976894354944272730666267894172313899915111,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1147949635331154255068131742370838770286626056686578445368495537829232057239.5800527156920358638182034688285765378247985976894354944272730666267894172313899915111');
    T('522894403668783,76766087553931', '6.811528636280055944192548602728291825591944197128281470842', '78240520039957');
    T('49845253705153083946560757047901116693643415419397420205639209376536358217548291172350660224299549,50000000000000000', '996905074103061678931215140958022333872868308387948404112784187530727164350965823.44701320448599098');
    T('338182838018828865464683020834510805289351205546397745297958338040061441,1000000000000000000000000000000000000', '338182838018828865464683020834510805.2893512055463977452979583380400614410', '14883484065364458852385532901646978224973759592019109846023434280302742');
    T('24649273314632626486164577096640564994348428417490664500095726069181221187821380966404579912254,649340013780256804334593736868432636453000213699945814874643231671', '37960502651195292960651209085.203374798057389753139268671210018169153661152337307012345644628714331515037827618880005930554', '908045485391832255020423148219972840921185441341969856690994256775');
    T('6891475120622508402719679996127417743765330884895739499430668315556332397523619642485183195127062533269943127231396897872180491292719,1000000000000000000000000000000000000000000', '6891475120622508402719679996127417743765330884895739499430668315556332397523619642485183195.127062533269943127231396897872180491292719');
    T('203868258448469033290239608200737201654430772057168471884887178288425302193616031248093726182688629906841814588980515821139,100000000000000000000000000000000000000000000000000000000000000000000000', '2038682584484690332902396082007372016544307720571684.71884887178288425302193616031248093726182688629906841814588980515821139', '358166259674835884479717432826072608521435320683484082553756957458496483');
    T('280498230216490169693952183821575487120350714514115741724266322855990846771014739347659202212007477,50', '5609964604329803393879043676431509742407014290282314834485326457119816935420294786953184044240149.54');
    T('205290587103533238565519893362134986807657,10000000000000000000000000000000000', '20529058.7103533238565519893362134986807657', '308243502694779752750195623211608037227150609235612049765729131496');
    T('863541229244183093415998713738595363675147376202617109427307037227836732506179435314860664537757768564421195844506348962328092938455202919,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '86354122924418309341599871373859536367514737.62026171094273070372278367325061794353148606645377577685644211958445063489623280929384552029190');
    T('1332503354412117081514650043329530553988390453,1000000000000000000000', '1332503354412117081514650.043329530553988390453', '234289217752670763502822022564923291688489066732789627701326549148438368853698800840847667515210806058742602452589237732');
    T('149231253658963411543221910342591002434468175057834645096361910659915438708229394346182595835161154220171761179324657602082487624478039438336283,20000000000000000000000000000000000000000000000', '7461562682948170577161095517129550121723408752891732254818095532995771935411469717309129791758057.71100858805896623288010412438122390197191681415');
    T('14671370523489352352179903919978798858628437740812967,100000000000000000000', '146713705234893523521799039199787.98858628437740812967');
    T('31972694535122492349621199531999,100000000000000000000', '319726945351.22492349621199531999', '444814411628813210359733486923268705561135');
    T('7671901743570237148323156463911986166241861750380367,1000000000000000000', '7671901743570237148323156463911986.166241861750380367', '4714079218556258364560566691433682958400719072779641399659759994605653409470820385182612');
    T('64914904653370646630817723806298619262853179362421159280052334161809239241810441802367897185624994804940558409456741,100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '649149046533706466.30817723806298619262853179362421159280052334161809239241810441802367897185624994804940558409456741', '52612631034775473006883291270936837923724549069552008161180077154644088916527893547656086834490317497155689233015');
    T('9622068526799441010744442228965559266999862472192613554951,100000', '96220685267994410107444422289655592669998624721926135.54951', '1440111784139735092809063761962639647793103214120990');
    T('8017871525312353806512067050596604565618790,315548414175204512371056357868162059', '25409322.83329595752563567210426781867380735708550429758298495482482043094943060', '549675823319634926616271312386674851');
    T('461120259670854299006123584618545607266420812575100938736117933259736622570867419,500000000000000000000000000000000000000000000000000000000000000000000000', '922240519.3417085980122471692370912145328416251502018774722358665194732451417348380');
    T('139792040108338252978453845748225898765036739532955399942037889179464005555009808455320117,25000000000000000000000000000000000', '5591681604333530119138153829929035950601469581318215997.68151556717856022220039233821280468');
    T('5894159987779675527812606527112902422823,10000000000000000000000000000000000000000', '0.5894159987779675527812606527112902422823');
    T('10794209997399872663981,500000000', '21588419994799.745327962');
    T('161167944446100079215361865141648945894914926470231878901891105444533027123352649883069305476504208232020922464999657127237239,250000000000000000000000000000000000', '644671777784400316861447460566595783579659705880927515607564421778132108493410599532277221.906016832928083689859998628508948956');
    T('65297284415319840892101117377284751412657466676091246251740082043,500000000000000000000000000000', '130594568830639681784202234754569502.825314933352182492503480164086');
    T('2766105154672097910919416707090692846083531091256961378583814595230638765028364556532292886699927313766545452665695161280216164748824152656732976954620185439709,500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '5532210309344195821838833414181385692167062182513922757167629190.461277530056729113064585773399854627533090905331390322560432329497648305313465953909240370879418');
    T('4965317900137877770176694971911414522545908917222619271011776655580494900114140891096653760381029022314448118382968242241409,500000000000000000000000000000000000000000000000', '9930635800275755540353389943822829045091817834445238542023553311160989800228.281782193307520762058044628896236765936484482818', '6087323064050560843067097059341387607572473063508563613294539460569998223927099513534061416866791');
    T('215007524870282058245020779477952115240341474039679043031777558877905242544399693717204716884024411270224399170135910743173,500000000000000000000000000000000000000000000000000000000000000000000000000000', '430015049740564116490041558955904230480682948.079358086063555117755810485088799387434409433768048822540448798340271821486346');
    T('33965356260862969077208625910361234343390562819287919971390918687360011980682938504819,100000000000000000000000000000000000000000000000000000000000', '339653562608629690772086259.10361234343390562819287919971390918687360011980682938504819');
    T('253321930539380453152708334768314839113869868931064250962440429142069655945019464486182962906443502943685473688729176287,50000000000000000000000000000000000000000000000000', '5066438610787609063054166695366296782277397378621285019248808582841393.11890038928972365925812887005887370947377458352574');
    T('1805858228685505996934094902160832337182222604423226849057516769855269416540373086132274874403135855391,2500000000000000000000000000000000000000000', '722343291474202398773637960864332934872889041769290739623006.70794210776661614923445290994976125434215640', '299931180540069025319209074997216426643301728798147429423616370144579300438276293226177325138489');
    T('31190292948951272069611342815544754687766064055983665546419,50000000000000000000000000000000000000000000000', '623805858979.02544139222685631089509375532128111967331092838', '56338321839749805341500571031690869993717263653260288651706865343950144409736437018194620914323');
    T('337956957987594659177352551047823442271367979011679862751244719933544405954450697376536645217973872506443,10000000000000000000000000000000000000000000000000000000000000000000000000000000', '33795695798759465917735255.1047823442271367979011679862751244719933544405954450697376536645217973872506443');
    T('482382437572212496064503056796905534583523906009551169968627497923338185298948,98881775', '4878375591176559037947114691225006171091931814629654149802908553404691.464114069554272473682452528283783605111951075878629019577756662152573343720077796', '99109659');
    T('139531491455761368238852756477562465476379871327657551278823131772010748498903090648551764744244326698930827028903816179520446707286815639062324792223595941248872160472784559239,62500000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2232503863292181891821644103640999447622077941242520820461170108352171975982449450376828235.907909227182893232462461058872327147316589050224997196675577535059981954567564552947824', '72224749819267684231620193401966551095054796101838472670695742640793352422104243890110175387436088259119888983940038219');
    T('1411416267549037105254657861275714068019729378454946755649335485716320411768092167436624477341575741093648180167816197631,2000000000000000000000000000000000000000000000000000000000000000000000000000', '705708133774518552627328930637857034009864689.2274733778246677428581602058840460837183122386707878705468240900839080988155');
    T('27827219481834304112493008583428333387209073306104984885967926767008978220794525219842633952626222316127104787702440880856041944988627062199717762582508863622603544795353020067572812970772493,50000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '556544389636686082249860171668566667744181466122099697719358535340179564415890504396852679052.52444632254209575404881761712083889977254124399435525165017727245207089590706040135145625941544986');
    T('553638437587991972935941089538503668489156735607582551300472526062673135513905787569392581,5000000000000000000000000000000000000000000', '110727687517598394587188217907700733697831347121.5165102600945052125346271027811575138785162');
    T('59464723510860089353595161485330798717117977688235446432820252084846370805317110927201609639990666929233,100000000000000000000000000000000000000000', '594647235108600893535951614853307987171179776882354464328202520.84846370805317110927201609639990666929233');
    T('7774943127923741543893327774453314221901409612145708347899080872861725693975133544163108094185354219016483147160029,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7774943127923741543.893327774453314221901409612145708347899080872861725693975133544163108094185354219016483147160029');
    T('2063769973984552389094577183907780446432714643266876167586915450871645665016033538837151166563922743418998809762073552642334105506792838094376118903308372743156366972007822263,5000000000000000000000000000000000000000000000000000000000000000000000000000000000', '412753994796910477818915436781556089286542928653375233517383090174329133003206707767430233312.7845486837997619524147105284668211013585676188752237806616745486312733944015644526');
    T('7836548872315924926855127765671663567146206518249512862581538745788328087691847420220642279030295292792647,1000000000000000000000000000000000000', '7836548872315924926855127765671663567146206518249512862581538745788328.087691847420220642279030295292792647', '96234594442279927365515769186347797337180500148188755841266605718724324934597679706160305024645');
    T('6276636774338099448991514147769685704475544055827320758088707616484824104853942959262327777197423596444308226835178887,100000000000000000000000000000000000000', '62766367743380994489915141477696857044755440558273207580887076164848241048539429.59262327777197423596444308226835178887');
    T('88804107107559930046092936712139620512040038382914013891624071753711226264013140153583717755112772233557425098002467830699052147734198261974270570978078001915494463782113496837,100000000000000000000000000000000000000000000000000000000000000000000000000000', '888041071075599300460929367121396205120400383829140138916240717537112262640131401535837177551127722.33557425098002467830699052147734198261974270570978078001915494463782113496837');
    T('115002442406703584178185294122325712122842192109474381450624873216924285090467595499234670133342099726329554064579,500000000000000000000000000000000000000000000000000000000000000000000000000000000', '230004884813407168356370588244651.4242456843842189487629012497464338485701809351909984693402666841994526591081291580');
    T('281153885629419726548824319427449841809079796787193695801572124693821870927,11125957', '25270085587192160328214851039551010471196302195594832498595143293635.0437923677037391020843520523715800957846185038636999961559631821400093846457966749', '26200508');
    T('107590545904878109399898544892791250425613052952813246657210241239393365761789609,1250000000000000000000000000000000000000000000000000000', '86072436723902487519918835.9142330003404904423622505973257681929915146926094316872');
    T('336691854215296851780405900296031485870297547905657610287048904912614725160314592634910900322215844608347716185906299548473017105494879354044791,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '336691854215296851780405900296031485870297547905.657610287048904912614725160314592634910900322215844608347716185906299548473017105494879354044791');
    T('371480560684277082668491524150108281529274363544360285467612385157153460856295638808810364012946376958259738430744167803245930749900255140112481712729944733952142996121247846362608579431806857,500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '742961121368554165336983048300216563058548727088720570935224770314306921712591277617620728025892.753916519476861488335606491861499800510280224963425459889467904285992242495692725217158863613714');
    T('90156107022576191282253438844768882560380632069595603307941806727485440731711227686222691231,1', '90156107022576191282253438844768882560380632069595603307941806727485440731711227686222691231.0', '5480298739494692404113665279307860287265772');
    T('676207065438806757306657797270701640910718687363106,1', '676207065438806757306657797270701640910718687363106.232707553596649377152910871250455082640109949938595144796401', '2');
    T('307627784692122693008294108656835628137921414014386937549059014929182112328861899672419,100000000000000000', '3076277846921226930082941086568356281379214140143869375490590149291821.12328861899672419');
    T('1019112032522319011513743334667754449108139023710934695597532972093019590416143731973703239996332828727266689906936150481584706243926489450173712530253,17054304152143118034178339067126344617804666449539944765504311916945546205137312910252', '59756881513940453622961763561761551483631213107634476215190193531.3165867394473728188428531421874167961311863141432393149058157078094433891826691976659580244336159690', '24149837024363500548775403668943393737605180826594994856002909548103707523924658663721');
    T('24320206632303877337018150776534830120127412621279410316049997868506080584897,500000000000000000000000000000000000000000000000000000000000000', '48640413264607.7546740363015530696602402548252425588206320999957370121611697940');
    T('632068538992,299627248623', '2.10951621355135030628959798316775030242', '667715230026');
    T('30723874502935040146583633833130458750279615551530417455389485335564411003264654722578476707999808118764177718404903122118992393,1000000000000000000000000000000000000000000000000000000000000', '30723874502935040146583633833130458750279615551530417455389485335564.411003264654722578476707999808118764177718404903122118992393');
    T('286178536955337099327464576728331587889607911392190634657903988794024067683540787647737271834874819490786147362267100761724779889,400000000000000000000000000000000000000000000000000000000', '715446342388342748318661441820828969724019778480476586644759971985060169.2088519691193431795871870487269653684056677519043119497225');
    T('225716760325402765480453148877447994008859292645269220156515793592565178478504694872477733969169483366031115902552261152263,100000000000000000000000000000000000000000000000000000000000000000000000', '2257167603254027654804531488774479940088592926452692.20156515793592565178478504694872477733969169483366031115902552261152263');
    T('166422076347679754457402925324150208100304102197597626198641249881291,500000000000000000', '332844152695359508914805850648300416200608204395195.252397282499762582', '595938455117353715726533211799175395565247198980519036774725159');
    T('92474208223036180278717843071595801935199978666019361143552458912542268609094249043138966113404623,100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '924742082.230361802787178430715958019351999786660193611435524589125422686090942490431389661134046230', '384893898044272576757884146625767491425042466631820743697266238210622119815334941457068522319571859800723072');
    T('4801579045833748240620308947950089769875299076650997798853,1000000', '4801579045833748240620308947950089769875299076650997.798853');
    T('92818162596390473570913489263772149216048983528220989923030438739602367016857612844868557954427,10000000000000000000000000000000000000000000000000000000', '9281816259639047357091348926377214921604.8983528220989923030438739602367016857612844868557954427');
    T('1054044512154426314878240206476736860691603532279813559806376256940851271707,20000000000000000000000000000000000000000000000000000000000000000000', '52702225.607721315743912010323836843034580176613990677990318812847042563585350', '132750239272960379396269149467880829933650132793415257290257220693012');
    T('1950260986744420612627266461252568941645104794448107,1000000000000000000000000000000000000000000000', '1950260.986744420612627266461252568941645104794448107');
    T('3420599355260725534580884522073225022198987022228846435295522221954135336191942043090733084633992155497809109068153537458117479579431129,500000000000000000000000000000000000000000000000000000000000000000000000000000000', '6841198710521451069161769044146450044397974044457692870.591044443908270672383884086181466169267984310995618218136307074916234959158862258');
    T('21801726759466625902075716247189780705759631914594809868496882689119821892069323995653346876539886660866871,250000000000000000000000000000000000000000000000000', '87206907037866503608302864988759122823038527658379239473.987530756479287568277295982613387506159546643467484');
    T('8662514288504709301163055702408996483964651352163407517985146778131926637288811147922076068037312182374897258627079219878971,10000000000000000000000000000000000000000000000000000000000000000000000000000000000', '866251428850470930116305570240899648396465.1352163407517985146778131926637288811147922076068037312182374897258627079219878971');
    T('2335156819025320456542491284916593277514846432649030650406781161541460408081740604535185151601367656917158704090094377170340940777294656240040982897,2500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '934062727610128182616996513966637311005938573059612.2601627124646165841632326962418140740606405470627668634816360377508681363763109178624960163931588');
    T('63654450465180016591742773920768518412294924364672171032800586170759569,100000000', '636544504651800165917427739207685184122949243646721710328005861.70759569');
    T('67017226545297291596788980559120386755737203907750761130545403,1250000000000000000000000000000000000000000000000000000', '53613781.2362378332774311844472963094045897631262006089044363224');
    T('1834202980794378523142996418618158434266769632338075275566234011547291333605095187322588266592801041375,2728622727250961344924048054555626447313315476626465983644676145928438417252418810441', '672208349830137661.0942704654707426447364588742190615418045766720625819944644489835174706718477671949019962898837174', '3060324951336066473403966327368318133634181672878042528628461545369720500607838478698');
    T('724151319188526857516015364911054452983796479888236151875899188098296091563373,2', '362075659594263428758007682455527226491898239944118075937949594049148045781686.5', '3728535308064241709642992906437162149848');
    T('913161203334294301947,1000000000', '913161203334.2943019470');
    T('1000730618593668355073183788658419468874098478193537979324951367,1000000000000', '1000730618593668355073183788658419468874098478193537.979324951367');
    T('98578065477380636870100743410132239138150603712881520,1', '98578065477380636870100743410132239138150603712881520.0');
    T('1593653227528651554398919030088527404972981723686120202542946612603975281246782283952373738247671009241097893,244376382898878709186556743314810099835899136846', '6521306226993688084262488516142465500275896529315090407774915.4390277153651041272996824372692207805951482614581375281886491067796', '320896389969383038785219513263634670827380910560');
    T('1098576836684484704997452093434476835830575095945740572130146590994553924253484329785345800790170709555710672234900297863234859386093535546823682473,125000000000000000000000000000000000000000000000000000000000', '8788614693475877639979616747475814686644600767565924577041172727956431394027874638282766.406321365676445685377879202382905878875088748284374589459784', '580755848202156326016924884077059025897828282924714692042700832');
    T('98856015342574146019060178479759793759304710950289379284070197,10000000000000000000000000000000000000000', '9885601534257414601906.0178479759793759304710950289379284070197');
    T('580368338794671752821269385270342578980827601542629307885673919782902096378604694343656768007757459380781031973411772409250979813132067528072149194522614456969,1250000000000000000000000000000000000000000000000000000000000000000000000000', '464294671035737402257015508216274063184662081234103446308539135826321677102883755474.9254144062059675046248255787294179274007838505056540224577193556180915655752');
    T('69982084220851887614167671807406656266816171302967221742593895150687,100', '699820842208518876141676718074066562668161713029672217425938951506.87', '54403392513559426721299504443157825252617408149349867483376440447');
    T('2018853736861528670763599026530020008195846168939825002390514885917432739031000764700018894935539479830625328771187655839828813167,29884853861934945378863310572572196981269', '67554412217922573204803437390939478181208940016133763859817864059085840485469282963181019.19791615300926846665465703968492270928565365268908436662715', '33392687306924808799500742211568775440536');
    T('530027912908014643675270751164750719833056592215023870512832293271045526003960718599676622511253038980627258993911327348577365854850830043924340793,1000000000000000000000000000000000000000000000000000000000000000000000000000', '530027912908014643675270751164750719833056592215023870512832293271045526.003960718599676622511253038980627258993911327348577365854850830043924340793');
    T('8734735018342228299296090238863640379760418043064479934394562234031967281491808567345771710811322186275692378629728220679938160003692820341,100000000000000000000000000000000000000000000000000000000000000000', '87347350183422282992960902388636403797604180430644799343945622340319672814.91808567345771710811322186275692378629728220679938160003692820341');
    T('6722788254690658393400979717508508269239637599135817059,200000000000', '33613941273453291967004898587542541346198187.9956790852950');
    T('7437180918830993537032395884958073653904626738419559962059983,1000000000000000000000000000000', '7437180918830993537032395884958.073653904626738419559962059983');
    T('678608120908470140173991795540669892372847725269378282378365833145325640439329229307313633722843810507520881589650753115173369715723114474748456801385922995060079,1000000000000000000000000000000000000000000000000000000000000000000', '678608120908470140173991795540669892372847725269378282378365833145325640439329229307313633722843.810507520881589650753115173369715723114474748456801385922995060079');
    T('86894722198923597840560746168874947033616752839619628508272485307838443277363083879320271424429852793,1000', '86894722198923597840560746168874947033616752839619628508272485307838443277363083879320271424429852.793', '3631197556693210653073184532428485425255348157219832901774');
    T('17970721880705870085054361726421717606722108915965471930329036480272239344308991597936504664878045388124735330004009910921,20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '898536094035293504252718086321085.88033610544579827359651645182401361196721544957989682523324390226940623676650020049554605');
    T('5948697443135706703788917312367,100000', '59486974431357067037889173.12367');
    T('3946733843989140918050146097267216988669431771,577886291193042952', '6829602820030790847543876934.02258790700574374198489118872645183303458692610035440156215891630548679719783', '667417859486950026');
    T('493858380487330334553899939554971082426451967753112354596468356557,5000000000000000000000000000000000000', '98771676097466066910779987910.9942164852903935506224709192936713114');
    T('123084490293475602607614884133531998828862861585787351245614755708686612116721277769653301046242789968234508397289909158055602802061,20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6154224514673780130380744206676599941443143.07928936756228073778543433060583606388848266505231213949841172541986449545790278014010305');
    T('6927482597114680607834922765933648950334814075397606193627167620391317958568821118589203,100000000000000', '69274825971146806078349227659336489503348140753976061936271676203913179585.68821118589203', '41718700671033927886513136610866596633645608053066723832100884');
    T('95538944758612146141176396275398005954352758782334733053038291386324885989961287864210,11', '8685358614419286012834217843218000541304796252939521186639844671484080544541935260382.727294449076880196380605689453495898969668417125590960428365196683178504081224393284', '570');
    T('32993752053684425807747294778148085597692415739709994,638184719', '51699376483635964037784011525584782291994076.6236209331737383706493508641321733305200624022422', '2141040464');
    T('223942440246,248225668363', '0.9021727757764007', '609899431382');
    T('16686579847373982176404589194403513916267058244556986258849490594859727370843269811410258544438506162845429784416595935602473679945761077,20000000000000000000000000000000000000000000000000000000000000000000', '834328992368699108820229459720175695813352912227849312942474529742986.36854216349057051292722192530814227148922082979678012368399728805385');
    T('7453839315378838224947273562889200300255201795789768843276848020086237609421,100000000000000000000000000000000000000000000000000000000000000000000000', '74538.39315378838224947273562889200300255201795789768843276848020086237609421');
    T('88186134153847200319907838505055317969004545179501640348910365893233471389254602659518381116512509844213870327,9765625000000000000000000000000000000', '9030260137353953312758562662917664560026065426380967971728421467467107470.2596713123346822263308810080475003214848', '833555353746649953279923803074475202769062814458423077827564033682942755');
    T('291651801565673891735,441799', '660145906997693.27620705343909321636692', '475636');
    T('3943935476569904891723785189074953927414878941370369202289012846241086155860318787858564276558320735525331470755293103519133817,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3943935476569904891723785189074953927414878.941370369202289012846241086155860318787858564276558320735525331470755293103519133817');
    T('75302834320878879457221652547914488454737262753632963871692491571975871172248660219123955075959110697547,100000000000000000000000000000000000', '753028343208788794572216525479144884547372627536329638716924915719758.71172248660219123955075959110697547');
    T('286254617761534858369612111554416,44902328951267285921', '6375050569697.807323576944467346526062905538962967762891749230526034972918', '44929005448766457238');
    T('3746139574442736600133406409787136253033944992476571372451349971463733707013337603307113483922358784475865353830238500792008470975355809924470290472421506149,5000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '749227914888547320026681281957427250606788998495314274490269994292.7467414026675206614226967844717568951730707660477001584016941950711619848940580944843012298');
    T('37406858419305938135286171702693320249855935205139647062682249364933852246421767685483,500000000000000000000000000000000000000000000000000000000000000000000000000000000000', '74.813716838611876270572343405386640499711870410279294125364498729867704492843535370966');
    T('11159601476231535654741788412416202881635877421391863561941550761560649976787093908100772767774222070807449142387341,100000000000000000000000000000000000000', '111596014762315356547417884124162028816358774213918635619415507615606499767870.9390810077276777422207080744914238734100');
    T('5202314789815646277150835429404879071603050113822553743596059100657262453703715303,40000', '130057869745391156928770885735121976790076252845563843589901477516431561342592.882575', '66027873');
    T('304974437931081602650032524760965871551608292495930150945863595349787675349596911689176201582031187229344132110281,1357797756240047452202752179724701484120084840431181047509757564399930443993347892330', '224609619900686178223912328077.7443121694198187253469840096889635680497517685978187200433620475779622573500912322765762965524609863', '4531192428741997048411481671100163403784805772178484102146612877029644423079894127823');
    T('3766014359237,800000000', '4707.51794904625', '7853235314974998');
    T('311475636684241059876365851605648547902228954687,5000000000000000000000000000000000000000', '62295127.3368482119752731703211297095804457909374');
    T('95942011683584513899200680778652795117868885339,1000000000000000000000000000000000000000000000', '95.942011683584513899200680778652795117868885339');
    T('12766155608271430221647976553149893759584816809408489139986361341082789186923046040729403013493459196737462331526072870887109,6250000000000000000000000000000000000000000', '2042584897323428835463676248503983001533570689505358262397817814573246269907687366.51670448215895347147799397304417165934193744', '93126923446050929294103459255648989880852923558235441052339284471205926790072168747045165');
    T('5050978417270036929890556014363105586807333767534651101340315552268579329086313825980085677972688558518469677967261525650390108135458820811,250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '20203913669080147719562224057452422347229335070138.604405361262209074317316345255303920342711890754234073878711869046102601560432541835283244');
    T('514785480892443581111011049635898989522189108839752679508896007986075563249680148629967905105747624723273024943,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '514785480892.443581111011049635898989522189108839752679508896007986075563249680148629967905105747624723273024943');
    T('267737767017891684255898230028455429260991473930007229195590053591931280617076656188214376062161831184386528376881147906819687668809677563924650021,50000000000000000000000000000000000000000000000000000000', '5354755340357833685117964600569108585219829478600144583911801071838625612341533123764287521.243236623687730567537622958136393753376193551278493000420');
    T('248259504263040834608233334198108145412348875247144730132530909287139514727494049122871346,47732130929763380841', '5201098283844657888397210939987325835547277037459741568044731851516514.73634681073130117711020811237650578307976297729277805241093095474959673783840483910867', '5024411020558340543312');
    T('55464558317631039321559765849925798648390666160464586486280863822713088834539695339682241860828114727160148856646089617,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '55464558317631039321559.7658499257986483906661604645864862808638227130888345396953396822418608281147271601488566460896170');
    T('15421483144336270077371210750899333898679645561504523633209496264965010394930803328016197705616989442784785,49984940267107918826', '308522588242127401197353099520103594922877070048176300229719294528874975525549904675985.9485359329604023169832489554441000991', '61355845435674844526');
    T('6855440895600044261671405027414507429789800105303059815241230707817129262097559133424765895800883527246883701308438021,100000000000000000000000', '68554408956000442616714050274145074297898001053030598152412307078171292620975591334247658958008.83527246883701308438021');
    T('27263704632607142270401184496759949558690923110949180431533606763,20000', '1363185231630357113520059224837997477934546155547459021576680.33815', '606624887961647054532897310953779853618934383968313019879350462126175949928993363987651814');
    T('90954990859482993959724846045748010087463959,125', '727639926875863951677798768365984080699711.672', '588824757782012773261640878838777321142728950393257882912418442907031819512234');
    T('1493569539393627900541858134137603447531077013858830325785084098585732603941651348853545566104365474803,1000000', '1493569539393627900541858134137603447531077013858830325785084098585732603941651348853545566104365.474803', '3182820534541090775056684943977433195872567458734700994010822943243520679220246426898');
    T('324725691215801921476094759101909541875602304206478416528531250398201107463367988182933405249982011662330601,50000000000000000000000000', '6494513824316038429521895182038190837512046084129568330570625007964022149267359763.65866810499964023324661202');
    T('25308321535355028217216957904491,5569187114692733093840', '4544347498.863190164819697293209529531404911058684', '7723635601139758565843');
    T('8658777803281975609968463280524071823375642811421607061029953506904956846061159429058371,100000000000000000000000000000000000000000000', '86587778032819756099684632805240718233756428.11421607061029953506904956846061159429058371');
    T('4215726801070323444647630281510808779143459576242997643967294379158258415984350576313270248295291231861927178054,8801198818356547303082759', '478994610629365197012477473717361789088233414206341594179220175540708293059656499309131.1197482000861233142996023980139256469694810', '9219447145070526604874006');
    T('350219027018265521256189312491936248573915564006546709163466462009603052479301023519364445626038090785927171839374456992889152646903427846290756941,50000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7004380540365310425123786249838724971478311280130934183269329240.19206104958602047038728891252076181571854343678748913985778305293806855692581513882');
    T('18838265670959180747789613494901603955905673528214021208240108707414144737352242491390744711159628979832028357517923496753376897134,2584402836812084642754771195541804154926099055589080760558497', '7289214128164546673077585428995457658876312767922421632262422308769050.379657338235342151193256041128846262031912288379823584158769838474025799337151989092013277121571615', '5322144155931111466743034122299554141788673340121784975895214');
    T('110540403222773032054929356513175003594418655601576752219028431256243715401738728452564148356781548250303,50000000000000000000000000000000000000000000000000', '2210808064455460641098587130263500071888373112031535044.3805686251248743080347745690512829671356309650060600');
    T('34521678049086112732982603617637423488030098370864976367418263782422677974076092812569932135807092991,10000000000000000000000000000', '3452167804908611273298260361763742348803009837086497636741826378242267797.4076092812569932135807092991', '9565990599377329484423005085641091352736216008630315082396889620817076025190521731088325462643453265267584');
    T('686981974413884751293690490784988550329391377445018867980503873317,50000000000000000000000000000000000000000000000000000000000000000', '13.73963948827769502587380981569977100658782754890037735961007746634');
    T('39967009834238797427182253133070010605320897536027467147,125000000', '319736078673910379417458025064560084842567180288.219737176', '3546986130406819267454979773897918044730322031561726889492515191671301');
    T('2054848767576587955462470796182561489545688219952205806313236406797613009606793513761184547627264318510083497,500000000000000000000000000000000000000000000000000000000000000000000000000000000', '4109697535153175910924941592.3651229790913764399044116126264728135952260192135870275223690952545286370201669940', '19171208630106026353275589645613967733224343436112425066800794894453677875314261981745825449319');
    T('1258311307164311617381367950628372981420522447544944439693149609041298843621306748572743262432414156764952221586409072251176086979,250000000000000000000000000000000000', '5033245228657246469525471802513491925682089790179777758772598436165195374485226994290973049729.656627059808886345636289004704347916');
    T('492541774159879640437936287616718070610370129799840600955174836508798297733803571061834516664670316321397051566302739470908697122858665682327,500000000000000000000000000000000000000000000000000000000', '985083548319759280875872575233436141220740259599681201910349673017596595467607142123.669033329340632642794103132605478941817394245717331364654', '772690036339621488428269775437822570422870894506512284167339990445682547198933662923265581');
    T('413875835951731659277384103388688394090011015604931352043,6250000', '66220133752277065484381456542190143054401762496789.01632688', '267799856219808852183');
    T('25623638455508129716975373710250219093690370694168048889682125687,728796356236789336532397', '35158845452820691695331252389119574225619.5643086760770102207250792274157899888147246542964984154489942046175778073820104', '4682678030669483013541263');
    T('12367863144967384872529417463063119791986560777209539757,2500000000000000000000000', '4947145257986953949011766985225.2479167946243108838159028', '19780730703790873838347852973574677540986567656865706078825739326608066836976512951818878167033944417505061150717024');
    T('9971596879982164667,1000000000', '9971596879.982164667');
    T('338345846975325139996317671732776306269185647762097932941355508406167695219190203003609,40587706508395974705401', '8336165703409107474709453480980066107483661912962839864091523498.0126686226890405486513889705767265041180045', '41022537112870625129272');
    T('127928935759871381312732087750566099177874361325298055925972104481990091044917887846249290063171763251,50000000000', '2558578715197427626254641755011321983557487226505961118519442089639801820898357756924985801.26343526502', '621875728720764263561029044312241358762794797665106621297993145788249024700');
    T('16626060161524464574713076045625021016208464957298152388988528750805470496009977748891419,25000000000000000000000', '665042406460978582988523041825000840648338598291926095559541150032.21881984039910995565676');
    T('139835943154689530164226571,16000000000', '8739746447168095.6352641606875');
    T('655492388278806204545768064730065599657599852803336783,100000000000000000', '6554923882788062045457680647300655996.57599852803336783', '976108250785731949679925001521155265790958652927658681941');
    T('810011346800097039139210061890622091915594232858534709414710566706428201195198961960548291930674225613464775148923720691217878393179,10000000000000000000000000000000000000000000', '81001134680009703913921006189062209191559423285853470941471056670642820119519896196054829.19306742256134647751489237206912178783931790', '659255936975989889409223375243561380964658820807429254878323937252539411092735435865211779');
    T('767010060358818890783562424457154229236761908349178661946491180919697196736640696221322466225137922274445553575680781328318284000710534431,125000000000000000000000000000000000000000000000000000000000000000000', '6136080482870551126268499395657233833894095266793429295571929447357577.573893125569770579729801103378195564428605446250626546272005684275448');
    T('6251274083692311423733916625297741079068776687255993282033945555286360790665510481,1000000000000000000000000000000000000000000000000000000', '6251274083692311423733916625.297741079068776687255993282033945555286360790665510481');
    T('473620934850848082187223138350602125191020210530458123567437931094543854409630864168927727,10000000000000000000000000000000000000000000000000000000000000000', '47362093485084808218722313.8350602125191020210530458123567437931094543854409630864168927727');
    T('28392638992177631538735383923909742173497869442249044705049559217795836913231369776653059449522857321829651,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '28392638992177631538735.383923909742173497869442249044705049559217795836913231369776653059449522857321829651', '5028820610639781639325654949257292223081041411494147435152768836878813294805858397173957511962736507674020');
    T('9552496340020091725697433,10000000', '955249634002009172.5697433', '681734365310500930436632536264485543390585757982517774887009602160278649869690646');
    T('191407226365712801218816880714794123311237815231478951057699351929786915001765279771136714191232777459076067241766801466017951665819488454219712701784568617525439,250000000000000000000000000000000000000000000000000000000000000', '765628905462851204875267522859176493244951260925915804230797407719147660007061119084546856764931109.836304268967067205864071806663277953816878850807138274470101756', '796843914588374585605530216500278077281516834694785006819875719121');
    T('973708114707156900848575306441883115282262041526510308450033867296915537409831603407246049399369228564061,25000000000000000000000000000000000000000000000000000000000000000000000000000000000', '38948324588286276033943.01225767532461129048166106041233800135469187662149639326413628984197597476914256244');
    T('41176818016861564738549999724345149794954792976459967902179568613277,500000000000', '82353636033723129477099999448690299589909585952919935804.359137226554', '669011453192737766662797002206221194892295529858894718227759009576786576439366369655530541');
    T('2297280934071181232289100315902073111835117061805828135181986873561508623403722194338276074897169459,250000000000000000000000000000000000000000000000000000000000000000000', '9189123736284724929156401263608.292447340468247223312540727947494246034493614888777353104299588677836');
    T('493865232957959844369851283310738555249442370187169054736707861015181,1000000000000000000000000000000000000000', '493865232957959844369851283310.738555249442370187169054736707861015181', '670693117968294371193270720647906750197834156929685579183197442866318330576022797208');
    T('409271320054752626250953921458862574122032388782159837960166261432376032498288772914987357485363480329191,20000000000000000000000000000000000000000000', '20463566002737631312547696072943128706101619439107991898008313.071618801624914438645749367874268174016459550', '1857697448678073248864111007805323144660940247856110752402597224275056054503062819540715261827115353176657605923020695');
    T('1091233021200519149409087486543248004149493427849573405514209173739773291315508685002542487733353392960642990271133459767,4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '272808255300129787352271871635812.00103737335696239335137855229343494332282887717125063562193333834824016074756778336494175');
    T('1595690121196871153886729,200000000000000', '7978450605.984355769433645', '6043921481194682383933679927');
    T('35198307665288379989739763374405377831198720275886562170938556931298244697897264380623354673785632771,10000000000000000000000000000000000000000000000000000000000000000000000000', '3519830766528837998973976337.4405377831198720275886562170938556931298244697897264380623354673785632771');
    T('23696485864014344411749757864467460969465379680156310297289316481649993164859,12500000000000000000000000000000000000000000000000000000000000', '1895718869121147.55293998062915739687755723037441250482378314531853199945318872', '88169772318726534478097482287174902361436232507508528249890047249859106140720080138823592262180105');
    T('2470252623431054313731331443175884514511972829541093697927860344370604828517286139353493663429245331486666,5108090079928118985489686671', '483596135694188011378667156401843144027164732844551280707660248632923225254800.587199079677953937875639324206276895271235823612547506378684214987948118274320359990368', '5382458010937250568880673669');
    T('114221349590728832380263984994984635861738747311983631781352559687519165032758871,100000000000000000000000000000000000000000000000', '1142213495907288323802639849949846.358617387473119836317813525596875191650327588710');
    T('3284945592812373184344753972758476891953304894259,4000000000000000000000000000000000000000', '821236398.20309329608618849318961922298832622356475');
    T('2256552538268324957101459584274765124413730131360493850955441001778068390578610403086691841111551405816799647689962152222209,500000000000000000000000000000000000000', '4513105076536649914202919168549530248827460262720987701910882003556136781157220806173.383682223102811633599295379924304444418');
    T('45102018400266989728931638141595361761071295617811579289997697670832298744506116516722877037008568752523755414709825779737363977291512033198315936360667277,10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4510201840026698972893163814159536176107129561781157928.9997697670832298744506116516722877037008568752523755414709825779737363977291512033198315936360667277');
    T('2991118205134544635217224882676124745244670098171611353811722020713540132602932667617729650082673649312361632555982015642938286243972586805717247022507624073645853595324932716067929,400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7477795512836361588043062206690311863111675245429028384529305051783850331507331669.0443241252066841232809040813899550391073457156099314670142931175562690601841146339883123317901698225');
    T('43406214460275172698038479789612757524898533646080070899355546034286693170983145909731585902643,18687745885171468234775664818563711', '2322710011522446475757338874013239466938178964161018202462196.53231516923374798133674231744884434422245047240330215423739416673515083141853518', '24830247378717343908862228854084190');
    T('4682650609411899735589035611734839738748275802992269558418422379673582726796923610465719356507461769677666289014200384110046607144919937994870547743121,5000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '936530121882379947117807122346967947749655160598453911.6836844759347165453593847220931438713014923539355332578028400768220093214289839875989741095486242');
    T('6343504805835314981042161153519341413408814287852266648003279622224288264182025200490500813492402339458961013,1000000000000000000', '6343504805835314981042161153519341413408814287852266648003279622224288264182025200490500813.492402339458961013');
    T('2043755907556873690066829609994954302797373554833237131297,5000000000000000', '408751181511374738013365921998990860559474.7109666474262594', '1956664943557976499821087345250316684037359526871385211708133002423524');
    T('313284563452294441735736652942879418320584421214021592208381792517413756116801965056033499858686861805057047003471,2000000000000000000000000000000000000000000000000000000000000', '156642281726147220867868326471439709160292210607010796.1041908962587068780584009825280167499293434309025285235017355');
    T('11471639769771687066577323756368359399412710207010171779949298240855743839316944634591329633416155194817943709829531,200000000000000000000000000', '57358198848858435332886618781841796997063551035050858899746491204278719196584723172956648.167080775974089718549147655');
    T('23776195197101022472023159066867591626860322971795239505778686401286942957171879146898109574605597407608118187436758896473157262424044809239822026232771581,250000000000000000000000000000000000000000000000000000000000000000000', '95104780788404089888092636267470366507441291887180958023114745605147771828687516587592.438298422389630432472749747035585892629049696179236959288104931086324', '170109268958634877909065036709787962672848011126048129765200921441020133297498972167180624');
    T('35518282899106761300653224275853720735244906250293433645349674469329930654366141,100000000000000000000000', '355182828991067613006532242758537207352449062502934336453.49674469329930654366141');
    T('853774591279175110534841081508464694413901769,25000000000000000', '34150983651167004421393643260.33858777655607076');
    T('434880828240088350029098564204588579863154654848110708450994020630614580434999236881672317324560458832813078468886616827281212017480013,25000000000000000000000000000000000000', '17395233129603534001163942568183543194526186193924428338039760825224583217399969475266892692982418.35331252313875546467309124848069920052', '739293900270272912650169319408236065394650909425013543189206691255522429');
    T('11526403854413902548795861447,6250000000000', '1844224616706224.40780733783152', '307840150789672937492642930347274685995354235175108309976189389250759848');
    T('702765355056540457405882229774275623914171478700737088553544569128446928397,10000000000000000000000000000000000000000000000000000', '70276535505654045740588.2229774275623914171478700737088553544569128446928397', '2994201467352227469980411736484219035208598424332283240168638173739557964288699659349793900394590666632');
    T('54825256449234689277507377,1000000000000000', '54825256449.2346892775073770', '7247599320592380995092818422');
    T('52241358526638931241686750736713811006187403081280396420418024952453139299798099747,1000000000000000000000000000000000000000000000000000000000000000000000000', '52241358526.638931241686750736713811006187403081280396420418024952453139299798099747');
    T('56621644568088128185885031935431,100000000000000000000000000000', '566.21644568088128185885031935431');
    T('39372223523562980543199487796419847872440851625907503200333,50', '787444470471259610863989755928396957448817032518150064006.66', '955541553332758813213626519789460543325290905244676838372840424628529596179789208585303701552552351175041044662668537427');
    T('325293311478566577489170522256284017692295100018580428088669389347157000201893827964335714244323280261021551781493573554200870379941749285031284525050456990337767,500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '650586622957133154978341044512568035384590200037160856177338778.694314000403787655928671428488646560522043103562987147108401740759883498570062569050100913980675534');
    T('614864078480713483874128578932501548981704516588520712161898329483464331757759954227806822474378198285515275589259000330713642481603365393389422940141019017401109,1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '614864078480713483874128578932501548981704516588520712161898329483464.331757759954227806822474378198285515275589259000330713642481603365393389422940141019017401109');
    T('3829270893665970578184851197459824275528243394573338346473491354929939788896655873285990520983852622278895790953125487430033,50000000000000000000000000000', '76585417873319411563697023949196485510564867891466766929469827098598795777933117465719810419677.05244557791581906250974860066');
    T('780053005909401364919286689621630092740741364968611783815095760045292,1', '780053005909401364919286689621630092740741364968611783815095760045292.0', '45690752976547956116643254519509539579394298044534876843625082629489');
    T('11115161271892961338110886311733472538474225695783394064891231044512131866645638610465780143254877130975043621543307573319,117372832939055230293151', '94699608023130933036348120811996198905372415393501407434579046671460258797730199936885542957363696.08748184584119788329165464491688408482957311641741', '959631860555403859564786');
    T('269442874752488639176619580396174881364833,10000000000000000000000000000000', '26944287475.2488639176619580396174881364833', '8371945805788249772750247654848998754940863393299617894213772127884912858370833994661593347');
    T('64668573815841734344417489298502668067859,1000000000000000000000000000000', '64668573815.841734344417489298502668067859');
    T('3761571027751684615243460444230696668509504743538045995257317822557431413971266200889927052756417,10000000000000000000000000000000000000000000000000000000', '376157102775168461524346044423069666850950.4743538045995257317822557431413971266200889927052756417', '902612837213836957407294100697258033215360383465558379387665');
    */

    T('2469,20', '12.345e1', null);
    T('2469,20', '12.345e1', u);
    T('2469,20', '12.345e1', '2.1e1');
    T('1111,9',  '12.345e1', new BigNumber(10));
    T('2469,20', '12.345e1', new BigNumber('123e399'));

    assertException(function () {new BigNumber('12.3e1').toFraction('')}, ".toFraction('')");
    assertException(function () {new BigNumber('12.3e1').toFraction(' ')}, ".toFraction(' ')");
    assertException(function () {new BigNumber('12.3e1').toFraction('\t')}, ".toFraction('\t')");
    assertException(function () {new BigNumber('12.3e1').toFraction(NaN)}, ".toFraction(NaN)");
    assertException(function () {new BigNumber('12.3e1').toFraction('NaN')}, ".toFraction('NaN')");
    assertException(function () {new BigNumber('12.3e1').toFraction('hello')}, ".toFraction('hello')");
    assertException(function () {new BigNumber('12.3e1').toFraction([])}, ".toFraction([])");
    assertException(function () {new BigNumber('12.3e1').toFraction({})}, ".toFraction({})");
    assertException(function () {new BigNumber('12.3e1').toFraction(true)}, ".toFraction(true)");
    assertException(function () {new BigNumber('12.3e1').toFraction(false)}, ".toFraction(false)");
    assertException(function () {new BigNumber('12.3e1').toFraction(function (){})}, ".toFraction(function (){})");
    assertException(function () {new BigNumber('12.3e1').toFraction(new Number)}, ".toFraction(new Number)");
    assertException(function () {new BigNumber('12.3e1').toFraction(new String)}, ".toFraction(new String)");
    assertException(function () {new BigNumber('12.3e1').toFraction(new Date)},".toFraction(new Date)");
    assertException(function () {new BigNumber('12.3e1').toFraction(new RegExp)}, ".toFraction(new RegExp)");

    assertException(function () {new BigNumber('12.3e1').toFraction(7.5)}, ".toFraction(7.5)");
    assertException(function () {new BigNumber('12.3e1').toFraction(new BigNumber('225.435435'))}, ".toFraction(new BigNumber('225.435435'))");
    assertException(function () {new BigNumber('12.3e1').toFraction(0)}, ".toFraction(0)");
    assertException(function () {new BigNumber('12.3e1').toFraction('0.99')}, ".toFraction('0.99')");
    assertException(function () {new BigNumber('12.3e1').toFraction(-1)}, ".toFraction(-1)");
    assertException(function () {new BigNumber('12.3e1').toFraction(-23)}, ".toFraction(-23)");
    assertException(function () {new BigNumber('12.3e1').toFraction(Infinity)}, ".toFraction(Infinity)");
    assertException(function () {new BigNumber('12.3e1').toFraction('-Infinity')}, ".toFraction('-Infinity')");

    BigNumber.config({ERRORS : false});

    T('2469,20', '12.345e1', null);
    T('2469,20', '12.345e1', u);
    T('2469,20', '12.345e1', undefined);
    T('2469,20', '12.345e1', '');
    T('2469,20', '12.345e1', ' ');
    T('2469,20', '12.345e1', '\t');
    T('2469,20', '12.345e1', NaN);
    T('2469,20', '12.345e1', 'NaN');
    T('2469,20', '12.345e1', []);
    T('2469,20', '12.345e1', {});
    T('2469,20', '12.345e1', new Date);
    T('2469,20', '12.345e1', new RegExp);
    T('864,7',   '12.345e1', '7.5');
    T('2469,20', '12.345e1', 0);
    T('2469,20', '12.345e1', -23);
    T('2469,20', '12.345e1', '2.1e1');
    T('2469,20', '12.345e1', Infinity);
    T('2469,20', '12.345e1', '-Infinity');
    T('1111,9',  '12.345e1', new BigNumber(10));
    T('2469,20', '12.345e1', new BigNumber('123e399'));
    T('2469,20', '12.345e1', new BigNumber(Infinity));

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;